/****************************************************************************
**
** A Qt PostScript widget.
**
** Copyright (C) 1997 by Mark Donohoe.
** Based on original work by Tim Theisen.
**
** This code is freely distributable under the GNU Public License.
**
*****************************************************************************/

#include <config.h>

#include <qdrawutil.h>
#include <kapp.h>
#include <kdebug.h>

#include "interpreterdialog.h"
#include "kpswidget.h"
#include "kpswidget.moc"

#define BUFFER_SIZE (8192)

int handler(Display *d, XErrorEvent *e)
{
   char msg[80], req[80], number[80];

   XGetErrorText(d, e->error_code, msg, sizeof(msg));
   sprintf(number, "%d", e->request_code);
   XGetErrorDatabaseText(d, "XRequest", number, "<unknown>", req, sizeof(req));

   //fprintf(stderr, "kghostview: %s(0x%lx): %s\n", req, e->resourceid, msg);

   return 0;
}

KPSWidget::KPSWidget( QWidget *parent ) : QWidget( parent )
{
   proc = 0;

   XSetErrorHandler(handler);
  
   fullView = new KFVWidget ( this );
   gs_window = fullView->winId();
   gs_display = fullView->x11Display();

   topFrame = new QFrame ( this );
   topFrame->setFrameStyle( QFrame::HLine | QFrame::Raised );
   topFrame->setLineWidth(1);
  
   leftFrame = new QFrame ( this );
   leftFrame->setFrameStyle( QFrame::VLine | QFrame::Raised );
   leftFrame->setLineWidth(1);
  
   rightFrame = new QFrame ( this );
   rightFrame->setFrameStyle( QFrame::VLine | QFrame::Sunken );
   rightFrame->setLineWidth(1);
  
   bottomFrame = new QFrame ( this );
   bottomFrame->setFrameStyle( QFrame::HLine | QFrame::Sunken );
   bottomFrame->setLineWidth(1);
  
   horScrollBar = new QScrollBar( QScrollBar::Horizontal, this );
   connect( horScrollBar, SIGNAL( valueChanged(int) ),
                          SLOT( slotHorScroll(int) ) );
  
   vertScrollBar = new QScrollBar( QScrollBar::Vertical, this );
   connect( vertScrollBar, SIGNAL(valueChanged(int)),
                           SLOT( slotVertScroll(int) ) );
		
   patch = new QWidget( this );
  
   //
   // Hide all widgets except for the fullView at startup
   //
   topFrame->hide();
   leftFrame->hide();
   rightFrame->hide();
   bottomFrame->hide();
   horScrollBar->hide();
   vertScrollBar->hide();
   patch->hide();
  
   //
   //	MESSAGES DIALOG
   //
   messages = new MessagesDialog( 0, "messages" );
  
   //
   //	INTERPRETER CONFIG DIALOG
   //
   intConfig = new InterpreterDialog( topLevelWidget(), "intConfig" );
  
   // Initialise class variables
   // I usually forget a few important ones resulting in nasty seg. faults.
  
   setDocumentPresent( false );
  
   background_pixmap = None;
   llx = 0;
   lly = 0;
   urx = 0;
   ury = 0;
   left_margin = 0;
   right_margin = 0;
   bottom_margin = 0;
   top_margin = 0;
   foreground = 0;
   background_pixel = 1;
   scroll_x_offset=0;
   scroll_y_offset=0;
   antialias=true;
   show_messages=false;
   xdpi=75.0;
   ydpi=75.0;
   disable_start = false;
   stdin_ready = false;
   interpreter_ready = false;

   input_buffer = 0;
   bytes_left = 0;
   ps_input = 0;
   orientation = 1;
   changed = false;
   busy = false;
   fullView->setCursor( arrowCursor );

   layout();
}

KPSWidget::~KPSWidget()
{
   delete proc; proc = 0;
   stopInterpreter();
   if ( background_pixmap != None )
      XFreePixmap( gs_display, background_pixmap );
   delete intConfig;
   delete messages;
}

// ************************************************************
//
//	PUBLIC METHODS
//
// ************************************************************

void KPSWidget::setDocumentPresent( bool state )
{
   mDocumentPresent = state;
   if( mDocumentPresent == false )
   {
      fullView->setBackgroundMode( PaletteMid );
   }
   //
   // Espen 2000-04-19
   // I tried earlier to set the setBackgroundMode to FixedPixmap
   // when there is a document present, but 1) it is not allowed
   // according to the Qt docs and 2) it is not necessary.
   //
}


void KPSWidget::disableInterpreter()
{
   disable_start = true;
   stopInterpreter();
}


void KPSWidget::enableInterpreter()
{
   disable_start = false;
   startInterpreter();
}


bool KPSWidget::isInterpreterReady()
{
   return isInterpreterRunning() && interpreter_ready;
}


bool KPSWidget::isInterpreterRunning()
{
   return proc && proc->isRunning();
}

bool KPSWidget::nextPage()
{
   XEvent ev;

   if (!isInterpreterRunning()) return false;
   if (fullView->mwin == None) 
   {
      kdDebug() << "kghostview: communication window unknown!" << endl;
      return false;
   }
	
   if (interpreter_ready) {
      interpreter_ready = false;
      fullView->setCursor( waitCursor );

      ev.xclient.type = ClientMessage;
      ev.xclient.display = gs_display;
      ev.xclient.window = fullView->mwin;
      ev.xclient.message_type = fullView->gs_next;
      ev.xclient.format = 32;
		
      XSendEvent(gs_display, fullView->mwin, false, 0, &ev);
      XFlush(gs_display);

      return true;
   } 
   else 
   {
      return false;
   }
}

bool KPSWidget::sendPS( FILE *fp, long begin, unsigned int len, bool close )
{
   struct record_list *ps_new;

   if (!isInterpreterRunning()) return false;

   ps_new = (struct record_list *) malloc(sizeof (struct record_list));
   ps_new->fp = fp;
   ps_new->begin = begin;
   ps_new->len = len;
   ps_new->seek_needed = true;
   ps_new->close = close;
   ps_new->next = 0;

   if (input_buffer == 0) 
   {
      input_buffer = (char *) malloc(BUFFER_SIZE);
   }

   if (ps_input == 0) 
   {
      bytes_left = len;
      ps_input = ps_new;
   } else {
      struct record_list *p = ps_input;
      while (p->next != 0) {
         p = p->next;
      }
      p->next = ps_new;
   }
   if (stdin_ready)
      gs_input();
   
   return false;
}   

// *****************************************************************
//
//	QT WIDGET REIMPLIMENTED METHODS
//
// *****************************************************************

void KPSWidget::paintEvent(QPaintEvent *)
{
}

void KPSWidget::resizeEvent(QResizeEvent *)
{
   if( mDocumentPresent == false )
   {
      emit viewSizeChanged( size() );
      //fullView->setGeometry( 0, 0, width()-1, height()-1 );
      return;
   }
   int frame_width, frame_height;
   int sbw = QApplication::style().scrollBarExtent().width();
   int sbh = QApplication::style().scrollBarExtent().height();
	
   if( width() > fullView->width() &&  height() > fullView->height() ) 
   {
      horScrollBar->hide();
      vertScrollBar->hide();
      clip_x= (int)( width()- fullView->width() )/2;
      clip_y= (int)( height()- fullView->height() )/2;
      clip_width= fullView->width();
      clip_height= fullView->height();
   } 
   else if ( width() > fullView->width()+sbw ) 
   {
      horScrollBar->hide();
      vertScrollBar->show();
      clip_x= (int)( width()-( fullView->width()+sbw) )/2;
      clip_y= 0;
      clip_width= width()-sbw;
      clip_height= height();
   }
   else if ( height() > fullView->height()+sbh ) 
   {
      horScrollBar->show();
      vertScrollBar->hide();
      clip_x= 0;
      clip_y= (int)( height()-( fullView->height()+sbh ))/2;
      clip_width= width();
      clip_height= height()-sbh;
   }
   else 
   {
      horScrollBar->show();
      vertScrollBar->show();
      clip_x= 0;
      clip_y= 0;
      clip_width= width()-sbw;
      clip_height= height()-sbh;
   } 
   
   horScrollBar->setGeometry( 0, height()-sbh, clip_width, sbh);

   if(fullView->width()- width() > 0) 
   {
      if( vertScrollBar->isVisible() )
         horScrollBar->setRange( 0, fullView->width() - width() +
                                    vertScrollBar->width() );
      else
         horScrollBar->setRange( 0, fullView->width() - width() );
   } 
   else 
   {
      horScrollBar->setRange( 0, 0 );
   }
   horScrollBar->setSteps( (int)( fullView->width()/50),  width() );
   
   vertScrollBar->setGeometry(width()-sbw, 0, sbw, clip_height);
   if( fullView->height()- height() > 0 ) 
   {
      if( horScrollBar->isVisible() )
         vertScrollBar->setRange( 0,  fullView->height() - height() +
                                      horScrollBar->height());
      else
         vertScrollBar->setRange( 0,  fullView->height() - height() );
   } 
   else 
   {
      vertScrollBar->setRange( 0,  0 );
   }
   vertScrollBar->setSteps( (int)( fullView->height()/50),  height() );
   
   if ( vertScrollBar->isVisible() && horScrollBar->isVisible() ) 
   {
      patch->show();
      patch->setGeometry( vertScrollBar->x(),
                          vertScrollBar->y()+vertScrollBar->height(),
                          vertScrollBar->width(), horScrollBar->height() );
   } 
   else
   {
      patch->hide();
   }
   
   if( clip_width == fullView->width() ) 
   {
      frame_width = width();
      frame_height = height();
   } 
   else 
   {
      frame_width = clip_width;
      frame_height = clip_height;
   }
   
   topFrame->setGeometry(0, 0, frame_width, 1);
      
   leftFrame->setGeometry(0, 0, 1, frame_height);
      
   rightFrame->setGeometry(frame_width-1, 0, 1, frame_height);
      
   bottomFrame->setGeometry(0, frame_height-1, frame_width, 1);
   
   movePage(); 
   emit viewSizeChanged( size() );
}

void KPSWidget::wheelEvent( QWheelEvent *e )
{
   if( vertScrollBar->isVisible() == true )
   {
      QApplication::sendEvent( vertScrollBar, e );
   }
}



void KPSWidget::layout()
{
   bool sizeChange = computeSize();
   //  printf ("FV1:  %d %d\n", fullView->width(), 
   //  fullView->height());
   if( sizeChange ) 
   {
      resizeEvent(0); // Dutta 16/3/98
      repaint();
      setup();
      emit pageSizeChanged( QSize(fullView->width(),
                                  fullView->height()) );
   }
}

/**********************************************************************************
 *
 *	SLOTS
 *
 **********************************************************************************/

bool KPSWidget::readDown()
{
   int new_value;

   if( vertScrollBar->value() == vertScrollBar->maxValue() )
      return false;

   new_value = vertScrollBar->value()+height()-50;
   if(new_value > vertScrollBar->maxValue())
      new_value = vertScrollBar->maxValue();
   
   vertScrollBar->setValue( new_value );
   return true;
}

void KPSWidget::scrollRight()
{
   horScrollBar->addLine();
}

void KPSWidget::scrollLeft()
{
   horScrollBar->subtractLine();
}

void KPSWidget::scrollDown()
{
   vertScrollBar->addLine();
}

void KPSWidget::scrollUp()
{
   vertScrollBar->subtractLine();
}

void KPSWidget::scrollTop()
{
   vertScrollBar->setValue( vertScrollBar->minValue() );
}

void KPSWidget::movePage()
{
   fullView->setGeometry(-scroll_x_offset+clip_x, 
                         -scroll_y_offset+clip_y, 
                         fullView->width(),
                         fullView->height());
   repaint();
   emit currentPosChanged( QPoint(scroll_x_offset,scroll_y_offset) );
}
   
bool KPSWidget::configure()
{
   intConfig->setup();
   if (intConfig->exec())
   {
      setup();
      return true;
   }

   return false; 
}

void
KPSWidget::writeSettings()
{
   intConfig->writeSettings();
}

void
KPSWidget::slotVertScroll(int value)
{
   scroll_y_offset =  value;
   movePage();
}

void
KPSWidget::slotHorScroll(int value)
{
   scroll_x_offset = value;
   movePage();
}

void KPSWidget::slotScroll( QPoint point )
{
   horScrollBar->setValue( point.x() );
   vertScrollBar->setValue( point.y() );
}

// *****************************************************************************
//
//	PRIVATE METHODS
//
// ******************************************************************************


bool
KPSWidget::computeSize()
{
   int newWidth=0, newHeight=0;
   bool change = false;
  
   int old_orient_angle = orient_angle;

   switch (orientation) 
   {
   case 1: //PORTRAIT
      orient_angle=0;
      newWidth = (int) ((urx - llx) / 72.0 * xdpi + 0.5);
      newHeight = (int) ((ury - lly) / 72.0 * ydpi + 0.5);
      break;
   case 2: //LANDSCAPE
      orient_angle=90;
      newWidth = (int) ((ury - lly) / 72.0 * xdpi + 0.5);
      newHeight = (int) ((urx - llx) / 72.0 * ydpi + 0.5);
      break;
   case 3: //UPSIDEDOWN
      orient_angle=180;
      newWidth = (int) ((urx - llx) / 72.0 * xdpi + 0.5);
      newHeight = (int) ((ury - lly) / 72.0 * ydpi + 0.5);
      break;
   case 4: //SEASCAPE
      orient_angle=270;
      newWidth = (int) ((ury - lly) / 72.0 * xdpi + 0.5);
      newHeight = (int) ((urx - llx) / 72.0 * ydpi + 0.5);
      break;
   }
  
   //printf("CS: x offset %d, width %d, y offset %d, height %d\n", llx, urx, lly,
   //ury);
  
   //printf("CS: newWidth = %d, new Height = %d\n", newWidth, newHeight );
  
   if( (newWidth != fullView->width()) || 
       (newHeight != fullView->height()) )
   {
      fullView->resize (newWidth, newHeight);
      change = true;
   }

   return change | (old_orient_angle != orient_angle);
}

static bool alloc_error;
static XErrorHandler oldhandler;

static int catch_alloc (Display *dpy, XErrorEvent *err)
{
   if (err->error_code == BadAlloc) {
      alloc_error = true;
   }
   if (alloc_error) return 0;
   return oldhandler(dpy, err);
}

void
KPSWidget::setup()
{
   Pixmap bpixmap = None;
  
   // NO stop interpreter ?
  
   stopInterpreter();
  
   // NO test of actual change ?
  
   if (background_pixmap != None) 
   {
      XFreePixmap(gs_display, background_pixmap);
      background_pixmap = None;
      XSetWindowBackgroundPixmap(gs_display, gs_window, None);
   }
  
   if( intConfig->backingStoreType() ==InterpreterDialog::PIX_BACKING ) 
   {
      if (background_pixmap == None) 
      {
         XSync(gs_display, false); 
         oldhandler = XSetErrorHandler(catch_alloc);
         alloc_error = false;
         bpixmap = XCreatePixmap(
                        gs_display, gs_window,
                        fullView->width(), fullView->height(),
                        DefaultDepth( gs_display, DefaultScreen( gs_display) ) );
         XSync(gs_display, false); 
         if (alloc_error) 
         {
            //printf("BadAlloc\n");
            //XtCallCallbackList(w, gvw->ghostview.message_callback,
            //         "BadAlloc");
            if (bpixmap != None) 
            {
               XFreePixmap(gs_display, bpixmap);
               XSync(gs_display, false); 
               bpixmap = None;
            }
         }
         oldhandler = XSetErrorHandler(oldhandler);
         if (bpixmap != None) 
         {
            background_pixmap = bpixmap;
            XSetWindowBackgroundPixmap(gs_display, gs_window,
                                       background_pixmap);
         }
      }
      else 
      {
         bpixmap = background_pixmap;
      }
   }
  
   XSetWindowAttributes xswa;
  
   if (bpixmap != None) 
   {
      xswa.backing_store = NotUseful;
      XChangeWindowAttributes(gs_display, gs_window,
                              CWBackingStore, &xswa);
   } 
   else 
   {
      xswa.backing_store = Always;
      XChangeWindowAttributes(gs_display, gs_window,
                              CWBackingStore, &xswa);
   }
  
   fullView->ghostview = (Atom) XInternAtom(gs_display, "GHOSTVIEW", false);
   fullView->gs_colors = (Atom) XInternAtom(gs_display, "GHOSTVIEW_COLORS", false);
   fullView->gs_next = (Atom) XInternAtom(gs_display, "NEXT", false);
   fullView->gs_page = (Atom) XInternAtom(gs_display, "PAGE", false);
   fullView->gs_done = (Atom) XInternAtom(gs_display, "DONE", false);
  
   char buf[512];

   sprintf(buf, "%ld %d %d %d %d %d %g %g %d %d %d %d",
           background_pixmap,
           orient_angle,
           llx, lly,
           urx, ury,
           xdpi, ydpi,
           left_margin, bottom_margin,
           right_margin, top_margin);
  
   //printf("%s\n", buf);
  
   XChangeProperty(gs_display, gs_window,
                   fullView->ghostview,
                   XA_STRING, 8, PropModeReplace,
                   (unsigned char *)buf, strlen(buf));
  
   sprintf(buf, "%s %d %d",
           intConfig->paletteType() == 
              InterpreterDialog::MONO_PALETTE  ? "Monochrome" :
           intConfig->paletteType() == 
              InterpreterDialog::GRAY_PALETTE  ? "Grayscale" :
           intConfig->paletteType() == 
              InterpreterDialog::COLOR_PALETTE ? "Color" : "?",
           (int) BlackPixel(gs_display, DefaultScreen(gs_display)),
           (int) WhitePixel(gs_display, DefaultScreen(gs_display)) );
  
   //printf("%s\n", buf);
  
   XChangeProperty(gs_display, gs_window,
                   fullView->gs_colors,
                   XA_STRING, 8, PropModeReplace,
                   (unsigned char *)buf, strlen(buf));
  
   XSync(gs_display, false);  // Be sure to update properties 
} 

void KPSWidget::startInterpreter()
{
   GC gc;
   XGCValues values;

   values.foreground = WhitePixel(gs_display, DefaultScreen( gs_display) );
   values.background = BlackPixel(gs_display, DefaultScreen( gs_display) );

   gc = XCreateGC ( gs_display,
                    RootWindow( gs_display, DefaultScreen( gs_display ) ),
                    ( GCForeground | GCBackground ), &values );

   stopInterpreter();

   if ( background_pixmap != None ) 
   {
      XFillRectangle( gs_display, background_pixmap,
                      gc /* DefaultGC( gs_display, DefaultScreen( gs_display ) ) */,
                      0, 0, fullView->width(), fullView->height() );
   }
   
   fullView->erase();

   if (disable_start) return;

   proc = new KProcess;

   *proc << "gs";
   if( intConfig->antiAlias() ) 
      *proc << "-sDEVICE=x11alpha";
   else
      *proc << "-sDEVICE=x11";

   if( !intConfig->platformFonts() )
      *proc << "-dNOPLATFONTS";

   *proc << "-dNOPAUSE";
   *proc << "-dQUIET";
   *proc << "-dSAFER";
   if (filename.isEmpty())
      *proc << "-";
   else
      *proc << filename;

   changed = false;
   busy = true;
   fullView->setCursor( waitCursor );

   // WABA: Save & restore this.
   char buf[512];
   sprintf(buf, "%d", (int) gs_window);
   setenv("GHOSTVIEW", buf, true);
   setenv("DISPLAY", XDisplayString(gs_display), true);

   connect(proc, SIGNAL(processExited(KProcess *)), 
           this, SLOT(interpreterFailed()));

   connect(proc, SIGNAL(receivedStdout(KProcess *,char *, int )), 
           this, SLOT(gs_output(KProcess *,char *, int)));

   connect(proc, SIGNAL(receivedStderr(KProcess *,char *, int )), 
           this, SLOT(gs_output(KProcess *,char *, int)));

   connect(proc, SIGNAL(wroteStdin(KProcess *)), 
           this, SLOT(gs_input()));

   XClearArea(gs_display, gs_window,
         0, 0, fullView->width(), fullView->height(), false);

   kapp->flushX();

   proc->start(KProcess::NotifyOnExit, 
               filename.isEmpty() ? KProcess::All : KProcess::AllOutput );

   stdin_ready = true;
   interpreter_ready = false;
}

void KPSWidget::stopInterpreter()
{
   if (!busy) return;
   busy=false;

   if (isInterpreterRunning())
   {
      proc->kill(SIGTERM);
      while(proc->isRunning())
      {
         kapp->processEvents();
      }
   }

   delete proc;
   proc = 0;

   while (ps_input) {
      struct record_list *ps_old = ps_input;
      ps_input = ps_old->next;
      if (ps_old->close) 
         fclose(ps_old->fp);
      free((char *)ps_old);
   }
   
   fullView->setCursor( arrowCursor );
}

void KPSWidget::interpreterFailed()
{
   stopInterpreter();
} 

/* Output - receive I/O from ghostscript's stdout and stderr.
 * Pass this to the application via the output_callback. */
 
void KPSWidget::gs_output( KProcess *, char *buffer, int len )
{
   QString line = QString::fromLocal8Bit(buffer, len);

   if (line.isEmpty()) return;
   if( intConfig->showMessages() ) 
   {
      messages->show();
      messages->cancel->setFocus();
      messages->messageBox->append( line );
      //messages->setCursorPosition( messages->numRows(), 0 );
   }
}


void KPSWidget::gs_input()
{
   stdin_ready = true;

   do 
   {
      // Close old buffer
      if (ps_input && bytes_left == 0) 
      {
         struct record_list *ps_old = ps_input;
         ps_input = ps_old->next;
         if (ps_old->close) 
            fclose(ps_old->fp);
         free((char *)ps_old);
      } 

      // Open new buffer
      if (ps_input && ps_input->seek_needed) 
      {
         if (ps_input->len > 0)
            fseek(ps_input->fp, ps_input->begin, SEEK_SET);
         ps_input->seek_needed = false;
         bytes_left = ps_input->len;
      }
   }
   while(ps_input && (bytes_left == 0)); // Skip empty blocks.

   int buflen = 0;
   if (bytes_left > BUFFER_SIZE) 
   {
      buflen = fread(input_buffer, sizeof (char), BUFFER_SIZE, ps_input->fp);
   }
   else if (bytes_left > 0) 
   {
      buflen = fread(input_buffer, sizeof (char), bytes_left, ps_input->fp);
   }

   if (bytes_left > 0 && buflen == 0) 
   {
      interpreterFailed();
      return;
   }
   bytes_left -= buflen;
   
   if (buflen > 0) 
   {
      if (!proc->writeStdin(input_buffer, buflen))
      {
         interpreterFailed();
         return;
      } 
      stdin_ready = false;
   }   
   else
   {
      interpreter_ready = true;
   }
}  

/*
void
KPSWidget::sync()
{
   emit viewSizeChanged( size() );  
   emit pageSizeChanged( QSize(fullView->width(),
                         fullView->height()) );
   emit currentPosChanged( QPoint(scroll_x_offset,scroll_y_offset) );
}
*/


KFVWidget::KFVWidget (QWidget *parent) :
  QWidget (parent)
{
   //TODO - make a nice hand cursor
   //  fullView->setCursor (hand);
   mwin = None;
}

void
KFVWidget::mousePressEvent (QMouseEvent *me)
{
   dragstartx = me->globalPos().x();
   dragstarty = me->globalPos().y();
   startoffx = ((KPSWidget *)parent())->scroll_x_offset;
   startoffy = ((KPSWidget *)parent())->scroll_y_offset;
}

void
KFVWidget::mouseMoveEvent (QMouseEvent *me)
{
   if (me->state()!=LeftButton)
      return;

   //range is checked by scroll bars
   ((KPSWidget *)parent())->slotScroll 
              ( QPoint ( startoffx + dragstartx - me->globalPos().x(),
                startoffy + dragstarty - me->globalPos().y() ));
}


bool 
KFVWidget::x11Event( XEvent *ev )
{
   if(ev->xany.type == ClientMessage) 
   {
      mwin = ev->xclient.data.l[0];

      XClientMessageEvent *cme = ( XClientMessageEvent * ) ev;

      if(cme->message_type == gs_page) 
      {
// kdDebug() << "kghostview: got gs_page" << endl;
//         kg->page->busy=False;
         setCursor( arrowCursor );
         return TRUE;
      } 
      else if(cme->message_type == gs_done) 
      {
// kdDebug() << "kghostview: got done" << endl;
//         kg->page->disableInterpreter();
         return TRUE;
      }
   }
   return QWidget::x11Event(ev);  
}

