#include "artsmodules.h"
#include "artsbuilder.h"
#include "stdsynthmodule.h"
#include "objectmanager.h"
#include "connect.h"
#include "flowsystem.h"
#include "debug.h"
#include <fstream>
#include <math.h>

using namespace Arts;
using namespace std;

struct TSNote {
	MidiChannel channelObject;
	TimeStamp time;
	mcopbyte channel, note, velocity;	// velocity=0: release
	TSNote(MidiChannel co, const TimeStamp& ts,
				mcopbyte c, mcopbyte n, mcopbyte v) :
		channelObject(co), time(ts), channel(c), note(n), velocity(v)
	{
	}
};

static vector<TSNote> noteQueue;

class AutoMidiRelease : public TimeNotify {
public:
	vector<MidiReleaseHelper> impls;
	AutoMidiRelease()
	{
		Dispatcher::the()->ioManager()->addTimer(10, this);
	}
	~AutoMidiRelease()
	{
		Dispatcher::the()->ioManager()->removeTimer(this);
	}
	void notifyTime()
	{
		timeval tv;
		gettimeofday(&tv,0);
		vector<TSNote>::iterator n = noteQueue.begin();
		while(n != noteQueue.end())
		{
			TSNote& note = *n;
			if( tv.tv_sec > note.time.sec 
			|| ((tv.tv_sec == note.time.sec) && (tv.tv_usec > note.time.usec)))
			{
				MidiChannel& ch = note.channelObject;

				if(note.velocity > 0)
					ch.noteOn(note.channel,note.note,note.velocity);
				else
					ch.noteOff(note.channel,note.note);
				noteQueue.erase(n);
				n = noteQueue.begin();
			}
			else n++;
		}


		vector<MidiReleaseHelper>::iterator i = impls.begin();
		while(i != impls.end())
		{
			if(i->terminate())
			{
				MidiReleaseHelper& helper = *i;

				// put the MidiReleaseHelper and the voice into the ObjectCache
				// (instead of simply freeing it)
				ObjectCache cache = helper.cache();
				SynthModule voice = helper.voice();
				voice.stop();
				cache.put(voice,helper.name());
				impls.erase(i);
				return;
			} else i++;
		}
	}
}	*autoMidiRelease;

// cache startup & shutdown
static class AutoMidiReleaseStart :public StartupClass
{
public:
	void startup() { autoMidiRelease = new AutoMidiRelease(); }
	void shutdown() { delete autoMidiRelease; }
} autoMidiReleaseStart;

class MidiReleaseHelper_impl : virtual public MidiReleaseHelper_skel,
							   virtual public StdSynthModule
{
protected:
	bool _terminate;
	SynthModule _voice;
	ObjectCache _cache;
	string _name;

public:
	MidiReleaseHelper_impl()
	{
		autoMidiRelease->impls.push_back(MidiReleaseHelper::_from_base(_copy()));
	}
	~MidiReleaseHelper_impl() {
		artsdebug("MidiReleaseHelper: one voice is gone now\n");
	}


	SynthModule voice() { return _voice; }
	void voice(SynthModule voice) { _voice = voice; }

	ObjectCache cache() { return _cache; }
	void cache(ObjectCache cache) { _cache = cache; }

	string name() { return _name; }
	void name(const string& name) { _name = name; }

	bool terminate() { return _terminate; }
	void streamStart() { _terminate = false; }

	void calculateBlock(unsigned long samples)
	{
		if(done[0] > 0.5)
			_terminate = true;
	}
};
REGISTER_IMPLEMENTATION(MidiReleaseHelper_impl);

static int XcleanReference(const string& reference)
{
	Object test;
	test = Reference("global:"+reference);
	if(test.isNull())
	{
		Dispatcher::the()->globalComm().erase(reference);
		return 1;
	}
	else
		return 0;
}

class Synth_MIDI_TEST_impl : virtual public Synth_MIDI_TEST_skel,
							 virtual public StdSynthModule {
protected:
	StructureDesc instrument;
	StructureBuilder builder;
	SynthModule voice[128];
	ObjectCache cache;

	string _filename;
public:
	Synth_MIDI_TEST self() { return Synth_MIDI_TEST::_from_base(_copy()); }

	Synth_MIDI_TEST_impl();

	void filename(const string& newname);
	string filename()
	{
		return _filename;
	}
	void noteOn(mcopbyte channel, mcopbyte note,
											mcopbyte volume);
	void noteOff(mcopbyte channel, mcopbyte note);

	TimeStamp time()
	{
		timeval tv;
		gettimeofday(&tv, 0);
		return TimeStamp(tv.tv_sec, tv.tv_usec);
	}
	void noteOnAt(const TimeStamp& time, mcopbyte channel,
			mcopbyte note, mcopbyte volume)
	{
		noteQueue.push_back(TSNote(self(),time,channel,note,volume));
	}
	void noteOffAt(const TimeStamp& time, mcopbyte channel, mcopbyte note)
	{
		noteQueue.push_back(TSNote(self(),time,channel,note,0));
	}
};
REGISTER_IMPLEMENTATION(Synth_MIDI_TEST_impl);

void Synth_MIDI_TEST_impl::filename(const string& newname)
{
	ifstream infile(newname.c_str());
	string line;
	vector<string> strseq;

	while(getline(infile,line))
		strseq.push_back(line);

	instrument.loadFromList(strseq);
	_filename = newname;
}

Synth_MIDI_TEST_impl::Synth_MIDI_TEST_impl()
{
	// create instrument

	if(!ObjectManager::the()->addGlobalReference(self(),"Arts_MidiChannel"))
	{
		XcleanReference("Arts_MidiChannel");
		if(!ObjectManager::the()->addGlobalReference(self(),"Arts_MidiChannel"))
			cerr << "already running, will not work" << endl;
	}

	// initialize all voices with NULL objects (no lazy create)
	for(int i = 0; i < 128; i++) voice[i] = SynthModule::null();
}

void Synth_MIDI_TEST_impl::noteOn(mcopbyte channel, mcopbyte note,
											mcopbyte volume)
{
	if(!voice[note].isNull())
	{
		noteOff(channel,note);
		cout << "duplicate noteOn (mixed channels?)" << endl;
	}

	Object structureObject = cache.get(_filename);
	if(structureObject.isNull())
	{
		cout << "creating new structure" << endl;
		structureObject = builder.createObject(instrument);
	}
	else cout << "used cached structure" << endl;

	SynthModule structure = DynamicCast(structureObject);
	assert(!structure.isNull());

	float outfrequency = 261.63 * pow(2,(float)note/12)/32.0;
	setValue(structure,"frequency",outfrequency);
	setValue(structure,"velocity",0.7);
	setValue(structure,"pressed",1.0);
	structure.start();

	voice[note] = structure;
}

void Synth_MIDI_TEST_impl::noteOff(mcopbyte channel, mcopbyte note)
{
	if(!voice[note].isNull())
	{
		setValue(voice[note],"pressed",0.0);

		MidiReleaseHelper h;

		h.voice(voice[note]);
		h.cache(cache);
		h.name(_filename);

		connect(voice[note],"done",h,"done");
		h.start();
		assert(!h.terminate());
		voice[note] = SynthModule::null();
	}
}
