/*
   Copyright (c) 2000 Stefan Schimanski <1Stein@gmx.de>
                 1999-2000 Christian Esken (esken@kde.org)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <kfiledialog.h>
#include <stdlib.h>
#include <unistd.h>

#include <kapp.h>
#include <kwin.h>
#include <kglobal.h>
#include <klocale.h>
#include <dcopclient.h>
#include <kstddirs.h>
#include <kmessagebox.h>
#include <kcmdlineargs.h>
#include <kconfig.h>

#include <qpixmap.h>
#include <qbitmap.h>
#include <qimage.h>
#include <qdropsite.h>
#include <qdragobject.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>

#include <X11/X.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/shape.h>

#include <iostream.h>
#include <stdlib.h>
#include <math.h>

#include "kaiman.h"
#include "playlistwin.h"
#include "pref.h"


#define DEFAULT_SKIN "car-preset"
#define STATUS_POLL_DELAY 200
#define ARTS_POLL_DELAY 500

bool artswrapper_check()
{
    if(system("artswrapper check") == 0)
        return true;
    return false;
}

using namespace Arts;

Kaiman::Kaiman( const QString &skin )
    : KMainWindow(0), _style(0), _prefDlg(0)
{
    KWin::setType( this->winId(), NET::Override );
    //setWFlags( Qt::WStyle_Customize | Qt::WStyle_NoBorderEx );
    setBackgroundMode( NoBackground );
    setAcceptDrops(true);

    // init config
    KConfig *config = kapp->config();
    config->setGroup(0);

    _shuffle = config->readBoolEntry( "Shuffle", false );
    _looping = config->readBoolEntry( "Loop", false );
    _altSkin = config->readBoolEntry( "AltSkin", false );
    _mixerCommand = "kmix"; // Default mixer
    _seeking = false;

    // initialize aRts references
    _playObject = Arts::PlayObject::null();
    _playObjectFactory = Arts::PlayObjectFactory::null();
    _volumeControl = Arts::StereoVolumeControl::null();
    _soundServer = Arts::SimpleSoundServer::null();
    _lastPolledState = Arts::posIdle;

    // load skin
    QString skinName;
    if ( skin.length()>0 )
        skinName = skin;
    else
        skinName = config->readEntry( "Skin", DEFAULT_SKIN );

    if ( !loadStyle( skinName, _altSkin ? "alt_skindata" : "skindata" ) ) {
        if ( !loadStyle(skinName, "skindata") ) {
            KMessageBox::sorry( this, i18n("Can't load skin %1. Switching to default skin.").arg(skinName) );
            if ( !loadStyle( DEFAULT_SKIN, "skindata" ) ) {
                KMessageBox::error( this, i18n("Can't load default skin %1.").arg(DEFAULT_SKIN) );
                exit( 1 );
            }
        }
    }

    // playlist initializations
    _mediaManager = new MediaManager( "Media Manager" );
    connect( _mediaManager, SIGNAL(stop()), this, SLOT(stop()) );
    connect( _mediaManager, SIGNAL(play(MediaFile*)), this, SLOT(play(MediaFile*)) );

    QString oldPlaylist = config->readEntry( "Playlist" );
    if ( oldPlaylist.length()>0 ) {
        _mediaManager->loadPlaylist( KURL(oldPlaylist) );
    } else {
        _mediaManager->addPlaylist(
            kapp->dirs()->saveLocation("data", "kaiman/") + "default.m3u" );
    }

    // set default volume
    setVolume( config->readNumEntry( "Volume", 100 ) );

    // start update timer
    _artsPollTimer.start( ARTS_POLL_DELAY );
    connect( &_artsPollTimer, SIGNAL(timeout()), this, SLOT(pollArts()));

    _statusPollTimer.start( STATUS_POLL_DELAY );
    connect( &_statusPollTimer, SIGNAL(timeout()), this, SLOT(updateStatusDisplay()));
}


Kaiman::~Kaiman()
{
    // save config
    KConfig *config = kapp->config();
    config->setGroup(0);

    config->writeEntry( "Shuffle", _shuffle );
    config->writeEntry( "Loop", _looping );
    config->writeEntry( "AltSkin", _altSkin );
    config->writeEntry( "Volume", _volume );
    config->writeEntry( "Skin", _style->skinName() );
    config->writeEntry( "Playlist", _mediaManager->fileName().url() );

    // remove effect from effect chain
    if ( !_soundServer.isNull() && !_soundServer.error() &&
         !_volumeControl.isNull() && !_volumeControl.error() )
        {
        _soundServer.outstack().remove( _volumeEffectID );
        }

    // destroy aRts objects
    _playObject = Arts::PlayObject::null();
    _playObjectFactory = Arts::PlayObjectFactory::null();
    _volumeControl = Arts::StereoVolumeControl::null();
    _soundServer = Arts::SimpleSoundServer::null();

    // destroy playlist
    delete _mediaManager;
    _mediaManager = 0;
}

bool Kaiman::initArts()
{
    if ( _soundServer.isNull() || _soundServer.error()  ) {
        _volumeControl = Arts::StereoVolumeControl::null();
        _soundServer = Reference("global:Arts_SimpleSoundServer");
        _playObjectFactory = Arts::Reference("global:Arts_PlayObjectFactory");

        if( _soundServer.isNull() || _soundServer.error() )
        {
            // aRts seems not to be running, let's try to run it

            // First, let's read the configuration as in kcmarts
            KConfig *config = new KConfig("kcmartsrc");
            QCString cmdline;

            config->setGroup("Arts");

            bool rt = config->readBoolEntry("StartRealTime",false);
            bool x11Comm = config->readBoolEntry("X11GlobalComm",false);

            /* put the value of x11Comm into .mcoprc */
            KConfig *X11CommConfig = new KConfig(QDir::homeDirPath()+"/.mcoprc");

            if(x11Comm)
                X11CommConfig->writeEntry("GlobalComm","Arts::X11GlobalComm");
            else
                X11CommConfig->writeEntry("GlobalComm","Arts::TmpGlobalComm");

            X11CommConfig->sync();
            delete X11CommConfig;

            cmdline = QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("kdeinit_wrapper")));
            cmdline += " ";

            if (rt)
              cmdline += QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("artswrapper")));
            else
              cmdline += QFile::encodeName(KStandardDirs::findExe(QString::fromLatin1("artsd")));

            cmdline += " ";
            cmdline += config->readEntry("Arguments", "-F 5 -S 8192").utf8();

            int status=system(cmdline);

            if ( status!=-1 && WIFEXITED(status) )
            {
                // We could have a race-condition here. The correct way to do it is to
                // make artsd fork-and-exit after starting to listen to connections
                // (and running artsd directly instead of using kdeinit),
                // but this is better than nothing.
                int time = 0;
                do
                {
                    sleep(1);
                    _soundServer = Reference("global:Arts_SimpleSoundServer");
                    _playObjectFactory = Arts::Reference("global:Arts_PlayObjectFactory");
                } while(++time < 5 && (_soundServer.isNull() || _playObjectFactory.isNull()));
            }

            if( _playObjectFactory.isNull() )
            {
                KMessageBox::error( 0, i18n("Connection to the soundserver failed - make sure that artsd is really running.") );
                return false;
            }
        }

        if ( !_soundServer.isNull() ) {
            // create a new stereo volume control object on the server
            _volumeControl = DynamicCast(_soundServer.createObject("Arts::StereoVolumeControl"));
            _volumeControl.start();
            _volumeEffectID = _soundServer.outstack().insertBottom(_volumeControl, "Volume Control");
            _volumeControl.scaleFactor( _volume/100.0 );
        }
    }

    _playObject = Arts::PlayObject::null();

    return true;
}

bool Kaiman::loadStyle( const QString &style, const QString &desc )
{
    if ( _style ) delete _style;
    _style = new KaimanStyle(this);
    if ( !_style->loadStyle( style, desc ) )
    {
        delete _style;
        _style = 0;
        return false;
    }

    if (  _style->Mask()!=0) {
        // Set the shaped window form
        XShapeCombineMask( qt_xdisplay(), winId(), ShapeBounding, 0,0,
                           _style->Mask()->handle(), ShapeSet );
    }

    KaimanStyleElement* item = _style->find("Background");
    setBackgroundMode(QWidget::NoBackground);
    if ( item!=0 ) {
        _style->resize( item->width(), item->height());
        resize( item->width(), item->height());
        setFixedSize( item->width(), item->height());
    }

    item = _style->find("Playlist_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(togglePlaylist()) );

    item = _style->find("Play_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(play()) );

    item = _style->find("Pause_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(pause()) );

    item = _style->find("Stop_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(stop()) );

    item = _style->find("Next_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(next()) );

    item = _style->find("Prev_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(prev()) );

    item = _style->find("Exit_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), kapp, SLOT(quit()) );

    item = _style->find("Mixer_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(execMixer()) );

    item = _style->find("Iconify_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(showMinimized()) );

    item = _style->find("Alt_Skin_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(toggleSkin()) );

    item = _style->find("Repeat_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(toggleLoop()) );

    item = _style->find("Shuffle_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(toggleShuffle()) );

    item = _style->find("Config_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this, SLOT(preferences()) );

    item = _style->find("Volume_Up_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this,SLOT(volumeUp()));

    item = _style->find("Volume_Down_Button");
    if( item!=0 ) connect( item, SIGNAL(clicked()), this,SLOT(volumeDown()));

    item = _style->find("Position_Slider");
    if( item!=0 ) {
        connect( item, SIGNAL(newValueDrag(int)), this, SLOT(seekStart(int)) );
        connect( item, SIGNAL(newValue(int)), this, SLOT(seekDrag(int)) );
        connect( item, SIGNAL(newValueDrop(int)), this, SLOT(seekStop(int)) );
    }


    KaimanStyleSlider* l_elem_volslider = static_cast<KaimanStyleSlider*>(_style->find("Volume_Slider"));
    if ( l_elem_volslider!=0 ) {
        connect(l_elem_volslider,SIGNAL(newValue(int)),this,SLOT(setVolume(int)));
        l_elem_volslider->setValue( _volume, 0, 100 );
    }

    KaimanStyleValue* volItem = static_cast<KaimanStyleValue*>(_style->find("Volume_Item"));
    if ( volItem ) volItem->setValue( _volume, 0, 100 );

    KaimanStyleText* titleItem = static_cast<KaimanStyleText*>(_style->find("Title"));
    if ( titleItem ) titleItem->startAnimation( 300 );

    return true;
}


void Kaiman::togglePlaylist()
{
    if ( _mediaManager->isVisible() )
        _mediaManager->hide();
    else
        _mediaManager->show();
}


bool Kaiman::play( const KURL &url ) {
    QString file = url.path();

    if ( !_playObjectFactory.isNull() ) {
        _playObject = _playObjectFactory.createPlayObject( file.ascii() );
        if ( !_playObject.isNull() ) {
            _playObject.play();
            return _playObject.state()==Arts::posPlaying;
        } else
            return false;
    } else
        return false;
}


bool Kaiman::play( MediaFile *media )
{
    // reconnect to arts
    if ( !initArts() ) {
        stop();
        return false;
    }

    if ( !play(media->url()) ) {
        stop();
        return false;
    }

    return true;
}


bool Kaiman::playNext( bool reverse )
{
    // reconnect to arts
    if ( !initArts() ) return false;

    // try to play files
    for ( int tries = _mediaManager->count();
          tries>0;
          tries-- )
    {
        // try to play active media file
        MediaFile *media = _mediaManager->active();
        if ( media && play(media->url()) )
            return true;

        // playing failed -> skip file
        if (_shuffle)
            _mediaManager->shuffle();
        else
            _mediaManager->next( reverse, _looping );
    }

    return false;
}


void Kaiman::play()
{
    if( !_playObject.isNull() ) {
        if ( _playObject.state()==Arts::posPlaying )
            pause();
        else
            _playObject.play();
    } else {
        _mediaManager->start( false );
        if ( _shuffle ) _mediaManager->shuffle();

        if ( !playNext(false) )
            stop();
    }
}


void Kaiman::pause()
{
    if( !_playObject.isNull() ) {
        if ( _lastPolledState==Arts::posPaused )
            _playObject.play();
        else
            _playObject.pause();
    }
}


void Kaiman::stop()
{
    if( !_playObject.isNull() ) {
        Arts::poTime l_t_zero( 0, 0, 0, "samples" );
        _playObject.seek( l_t_zero );
        _playObject.halt();
    }

    _playObject = Arts::PlayObject::null();
    _lastPolledState = Arts::posIdle;

    _mediaManager->reset();
}


void Kaiman::clear()
{
    stop();
    if ( _mediaManager )
        _mediaManager->clear();
}


void Kaiman::next()
{
    if ( _shuffle ) _mediaManager->shuffle(); else _mediaManager->next( false, _looping );
    if ( !playNext(false) ) stop();
}


void Kaiman::prev()
{
    if ( _shuffle ) _mediaManager->shuffle(); else _mediaManager->next( true, _looping );
    if ( !playNext(true) ) stop();
}


void Kaiman::seekStart( int /*value*/ )
{
    _seeking = true;
}


void Kaiman::seekDrag( int value )
{
    KaimanStyleValue* posItem =
        static_cast<KaimanStyleValue*>(_style->find("Position_Item"));
    if ( posItem ) posItem->setValue( value );

    KaimanStyleSlider* posSlider =
        static_cast<KaimanStyleSlider*>(_style->find("Position_Slider"));
    if ( posSlider ) posSlider->setValue( value );
}


void Kaiman::seekStop( int value )
{
    seek( value );
    _seeking = false;
}


void Kaiman::seek( int sec )
{
   if ( !_playObject.isNull() ) {
        Arts::poTime t;
        t.seconds = sec;
        t.ms = 0;
        t.custom = sec;
        t.customUnit = "seconds";
        _playObject.seek( t );
    }
}


void Kaiman::toggleSkin()
{
    _altSkin = !_altSkin;

    QString skinName = _style->skinName();

    QString oldDesc, newDesc;
    if ( _altSkin ) {
        oldDesc = QString::fromLatin1("skindata");
        newDesc = QString::fromLatin1("alt_skindata");
    } else {
        newDesc = QString::fromLatin1("skindata");
        oldDesc = QString::fromLatin1("alt_skindata");
    }

    hide();
    if ( !loadStyle(skinName, newDesc) )
        loadStyle(skinName, oldDesc);
    show();
}


void Kaiman::toggleLoop()
{
    _looping = !_looping;
}


void Kaiman::toggleShuffle()
{
    _shuffle = !_shuffle;
}


void Kaiman::setVolume( int vol )
{
    if ( vol<0 ) vol=0;
    if ( vol>=100 ) vol=100;
    _volume = vol;

    if ( !_volumeControl.isNull() )
        {
        _volumeControl.scaleFactor(vol/100.0);
        }

    KaimanStyleSlider* l_elem_volslider = static_cast<KaimanStyleSlider*>(_style->find("Volume_Slider"));
    KaimanStyleValue* l_elem_volitem = static_cast<KaimanStyleValue*>(_style->find("Volume_Item"));
    if ( l_elem_volslider!=0 ) l_elem_volslider->setValue( _volume, 0, 100 );
    if ( l_elem_volitem!=0 ) l_elem_volitem->setValue( _volume, 0, 100 );
}


void Kaiman::volumeUp()
{
    setVolume( _volume+10 );
}


void Kaiman::volumeDown()
{
    setVolume( _volume-10 );
}


void Kaiman::execMixer() {
    if ( fork() == 0 ) {
        execlp( QFile::encodeName(_mixerCommand), QFile::encodeName(_mixerCommand), 0 );
        // We should never pass this line. If yes, execlp() failed.
        // Please remember that we are here in the forked process,
        // so doing exit() is OK. But we cannot open a dialog
        // box here, can we?!?
        fprintf(stderr,"Error: Starting mixer failed.\n");
        exit( 1 );
    }
}


void Kaiman::preferences()
{
    if ( _prefDlg==0 ) {
        // create dialog
        _prefDlg = new KaimanPrefDlg( this, "KaimanPrefDlg", false );
        connect( _prefDlg, SIGNAL(applyClicked()), this, SLOT(applyPref()) );
        connect( _prefDlg, SIGNAL(okClicked()), this, SLOT(okPref()) );

        // set settings
        _prefDlg->setSkin( _style->skinName() );

        // show dialog
        _prefDlg->show();
    } else {
        _prefDlg->hide();
        _prefDlg->show();
    }
}

void Kaiman::okPref()
{
    if ( _prefDlg ) {
        applyPref();
        delete _prefDlg;
        _prefDlg = 0;
    }

}



void Kaiman::applyPref()
{
    if ( _prefDlg ) {
        // apply skin
        if ( _style->skinName()!=_prefDlg->skin() ) {

            QString oldStyle = _style->skinName();

            hide();
            if ( !loadStyle(_prefDlg->skin()) ) {
                loadStyle( oldStyle );
            }
            show();
        }
    }
}


void Kaiman::dragEnterEvent( QDragEnterEvent *event )
{
  event->accept( QUriDrag::canDecode(event) );
}


void Kaiman::dropEvent( QDropEvent *event )
{
    doDropEvent(event);
}


void Kaiman::doDropEvent(QDropEvent *event) {
    if ( QUriDrag::canDecode(event) ) {
        QStrList  urls;
        if ( QUriDrag::decode(event, urls) ) {
            for ( char *urlStr=urls.first(); urlStr!=0; urlStr=urls.next() ) {
                KURL url(urlStr);
                _mediaManager->addURL( url );
            }
        }
    }
}


void Kaiman::addURL( KURL url )
{
    _mediaManager->addURL( url );
}


void Kaiman::updateArtsDisplay()
{
    long sec = 0;
    long all = 0;
    int percent = 0;
    if ( !_playObject.isNull() ) {
        int newPolledState = _playObject.state();
        if ( newPolledState==Arts::posPaused || newPolledState==Arts::posPlaying )
        {
            sec = _playObject.currentTime().seconds;
            all = _playObject.overallTime().seconds;
            percent = 100*sec;
            if ( all ) percent/=all; else percent=0;
        }
    }

    // update play position
    if ( !_seeking ) {
        KaimanStyleValue* posItem =
            static_cast<KaimanStyleValue*>(_style->find("Position_Item"));
        if ( posItem ) posItem->setValue( sec, 0, all );

        KaimanStyleSlider* posSlider =
            static_cast<KaimanStyleSlider*>(_style->find("Position_Slider"));
        if ( posSlider ) posSlider->setValue( sec, 0, all );
    }

    // update time
    KaimanStyleNumber* numItem = static_cast<KaimanStyleNumber*>(_style->find("Hour_Number"));
    if ( numItem ) numItem->setValue( sec/3600 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Minute_Number"));
    if ( numItem ) numItem->setValue( sec/60%60 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Second_Number"));
    if ( numItem ) numItem->setValue( sec%60 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Hour_Total_Number"));
    if ( numItem ) numItem->setValue( all/3600 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Minute_Total_Number"));
    if ( numItem ) numItem->setValue( all/60%60 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Second_Total_Number"));
    if ( numItem ) numItem->setValue( all%60 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Song_Second_Number"));
    if ( numItem ) numItem->setValue( all%60 );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Song_Minute_Number"));
    if ( numItem ) numItem->setValue( all/60%60 );
}


void Kaiman::updateStatusDisplay()
{
    // update title
    KaimanStyleText* titleItem = static_cast<KaimanStyleText*>(_style->find("Title"));
    if ( titleItem ) {
        MediaFile *media = _mediaManager->active();
        if ( media )
            titleItem->setValue( media->title() );
        else
            titleItem->setValue( "Kaiman" );
    }

    // update status
    KaimanStyleState* stateItem = static_cast<KaimanStyleState*>(_style->find("Stereo_Item"));
    if ( stateItem ) stateItem->setValue( 0 );

    stateItem = static_cast<KaimanStyleState*>(_style->find("Shuffle_Item"));
    if ( stateItem ) stateItem->setValue( _shuffle ? 1 : 0 );

    KaimanStyleButton* butItem = static_cast<KaimanStyleButton*>(_style->find("Shuffle_Button"));
    if ( butItem ) butItem->setLit( _shuffle );

    stateItem = static_cast<KaimanStyleState*>(_style->find("Repeat_Item"));
    if ( stateItem ) stateItem->setValue( _looping ? 1 : 0 );

    butItem = static_cast<KaimanStyleButton*>(_style->find("Repeat_Button"));
    if ( butItem ) butItem->setLit( _looping );

    stateItem = static_cast<KaimanStyleState*>(_style->find("Minus_Item"));
    if ( stateItem ) stateItem->setValue( 1 ); // counts up

    stateItem = static_cast<KaimanStyleState*>(_style->find("Total_Item"));
    if ( stateItem ) stateItem->setValue( 0 ); // current song

    KaimanStyleNumber *numItem = static_cast<KaimanStyleNumber*>(_style->find("Total_Number"));
    if ( numItem ) numItem->setValue( _mediaManager->count() );

    numItem = static_cast<KaimanStyleNumber*>(_style->find("Song_Number"));
    if ( numItem ) numItem->setValue( _mediaManager->position() );
}


void Kaiman::pollArts() {
    KaimanStyleButton* l_elem_pause =
            static_cast<KaimanStyleButton*>(_style->find("Pause_Button"));
    KaimanStyleButton* l_elem_play =
            static_cast<KaimanStyleButton*>(_style->find("Play_Button"));
    KaimanStyleState* status = static_cast<KaimanStyleState*>(_style->find("Status_Item"));

    if ( !_playObject.isNull() ) {
        // check play object state and update buttons
        int oldPolledState = _lastPolledState;
        _lastPolledState = _playObject.state();
        switch ( _lastPolledState ) {
            case Arts::posIdle:
                if ( l_elem_pause!=0 ) l_elem_pause->setLit(false);
                if ( l_elem_play!=0 ) l_elem_play->setLit(false);
                if ( status ) status->setValue( 0 );

                if ( oldPolledState==Arts::posPlaying ) {
                    if ( _mediaManager->active() )
                        next();
                }
                break;

            case Arts::posPaused:
                if ( l_elem_pause!=0 ) l_elem_pause->setLit(true);
                if ( status ) status->setValue( 1 );
                break;

            case Arts::posPlaying:
                if(l_elem_pause != 0) l_elem_pause->setLit(false);
                if(l_elem_play != 0) l_elem_play->setLit(true);
                if ( status ) status->setValue( 2 );
                break;
        }
    } else {
        if ( _lastPolledState!=Arts::posIdle ) stop();

        if ( l_elem_pause!=0 ) l_elem_pause->setLit(false);
        if ( l_elem_play!=0 ) l_elem_play->setLit(false);
        if ( status ) status->setValue( 0 );
    }

    updateArtsDisplay();
}

#include "kaiman.moc"
