/*
  wrapper for X11 Window
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "ditherWrapper.h"

/* 
   Flag for gamma correction 
   Makes images brighter/darker. 
   It's in the source but not activated (for now)
*/
int gammaCorrectFlag = 0;
double gammaCorrect = 1.0;

/* 
   Flag for chroma correction.
   reduce the color intensity..
   It's in the source but not activated (for now)
*/
int chromaCorrectFlag = 0;
double chromaCorrect = 1.0;



DitherWrapper::DitherWrapper(int bpp,unsigned int redMask,
			     unsigned int greenMask,unsigned int blueMask,
			     unsigned char pixel[256]) {

  this->bpp=bpp;
  this->redMask=redMask;
  this->greenMask=greenMask;
  this->blueMask=blueMask;


  dither8Bit=new Dither8Bit(pixel);
  dither16Bit=new Dither16Bit(redMask,greenMask,blueMask);
  dither32Bit=new Dither32Bit(redMask,greenMask,blueMask);
  ditherRGB_flipped=new DitherRGB_flipped();
  ditherRGB=new DitherRGB();
 
  
#ifdef INTEL
  lmmx=mm_support();
#else
  lmmx=false;
#endif


}


DitherWrapper::~DitherWrapper(){
  delete dither16Bit;
  delete dither8Bit;
  delete dither32Bit;
  delete ditherRGB_flipped;
  delete ditherRGB;
}





void  DitherWrapper::doDither(YUVPicture* pic,int depth,int ditherSize,
			      unsigned char* dest,int offset) {


  //
  // according to the input imageType and the output area
  // handle different dither methods
  //

  int inputType=pic->getImageType();

  if ( (inputType == PICTURE_YUVMODE_CR_CB) ||
       (inputType == PICTURE_YUVMODE_CB_CR) ) {
    doDitherYUV(pic,depth,ditherSize,dest,offset);
    return;
  }

  if ( (inputType == PICTURE_RGB) ||
       (inputType == PICTURE_RGB_FLIPPED) ){
    doDitherRGB(pic,depth,ditherSize,dest,offset);
    return;
  }
  
  cout << "unknown inputType:"<<inputType
       << " in DitherWrapper::doDither"<<endl;
}


void DitherWrapper::doDitherRGB(YUVPicture* pic,int depth,int ditherSize,
				unsigned char* dest,int offset) {
  
  int inputType=pic->getImageType();

  switch(inputType) {
  case PICTURE_RGB:
    doDitherRGB_NORMAL(pic,depth,ditherSize,dest,offset);
    break;
  case  PICTURE_RGB_FLIPPED:
    doDitherRGB_FLIPPED(pic,depth,ditherSize,dest,offset);
    break;
  default:
    cout << "unknown RGB type:"<<inputType<<" in DitherWrapper"<<endl;
    exit(0);
  }
}


void DitherWrapper::doDitherRGB_NORMAL(YUVPicture* pic,
				       int depth,int ditherSize,
				       unsigned char* dest,int offset) {
 
  int w=pic->getWidth();
  int h=pic->getHeight();

  unsigned char* src=pic->getImagePtr();

  switch (ditherSize) {
  case _SIZE_NORMAL: 
    ditherRGB->ditherRGBImage(dest,src,depth,w,h,offset);
    break;
  case _SIZE_DOUBLE:
    cout << "double not supported for RGB"<<endl;
    break;
  default:
    cout << "unknown size:"<<ditherSize<<" in DitherWrapper"<<endl;
    exit(0);
  }
}

void DitherWrapper::doDitherRGB_FLIPPED(YUVPicture* pic,
					int depth,int ditherSize,
					unsigned char* dest,int offset) {
 
  int w=pic->getWidth();
  int h=pic->getHeight();

  unsigned char* src=pic->getImagePtr();

  switch (ditherSize) {
  case _SIZE_NORMAL: 
  case _SIZE_DOUBLE:
    // flip images itsself
    ditherRGB_flipped->flipRGBImage(dest,src,depth,w,h,offset);
    // normal RGB dither
    break;
  default:
    cout << "unknown size:"<<ditherSize<<" in DitherWrapper"<<endl;
    exit(0);
  }
}



void DitherWrapper::doDitherYUV(YUVPicture* pic,int depth,int ditherSize,
				 unsigned char* dest,int offset) {

  switch (ditherSize) {
  case _SIZE_NORMAL:
    doDither_std(pic,depth,dest,offset);
    break;
  case _SIZE_DOUBLE:
    doDither_x2(pic,depth,dest,offset);
    break;
  default:
    cout << "unknown size:"<<ditherSize<<" in DitherWrapper"<<endl;
    exit(0);
  }
}


void DitherWrapper::doDither_std(YUVPicture* pic,int depth,
				 unsigned char* dest,int offset){
  
  int h=pic->getHeight();
  int w=pic->getWidth();
  unsigned char* lum=pic->getLuminancePtr();
  unsigned char* cr=pic->getCrPtr();
  unsigned char* cb=pic->getCbPtr();


  switch (depth) {
  case 8:
    dither8Bit->ditherImageOrdered(lum, cr, cb,dest , h, w);
    break;
  case 16:
    if (lmmx) {
      ditherBlock(lum,cr,cb,dest,h,w,offset);
    } else {
      dither16Bit->ditherImageColor16(lum,cr,cb,dest,h,w,offset);
    }

    break;
  case 24:
  case 32:
    if (lmmx) {
      dither32_mmx(lum, cr, cb,dest ,h,w,offset);
    } else {
      dither32Bit->ditherImageColor32(lum, cr, cb,dest ,h,w,offset);
    }


    break;
  default:
    cout << "cannot dither depth:"<<depth<<endl;
  }

}
  

void DitherWrapper::doDither_x2(YUVPicture* pic,int depth,
				unsigned char* dest,int offset){

  int h=pic->getHeight();
  int w=pic->getWidth();
  unsigned char* lum=pic->getLuminancePtr();
  unsigned char* cr=pic->getCrPtr();
  unsigned char* cb=pic->getCbPtr();


  switch (depth) {
  case 16:
    dither16Bit->ditherImageTwox2Color16(lum,cr,cb,dest,h,w,offset);
    break;
  case 24:
  case 32:
    if (lmmx) {
      //dither32x2_mmx(lum, cr, cb,dest ,h,w,offset);
      dither32Bit->ditherImageTwox2Color32(lum,cr,cb,dest,h,w,offset);
    } else {
      dither32Bit->ditherImageTwox2Color32(lum,cr,cb,dest,h,w,offset);
    }
    break;
  default:
    cout << "cannot dither depth:" << depth << endl;
  }
}
