/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: cmdpipe.c
 *
 * Description: Generic operations for manipulating commandline
 *              tools via pipes.  Replacement for libc popen() /
 *              pclose() tools.
 */

/*
 * WARNING! These functions are not entirely secure quite yet, nor
 *          are they POSIX compliant.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/types.h>

#include "cmdpipe.h"

struct cmdpipe_info {
    pid_t childpid;    /* child's process id */
    FILE *file[3];     /* parent's files */
};

/* cmdpipeopen() - Open up pipes to a shell command.
 *
 * The child process executes in the current process's environment.
 * It may inherit things that we don't want it to, and it also relies
 * on the shell to parse arguments and locate the program in the
 * path.  It could be possible for a user to modify which program is
 * launched simply by writing a script in the current command path.
 * It is therefore safer if the calling application provides an
 * absolute path.
 *
 * Synopsis:
 *    void * cmdpipeopen(
 *        char *shellcommand, passed to system shell (sh)
 *        FILE **newstdin,    if NULL, uses stdin
 *        FILE **newstdout,   if NULL, uses stdout
 *        FILE **newstderr,   if NULL, uses stderr
 *                            may equal newstdout
 *
 * If unable to open certain streams, will return success but
 * some streams will be set to NULL.
 * The application should detect this condition and decide whether
 * to terminate the child program or not.
 */
void *cmdpipeopen(const char *shellcommand,
    FILE **newstdin, FILE **newstdout, FILE **newstderr)
{
    volatile int pipefd_in[2]  = {-1, -1}; /* redirect stdin  */
    volatile int pipefd_out[2] = {-1, -1}; /* redirect stdout */
    volatile int pipefd_err[2] = {-1, -1}; /* redirect stderr */
    int erronstdout = 0;   /* stderr on stdout? */
    pid_t childpid;    /* child's process id */

    struct cmdpipe_info *p;
    int t;

    /* Allocate memory */
    p = malloc(sizeof(struct cmdpipe_info));
    if (! p) return NULL;

    /* Setup pipes */
    t = 1;
    if (newstdin) t = (pipe((int *)pipefd_in) >= 0);
    if (t) {
        if (newstdout) t = (pipe((int *)pipefd_out) >= 0);
        if (t) {
            if (newstderr) {
                erronstdout = (newstderr == newstdout);
                if (! erronstdout) t = (pipe((int *)pipefd_err) >= 0);
            }
            if (t) {
                /* Launch */
                childpid = fork();
                if (childpid == 0) {
                    /*** In child ***/
                    /* redirect stdin */
                    if (pipefd_in[0] >= 0) { /* not parent's stdin? */
                        dup2(pipefd_in[0], 0);
                        close(pipefd_in[1]);
                    }
                    /* redirect stdout */
                    if (pipefd_out[1] >= 0) { /* not parent's stdout? */
                        dup2(pipefd_out[1], 1);
                        close(pipefd_out[0]);
                    }
                    /* redirect stderr */
                    if (erronstdout) {
                        dup2(1, 2); /* put errors on stdout */
                    } else if (pipefd_err[1] >= 0) { /* not parent's stderr? */
                        dup2(pipefd_err[1], 2);
                        close(pipefd_err[0]);
                    }
                    /* execute command... */
                    execl("/bin/sh", "sh", "-c", shellcommand, NULL);
                    /* if all went well, we won't get here... */
                    exit(127);
                }
                /*** In parent ***/
                if (childpid > 0) {
                    p->childpid = childpid;
                    goto makestreams; /* get out of this totem-pole! */
                }
            }
            if (pipefd_err[0] >= 0) close(pipefd_err[0]);
            if (pipefd_err[1] >= 0) close(pipefd_err[1]);
        }
        if (pipefd_out[0] >= 0) close(pipefd_out[0]);
        if (pipefd_out[1] >= 0) close(pipefd_out[1]);
    }
    if (pipefd_in[0] >= 0) close(pipefd_in[0]);
    if (pipefd_in[1] >= 0) close(pipefd_in[1]);
    free(p);
    return NULL;

    /* Child launched okay... setup file streams */
    /* goto is cleaner than the alternative... */
makestreams:
    /* setup stdin stream */
    if (pipefd_in[0] >= 0) {
        *newstdin = p->file[0] = fdopen(pipefd_in[1], "w");
        t = t && (*newstdin != NULL);
        close(pipefd_in[0]);
    } else p->file[0] = NULL;
    /* setup stdout stream */
    if (pipefd_out[1] >= 0) {
        *newstdout = p->file[1] = fdopen(pipefd_out[0], "r");
        t = t && (*newstdout != NULL);
        close(pipefd_out[1]);
    } else p->file[1] = NULL;
    /* setup stderr stream */
    if (pipefd_err[1] >= 0) {
        *newstderr = p->file[2] = fdopen(pipefd_err[0], "r");
        t = t && (newstderr != NULL);
        close(pipefd_err[1]);
    } else p->file[2] = NULL;

    if (t) return (void *)p; /* success! */

    /* otherwise we have a dilemma... the child has started,
     * but we don't have a complete set of file streams...
     *
     * Rather than risk killing the application, we'll just
     * return these streams to the application for it to deal
     * with.
     */
    return (void *)p;
}

/* cmdpipeclose() - Wait for the child to exit, with timeout.
 *
 * Waits for a process started by cmdpipeopen() to exit.
 * Returns the process return code on success.
 * Returns -2 if the timer expired.
 * Returns -1 if the process returned nothing.
 *
 * Synopsis:
 *    int cmdpipeclose(
 *        void *pipehandle        struct returned by cmdpipeopen()
 *        int   timeout_secs      timeout in seconds (0 = indefinite)
 *        );
 */
int cmdpipeclose(void *pipehandle, int timeout_secs)
{
    struct cmdpipe_info *p;
    int   status;
    pid_t pid;

    /* Check params */
    p = (struct cmdpipe_info *) pipehandle;
    if (! p) return -1;

    /* Wait for process to die */
    /* block? */
    if (timeout_secs <= 0) {
        pid = waitpid(p->childpid, &status, 0);
    } else {
        /* simple polling mechanism for now...
         * should be replaced with an alarm / timer
         */
        for (;;) {
            pid = waitpid(p->childpid, &status, WNOHANG);
            if (pid != 0) break;
            if (timeout_secs-- == 0) return -2;
            sleep(1);
        }
    }
    /* May get here on success or error...
     * assume child has exited, or is no longer available.
     */

    /* Close files */
    if (p->file[0]) fclose(p->file[0]);
    if (p->file[1]) fclose(p->file[1]);
    if (p->file[2]) fclose(p->file[2]);

    /* Free struct */
    free(p);
    return (pid == -1) ? -1 : WEXITSTATUS(status);
}

/* cmdpipegetpid() - Get PID of a child program
 *
 * Simply returns process id of child.
 *
 * Synopsis:
 *    pid_t cmdpipegetpid(
 *        void *pipehandle        struct returned by cmdpipeopen()
 *        );
 */
pid_t cmdpipegetpid(void *pipehandle)
{
    struct cmdpipe_info *p;
    p = (struct cmdpipe_info *) pipehandle;

    return (p) ? p->childpid : -1;
}
