/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: debug.h
 *
 * Description: Debugging tools (see internal documentation)
 *
 */

#ifndef DEBUG_H
#define DEBUG_H

/* If debugging enabled...
 */
#if (APSCFG_DEBUG)

/* Import the array of result identifier strings */
extern const char *DebugResultCodeNames[];

/* ASSERT: run-time assertions */
#include <assert.h>
#define ASSERT(x) assert(x)

/* RESULTSTR: get name of an Aps_Result constant */
#define RESULTSTR(result) \
    (((result >= APS_SUCCESS) && (result < APS_NUMRESULTS)) ? \
    DebugResultCodeNames[result - APS_SUCCESS] : "unknown") \

/* Else, debugging disabled...
 */
#else
#define ASSERT(x)
#define RESULTSTR(x) "???"

#endif /* APSCFG_DEBUG */

/* If run-time debug messages enabled...
 */
#if (APSCFG_DEBUG && APSCFG_DEBUG_TRACE)

/* Import the array of channel enable flags.
 * Each channel is represented by an index into this array, and
 * each class is assigned a single bit in each channel's storage byte.
 *
 * If the bit is 1, then the message class for that channel is enabled.
 */
extern unsigned char DebugChannelFlags[];
extern int DebugChannelsInvalid;

/* Import functions we need */
extern int DebugPrintHeader(int channelIndex, int classIndex,
    const char *locFunction, const char *locFile, int locLine);
extern void DebugPrintMessage(const char *fmt, ...);

/* Each message class flag is bitmapped according to the enumeration
 * below.  Bits are assigned from right to left such that
 * DEBUG_CLASS_BIT_msg = 1<<DEBUG_CLASS_INDEX_msg.
 */
enum {
    DEBUG_CLASS_INDEX_msg = 0,
    DEBUG_CLASS_INDEX_warn,
    DEBUG_CLASS_INDEX_error,
    DEBUG_CLASS_INDEX_fixme,
    DEBUG_CLASS_INDEX_trace,
    DEBUG_CLASS_INDEX_all,
    DEBUG_CLASS_INDEX_END
};

/* List the names of each message class.  Each entry is 8 characters long
 * to permit future expansion and ease indexing.
 */
#define DEBUG_CLASS_NAMES ( /* display names case insensitive */ \
    "MSG     " \
    "WARN    " \
    "ERROR   " \
    "FIXME   " \
    "TRACE   " \
    "ALL     " )
#define DEBUG_CLASS_NAME_SIZE 8

/* Enumerate channels by id # and list corresponding names...
 * Each name is 8 characters long so we can simply index things by
 * multiplying by 8 with the stipulation that the names be padded by
 * spaces.  This speeds name formatting, but slows name extraction
 * just a tad.
 * Note: We will not define these things if tracing is not enabled as
 *       anything that uses these values should be compiled out.  Doing
 *       this lets us locate places where debug code may not be compiled
 *       out correctly...
 *
 * Suggestion: Keeping these in alphabetical order will make the list
 *             easier to maintain.
 */
/* These enum values will only be used directly by debugging macros below */
enum { /* must be lowercase */
    DEBUG_CHANNEL_INDEX_all = 0,
    DEBUG_CHANNEL_INDEX_api,
    DEBUG_CHANNEL_INDEX_attr,
    DEBUG_CHANNEL_INDEX_attrprov,
    DEBUG_CHANNEL_INDEX_cfg,
    DEBUG_CHANNEL_INDEX_cups,
    DEBUG_CHANNEL_INDEX_filter,
    DEBUG_CHANNEL_INDEX_lpr,
    DEBUG_CHANNEL_INDEX_job,
    DEBUG_CHANNEL_INDEX_mem,
    DEBUG_CHANNEL_INDEX_model,
    DEBUG_CHANNEL_INDEX_object,
    DEBUG_CHANNEL_INDEX_printer,
    DEBUG_CHANNEL_INDEX_printcap,
    DEBUG_CHANNEL_INDEX_queue,
    DEBUG_CHANNEL_INDEX_transprt,
    DEBUG_CHANNEL_INDEX_utils,
    DEBUG_CHANNEL_INDEX_END
};

/* Index size for names */
#define DEBUG_CHANNEL_NAME_SIZE 8

/* Total name string size, use when declaring a constant instance of the
 * string for error checking -- it'll catch some problems arising from the
 * lists being out of sequence */
#define DEBUG_CHANNEL_NAME_TOTSIZE \
    (DEBUG_CHANNEL_NAME_SIZE * DEBUG_CHANNEL_INDEX_END + 1)

/* These names must all be exactly DEBUG_CHANNEL_NAME_SIZE chars
 * long and must be padded with spaces.  Their order must also be the
 * same as the list of enumerated constants above */
#define DEBUG_CHANNEL_NAMES ( /* display names case insensitive */ \
    "ALL     " \
    "API     " \
    "ATTR    " \
    "ATTRPROV" \
    "CFG     " \
    "CUPS    " \
    "FILTER  " \
    "LPR     " \
    "JOB     " \
    "MEM     " \
    "MODEL   " \
    "OBJECT  " \
    "PRINTER " \
    "PRINTCAP" \
    "QUEUE   " \
    "TRANSPRT" \
    "UTILS   " )

/*** DEBUG_CHANNEL : Declare a channel that will be used in this source file
 * Makes the specified channel available for use for debug message
 * output.
 */
#define DEBUG_CHANNEL(channel) \
  static const int DebugChannelIndex_##channel = \
      DEBUG_CHANNEL_INDEX_##channel;

/*** DEBUG_CHANNEL_DEFAULT : Declare default channel for this source file
 * Declares a particular channel as being default for debug message
 * output.  This also makes channel "default" usable in TRACE_(channel)-style
 * macros.
 */
#define DEBUG_CHANNEL_DEFAULT(channel) \
  DEBUG_CHANNEL(channel) \
  static const int DebugChannelIndex_default = \
      DEBUG_CHANNEL_INDEX_##channel ;

/*** DEBUG_PRINT : Print out a message
 * Calls the appropriate functions in debug.c to generate a trace message
 */
#define DEBUG_PRINT(classIndex, channelIndex) \
  if (DEBUG_STATE(classIndex, channelIndex)) \
    if (DebugPrintHeader(channelIndex, classIndex, __FUNCTION__, \
      __FILE__, __LINE__)) DebugPrintMessage

/*** DEBUG_STATE : Get status of a debug channel and message class
 * Returns TRUE if that channel/class is enabled.
 */
#define DEBUG_STATE(classIndex, channelIndex) \
    ((DebugChannelFlags[channelIndex] & (1<<classIndex)) || \
     DebugChannelsInvalid)

/* Else, all run-time debug messages disabled...
 */
#else
#define DEBUG_CHANNEL_DEFAULT(channel)
#define DEBUG_CHANNEL(channel)
#endif /* APSCFG_DEBUG_TRACE */

/* Dummy printf
 * this causes the trailing parameter list to be optimized right out!
 */
#define DEBUG_PRINT_DUMMY \
    1 ? (void)0 : (void)((int (*)(char *, ...))NULL)

/* Conditionally enable particular message types
 */

/*** MSG(fmt, ...)           : generate message on default channel
 *** MSG_(channel)(fmt, ...) : generate message on specified channel
 *** MSG_ON_(channel)        : returns TRUE if output enabled
 * Ditto for all the other types...
 *
 * This looks messy but short of having recursively expanding macros...
 */
#if ((APSCFG_DEBUG) && (APSCFG_DEBUG_TRACE & 1))
#define MSG \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_msg, DebugChannelIndex_default)
#define MSG_(channel) \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_msg, DebugChannelIndex_##channel)
#define MSG_ON_(channel) \
    DEBUG_STATE(DEBUG_CLASS_INDEX_msg, DebugChannelIndex_##channel)
#else
#define MSG DEBUG_PRINT_DUMMY
#define MSG_(channel) DEBUG_PRINT_DUMMY
#define MSG_ON_(channel) 0
#endif

#if ((APSCFG_DEBUG) && (APSCFG_DEBUG_TRACE & 2))
#define WARN \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_warn, DebugChannelIndex_default)
#define WARN_(channel) \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_warn, DebugChannelIndex_##channel)
#define WARN_ON_(channel) \
    DEBUG_STATE(DEBUG_CLASS_INDEX_warn, DebugChannelIndex_##channel)
#else
#define WARN DEBUG_PRINT_DUMMY
#define WARN_(channel) DEBUG_PRINT_DUMMY
#define WARN_ON_(channel) 0
#endif

#if ((APSCFG_DEBUG) && (APSCFG_DEBUG_TRACE & 4))
#define ERROR \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_error, DebugChannelIndex_default)
#define ERROR_(channel) \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_error, DebugChannelIndex_##channel)
#define ERROR_ON_(channel) \
    DEBUG_STATE(DEBUG_CLASS_INDEX_error, DebugChannelIndex_##channel)
#else
#define ERROR DEBUG_PRINT_DUMMY
#define ERROR_(channel) DEBUG_PRINT_DUMMY
#define ERROR_ON_(channel) 0
#endif

#if ((APSCFG_DEBUG) && (APSCFG_DEBUG_TRACE & 8))
#define FIXME \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_fixme, DebugChannelIndex_default)
#define FIXME_(channel) \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_fixme, DebugChannelIndex_##channel)
#define FIXME_ON_(channel) \
    DEBUG_STATE(DEBUG_CLASS_INDEX_fixme, DebugChannelIndex_##channel)
#else
#define FIXME DEBUG_PRINT_DUMMY
#define FIXME_(channel) DEBUG_PRINT_DUMMY
#define FIXME_ON_(channel) 0
#endif

#if ((APSCFG_DEBUG) && (APSCFG_DEBUG_TRACE & 16))
#define TRACE \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_trace, DebugChannelIndex_default)
#define TRACE_(channel) \
    DEBUG_PRINT(DEBUG_CLASS_INDEX_trace, DebugChannelIndex_##channel)
#define TRACE_ON_(channel) \
    DEBUG_STATE(DEBUG_CLASS_INDEX_trace, DebugChannelIndex_##channel)
#else
#define TRACE DEBUG_PRINT_DUMMY
#define TRACE_(channel) DEBUG_PRINT_DUMMY
#define TRACE_ON_(channel) 0
#endif

#endif /* !APSINTERNAL_H */
