/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: filter.c
 *
 * Description: APIs for composing and applying filters to objects.
 *              Used by queues and jobs; perhaps others later.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "filter.h"

DEBUG_CHANNEL_DEFAULT(filter)

/* Local protos */
static void FilterDeleteLinks(Filter *filter);
static Aps_Result FilterScanRecursive(Filter *haystack,
    Aps_FilterHandle needle);

/* ---------------------------------------------------------------------------
 * FilterGetPtrFromHandle()
 *
 * Obtains a pointer to a Filter object, given an Aps_FilterHandle.
 *
 * Parameters: handle - The handle for which the corresponding Filter object
 *                      should be obtained.
 *
 *     Return: A pointer to a Filter object, or NULL on failure.
 */
Filter *FilterGetPtrFromHandle(Aps_FilterHandle handle)
{
    Filter *filter;

    if (handle == NULL)
        return (NULL);

    filter = (Filter *) handle;

    /* Check that this is actually a filter object, and not something else. */
    if (filter->baseClass.identifier != FILTER_HANDLE)
        return (NULL);

    return (filter);
}

/* ---------------------------------------------------------------------------
 * FilterGetHandleFromPtr()
 *
 * Obtain the handle for a specified Filter object.
 *
 * Parameters: filter - The Filter for which the corresponding handle should be
 *                     obtained.
 *
 *     Return: An Aps_FilterHandle.
 */
Aps_FilterHandle FilterGetHandleFromPtr(Filter * filter)
{
    ASSERT(filter != NULL);

    return ((Aps_FilterHandle) filter);
}

/* --------------------------------------------------------------------
 * FilterLink()
 *
 * Link a filter qualifier into a filter object
 *
 * Parameters : filterHandle - handle of filter in which to link
 *              filterQual   - filter qualifier structure (may be NULL)
 * Result     : (inparam) filterHandle - new filter handle, or NULL on failure
 *
 * Creates a filter object if none exists at the time
 */
Aps_Result FilterLink(Aps_FilterHandle *filterHandle, FilterQual *filterQual)
{
    Aps_Result  result;
    Filter     *filter;

    /* Check params */
    ASSERT(filterHandle);

    /* If no filter, create one first */
    if (! *filterHandle) {
        /* alloc object */
        filter = malloc(sizeof(Filter));
        if (! filter) return APS_OUT_OF_MEMORY;
        /* setup baseclass */
        result = ObjectInitialize(& filter->baseClass, FILTER_HANDLE);
        if (result == APS_SUCCESS) {
            /* setup handle */
            *filterHandle = FilterGetHandleFromPtr(filter);
            ASSERT(*filterHandle);

            /* setup other internal fields */
            filter->tail = NULL;
        } else { /* failed init */
            free(filter);
            return result;
        }
    } else {
        /* Else locate object */
        filter = FilterGetPtrFromHandle(*filterHandle);
        if (! filter) return APS_INVALID_HANDLE;
    }

    /* Link into the filter if filterQual specified */
    if (filterQual) {
        if (filter->tail) {
            /* insert node into circularly linked list */
            filterQual->next = filter->tail->next;
            filter->tail->next = filterQual;
        } else {
            /* single node */
            filterQual->next = filterQual;
        }
        filter->tail = filterQual;
    }
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * FilterDeleteLinks()
 *
 * Destroy a filter's linked list
 *
 * Parameters : filter - pointer to filter object
 * Result     : none
 *
 * Deletes all elements of linked list and frees all associated resources.
 */
void FilterDeleteLinks(Filter *filter)
{
    ASSERT(filter);

    /* Scan list and delete all elements */
    if (filter->tail) {
        FilterQual *seek = filter->tail->next;
        /* mark end of list */
        filter->tail->next = NULL;
        for (; seek; seek = seek->next) {
            switch (seek->filterType) {
                case FILTER_TYPE_NONE:
                case FILTER_TYPE_ALL:
                case FILTER_TYPE_SPECIAL:
                case FILTER_TYPE_FUNCTION:
                    /* all associated data is grouped with structure */
                    break;
                case FILTER_TYPE_EMBED:
                    /* release embedded handle */
                    Aps_ReleaseHandle( ((Aps_FilterHandle)seek->data) );
                    break;
            }
        }
    }
}

/* --------------------------------------------------------------------
 * FilterAllocQual()
 *
 * Create/init a qualifier structure and allocate space for auxiliary
 * data.
 *
 * Parameters : filterType    - type of qualifier (stored in struct)
 *              filterSpecial - special field (stored in struct)
 *              filterMode    - special modes (stored in struct)
 *              sizeAuxData   - size of auxiliary data space
 * Result     : returns initialized structure or NULL (out of memory)
 */
FilterQual *FilterAllocQual(FilterType filterType,
    FilterSpecial filterSpecial, Aps_FilterMode filterMode,
    size_t sizeAuxData)
{
    FilterQual *qual;
    /* allocate structure */
    qual = calloc(1, sizeof(FilterQual) + sizeAuxData);
    if (qual) {
        /* initialize structure */
        qual->next = NULL;
        qual->filterType = filterType;
        qual->filterSpecial = filterSpecial;
        qual->filterMode = filterMode;
        qual->data = (void *)(((char *)qual) + sizeof(FilterQual));
    }
    return qual;
}

/* --------------------------------------------------------------------
 * FilterDelete()
 *
 * Destroy a Filter object (low-level)
 *
 * Parameters : filterHandle - handle of filter to delete
 * Result     : (inparam) filterHandle - new filter handle, or NULL on failure
 *
 * The returned handle must be freed by Aps_ReleaseHandle().
 * On failure *filterHandle set to NULL and detailed result returned.
 *
 * Deletes all elements of linked list and frees all associated resources.
 */
void FilterDelete(Filter *filter)
{
    /* Free associated resources */
    FilterDeleteLinks(filter);

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    filter->baseClass.identifier = INVALID_HANDLE;
    free(filter);
}

/* --------------------------------------------------------------------
 * Aps_AttachFilter()
 *
 * Attach a Filter to an object. By default, many actions of the
 * target object will be affected by this filter.
 *
 * Parameters : objHandle    - handle of object for which to attach filter
 *              filterHandle - handle of filter to attach
 *              filterOpts   - options for filter comparisons
 *
 * Result     : Standard APS result code.
 */
Aps_Result Aps_AttachFilter(Aps_Handle objHandle,
    Aps_FilterHandle filterHandle, Aps_FilterOptions filterOpts)
{
    ApsObject *obj;
    Filter    *filter;
    Aps_Result result;

    /* Check params */
    obj = ObjectGetPtrFromHandle(objHandle);
    if (! obj) return APS_INVALID_HANDLE;

    filter = FilterGetPtrFromHandle(filterHandle);
    if (! filter) return APS_INVALID_HANDLE;

    /* Detach old filter */
    if (obj->filterHandle) Aps_ReleaseHandle(obj->filterHandle);
    obj->filterHandle = NULL;
    obj->filterOpts = 0;

    /* Attach new filter */
    if (filterHandle) {
        /* Attach new filter */
        result = Aps_AddRef(filterHandle);
        if (result != APS_SUCCESS) return result;
        obj->filterHandle = filterHandle;
        obj->filterOpts = filterOpts;
    }
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * Aps_DetachFilter()
 *
 * Detach a Filter from an object. Future operations on this object will
 * no longer be affected by the filter.
 *
 * Parameters : objHandle    - handle of object for which to detach filter
 *
 * Result     : filterHandle - old filter handle
 *              filterOpts   - old filter options
 */
Aps_Result Aps_DetachFilter(Aps_Handle objHandle,
    Aps_FilterHandle *oldFilterHandle, Aps_FilterOptions *oldFilterOpts)
{
    ApsObject *obj;

    /* Check params */
    obj = ObjectGetPtrFromHandle(objHandle);
    if (! obj) return APS_INVALID_HANDLE;

    /* Detach old filter */
    if (obj->filterHandle) {
        Aps_ReleaseHandle(obj->filterHandle);
    } else obj->filterOpts = 0; /* for consistent results */

    /* Return old values */
    if (oldFilterHandle) *oldFilterHandle = obj->filterHandle;
    if (oldFilterOpts) *oldFilterOpts = obj->filterOpts;

    /* Cleanup */
    obj->filterHandle = NULL;
    obj->filterOpts = 0;
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * Aps_FilterClear()
 *
 * Clear an existing filter definition.  (Creates a blank one if none).
 *
 * Parameters : filterHandle - pointer to handle of filter to modify
 * Result     : Standard APS result code.
 */
Aps_Result Aps_FilterClear(Aps_FilterHandle *filterHandle)
{
    /* Check params */
    if (! filterHandle) return APS_INVALID_PARAM;

    /* Must we clear filter definition? */
    if (*filterHandle) {
        Filter *filter = FilterGetPtrFromHandle(*filterHandle);
        if (! filter) return APS_INVALID_HANDLE;

        FilterDeleteLinks(filter);
        return APS_SUCCESS;
    /* Else, we will create a new blank filter */
    } else {
        return FilterLink(filterHandle, NULL);
    }
}

/* --------------------------------------------------------------------
 * Aps_FilterWithFunction()
 *
 * Create a filter definition from a user callback function.
 *
 * Parameters: filterHandle  - pointer to handle storage location, if not
 *                             NULL, filter is appended to old one
 *             filterMode    - one of Aps_FilterMode modes
 *             matchFunction - user callback function (should be thread-safe)
 *             userArg       - passed to callback function
 *
 *     Result: A standard APS result code indicating the cause of failure.
 */
Aps_Result Aps_FilterWithFunction(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, Aps_FilterFunc matchFunction,
    void *userArg)
{
    Aps_Result  result;
    FilterQual *qual;

    /* Check params */
    if (! matchFunction) return APS_INVALID_PARAM;
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_FUNCTION, FILTER_SPECIAL_NONE,
        filterMode, sizeof( FilterData_Function));
    if (! qual) return APS_OUT_OF_MEMORY;

    /* Fill in data */
    ((FilterData_Function *)qual->data)->arg = userArg;
    ((FilterData_Function *)qual->data)->func = matchFunction;

    /* Associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}

/* --------------------------------------------------------------------
 * Aps_FilterAll()
 *
 * Create a filter definition that matches everything.
 *
 * Parameters: filterHandle - pointer to handle storage location, if not
 *                            NULL, filter is appended to old one
 *             filterMode   - one of Aps_FilterMode modes
 *
 *     Result: A standard APS result code indicating the cause of failure.
 */
Aps_Result Aps_FilterAll(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode)
{
    Aps_Result  result;
    FilterQual *qual;

    /* Check params */
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_ALL, FILTER_SPECIAL_NONE,
        filterMode, 0);
    if (! qual) return APS_OUT_OF_MEMORY;

    /* Associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}

/* --------------------------------------------------------------------
 * Aps_FilterMerge()
 *
 * Create a filter definition by embedding some other filter into the
 * current one.
 *
 * Parameters: filterHandle - pointer to handle storage location, if not
 *                            NULL, filter is appended to old one
 *             filterMode   - one of Aps_FilterMode modes
 *             srcHandle    - handle of filter to be embedded
 *
 *     Result: A standard APS result code indicating the cause of failure.
 *
 * Checks for recursive definitions.
 */
Aps_Result Aps_FilterMerge(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, Aps_FilterHandle srcHandle)
{
    Aps_Result  result;
    FilterQual *qual;
    Filter     *srcFilter;

    /* Check params */
    if (! srcHandle) return APS_SUCCESS; /* ignored */
    if (! filterHandle) return APS_INVALID_PARAM;
    srcFilter = FilterGetPtrFromHandle(srcHandle);
    if (! srcFilter) return APS_INVALID_HANDLE;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Search for recursive definition */
    if (*filterHandle) {
        Filter *filter;

        /* check most basic type of error: passing same handle */
        if (srcHandle == *filterHandle) return APS_INVALID_PARAM;

        /* get handle */
        filter = FilterGetPtrFromHandle(*filterHandle);
        if (! filter) return APS_INVALID_HANDLE;
        /* scan recursively */
        result = FilterScanRecursive(srcFilter, *filterHandle);
        if (result != APS_SUCCESS) return result;
    }

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_EMBED, FILTER_SPECIAL_NONE,
        filterMode, sizeof(Aps_FilterHandle));
    if (! qual) return APS_OUT_OF_MEMORY;

    /* Lock handle and store */
    result = Aps_AddRef(srcHandle);
    if (result == APS_SUCCESS) {
        *((Aps_FilterHandle *)qual->data) = srcHandle;
        /* Associate with filter */
        result = FilterLink(filterHandle, qual);
    } else free(qual);
    return result;
}

/* --------------------------------------------------------------------
 * FilterScanRecursive()
 *
 * Scan a filter recursively for some other filter,
 *
 * Parameters : haystack - pointer to filter object to search in
 *              needle   - handle to search for
 * Result     : APS_INVALID_PARAM if found, else APS_SUCCESS.
 */
Aps_Result FilterScanRecursive(Filter *haystack, Aps_FilterHandle needle)
{
    FilterQual *qual;

    /* check params */
    if (! haystack) return APS_GENERIC_FAILURE;
    if (! haystack->tail) return APS_SUCCESS;

    /* Check object using filter def'n */
    /* iterate through list */
    qual = haystack->tail->next;
    do {
        /* check embedded handles */
        if (qual->filterType == FILTER_TYPE_EMBED) {
            Aps_FilterHandle filterHandle = *((Aps_FilterHandle *)qual->data);
            Aps_Result result;

            if (filterHandle == needle) return APS_INVALID_PARAM;
            /* recurse */
            result = FilterScanRecursive(FilterGetPtrFromHandle(filterHandle),
                needle);
            if (result != APS_SUCCESS) return result;
        }
        qual = qual->next;
    } while (qual != haystack->tail->next);
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * Aps_FilterCheck()
 *
 * Run an object through a filter to determine if it is a match or not.
 *
 * Parameters : filterHandle - handle of filter to use
 *              filterOpts   - filtering options
 *              objHandle    - handle of object to check
 *
 * Result     : matchesFilter - set to TRUE if match, else FALSE.
 */
Aps_Result Aps_FilterCheck(Aps_FilterHandle filterHandle,
    Aps_FilterOptions filterOpts, Aps_Handle objHandle,
    int *matchesFilter)
{
    Filter        *filter;
    Aps_Result     result = APS_SUCCESS;
    int            match = TRUE;
    Aps_FilterOptions cleanOpts = filterOpts &
        (~ (APS_FILTER_INVERT | APS_FILTER_FALLBACK)); /* remove irrelevant */

    /* Check params */
    if ((! matchesFilter)||(! filterHandle)) return APS_INVALID_PARAM;
    if (filterHandle) { /* not blank filter */
        filter = FilterGetPtrFromHandle(filterHandle);
        if (! filter) return APS_INVALID_HANDLE;
    }
    if ((filter) && (filter->tail)) {
        FilterQual *qual;

        /* Check object using filter def'n */
        /* iterate through list */
        qual = filter->tail->next;
        do {
            Aps_FilterMode mode;
            /* dispatch filter checks */
            switch (qual->filterType) {
                /* matches nothing */
                case FILTER_TYPE_NONE:
                    match = FALSE;
                    break;
                /* matches everything */
                case FILTER_TYPE_ALL:
                    break;
                /* matches by embedded filter -- recursive */
                case FILTER_TYPE_EMBED:
                    ASSERT(qual->data);
                    result = Aps_FilterCheck(*((Aps_FilterHandle *)qual->data),
                        cleanOpts, objHandle, & match);
                    break;
                /* matches by user callback function */
                case FILTER_TYPE_FUNCTION:
                    ASSERT(qual->data);
                    match = ((FilterData_Function *)qual->data)->func(
                        objHandle, ((FilterData_Function *)qual->data)->arg);
                    break;
                /* matches by special field */
                case FILTER_TYPE_SPECIAL:
                    result = FilterCheckSpecial(qual, objHandle, & match);
                    break;
            }
            /* handle filter modes */
            /* we also set the default match mode to respect the
             * rules if last */
            mode = qual->filterMode & APS_FILTER_MODE_MASK;
            if (mode == APS_FILTER_INCLUDE) { /* OR */
                if (match) break;
                match = FALSE; /* if we fall-through, we have failed */
            } else if (mode == APS_FILTER_REQUIRE) { /* AND */
                if (! match) break;
                /*match = TRUE;*/ /* if we fall-through, we have succeeded */
            } else if (mode == APS_FILTER_EXCLUDE) { /* AND NOT */
                match = ! match;
                if (! match) break;
                /*match = TRUE;*/ /* if we fall-through, we have succeeded */
            } else { /* ERROR! */
                result = APS_GENERIC_FAILURE;
                match = FALSE;
                break;
            }
        /* goto next */
        qual = qual->next;
        } while (qual != filter->tail->next);
    }

    /* Set match according to options */
    *matchesFilter = (filterOpts & APS_FILTER_INVERT) ?
        (! match) : match;
    return result;
}
