/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: job.h
 *
 * Description: Definition of the job structure. There is one instance
 *              of this structure for each active job.
 *
 */

#ifndef JOB_H
#define JOB_H

typedef struct Job_ Job;

#include "aps.h"
#include "apsinternal.h"
#include "printer.h"
#include "jobattributes.h"
#include "object.h"
#include "transport.h"

/* Job object:
 *
 * Stores the context of a job so that operations may be performed.
 * Unless the Job is currently open (and even then, in some cases),
 * this structure may become out of date and it is necessary for it
 * to be partially regenerated by the transport.
 *
 * To assist APS queue operations, this structure uses Aps_QuickJobInfo
 * as the basis of most of its internal storage.  If the user requests
 * information about the job, a copy is made from this structure rather
 * than passing it directly.
 */
struct Job_ {
    ApsObject baseClass;

/*** This stuff is for internal bookkeeping only */
    void *transportDefaultData; /* for the transport's use */
    void *transportData;        /* for the transport's use */

    Transport  *transport; /* filled in when first open or queried */
    Printer    *printer;   /* filled in when first open or queried */

/*** This stuff looks like what we export to the application ***/
    Aps_QuickJobInfo *info;
};

/* Object handle operations */
Job *JobGetPtrFromHandle(Aps_JobHandle handle);
Aps_JobHandle JobGetHandleFromPtr(Job * job);

/* Object creation / disposal */
Aps_Result JobCreate(Aps_PrinterHandle printerHandle,
    Aps_JobHandle *jobHandle);
Aps_Result JobSetup(Job *job, const char *format, const char *name,
    const char *filename);
void JobDelete(Job * job);

/* IsOperationAvailable support */
Aps_Result JobIsOperationAvailable(Job *job,
    Aps_OperationID operation, Aps_Result *anticipatedResult);

/* Job properties */
Aps_Result JobSetAttributes(Job *job, Aps_JobAttrHandle attrHandle);

/* Working on active jobs */
Aps_Result JobUpdate(Job *job);

/* Manipulations for managing QuickJobInfo's */
Aps_Result QuickJobInfoInit(Aps_QuickJobInfo *info, int version);
Aps_Result QuickJobInfoCleanup(Aps_QuickJobInfo *info);
Aps_Result QuickJobInfoRecycle(Aps_QuickJobInfo *info);
Aps_Result QuickJobInfoCreatePackage(
    Aps_QuickJobInfo **dest, const Aps_QuickJobInfo *src);
int QuickJobInfoComputePackageSize(const Aps_QuickJobInfo *src);
void QuickJobInfoFillInPackage(Aps_QuickJobInfo *dest, void *storage,
    const Aps_QuickJobInfo *src);

#endif /* !JOB_H */
