/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: jobattributes.c
 *
 * Description: Generic printer JobAttributes implementation.
 *
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "ppdattrprovider.h"
#include "gsattrprovider.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(attr)

/* Local private function prototypes. */
static void JobAttrDeleteAllProviders(JobAttributes * attributes);
static Aps_Result JobAttrCreateRequiredProviders(JobAttributes * attributes,
                                                 Printer * printer);
static Aps_Result JobAttrAddProvider(JobAttributes * attributes,
                                     AttrProvider * provider);
static Aps_Result JobAttrIsFieldSet(Printer * printer, const char *field,
                                    BOOL *isSet);

/* ---------------------------------------------------------------------------
 * JobAttrGetPtrFromHandle()
 *
 * Obtains a pointer to a JobAttributes instance given an Aps_JobAttrHandle.
 *
 * Parameters: handle - The Aps_JobAttrHandle to be converted.
 *
 *     Return: A JobAttributes pointer; NULL on failure.
 */
JobAttributes *JobAttrGetPtrFromHandle(Aps_JobAttrHandle handle)
{
    JobAttributes *attributes;

    if (handle == NULL)
        return (NULL);

    attributes = (JobAttributes *) handle;

    /* Ensure that we have a valid pointer to a JobAttributes instance. */
    if (attributes->baseClass.identifier != JOB_ATTRIBUTES_HANDLE)
        return (NULL);

    return (attributes);
}

/* ---------------------------------------------------------------------------
 * JobAttrGetHandleFromPtr()
 *
 * Obtains a handle that the client application can use to identify a job
 * attributes instance.
 *
 * Parameters: attributes - The JobAttributes instance for which a 
 *                          corresponding handle should be obtained.
 *
 *     Return: An Aps_JobAttrHandle; NULL on failure.
 */
Aps_JobAttrHandle JobAttrGetHandleFromPtr(JobAttributes * attributes)
{
    return ((Aps_JobAttrHandle) attributes);
}

/* ---------------------------------------------------------------------------
 * JobAttrCreate()
 *
 * Allocates an empty JobAttributes instance. All new JobAttributes instances
 * have an initial reference count of 1.
 *
 * Parameters: None.
 *
 *     Return: A new allocated JobAttributes, or NULL if there is insufficient
 *             memory.
 */
JobAttributes *JobAttrCreate()
{
    JobAttributes *newAttr = calloc(1, sizeof(JobAttributes));
    Aps_Result result;

    if (newAttr == NULL)
        return NULL;

    /* Initialize the base class information for this APS object, marking */
    /* it as a printer object.                                            */
    result = ObjectInitialize(&newAttr->baseClass, JOB_ATTRIBUTES_HANDLE);
    if (result != APS_SUCCESS) {
        /* TBD: Propogate reason for failure back up the call stack. */
        return NULL;
    }

    /* Initially we have no attribute providers. */
    newAttr->numProviders = 0;

    return newAttr;
}

/* ---------------------------------------------------------------------------
 * Aps_AttrCreateCopy()
 *
 * Creates a new Aps_JobAttrHandle that is an exact (deep) copy of another.
 *
 * Parameters: existingAttributes - The existing job attributes handle to
 *                                  be copied.
 *
 *             newAttributes      - The address of an Aps_JobAttrHandle to
 *                                  receive a handle to the new job
 *                                  attributes object.
 *
 *     Return: APS_SUCCESS on success, or another result code on failure.
 */
Aps_Result Aps_AttrCreateCopy(Aps_JobAttrHandle existingAttributes,
                              Aps_JobAttrHandle *newAttributes)
{
    JobAttributes *src;
    JobAttributes *dest;

    /* Check for reasonable parameter values. */
    if (newAttributes == NULL)
        return APS_INVALID_PARAM;

    /* Set output parameters to default values in case of failure. */
    *newAttributes = NULL;
        
    /* Obtain the JobAttributes instance corresponding to the existing job
     * attributes handle.
     */
    src = JobAttrGetPtrFromHandle(existingAttributes);
    if (src == NULL)
        return APS_INVALID_HANDLE;

    /* Create an exact copy of this job attributes object. */
    dest = JobAttrCreateCopy(src);
    if (dest == NULL)
        return APS_OUT_OF_MEMORY;
        
    /* Provide the caller with a handle to the new object. */
    *newAttributes = JobAttrGetHandleFromPtr(dest);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrCreateCopy()
 *
 * Creates a new JobAttributes instance that is an exact (deep) copy of
 * another.
 *
 * Parameters: source - A previously allocated JobAttributes instance to copy
 *                      from.
 *
 *     Return: A pointer to the newly allocated JobAttributes instance, or NULL
 *             if there was insufficient memory.
 */
JobAttributes *JobAttrCreateCopy(const JobAttributes * source)
{
    JobAttributes *newAttr;
    AttrProvider *srcProvider;
    AttrProvider *newProvider;
    int i;

    /* We assume that the source object is internally consistent. */
    ASSERT(source != NULL);
    ASSERT(source->baseClass.identifier == JOB_ATTRIBUTES_HANDLE);

    /* Start with a clean new JobAttributes instance. */
    newAttr = JobAttrCreate();
    if (newAttr == NULL)
        return (NULL);

    /* Copy the attributes providers that compose this job attributes object.
     */
    for (i = 0; i < source->numProviders; ++i) {
        srcProvider = source->attributeProviders[i];
        
        /* Create a copy of this attribute provider. */
        if (srcProvider->vtbl->AttrCreateCopy(srcProvider, &newProvider,
                                              newAttr) != APS_SUCCESS) {
            JobAttrDelete(newAttr);
            return NULL;
        }
        
        /* Attempt to add the new attribute provider to the new job attributes
         * object.
         */
        if (JobAttrAddProvider(newAttr, newProvider) != APS_SUCCESS) {
            JobAttrDelete(newAttr);
            AttrProvDelete(newProvider);
            return NULL;
        }
    }

    /* If we get to this point, then we've successfully created a complete */
    /* copy of the other attributes object, so return the new one to the */
    /* caller.                                                           */
    return (newAttr);
}

/* ---------------------------------------------------------------------------
 * JobAttrDelete()
 *
 * Deallocate a JobAttributes instance.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to be
 *                          deallocated.
 *
 *     Return: void
 */
void JobAttrDelete(JobAttributes * attributes)
{
    /* We assume that the attributes object is internally consistent. */
    ASSERT(attributes != NULL);
    ASSERT(attributes->baseClass.identifier == JOB_ATTRIBUTES_HANDLE);

    /* Delete the attribute providers. */
    JobAttrDeleteAllProviders(attributes);

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    attributes->baseClass.identifier = INVALID_HANDLE;

    /* Finally, delete the JobAttributes memory itself. */
    free(attributes);
}

/* ---------------------------------------------------------------------------
 * JobAttrDeleteAllProviders()
 *
 * Deletes any and all attribute providers that are currently providing
 * attributes to this job attributes object.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to be
 *                          cleaned out.
 *
 *     Return: void
 */
static void JobAttrDeleteAllProviders(JobAttributes * attributes)
{
    int i;

    /* Loop through all attribute providers that are contributing to this
     * job attributes object.
     */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Delete this attribute provider. */
        AttrProvDelete(attributes->attributeProviders[i]);
        
        attributes->attributeProviders[i] = NULL; /* For predictability. */
    }
    
    /* Record that we no longer have any attribute providers. */
    attributes->numProviders = 0;
}

/* ---------------------------------------------------------------------------
 * JobAttrCreateRequiredProviders()
 *
 * Creates the attribute providers required for the specified printer, based
 * on any configuration that has been made for that printer, along with
 * information obtained from the associated model object.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to setup.
 *
 *             printer    - Pointer to the printer that this job attributes
 *                          object should be setup for.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result JobAttrCreateRequiredProviders(JobAttributes * attributes,
                                                 Printer * printer)
{
    AttrProvider *provider;
    Aps_Result result;
    BOOL fieldIsSet;

    /* Start by removing any attribute providers that are currently attached
     * to this job attributes object.
     */
    JobAttrDeleteAllProviders(attributes);

    /* Check whether this printer or model have an associated PPD filename.
     * If so, we will create a PPD attribute provider.
     */
    result = JobAttrIsFieldSet(printer, "ppd", &fieldIsSet);
    if (result != APS_SUCCESS)
        return result;
    if (fieldIsSet) {
        /* Create the PPD attribute provider. */
        result = PPDAttrCreate((PPDAttrProvider **)&provider,
                               JobAttrGetHandleFromPtr(attributes));
        if (result != APS_SUCCESS)
            return result;

        /* Add this attribute provider to this job attributes object.
         */
        result = JobAttrAddProvider(attributes, provider);
        if (result != APS_SUCCESS)
            return result;
    }

    /* Check whether this printer's has a GhostScript driver associated
     * with it. If so, we want to create a GhostScript attribute provider
     * for it.
     */
    result = JobAttrIsFieldSet(printer, "gsdevice", &fieldIsSet);
    if (result != APS_SUCCESS)
        return result;
    if (fieldIsSet) {
        /* Create the GhostScript attribute provider. */
        result = GSAttrCreate((GSAttrProvider **)&provider,
                              JobAttrGetHandleFromPtr(attributes));
        if (result != APS_SUCCESS)
            return result;

        /* Add this attribute provider to this job attributes object.
         */
        result = JobAttrAddProvider(attributes, provider);
        if (result != APS_SUCCESS)
            return result;
    }

    /* We've successfully created any required attribute providers for this
     * printer.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrAddProvider()
 *
 * Adds an attribute provide to a particular job attributes instance.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to operate
 *                          on.
 *
 *             provider   - Pointer to an attribute provider to be added to
 *                          this job attributes. On success, the attribute
 *                          provider will become the property of the job
 *                          attributes object, and will be destroyed when the
 *                          job attributes object is destroyed.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result JobAttrAddProvider(JobAttributes * attributes,
                                     AttrProvider * provider)
{
    /* The current implementation allows for only a fixed maximum number of
     * job attribute providers.
     */
    ASSERT(attributes->numProviders < APSCFG_ATTR_MAX_PROVIDERS - 1);
    
    /* Store a pointer to the new attribute provider in this job attributes
     * object, and update the number of attribute providers in this job
     * attributes object.
     */
    attributes->attributeProviders[attributes->numProviders] = provider;
    attributes->numProviders++;

    /* In the current implementation, no memory allocation/etc. is performed by
     * this function, and so it will never fail.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrSetToModelDefaults()
 *
 * Resets this job attributes object to the defaults for the specified model
 * of printer, based on information retrieved from the model database.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to operate
 *                          on.
 *
 *             printer    - Pointer to the printer with which this
 *                          JobAttributes object is associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result JobAttrSetToModelDefaults(JobAttributes * attributes,
                                     Printer * printer)
{
    int i;
    Aps_Result result;
    BOOL partialSuccess = FALSE;

    /* Create the required attribute providers for this printer / model. */
    result = JobAttrCreateRequiredProviders(attributes, printer);
    if (result != APS_SUCCESS)
        return result;
    
    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this request on to this attribute provider. */
        result = provider->vtbl->AttrResetToModelDefaults(provider, printer);

        /* If the attribute provider encountered an error, then stop
         * and propogate the failure back to the caller.
         */
        if (result == APS_NOT_FOUND) {
            partialSuccess = TRUE;
        } else if (result != APS_SUCCESS) {
            return result;
        }
    }
    
    /* If we reach this point, all attribute providers successfully completed
     * this operation.
     */
    return partialSuccess ? APS_PARTIAL_SUCCESS : APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrSaveAsPrinterDefaults()
 *
 * Saves the settings in this job attributes object as the default for the
 * specified printer.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to operate
 *                          on.
 *
 *             printer    - Pointer to the printer with which this
 *                          JobAttributes object is associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result JobAttrSaveAsPrinterDefaults(JobAttributes * attributes,
                                        Printer * printer)
{
    int i;
    Aps_Result result;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this request on to this attribute provider. */
        result = provider->vtbl->AttrSaveAsPrinterDefaults(provider, printer);

        /* If the attribute provider encountered an error, then stop
         * and propogate the failure back to the caller.
         */
        if (result != APS_SUCCESS) {
            return result;
        }
    }
    
    /* If we reach this point, all attribute providers successfully completed
     * this operation.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrSetToPrinterDefaults()
 *
 * Initializes this job attributes object based on the defaults for the
 * specified printer.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to operate
 *                          on.
 *
 *             printer    - Pointer to the printer with which this
 *                          JobAttributes object is associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result JobAttrSetToPrinterDefaults(JobAttributes * attributes,
                                       Printer * printer)
{
    int i;
    Aps_Result result;
    BOOL partialSuccess = FALSE;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this request on to this attribute provider. */
        result = provider->vtbl->AttrSetToPrinterDefaults(provider, printer);

        /* If the attribute provider encountered an error, then stop
         * and propogate the failure back to the caller.
         */
        if (result != APS_SUCCESS) {
            partialSuccess = TRUE;
        }
    }
    
    /* If we reach this point, all attribute providers successfully completed
     * this operation.
     */
    return partialSuccess ? APS_PARTIAL_SUCCESS : APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrRemovePrinterDefaults()
 *
 * Remove any default attributes associated with the specified printer.
 *
 * Parameters: attributes - Pointer to the JobAttributes instance to operate
 *                          on.
 *
 *             printer    - Pointer to the printer with which this
 *                          JobAttributes object is associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result JobAttrRemovePrinterDefaults(JobAttributes * attributes,
                                       Printer * printer)
{
    int i;
    Aps_Result result;
    BOOL partialSuccess = FALSE;

    /* Pass on this request to all attribute providers. */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this request on to this attribute provider. */
        result = provider->vtbl->AttrRemovePrinterDefaults(provider, printer);

        /* If the attribute provider encountered an error, then stop
         * and propogate the failure back to the caller.
         */
        if (result != APS_SUCCESS) {
            partialSuccess = TRUE;
        }
    }
    
    /* If we reach this point, all attribute providers successfully completed
     * this operation.
     */
    return partialSuccess ? APS_PARTIAL_SUCCESS : APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * JobAttrIsFieldSet()
 *
 * Checks whether the specified field (such as "ppd" or "gsdevice") is set to
 * a non-empty value either in this printer's meta configuration or in its
 * associated model object.
 *
 * Parameters: printer - Pointer to the printer object to test.
 *
 *             field   - The name of the field to check for.
 *
 *             isSet   - On success this is set to TRUE if the specified
 *                       field is set, FALSE otherwise.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result JobAttrIsFieldSet(Printer * printer, const char *field,
                                    BOOL *isSet)
{
    char *value;
    int valueLen;
    Aps_Result result;

    /* Start by assuming that the field is not set unless we find it. */
    *isSet = FALSE;

    /* Check whether the printer's model object has a field by this name. */
    if (printer->model != NULL) {
        result = Aps_GetPropertyString(printer->model, field, &value);
        if (result == APS_SUCCESS) {
            /* If the string is non-empty, report that the desired field is
             * available.
             */
            valueLen = strlen(value);
            Aps_ReleaseBuffer(value);
            if (valueLen > 0) {
                *isSet = TRUE;
                return APS_SUCCESS;
            }
        } else if (result != APS_NOT_FOUND) {
            return result;
        }
    }

    /* This value wasn't set in the model object (if any) that is associated
     * with this printer, so check the printer's meta configuration for this
     * field.
     */
    result = MetaRead(&value, printer->name, field);
    if (result == APS_SUCCESS) {
        /* If the string is non-empty, report that the desired field is
         * available.
         */
        valueLen = strlen(value);
        free(value);
        if (valueLen > 0) {
            *isSet = TRUE;
            return APS_SUCCESS;
        }
    } else if (result != APS_NOT_FOUND) {
        return result;
    }

    /* We've confirmed that the requested field is not set. We've already
     * defaulted the isSet output parameter to FALSE, so just return with
     * success.
     */
    ASSERT(!*isSet);
    return APS_SUCCESS;
}
