/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: jobdispatch.c
 *
 * Description: APIs for job creation and dispatch.
 *              Covers jobs in APS_JOB_PHASE_PRODUCTION only.
 *
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"
#include "utils.h"
#include "jobattributes.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(job)

/* Will help us track down spots where notification should take place */
#define PRENOTIFY(x)
#define POSTNOTIFY(x)

/* ---------------------------------------------------------------------------
 * Aps_DispatchJob()
 *
 * Sends a job from a particular file or other entity in the file system
 * namespace.
 *
 * Parameters: printer       - An Aps_PrinterHandle handle identifying the
 *                             device to print to. NULL for default.
 *
 *             filename      - String containing the name of the file/etc to
 *                             be printed.
 *
 *             format        - Indentifies whether the file is in PostScript,
 *                             the device's native language, MIME-encoded,
 *                             or some other format.
 *
 *             jobAttributes - The settings to use for this job, or NULL to
 *                             use the printer's default job settings.
 *
 *             job           - An optional pointer to an Aps_JobHandle to
 *                             receive a handle to this job for future
 *                             monitoring, or NULL for none.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *
 * Note: Returned jobhandle MUST be passed to Aps_ReleaseHandle when finished.
 */
Aps_Result Aps_DispatchJob(Aps_PrinterHandle printer, const char *filename,
                           const char *format, Aps_JobAttrHandle jobAttributes,
                           Aps_JobHandle *jobxHandle)
{
    Aps_Result result;
    Aps_JobHandle jobHandle = NULL;

    /* Check parameters */
    if (!filename) return APS_INVALID_PARAM;
    if (!format) format = "";

    /* create the job */
    result = JobCreate(printer, & jobHandle);
    if (result == APS_SUCCESS) {
        Job *job = JobGetPtrFromHandle(jobHandle);
        ASSERT(job);
        /* set attribs */
        result = JobSetAttributes(job, jobAttributes);
        if (result == APS_SUCCESS) {
            /* setup job in praparation for writing */
            result = JobSetup(job, format, filename, filename);

            /* try to dispatch */
            result = job->transport->vtbl->JobDispatch(
                job->transport, job, filename);
        }
    }
    if (jobxHandle) *jobxHandle = jobHandle;
    else if (jobHandle) Aps_ReleaseHandle(jobHandle);
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterStartJob()
 *
 * Begins a job that will be sent by one or more write calls.
 *
 * Parameters: printerHandle - Aps_PrinterHandle identifying the printer that
 *                             will print the job. NULL for default.
 *
 *             format        - Describes what type of data will be sent, such
 *                             as PostScript code or code in the device's
 *                             native language.
 *
 *             jobHandle     - Receives a new Aps_JobHandle that can be used to
 *                             identify this job in the future.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterStartJob(Aps_PrinterHandle printerHandle,
                               const char *format, Aps_JobHandle *jobHandle)
{
    Aps_Result result;

    /* Check parameters and lock printer */
    if (!jobHandle) return APS_INVALID_PARAM;
    *jobHandle = NULL;

    /* create the job */
    result = JobCreate(printerHandle, jobHandle);
    if (result == APS_SUCCESS) {
        Job *job = JobGetPtrFromHandle(*jobHandle);
        ASSERT(job);
        /* set attribs */
        result = JobSetAttributes(job, NULL); /* use defaults */
        if (result == APS_SUCCESS) {
            /* setup job in praparation for writing */
            result = JobSetup(job, format, "(unnamed)", NULL);

            /* try to dispatch */
            result = job->transport->vtbl->JobStart(job->transport, job);
        }
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_JobWrite()
 *
 * Sends data to be spooled as part of a particular print job.
 *
 * Parameters: jobHandle  - An Aps_JobHandle identifying the job to write to.
 *
 *             data       - A pointer to a buffer containing the data to be
 *                          written.
 *
 *             size       - The number of bytes to be written from the buffer
 *                          pointed to by data.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobWrite(Aps_JobHandle jobHandle, const void *data, size_t size)
{
    Job       *job;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_SETUP) {
        PRENOTIFY(jobHandle);
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_SETUP)) | APS_JOB_SPOOLING;
        POSTNOTIFY(jobHandle);
    }
    if (!(job->info->jobStatus & APS_JOB_SPOOLING))
        return APS_OUT_OF_SEQUENCE;

    /* Ignore writes with zero length */
    if (size == 0) return APS_SUCCESS;

    /* Check for reasonable parameter values. */
    if ((size < 0) || (! data)) return APS_INVALID_PARAM;

    /* Go dispatch the operation to the transport */
    ASSERT(job->transport);
    return job->transport->vtbl->JobWrite(job->transport, job, data, size);
}

/* ---------------------------------------------------------------------------
 * Aps_JobWriteString()
 *
 * Sends data to be spooled as part of a particular print job, in the form
 * of a string.
 *
 * Parameters: jobHandle  - An Aps_JobHandle identifying the job to write to.
 *
 *             string     - A pointer to a '\0'-terminated string to write
 *                          to this job.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobWriteString(Aps_JobHandle jobHandle, const char *string)
{
    return Aps_JobWrite(jobHandle, string, strlen(string));
}

/* ---------------------------------------------------------------------------
 * Aps_JobWriteFile()
 *
 * Sends data to be spooled as part of a particular print job.  Reads the
 * source data from a file.  Depending on the implementation, the provided
 * file may be passed directly to the transport if no other writes are
 * pending.
 *
 * Parameters: jobHandle  - An Aps_JobHandle identifying the job to write to.
 *
 *             filename      - String containing the name of the file/etc to
 *                             be printed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobWriteFile(Aps_JobHandle jobHandle, const char *filename)
{
    Aps_Result result = APS_SUCCESS;
    Job   *job;
    int    fd;

    /* Check parameters */
    if (! filename) return APS_INVALID_PARAM;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_SETUP) {
        PRENOTIFY(jobHandle);
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_SETUP)) | APS_JOB_SPOOLING;
        POSTNOTIFY(jobHandle);
    }
    if (!(job->info->jobStatus & APS_JOB_SPOOLING))
        return APS_OUT_OF_SEQUENCE;

    /* Open file and create buffer */
    ASSERT(job->transport);
    errno = 0;
    fd = open(filename, O_RDONLY);
    if (fd) {
        char *buffer = malloc(APSCFG_COPY_BUFFER_SIZE);
        if (buffer) {
            /* Transfer data */
            int numRead;

            while ((numRead = read(fd, buffer,
                APSCFG_COPY_BUFFER_SIZE)) > 0) {
                result = job->transport->vtbl->JobWrite(job->transport, job,
                    buffer, numRead);
            }
            free(buffer); /* Free buffer */
        } else result = APS_OUT_OF_MEMORY;
        close(fd); /* Close file */
    } else result = GetResultFromErrno(); /* file not found */

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_JobWriteBlock()
 *
 * Causes a standard block of data appropriate for this device to be generated
 * and written into this job, such as a file header, file footer or PostScript
 * code for setup of device-specific settings as established in this job's
 * attributes.
 *
 * Parameters: job       - A handle to a print job that is currently being
 *                         spooled.
 *
 *             blockType - The type of block to be written, as identified by
 *                         a Aps_BlockType enumeration.
 *
 *     Return: APS_SUCCESS on success, or a standard result code on failure.
 */
Aps_Result Aps_JobWriteBlock(Aps_JobHandle jobHandle, Aps_BlockType blockType)
{
    Job *job;
    JobAttributes *attributes;
    int i;
    Aps_Result result;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_SETUP) {
        PRENOTIFY(jobHandle);
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_SETUP)) | APS_JOB_SPOOLING;
        POSTNOTIFY(jobHandle);
    }
    if (!(job->info->jobStatus & APS_JOB_SPOOLING))
        return APS_OUT_OF_SEQUENCE;

    attributes = JobAttrGetPtrFromHandle(job->info->jobAttr);
    if (attributes == NULL) {
        /* We should never have a job without an associated job attributes
         * object at this stage in its lifetime.
         */
        ASSERT(FALSE);
        return APS_GENERIC_FAILURE;
    }
    
    /* Give all attribute providers an opportunity to output data for this
     * block type.
     */
    for (i = 0; i < attributes->numProviders; ++i) {
        /* Obtain a pointer to the next attribute provider. */
        AttrProvider *provider = attributes->attributeProviders[i];

        /* Pass this request on to this attribute provider. */
        result = provider->vtbl->AttrWriteBlock(provider, job, blockType);
        if (result != APS_SUCCESS)
            return result;
    }

    return APS_SUCCESS;    
}

/* ---------------------------------------------------------------------------
 * Aps_JobEnd()
 *
 * Ends a job, writes any buffered data back to the spool and queues
 * the job for printing.  In some implementations, the job may already
 * be printing by this time.
 *
 * Parameters: jobHandle - An Aps_JobHandle handle identifying the job
 *                         which the application has finished sending.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobEnd(Aps_JobHandle jobHandle)
{
    Job       *job;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_PHASE_PRODUCTION) {
        PRENOTIFY(jobHandle);
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_PHASE_PRODUCTION)) |
            APS_JOB_QUEUEING;
        POSTNOTIFY(jobHandle);
    } else return APS_OUT_OF_SEQUENCE;

    /* Go dispatch the operation to the transport */
    ASSERT(job->transport);
    return job->transport->vtbl->JobEnd(job->transport, job);
}

/* ---------------------------------------------------------------------------
 * Aps_JobAbort()
 *
 * Discards a job that was started by apsStartJob().
 *
 * Parameters: jobHandle - An Aps_JobHandle handle identifying the job to
 *                         be cancelled.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_JobAbort(Aps_JobHandle jobHandle)
{
    Job       *job;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_PHASE_PRODUCTION) {
        PRENOTIFY(jobHandle);
        /* Set it to APS_JOB_QUEUEING just like if we were about to
         * call the transport's JobEnd(). In the event of failure,
         * the transport will continue as if JobEnd() had been called.
         * On success, it will set APS_JOB_ABORTED. (Or perhaps
         * APS_JOB_CANCELLED if the operation was deferred.)
         */
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_PHASE_PRODUCTION)) |
            APS_JOB_QUEUEING;
        POSTNOTIFY(jobHandle);
    } else return APS_OUT_OF_SEQUENCE;

    /* Go dispatch the operation to the transport */
    return job->transport->vtbl->JobAbort(job->transport, job);
}

/* ---------------------------------------------------------------------------
 * Aps_JobGetFileDescriptor()
 *
 * Get a file descriptor to allow an application to write data to the
 * printer spool much like a file.
 *
 * Parameters: jobHandle - An Aps_JobHandle handle identifying the job to
 *                         be cancelled.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             (in param) new file descriptor
 *
 * Assume that the app won't close the file descriptor until we're
 * through.
 */
Aps_Result Aps_JobGetFileDescriptor(Aps_JobHandle jobHandle, int *fd)
{
    Job       *job;

    /* Check params */
    if (! fd) return APS_INVALID_PARAM;
    *fd = -1;

    /* Obtain a pointer to the job structure. */
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return (APS_INVALID_HANDLE);

    /* Check sequence */
    if (job->info->jobStatus & APS_JOB_SETUP) {
        PRENOTIFY(jobHandle);
        job->info->jobStatus =
            (job->info->jobStatus & (~APS_JOB_SETUP)) | APS_JOB_SPOOLING;
        POSTNOTIFY(jobHandle);
    }
    if (!(job->info->jobStatus & APS_JOB_SPOOLING))
        return APS_OUT_OF_SEQUENCE;

    /* Go dispatch the operation to the transport */
    ASSERT(job->transport);
    return job->transport->vtbl->JobGetFileDescriptor(
        job->transport, job, fd);
}
