/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: lprdetect.h
 *
 * Description: Structures and constants needed for LPR implementation
 *              autodetection.
 *
 */

#ifndef LPRDETECT_H
#define LPRDETECT_H

typedef enum LprImplementation_ LprImplementation;
typedef struct LprImplInfo_ LprImplInfo;

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "utils.h"

/* This enumeration is used to descriminate between different
 * implementations of LPR.  While they are all fundamentally
 * very similar, they each have their own idiosyncrasies,
 * minor incompatibilities, and differing features.
 */
enum LprImplementation_ {
    LPR_IMPL_MINIMAL,   /* minimal compliance -- LPR only */
    LPR_IMPL_COMPLIANT, /* base compliance */
    LPR_IMPL_BSD,       /* Berkeley LPR compliant */
    LPR_IMPL_LPRNG      /* LPRng compliant */
};

/* This structure holds information which is filled in by
 * LprCreateImplInfo.  Typically this information is autodetected.
 */
struct LprImplInfo_ {
    LprImplementation impl;
    long int version;        /* version number as 0xMMmmllbb (MM.mm.ll Build bb) */
    char *cmd_lpr;           /* path/name of lpr  command */
    char *cmd_lprm;          /* path/name of lprm command */
    char *cmd_lpq;           /* path/name of lpq  command */
    char *cmd_lpc;           /* path/name of lpc  command */
    char *cmd_lpd;           /* path/name of lpd  command (may be NULL) */
    char *file_printcap;     /* absolute path of printcap file */
    char *file_masterfilter; /* absolute path of master printfilter */
    char *dir_spool;         /* default path of spool directory */
    char *dir_filters;       /* master directory of print filters */
};

/* Functions that manage LprImplInfo. */
Aps_Result LprCreateImplInfo(LprImplInfo **info);
Aps_Result LprFreeImplInfo(LprImplInfo *info);

/* IMPLEMENTATION COMPLIANCE: <minimum>
 *
 * Please refer to appropriate internal documentation for the most
 * up-to-date information.  This list is no longer maintained.
 *
 * [Only indicates options which are useful to APS]
 *
 * lpq options:
 *   - MINIMAL  : <not supported>
 *
 *   - COMPLIANT: -Pprinter  : specify printer
 *                -l         : verbose list
 *                ...        : job #
 *
 *   - BSD      : <no additional>
 *
 *   - LPRNG    : <and...>
 *                -A         : use authentication
 *                -l         : verbose list (different)
 *                -ll        : very verbose
 *                -s         : queue status
 *                -V         : version number
 *
 * lpr options:
 *   - MINIMAL  : -Pprinter  : specify printer
 *                ...        : filename(s)
 *
 *   - COMPLIANT: -Pprinter  : specify printer
 *                -l         : allow binary data
 *                -r         : remove original on completion
 *                -h         : suppress 'burst' (summary) page
 *                -#num      : number of copies
 *                ...        : filename(s)
 *
 *   - BSD      : <and...>
 *                -Jname     : job name
 *                -Ttitle    : job title
 *                ...        : filename(s)
 *
 *   - LPRng    : <lpr and...>
 *                -A         : use authentication
 *                -b         : alternate to -l above (binary)
 *                -Cpri      : priority from A-Z (high-low)
 *                -K         : alternate to -# above (copies)
 *                -k         : print immediately (while spooling)
 *                -V         : verbose (first 5 chars: 'LPRng')
 *                -VV        : very verbose
 *                ...        : filename(s)
 *
 * lpc options:
 *   - MINIMAL  : <not supported>
 *
 *   - COMPLIANT: <not supported>
 *
 *   - BSD      : command <all | printer> [...]
 *                enable     : permit new jobs
 *                disable    : inhibit new jobs
 *                start      : enable printing
 *                stop       : disable printing after job completes
 *                down msg...: stop & disable and set message
 *                up         : start & enable
 *                status     : display queue status
 *                topq printer <jobs...> : prioritize jobs in order
 *                             at top of queue
 *                clean      : clean temp files (do after purge)
 *
 *   - LPRng    : <lpr and...>
 *                -A         : use authentication
 *                -Pprinter  : printer to operate on
 *                -V         : version info
 *                hold <prt> <id> : hold job(s) until release
 *                release <prt> <id> : release job(s)
 *                holdall    : automatically hold new jobs
 *                noholdall  : disable auto holdall
 *                redo <prt> <id> : reprint the selected job
 *                move <prt> <id> <toprt> : move job from <prt> to <toprt>
 *
 * lprm options:
 *   - MINIMAL  : <not supported>
 *
 *   - COMPLIANT: -Pprinter  : specify printer
 *                jobs       : jobs (by #) to purge
 *
 *   - LPR      : <and...>
 *                users      : users to purge (root only)
 *                -          : (single dash) purge all for user
 *                                           everyone's for root
 *
 *   - LPRng    : <lpr and...>
 *                -A         : use authentication
 *                -a         : purge all user has access to
 *                -V         : version and verbose
 *
 */

#endif /* !LPRDETECT_H */
