/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: lprtransport.c
 *
 * Description: Implementation specific to the LPR transport abstraction
 *              layer.
 *
 */
#include "apsinternal.h"
#if (APSCFG_LPR_BUILD)

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "aps.h"
#include "lprtransport.h"
#include "lprprinter.h"
#include "lprdetect.h"
#include "lprprefs.h"
#include "job.h"
#include "jobattributes.h"
#include "utils.h"
#include "metaconfig.h"
#include "resultcode.h"
#include "modeldb.h"

DEBUG_CHANNEL_DEFAULT(lpr)

/* C++-like virtual functions for the lpr transport.
 * See transport.h for a summary of each function call.
 */
static int LprIsPrinterKnown(Transport * this, const char *name);
static Printer *LprCreatePrinterInstance(Transport * this, const char *name);
static Aps_Result LprJobInit(Transport *thisBase, Job *job);
static Aps_Result LprJobCleanup(Transport *thisBase, Job *job);
static Aps_Result LprJobDispatch(Transport * this, Job *job,
    const char *filename);
static Aps_Result LprJobUpdate(Transport * this, Job *job);
static Aps_Result LprJobIterate(Transport * this, Printer * printer, Job **job);
static Aps_Result LprRebuildQueue(Transport * this, Printer * printer);
static Aps_Result LprJobControl(Transport * this, Job *job,
    Aps_OperationID op, void *argin, void *argout);
static Aps_Result LprQueueControl(Transport * this, Printer *printer,
    Aps_OperationID op, void *argin, void *argout);
static Aps_Result LprJobIsOperationAvailable(Transport * this, Job *job,
    Aps_OperationID op, Aps_Result *anticipatedResult);
static Aps_Result LprQueueIsOperationAvailable(Transport * this,
    Printer *printer, Aps_OperationID op, Aps_Result *anticipatedResult);
static Aps_Result LprGetPrinters(Transport * this, TrackArray_PtrChar *names);
static char *LprGetDefaultPrinterName(Transport * this);
static char *LprGetPPDFileName(Transport * this, Printer * printer);
static Aps_Result LprSetPPDFileName(Transport * this, Printer * printer,
    const char *name);
static Aps_Result LprPrinterGetConnectInfo(Transport * thisBase,
    Printer *printer, Aps_ConnectionType *connectionType,
    char **location);
static Aps_Result LprPrinterSetConnectInfo(Transport * thisBase,
    Printer *printer, Aps_ConnectionType connectionType,
    const char *location);
static Aps_Result LprPrinterRename(Transport * thisBase,
                            Aps_PrinterHandle handle, const char *newName);
static Aps_Result LprPrinterSetMaxJobSize(Transport * thisBase,
                                  Aps_PrinterHandle handle, int maxSize);
static Aps_Result LprPrinterGetMaxJobSize(Transport * thisBase,
                                  Aps_PrinterHandle handle, int *maxSize);
static Aps_Result LprPrinterSetConfigFlags(Transport * thisBase,
    Printer *printer, long int flagMask, long int flagSet);
static Aps_Result LprPrinterGetConfigFlags(Transport * thisBase,
    Printer *printer, long int *configFlags);
static Aps_Result LprAddPrinter(Transport * thisBase, const char *name,
                         Aps_PrinterHandle * handle);
static Aps_Result LprPrinterRemove(Transport * thisBase, Aps_PrinterHandle  handle);
static Aps_Result LprPrinterSetAsDefault(Transport * thisBase,
                                  Aps_PrinterHandle handle);
static Aps_Result LprPrinterIsDefault(Transport * thisBase,
                               Aps_PrinterHandle handle, int *isDefault);
static Aps_Result LprProbeModelForPrinter(Transport *thisBase,
                                          Printer *printer,
                                          Aps_ModelHandle *model);
static Aps_Result LprPrinterAttrGetDefaults(Transport *thisBase,
    Printer *printer, JobAttributes *attr);
static Aps_Result LprPrinterAttrSetDefaults(Transport *thisBase,
    Printer *printer, JobAttributes *attr);

/* PRIVATE FUNCTIONS */
static BOOL LprUpdatePrintcapCache(Transport * thisBase);

static int LprScanQueue_Compare(const Aps_QuickJobInfo *src1,
    const Aps_QuickJobInfo *src2, LprImplementation impl);
static Aps_Result LprScanQueue_BuildTable(LprTransport *lpr,
    Printer *printer, Job *job, LprScanTable **table);
static Aps_Result LprScanQueue_BuildMap(LprTransport *lpr,
    Printer *printer, Job *xjob, LprScanTable *table);
static int LprScanQueue_Update(Aps_QuickJobInfo *dest,
    const Aps_QuickJobInfo *info, LprImplementation impl);
static int LprScanQueue_UpdatePrinter(Aps_QuickJobInfo *info,
    Aps_PrinterStatus status, LprImplementation impl);
static int LprScanQueue_UpdateNonExistant(Job *job);
static void LprScanQueue_DeleteTable(LprScanTable *table);
static Aps_Result LprDoLPCCommand(Transport *thisBase,
    Printer *printer, const char *fmt);

static BOOL IsPrinterNameUnique(const char *printerName,
                      Aps_Printcap_Info ** printcapInfo, int numberOfEntry);
static Aps_Result DeletePrinterEntryInPrintcap(const char *printerName,
                                               Transport * thisBase);
static int GetPrinterRecPositionInArray(const char *printerName,
                      Aps_Printcap_Info ** printcapInfo, int numberOfEntry);
static Aps_Result AddPrinterEntryInPrintcap(Aps_Printcap_Info * newEntry,
                                            Transport * thisBase);

/* VECTOR TABLE */
static TransportVtbl lprTransportVtbl =
{
    LprIsPrinterKnown,
    LprCreatePrinterInstance,
    LprJobInit,
    LprJobCleanup,
    DefaultBULK_JobStart,
    DefaultBULK_JobWrite,
    DefaultBULK_JobAbort,
    DefaultBULK_JobEnd,
    DefaultBULK_JobGetFileDescriptor,
    LprJobDispatch,
    LprJobUpdate,
    LprJobIterate,
    LprRebuildQueue,
    LprJobControl,
    LprQueueControl,
    LprJobIsOperationAvailable,
    LprQueueIsOperationAvailable,
    LprGetPrinters,
    LprGetDefaultPrinterName,
    LprGetPPDFileName,
    LprSetPPDFileName,
    LprAddPrinter,
    LprPrinterRemove,
    LprPrinterSetAsDefault,
    LprPrinterIsDefault,
    LprPrinterRename,
    LprPrinterSetMaxJobSize,
    LprPrinterGetMaxJobSize,
    LprPrinterSetConfigFlags,
    LprPrinterGetConfigFlags,
    LprPrinterGetConnectInfo,
    LprPrinterSetConnectInfo,
    LprProbeModelForPrinter,
    LprPrinterAttrGetDefaults,
    LprPrinterAttrSetDefaults
};

/*---------------------------------------------------------------------------
 * LprUpdatePrintcapCache()
 *
 * Determines if Lpr's printcap file has been changed, and if so updates
 * internal databases.  If the database cannot be accessed, it is assumed
 * not to have changed (eg. not reloaded).  This is not a concern since
 * commands issued to "dead" printers will fail eventually.
 *
 * Parameters: thisBase - Pointer to the this Transport object
 *
 *     Return: TRUE if the database has been updated;
 *             FALSE if it has not.
 */
static BOOL LprUpdatePrintcapCache(Transport * thisBase)
{
    int fd;                     /* Unix file descriptor for Printcap */
    LprTransport *lpr = (LprTransport *) thisBase;
    BOOL flagUpdated = FALSE;

    TRACE("Entered");
    ASSERT(lpr);

    /* First we need to find if the Printcap file has changed. If it
     * has, then we will reload it; otherwise we exit and return TRUE.
     * If for some reason we are unable to access the file, we will
     * exit without changing the database but return FALSE if we don't
     * have a valid one to use until the file become available again.
     */

    fd = open(lpr->config->file_printcap, O_RDONLY);
    if (fd == -1) {
        ERROR("I/O Error: open() on %s for READ failed",
            lpr->config->file_printcap);
        if (lpr->apsPrintcapInfo) {
            WARN("Assuming cached database is still valid");
        } else {
            WARN("No cached database -- leaving uninitialized");
            lpr->apsPrintcapInfo = NULL;
            lpr->numberOfPrintcapEntries = 0;
        }
        return (FALSE);
    } else {
        struct stat printcapStat;	/* filled in by fstat() */
        time_t newTime;

        /* We'll get the datestamp even if we know we must replace the
         * database since it's got to be stored with it for next time.
         */
        if (fstat(fd, &printcapStat)) {	/* fails on != 0 */
            ERROR("I/O Error: fstat() on PRINTCAP failed");
            return (FALSE);
        }
        /* check if contents modified (not attributes) */
        newTime = printcapStat.st_mtime;
        if ((!lpr->apsPrintcapInfo) ||
            (lpr->lastPrintcapInfoUpdate < newTime)) {
            /* File pointer for loading Printcap obtained from descriptor */
            FILE *fp;
            Aps_Printcap_Info **newInfo;
            int newNumEntries;

            /* Construct C filepointer for reading */
            fp = fdopen(fd, "r");
            ASSERT(fp);
            newNumEntries = LoadLprPrintcapFile(fp, &newInfo);
            /* < 0 on failure */
            fclose(fp);

            if (newNumEntries < 0) {
                /* Since LoadLprPrintcapFile does not tell us what caused the
                 * problem, we can't tell if the DB is wrong or if perhaps
                 * the file is incomplete. So we'll trust out internal DB until
                 * we know better.
                 */
                ERROR("LoadLprPrintcapFile failed");
                return (FALSE);
            }
            /* Assume everything went well and copy over the old data */
            /* free the old structure */
            if (lpr->apsPrintcapInfo) {
                int i;
                for (i = 0; i < lpr->numberOfPrintcapEntries; i++)
                    FreePrintcapInfoStructure(lpr->apsPrintcapInfo[i]);
                free(lpr->apsPrintcapInfo);
            }
            /* update information */
            lpr->lastPrintcapInfoUpdate = newTime;
            lpr->apsPrintcapInfo = newInfo;
            lpr->numberOfPrintcapEntries = newNumEntries;
            flagUpdated = TRUE;
            MSG("Locally cached printcap file updated");
        }
    }

    /* We're finished! Close files and exit with TRUE if we updated */
    close(fd);
    return (flagUpdated);
}

/*---------------------------------------------------------------------------
 * LprNotifyPrintcapWasModified()
 *
 * Notify system that printcap file changed.  Also update local cached
 * timestamp.
 *
 * Parameters: thisBase - pointer to the transport layer.
 *             fh       - file handle or NULL
 *             fd       - file descriptor or -1
 *
 * if ((fd == -1) && (fh == NULL)) then
 *   opens the file
 *   reads out the new time stamp
 *   closes the file
 * else
 *   reads the new time stamp
 *   closes the file
 * fi
 * notifies the daemon about the change
 *
 * ONLY ONE OF fh OR fd SHOULD BE SPECIFIED! The option is provided as
 * a conveniece for places where STDIO is used.
 *
 * N.B. There is a potential race condition when calling this with both
 *      fd && fd == NULL.  Whenever possible, this subroutine should be
 *      called with the file still open so that the cached time stamp
 *      is correct.
 */
void LprNotifyPrintcapWasModified(Transport * thisBase, FILE *fh, int fd)
{
    LprTransport *this = (LprTransport *) thisBase;
    LprImplementation impl = this->config->impl;
    char *cmdLPC = strdup(this->config->cmd_lpc);

    /* If we have a file handle, extract the file descriptor */
    if (fh) {
        fd = fileno(fh);
    /* If we don't have anything, open the file -- potential race condition!! */
    } else if (fd == -1) {

/* Define PRINTCAP_UPDATE_PARANOID if you are worried about race conditions
 * causing us to assume our cached printcap is correct when indeed it is
 * not.  This is a deficiency of the current system and should be rectified
 * soon by encapsulating all printcap-related elements into one tight
 * package which handles caching internally.
 */
#ifdef PRINTCAP_UPDATE_PARANOID
        LprUpdatePrintcapCache();
#else
        fd = open(this->config->file_printcap, O_RDONLY);
#endif
    }
    /* Stat the file and get timestamp, if possible */
    if (fd != -1) {
        struct stat printcapStat;
        fstat(fd, &printcapStat);
        this->lastPrintcapInfoUpdate = printcapStat.st_mtime;
    }
    /* Close the file now */
    if (fh) fclose(fh); else if (fd != -1) close(fd);

/* Define PRINTCAP_UPDATE_FANCY if you are worried that the procedure may
 * take some time.  This would be especially helpful in the event that it
 * becomes necessary to wait for a job to finish (eg. lpc stop all)
 */
#ifdef PRINTCAP_UPDATE_FANCY
    /* Do this asynchronously... fork! */
    if (fork() == 0) {
#endif
        switch (impl) {
            case LPR_IMPL_COMPLIANT:
            case LPR_IMPL_BSD:
/* This is apparently unnecessary but I'm not sure so we'll leave this
 * here in case we come up with a valid reason to dispose of it.
 */
#if 0
                /* lpc stop all */
                GetCommandOutput(cmdLPC, "stop all", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_STOPALL, NULL,
                    NULL, NULL, NULL); /* should perhaps wait indefinitely */
                /* lpc restart all */
                GetCommandOutput(cmdLPC, "restart all", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_RESTARTALL, NULL,
                    NULL, NULL, NULL);
                /* lpc start all -- BUG: ignores any paused/disabled flags */
                GetCommandOutput(cmdLPC, "start all", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_STARTALL, NULL,
                    NULL, NULL, NULL);
#endif
                break;
            case LPR_IMPL_LPRNG:
                /* lpc reread */
                GetCommandOutput(cmdLPC, "reread", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_REREAD, NULL,
                    NULL, NULL, NULL);
                break;
            default:
                break;
        }
#ifdef PRINTCAP_UPDATE_FANCY
        exit(0);
    }
#endif
}

/*---------------------------------------------------------------------------
 * LprIsPrinterKnown()
 *
 * Determines whether or not the specified printer is recognized by this
 * transport.
 *
 * Parameters: thisBase - Pointer to the Transport object for this
 *                        transport;
 *
 *             name     - The name of the printer to match.
 *
 *     Return: TRUE if the printer is known to this transport,
 *             FALSE if it is not.
 */
static int LprIsPrinterKnown(Transport * thisBase, const char *name)
{
    LprTransport *this = (LprTransport *) thisBase;
    int i;
    char *printerName;

    ASSERT(this != NULL);

    /* Update the internal printer name cache. Fail if we don't have
     * a proper one.
     */
    LprUpdatePrintcapCache(thisBase);
    if (!this->apsPrintcapInfo)
        return FALSE;           /* APS_NOT_FOUND; */

    /* Loop through all entries in the printcap database. */
    for (i = 0; i < this->numberOfPrintcapEntries; ++i) {
        /* Obtain a pointer to the printer name from this entry. */
        printerName = this->apsPrintcapInfo[i]->printerName;

        /* Assuming this entry has a name, compare it to the name being */
        /* queried.                                                     */
        if (printerName != NULL && strcmp(printerName, name) == 0)
            return TRUE;
    }

    /* If we reach this point, the specified printer name was not in the  
     * list of printcap entries, and so this printer is not known to this
     * transport.                                                         
     */
    return FALSE;
}

/*---------------------------------------------------------------------------
 * LprCreatePrinterInstance()
 *
 * Opens a printer accessible through this transport.
 *
 * Parameters: thisBase - Pointer to the Transport object for this
 *                        transport.
 *
 *             name      - The name of the printer to open.
 *
 *     Return: Pointer to a new Printer, or NULL on failure.
 */
static Printer *LprCreatePrinterInstance(Transport * thisBase,
                                         const char *name)
{
    LprTransport *this = (LprTransport *) thisBase;

    ASSERT(this);

    /* Update the internal printer name cache. Fail if we don't have
     * a proper one.
     */
    LprUpdatePrintcapCache(thisBase);
    if (!this->apsPrintcapInfo)
        return NULL;            /* APS_NOT_FOUND; */

    /* Create a new LprPrinter instance associated with this transport. */
    return ((Printer *) LprCreatePrinter(name, thisBase));
}

/*---------------------------------------------------------------------------
 * LprCreateTransport()
 *
 * Creates a new instance of the lpr transport.
 *
 * Parameters: None.
 *
 *     Return: A pointer to a Transport object.
 */
Transport *LprCreateTransport(void)
{
    Aps_Result result;
    LprTransport *transport = (LprTransport *) calloc(1,
                                                      sizeof(LprTransport));

    /* For now we will not load the printcap file. When we need to look
     * something up, we'll go check it out; not before.
     */
    transport->apsPrintcapInfo = NULL;      /* database not yet valid */
    transport->numberOfPrintcapEntries = 0;
    transport->lastPrintcapInfoUpdate = 0;

    /* Set up transport vector table */
    transport->baseClass.vtbl = &lprTransportVtbl;

    /* Detect settings for the transport
     * Locate program paths, config files, etc... */
    result = LprCreateImplInfo(&transport->config);
    if (result != APS_SUCCESS) {
        WARN("Failed to detect LPR: may be missing components");
        free(transport);
        transport = NULL;
    }
    return ((Transport *) transport);
}

/*---------------------------------------------------------------------------
 * LprGetPrinters()
 *
 * Obtains a list of all printers known to this transport.
 *
 * Parameters: thisBase - Pointer to the base Transport for the lpr transport.
 *
 *             names    - Pointer to a TrackArray of string pointers to
 *                        receive the names
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result LprGetPrinters(Transport * thisBase,
    TrackArray_PtrChar *names)
{
    LprTransport *this = (LprTransport *) thisBase;
    int i;
    char *sourceName;

    ASSERT(this != NULL);

    /* Update the internal printer name cache. Fail if we don't have
     * a proper one.
     */
    LprUpdatePrintcapCache(thisBase);

    /* Loop through all entries in the printcap database. */
    /* (Falls through if no entries available) */
    for (i = 0; i < this->numberOfPrintcapEntries; ++i) {
        /* Obtain a pointer to the printer name from this entry. */
        sourceName = this->apsPrintcapInfo[i]->printerName;

        /* Assuming this entry has a name, add it to the array. */
        if (sourceName != NULL) {
            char *newString;
            newString = (char *)TrackMemDupString(*names, sourceName, 0);
            if (! newString) return APS_OUT_OF_MEMORY;
            if (! TrackArrayAddLast_PtrChar(names, newString))
                return APS_OUT_OF_MEMORY;
        }
    }

    /* If we get to this point, we went through all printcap entries
     * with no problem.                                             
     */
    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * LprJobInit()
 *
 * Create any temporary structures needed by the transport.
 *
 * Parameters: thisBase       - Pointer to the base Transport for the lpr
 *                              transport.
 *
 *             job            - Pointer to a Job to operate on.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result LprJobInit(Transport * thisBase, Job * job)
{
    LprJobContext *jc;

    ASSERT(thisBase && job);

    /* Create job context structure */
    jc = malloc(sizeof(LprJobContext));
    if (! jc) return APS_OUT_OF_MEMORY;

    /* Setup context */
    job->transportData = (void *)jc;
    jc->locked = FALSE;

    /* Append to internal queue cache list */
    jc->queueNext = NULL;
    jc->queuePrev = ((LprPrinter *)job->printer)->queueTail;
    if (((LprPrinter *)job->printer)->queueTail) {
        LprJobContext *tail = (LprJobContext *)
            ((LprPrinter *)job->printer)->queueTail->transportData;
        ASSERT(tail);
        tail->queueNext = job;
    }
    if (! ((LprPrinter *)job->printer)->queueHead)
        ((LprPrinter *)job->printer)->queueHead = job;
    ((LprPrinter *)job->printer)->queueTail = job;
    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * LprJobCleanup()
 *
 * Cleanup and free any structures allocated by the transport in the
 * reserved space.
 *
 * Parameters: thisBase       - Pointer to the base Transport for the lpr
 *                              transport.
 *
 *             job            - Pointer to a Job to operate on.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result LprJobCleanup(Transport * thisBase, Job * job)
{
    LprJobContext *jc;

    ASSERT(thisBase && job);

    /* Locate context */
    jc = (LprJobContext *)job->transportData;
    if (! jc) return APS_SUCCESS;

    if (jc->locked) {
        ERROR("Job freed too many times with Aps_ReleaseHandle()");
        jc->locked = FALSE;
    }

    /* Unlink job from list */
    if (jc->queueNext) {
        LprJobContext *next = (LprJobContext *)jc->queueNext->transportData;
        if (! (next->queuePrev = jc->queuePrev) )
            ((LprPrinter *)job->printer)->queueHead = jc->queueNext;
    } else ((LprPrinter *)job->printer)->queueTail = jc->queuePrev;
    if (jc->queuePrev) {
        LprJobContext *prev = (LprJobContext *)jc->queuePrev->transportData;
        if (! (prev->queueNext = jc->queueNext) )
            ((LprPrinter *)job->printer)->queueTail = jc->queuePrev;
    } else ((LprPrinter *)job->printer)->queueHead = jc->queueNext;

    /* Free memory */
    job->transportData = NULL;
    free(jc);
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprJobDispatch()
 *
 * Dispatch a printing job to LPR as a file.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *             filename  - name of file to be printed
 *
 * Result    : A standard APS result code
 */
Aps_Result LprJobDispatch(Transport * thisBase, Job * job,
                          const char *filename)
{
    const char cmdargFormatString_Minimal[] =
        " -P%s \"%s\"";
    const char cmdargFormatString_Compliant[] =
        " %s-P%s \"%s\"";
    const char cmdargFormatString_LPR[] =
        " %s-P%s %s%s\"%s\"";
    const char cmdargFormatString_LPR_JobName[] =
        "-J\"%s\" ";
    const char cmdargFormatString_LPR_DocTitle[] =
        "-T\"%s\" ";
    Aps_Result result = APS_SUCCESS;
    LprTransport *lpr = (LprTransport *)thisBase;
    struct  stat fileinfo;
    char   *commandArgs;
    int     commandResult = 0;
    char  **buffer;
    int     bufferNumLines;

    ASSERT(thisBase && job && filename);

    /* Update local information */
    strupdatehostname(& job->info->localHost);
    strupdate(& job->info->localFile, filename);

    /* Find file size for job info struct
     * Also checks if file exists, is accessible, etc...
     */
    errno = 0;
    if ( stat(filename, &fileinfo) == 0) {
        job->info->jobSize = fileinfo.st_size; /* get actual filesize */
    }
    result = GetResultFromErrno();
    if (result != APS_SUCCESS) {
        ERROR("stat() failed with result code: %s", RESULTSTR(result));
        return result; /* uhoh! */
    }

    /* Avoid a buffer overrun on commandArgs. We'll allocate
     * the space we need here on the stack with alloca().
     * [alloca() works fine outside of C++ exception handling
     * mechanisms]
     */
    {
        int flagBinary = FALSE;

        /* Check format string and set flags */
        if (job->info->jobFormat) {
            if (strcmp(job->info->jobFormat, APS_FORMAT_NATIVE) == 0) {
                /* application/octet-stream */
                flagBinary = TRUE;
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_PDF) == 0) {
                /* application/pdf */
                flagBinary = TRUE;
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_POSTSCRIPT) == 0) {
                /* application/postscript */
                /* Check if there is a gs device available
                 * for this printer. If it goes run the passed file through
                 * gs and then send it to LPR.
                 */
                char* gsdevice; 
                char* resolution;
                Aps_JobAttrHandle attributes = NULL;
                
                result = Aps_JobGetAttributes( job, &attributes);
                if( result == APS_SUCCESS ){
                    result = Aps_AttrGetSetting(attributes, "gsdevice",
                                                &gsdevice);
                    if( result==APS_SUCCESS )
                        result = Aps_AttrGetSetting(attributes, "resolution",
                                                &resolution); 
                    if( result==APS_SUCCESS && gsdevice!=NULL && resolution!=NULL ){
                        /* Here we run the file through GS */
                        char* outputFile;
                        char* commandLine;
                        const char formatString[]= 
                              "-q -dBATCH -dSAFER -dQUIET -dNOPAUSE"
                              "r%s -sDEVICE=%s -sOutputFile=%s";
                        int   cmdResult = 0;
                        char  **buf;
                        int   bufNumLines;
   
                        /* Allocating space for <filename>.GS */
                        outputFile = calloc(1, strlen(filename)+4); 
                        sprintf(outputFile, "%s.GS", filename);
                        /* Allocating space for gs commandline.
                         * 60 is after manually counting the 
                         * length of format string.
                         */ 
                        commandLine = calloc( 1, strlen(formatString)
                                                +strlen(gsdevice)
                                                +strlen(resolution)
                                                +strlen(outputFile) + 1);
                        sprintf(commandLine,
                                formatString,
                                resolution, gsdevice, outputFile);
                        result = GetCommandOutput("gs", 
                            commandLine, NULL, APSCFG_LPR_TIMEOUT_GS_PROCESS,
                            &buf, &bufNumLines, &cmdResult);

                        /* Check the result to see if there was a 
                         * problem launching gs. Not checking anything 
                         * else. 
                         */
                        if (result == APS_SUCCESS){
                            /* Copy the newly created file to input file  
                             * for actual printing will copy using cp
                             * command executed using GetCommandOutput.
                             */
                            commandLine = realloc( commandLine,
                                                   strlen(filename)
                                                  +strlen(outputFile)
                                                  +4);/* for space and null char.*/
                            sprintf(commandLine, "%s %s", filename, outputFile);
                            result = GetCommandOutput("cp", 
                                commandLine, NULL, 0, &buf, &bufNumLines,
                                &cmdResult);
                            if(result != APS_SUCCESS)
                                result = APS_SUBPROGRAM_FAILED;
                            
                        }else{
                            result = APS_SUBPROGRAM_FAILED;
                        }
                        free(commandLine);
                    }
                }
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_JPEG) == 0) {
                /* image/jpeg */
                flagBinary = TRUE;
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_HTML) == 0) {
                /* text/html */
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_TEXT) == 0) {
                /* text/plain */
            } else if (strcmp(job->info->jobFormat, APS_FORMAT_MULTIPART) == 0) {
                /* multipart/mixed */
            } else {
                /* ASSUME: text/plain */
            }
        }

        /* Format arguments for command */
        switch (lpr->config->impl) {
            case LPR_IMPL_MINIMAL:
                commandArgs = alloca(
                    strlen(cmdargFormatString_Minimal) - 2 /* 2x %s */ + 1
                    + strlen(job->info->printerName)
                    + strlen(filename)
                    );
                sprintf(commandArgs, cmdargFormatString_Minimal,
                    job->info->printerName,     /* printer name */
                    filename);                  /* file to print */
                break;
            case LPR_IMPL_COMPLIANT:
                commandArgs = alloca(
                    strlen(cmdargFormatString_Compliant) - 6 /* 3x %s */ + 1
                    + ((flagBinary) ? 3 : 0)
                    + strlen(job->info->printerName)
                    + strlen(filename)
                    );
                sprintf(commandArgs, cmdargFormatString_Compliant,
                    (flagBinary) ? "-l " : "",   /* allow binary */
                    job->info->printerName,     /* printer name */
                    filename);                  /* file to print */
                break;
            case LPR_IMPL_BSD:
            case LPR_IMPL_LPRNG: {
                char *optJobName = "";
                char *optDocTitle = "";

                /* format jobName */
                if (strisdef(job->info->jobName)) {
                    optJobName = alloca(
                        strlen(cmdargFormatString_LPR_JobName) - 2 + 1
                        + strlen(job->info->jobName));
                    sprintf(optJobName, cmdargFormatString_LPR_JobName,
                        job->info->jobName);
                }
                /* format docTitle */
                if (strisdef(job->info->docTitle)) {
                    optDocTitle = alloca(
                        strlen(cmdargFormatString_LPR_DocTitle) - 2 + 1
                        + strlen(job->info->docTitle));
                    sprintf(optDocTitle, cmdargFormatString_LPR_DocTitle,
                        job->info->docTitle);
                }

                commandArgs = alloca(
                    strlen(cmdargFormatString_LPR) - 10 /* 5x %s */ + 1
                    + ((flagBinary) ? 3 : 0)
                    + strlen(job->info->printerName)
                    + strlen(optJobName)
                    + strlen(optDocTitle)
                    + strlen(filename)
                    );
                sprintf(commandArgs, cmdargFormatString_LPR,
                    (flagBinary) ? "-l " : "",  /* allow binary */
                    job->info->printerName,     /* printer name */
                    optJobName,                 /* job name */
                    optDocTitle,                /* document title */
                    filename);                  /* file to print */
                } break;
            default:
                return APS_GENERIC_FAILURE;
        }
    }

    /** CALL LPR TO QUEUE THE PRINT JOB
     *
     * LPR_IMPL_MINIMAL   : lpr -P[printer] [job]
     * LPR_IMPL_COMPLIANT : lpr {-l} -P[printer] [job]
     * LPR_IMPL_BSD       : lpr {-l} -P[printer] -J[jobName] -T[docTitle] [job]
     * LPR_IMPL_LPRNG     : lpr {-l} -P[printer] -J[jobName] -T[docTitle] [job]
     */
    result = GetCommandOutput(lpr->config->cmd_lpr, commandArgs, NULL,
        APSCFG_LPR_TIMEOUT_LPR_DISPATCH,
        &buffer, &bufferNumLines, &commandResult);

    /* Check the result to see if there was a problem launching LPR. */
    if (result != APS_SUCCESS) result = APS_SUBPROGRAM_FAILED;

    /* If return code is not 0, investigate... */
    /* If we're not sure, we'll have to assume success... */
    if (commandResult != 0) {
        TRACE("LPR returned %d : %s \\ %s\n", commandResult,
            (bufferNumLines > 0) ? buffer[0] : "<no output>",
            (bufferNumLines > 1) ? buffer[1] : "");
        switch (lpr->config->impl) {
            case LPR_IMPL_MINIMAL:
            case LPR_IMPL_COMPLIANT:
                break;
            case LPR_IMPL_BSD:
                if ((bufferNumLines > 0) &&
                    (strstr(buffer[0], "queue is disabled"))) {
                    result = APS_ACCESS_DENIED;
                }
                break;
            case LPR_IMPL_LPRNG:
                break;
        }
    }

    /* Check for problems...
     * Set status to APS_JOB_QUEUEING if not, or APS_JOB_FAILED on
     * error.
     */
    if (result != APS_SUCCESS) {
        job->info->jobStatus = APS_JOB_FAILED;
    } else {
        job->info->jobStatus = APS_JOB_QUEUEING;
    }

    /* Finally release buffer */
    if (buffer) Aps_ReleaseBuffer(buffer); /* may check this later */

    /* Update spool information */
    /* assumes spool is local, but this may not always be true
     * with LPRng forwarding. Should be tested in a later release.
     */
    strupdate(& job->info->spoolHost, job->info->localHost);
    strupdate(& job->info->spoolFile, "");

    /* Try to get additional information (like jobID)
     * This will hopefully correct our assumption above that LPR
     * has succeeded.
     */
    thisBase->vtbl->JobUpdate(thisBase, job);	/* discard result */

    return (result);
}

/* ---------------------------------------------------------------------------
 * LprJobIterate()
 *
 * Locate the next job in the list.
 * Whe passed NULL, this function builds a complete image of the jobs for
 * a given printer.  For jobs already in the cache, they are merely
 * updated; for those which are not, new job structures are created
 * and marked "discovered".  If jobs in the cache are not found, and
 * are marked discovered, their status is set to APS_JOB_COMPLETED
 * and they are not returned.
 *
 * Parameters: thisBase       - Pointer to the base Transport for the lpr
 *                              transport.
 *
 *             printer        - required to locate printer for first call.
 *
 *             job            - Pointer to a Job to operate on.
 *
 *     Return: Codes as defined in transport.h
 *             Job updated.
 */
Aps_Result LprJobIterate(Transport *thisBase, Printer *printer, Job **job)
{
    ASSERT(thisBase && printer && job);

    /* If we have a job now, go find the next one in the chain */
    if (*job) {
        LprJobContext *jc;
        TRACE("Got a valid pointer, fetching next job in list");
        Aps_ReleaseHandle(JobGetHandleFromPtr(*job));
        jc = (LprJobContext *)((*job)->transportData);
        *job = jc->queueNext;
    } else {
        /* If passed NULL, rebuild list first */
        TRACE("Got NULL, rebuilding list then returning first element");
        LprRebuildQueue(thisBase, printer); /* ignores failure */
        *job = ((LprPrinter *)printer)->queueHead;
    }
    if (*job) {
        Aps_AddRef(JobGetHandleFromPtr(*job));
        return APS_SUCCESS;
    } else return APS_NO_MORE_DATA;
}

/* ---------------------------------------------------------------------------
 * LprJobUpdate()
 *
 * Update the job status.
 *
 * Parameters: thisBase       - Pointer to the base Transport for the lpr
 *                              transport.
 *
 *             job            - Pointer to a Job to operate on.
 *
 *     Return: Codes as defined in transport.h
 */
Aps_Result LprJobUpdate(Transport * thisBase, Job * job)
{
    Aps_Result     result;
    LprScanTable  *table;
    LprTransport  *lpr = (LprTransport *)thisBase;
    int changed = FALSE;

    ASSERT(thisBase && job && job->printer);

    /* Build a table representing the queue */
    result = LprScanQueue_BuildTable(lpr, job->printer, job, & table);
    if (result != APS_SUCCESS) {
        ERROR("LprScanQueue_BuildTable() failed with result code: %s",
            RESULTSTR(result));
        return result;
    }

    /* Determine the mapping between known jobs and the ones in the table */
    result = LprScanQueue_BuildMap(lpr, job->printer, job, table);
    if (result == APS_SUCCESS) {
        int i;
        /* Find the right mapping */
        for (i = 0; i < table->size; i++) {
            if (job == table->map[i].job) break;
        }

        /* Update printer info */
        changed |=
            LprScanQueue_UpdatePrinter(job->info, table->printerStatus,
            lpr->config->impl);

        /* Mapping not found */
        if ((i == table->size) || (! table->map[i].info)) {
            changed |= LprScanQueue_UpdateNonExistant(job);
        /* Mapping found */
        } else {
            changed |= LprScanQueue_Update(job->info, table->map[i].info,
                lpr->config->impl);
        }
        if (! changed) result = APS_NO_CHANGE;
    } else {
        ERROR("LprScanQueue_BuildMap() failed with result code: %s",
            RESULTSTR(result));
    }
    LprScanQueue_DeleteTable(table);
    return result;
}

/* ---------------------------------------------------------------------------
 * LprRebuildQueue()
 *
 * Rebuilds the status of all jobs in the printer queue as if JobUpdate()
 * were called on all of the jobs in the queue at the same time.
 *
 * Parameters: thisBase       - Pointer to the base Transport for the lpr
 *                              transport.
 *
 *             printer        - Pointer to a Printer to operate on.
 *
 *     Return: Codes as defined in transport.h
 */
Aps_Result LprRebuildQueue(Transport *thisBase, Printer *printer)
{
    Aps_Result     result;
    LprScanTable  *table;
    LprTransport  *lpr = (LprTransport *)thisBase;
    Job           *job;

    ASSERT(thisBase && printer);

    /* Build a table representing the queue */
    result = LprScanQueue_BuildTable(lpr, printer, NULL, & table);
    if (result != APS_SUCCESS) {
        ERROR("LprScanQueue_BuildTable() failed with result code: %s",
            RESULTSTR(result));
        return result;
    }

    /* Clear all existing mappings */
    job = ((LprPrinter *)printer)->queueHead;
    while (job) {
        LprJobContext *jc = (LprJobContext *)job->transportData;
        jc->match = -1; /* clear mapping */
        job = jc->queueNext;
    }

    /* Determine the mapping between known jobs and the ones in the table */
    result = LprScanQueue_BuildMap(lpr, printer, NULL, table);
    if (result == APS_SUCCESS) {
        Job *next;
        int  i;
        /* Scan all jobs for mappings */
        next = ((LprPrinter *)printer)->queueHead;
        while ( (job = next) ) {
            LprJobContext *jc = (LprJobContext *)job->transportData;
            next = jc->queueNext;

            /* Update printer info */
            LprScanQueue_UpdatePrinter(job->info, table->printerStatus,
                lpr->config->impl);

            /* Mapping found */
            if (jc->match != -1) {
                ASSERT(table->map[jc->match].info);
                LprScanQueue_Update(job->info, table->map[jc->match].info,
                    lpr->config->impl);
            /* Mapping not found */
            } else {
                LprScanQueue_UpdateNonExistant(job);
            }
        }

        /* Scan all mappings for new jobs */
        for (i = 0; i < table->size; i++) {
            if (table->map[i].job) continue; /* skip those which are mapped */

            /* create new job -- retains lock */
            TransJobCreate(printer, table->map[i].info, & job);
            if (job) {
                LprJobContext *jc = (LprJobContext *)job->transportData;
                jc->locked = TRUE;
                jc->match = -1;
                Aps_AddRef(JobGetHandleFromPtr(job));

                LprScanQueue_UpdatePrinter(job->info, table->printerStatus,
                    lpr->config->impl);
            } else {
                ERROR("Failed to create job object, ignoring...");
            }
        }
    } else {
        ERROR("LprScanQueue_BuildMap() failed with result code: %s",
            RESULTSTR(result));
    }
    LprScanQueue_DeleteTable(table);
    return result;
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_BuildTable()
 *
 * Scan the queue and build a table of all the information structures
 * found, with blank mappings.
 *
 * Can be asked about many jobs, or just one to narrow down the search
 * (if possible). If search could not be narrowed, acts as if it had
 * not been requested.
 */
static Aps_Result LprScanQueue_BuildTable(LprTransport *lpr,
    Printer *printer, Job *job, LprScanTable **table)
{
    Aps_QuickJobInfo info;   /* for holding job information */
    Aps_Result       result;
    TrackArrayIndirect_QuickJobInfo array = NULL;

    ASSERT(lpr && printer && table);

    /* Not supported by minimal LPR */
    if (lpr->config->impl == LPR_IMPL_MINIMAL) {
        ERROR("Minimal conformance with LPR is insufficient for reading queue");
        return APS_NOT_SUPPORTED;
    }

    /* Initialize a QuickJobInfo for scanning purposes */
    QuickJobInfoInit(& info, 0);

    /* Setup printerName and printerHandle */
    if (strupdate(& info.printerName, printer->name)) {
        info.printerHandle = PrinterGetHandleFromPtr(printer);
        info.printerStatus = APS_PRINTER_UNKNOWN;

        array = TrackArrayIndirectNew_QuickJobInfo(NULL, 0);
        if (array) {
            /* Get printer status and build array */
            switch (lpr->config->impl) {
                case LPR_IMPL_COMPLIANT:
                case LPR_IMPL_BSD:
                    result = LprBSDGetQueueStatus(lpr, & info);
                    if (Aps_Succeeded(result)) {
                        result = LprBSDReadQueue(lpr, & info, & array);
                        if (result != APS_SUCCESS)
                            ERROR("Unable to read BSD queue contents,"
                                  "result code: %s", RESULTSTR(result));
                    } else ERROR("Unable to read BSD queue status,"
                                 "result code: %s", RESULTSTR(result));
                    break;
                case LPR_IMPL_LPRNG:
                    result = LprNGGetQueueStatus(lpr, & info);
                    if (Aps_Succeeded(result)) {
                        result = LprNGReadQueue(lpr, & info, & array);
                        if (result != APS_SUCCESS)
                            ERROR("Unable to read LPRng queue contents,"
                                  "result code: %s", RESULTSTR(result));
                    } else ERROR("Unable to read LPRng queue status,"
                                 "result code: %s", RESULTSTR(result));
                    break;
                default:
                    result = APS_NOT_SUPPORTED;
            }
        } else result = APS_OUT_OF_MEMORY;
    } else result = APS_OUT_OF_MEMORY;

    /* Build a table if succeeded */
    if (result == APS_SUCCESS) {
        LprScanTable *tab;
        int size = TrackArrayIndirectGetSize_QuickJobInfo(array);

        TRACE("Building table of jobs with blank mappings");
        /* Allocate space */
        *table = tab =
            malloc(sizeof(LprScanTable) + (size * sizeof(LprScanMap)));
        if (tab) {
            int i;

            /* Fill in tracking information */
            tab->array = array;
            tab->size = size;
            tab->printerStatus = info.printerStatus;
            /* Fill in blank mapping table array */
            for (i = 0; i < size; i++) {
                tab->map[i].job = NULL;
                tab->map[i].info = array[i];
                tab->map[i].closeness = -1;
            }
            return APS_SUCCESS;
        } else result = APS_OUT_OF_MEMORY;
    }

    /* Release the array on failure */
    if (array) TrackArrayIndirectDelete_QuickJobInfo(array);
    /* Release the resources in the QuickJobInfo */
    QuickJobInfoCleanup(& info);

    return result;
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_BuildMap()
 *
 * Build a mapping between jobs and their descriptions. Will stop
 * when the desired job is finally located and matched. If not
 * specified, tries to map all jobs in the local cache.
 */
static Aps_Result LprScanQueue_BuildMap(LprTransport *lpr,
    Printer *printer, Job *xjob, LprScanTable *table)
{
    int conflict;

    ASSERT(lpr && printer && table);

    /* Scans list until all pairs have been resolved */
    /* This is a slow O(n^2) algorithm... (worst case O(n^3) ) OUCH!! */
    do {
        Job *job, *next;
        int i;

        /* scan list */
        conflict = 0;
        next = ((LprPrinter *)printer)->queueHead;
        while ( (job = next) ) {
            int bestCloseness = 0;
            int bestMatch = -1;
            LprJobContext *jc = (LprJobContext *)job->transportData;
            next = jc->queueNext;

            /* perform a best-match lookup */
            for (i = 0; i < table->size; i++) {
                int closeness;
                closeness = LprScanQueue_Compare(job->info,
                        table->map[i].info, lpr->config->impl);
                if ((closeness > bestCloseness) &&
                    (closeness > table->map[i].closeness)) {
                    bestCloseness = closeness;
                    bestMatch = i;
                }
            }
            /* set match */
            if (bestCloseness > 0) {
                if (table->map[bestMatch].job) { /* existing match? */
                    LprJobContext *xjc = (LprJobContext *)
                        table->map[bestMatch].job->transportData;
                    xjc->match = -1; /* remove old match */
                    conflict++;
                }
                table->map[bestMatch].job = job; /* make association */
                table->map[bestMatch].closeness = bestCloseness;
                jc->match = bestMatch;
            }
        }
    } while (conflict);
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_DeleteTable()
 *
 * Deletes a table and all mapping information it contains
 */
static void LprScanQueue_DeleteTable(LprScanTable *table)
{
    if (table) {
        if (table->array) TrackArrayIndirectDelete_QuickJobInfo(table->array);
        free(table);
    }
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_Compare()
 *
 * Try to determine if a job matches another one and return a value indicating
 * how closely it does.  If any fields are found which do not match, -1
 * is returned.
 *
 * Compares and weights:
 *  - jobID
 *  - localFile   - LPRNG compares clipped strings
 *  - ownerName
 *
 * Compares src1 to src2. If any concessions must be made, they are made
 * when considering the information from src2. eg. src1 should be the more
 * "precise" of the two.
 */
static int LprScanQueue_Compare(const Aps_QuickJobInfo *src1,
    const Aps_QuickJobInfo *src2, LprImplementation impl)
{
    int closeness;

    ASSERT(src1 && src2);

    /*** Uses the method outlined in transport.h ***/
    /* 1. If job->transportData contains any useful internal
     *    information, use that to find the job. Unless the data
     *    is unreliable (why keep it then?), it is safe to
     *    return failure for the Update operation if the job could
     *    not be located with stored information.
     * 2a. Use job->jobID, if valid and no match then bail.
     * 2b. Check filenames (cross-check too if they got mixed up)
     * 2x. If nothing matched, bail out.
     * 3. Use any other reliable (stored) fields which can be
     *     located easily and verify uniqueness if unsure.
     */

    /* 1. -- We have no cached data */
    closeness = 0;

    /* 2a. -- Check jobID*/
    if (src1->jobID != -1) {
        if (src1->jobID == src2->jobID) closeness += 2048;
        else return -1;
    }

    /* 2b. -- Cross-check filenames */
    if (strisdef(src1->jobFilename)) {
        if (strisdef(src2->jobFilename) && (strcmp(src1->jobFilename,
           src2->jobFilename) == 0)) closeness += 1024;
        else if (strisdef(src2->localFile) && (strcmp(src1->jobFilename,
           src2->localFile) == 0)) closeness += 256;
        else if (strisdef(src2->spoolFile) && (strcmp(src1->jobFilename,
           src2->spoolFile) == 0)) closeness += 256;
    }
    if (strisdef(src1->localFile)) {
        if (strisdef(src2->localFile) && (strcmp(src1->localFile,
           src2->localFile) == 0)) closeness += 1024;
        else if (strisdef(src2->jobFilename) && (strcmp(src1->localFile,
           src2->jobFilename) == 0)) closeness += 256;
        else if (strisdef(src2->spoolFile) && (strcmp(src1->localFile,
           src2->spoolFile) == 0)) closeness += 256;
    }
    if (strisdef(src1->spoolFile)) {
        if (strisdef(src2->spoolFile) && (strcmp(src1->spoolFile,
           src2->spoolFile) == 0)) closeness += 1024;
        else if (strisdef(src2->jobFilename) && (strcmp(src1->spoolFile,
           src2->jobFilename) == 0)) closeness += 512;
        else if (strisdef(src2->localFile) && (strcmp(src1->spoolFile,
           src2->localFile) == 0)) closeness += 512;
    }

    /* 2x. -- Bail if not reliably matched */
    if (closeness == 0) return -1; /* must have at least one of the above */

    /* 3a. -- Check ownerName */
    if (strisdef(src1->ownerName) && strisdef(src2->ownerName)) {
        if  (strcmp(src1->ownerName, src2->ownerName) == 0) closeness += 256;
    }

    /* 3b. -- Make sure job status hasn't gone back in time */
    if ((src1->jobStatus & APS_JOB_PHASE_ENDED) &&
        (src2->jobStatus & (APS_JOB_PHASE_PRODUCTION | APS_JOB_PHASE_PENDING |
        APS_JOB_PHASE_WORKING))) return -1;
    if ((src1->jobStatus & APS_JOB_PHASE_WORKING) &&
        (src2->jobStatus & (APS_JOB_PHASE_PRODUCTION | APS_JOB_PHASE_PENDING)))
        return -1;
    if ((src1->jobStatus & APS_JOB_PHASE_PENDING) && (
        (src2->jobStatus & APS_JOB_PHASE_PRODUCTION))) return -1;

    /* 3c. -- Extra points for a good timestamp match */
    {
        int diff = abs(src1->jobCreationTime - src2->jobCreationTime);
        if (diff == 0) closeness += 512;
        else if (diff < 4) closeness += 256; /* +/- 4 secs */
    }

    /* 3d. -- Check if size matches */
    if (src1->jobSize == src2->jobSize) closeness += 512;

    /* Tell app about it */
    return closeness;
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_UpdatePrinter()
 *
 * Updates a job's printer status and job status information based on
 * printer status.
 */
static int LprScanQueue_UpdatePrinter(Aps_QuickJobInfo *info,
    Aps_PrinterStatus status, LprImplementation impl)
{
    Aps_JobStatus oldStatus = info->jobStatus;

    ASSERT(info);

    /* fix JobStatus printer-dependant flags */
    if (status != APS_PRINTER_UNKNOWN) {
        info->jobStatus &= ~APS_JOB_SUSPENDED;
        if (status & APS_PRINTER_SUSPENDED)
            info->jobStatus |= APS_JOB_SUSPENDED;
    }
    info->printerStatus = status;

    return (oldStatus != info->jobStatus);
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_Update()
 *
 * Updates a job's information based on data in an Aps_QuickJobInfo struct.
 *
 * Uses procedures outlined in transport.h
 */
static int LprScanQueue_Update(Aps_QuickJobInfo *dest,
    const Aps_QuickJobInfo *info, LprImplementation impl)
{
    int changed = FALSE;

    ASSERT(dest && info);

    /* jobStatus -- always retain APS_JOB_SUSPENDED */
    if ((info->jobStatus != APS_JOB_UNKNOWN) &&
       ((dest->jobStatus ^ info->jobStatus) & (~APS_JOB_SUSPENDED))) {
        dest->jobStatus = (dest->jobStatus & APS_JOB_SUSPENDED) |
            (info->jobStatus & (~APS_JOB_SUSPENDED));
        changed = TRUE;
    }
    /* jobHost -- if unknown */
    if (!strisdef(dest->jobHost)) {
        strupdate(& dest->jobHost, info->jobHost);
        changed = TRUE;
    }
    /* jobName -- if unknown */
    if (!strisdef(dest->jobName)) {
        strupdate(& dest->jobName, info->jobName);
        changed = TRUE;
    }
    /* jobFilename -- if unknown */
    if (!strisdef(dest->jobFilename)) {
        strupdate(& dest->jobFilename, info->jobFilename);
        changed = TRUE;
    }
    /* jobID */
    if (dest->jobID != info->jobID) {
        dest->jobID = info->jobID;
        changed = TRUE;
    }
    /* jobSize */
    if (dest->jobSize != info->jobSize) {
        dest->jobSize = info->jobSize;
        changed = TRUE;
    }
    /* jobCreationTime -- LPRng ONLY */
    if ((impl == LPR_IMPL_LPRNG) &&
        (dest->jobCreationTime != info->jobCreationTime)) {
        dest->jobCreationTime = info->jobCreationTime;
        changed = TRUE;
    }
    /* jobFormat -- unknown */
    /* jobOrder */
    if (dest->jobOrder != info->jobOrder) {
        dest->jobOrder = info->jobOrder;
        changed = TRUE;
    }
    /* jobPriority -- LPRng ONLY */
    if ((impl == LPR_IMPL_LPRNG) &&
        (dest->jobPriority != info->jobPriority)) {
        dest->jobPriority = info->jobPriority;
        changed = TRUE;
    }
    /* printerHandle -- DO NOT TOUCH */
    /* printerStatus -- DO NOT TOUCH */
    /* printerName -- DO NOT TOUCH */
    /* docTitle -- LPRng ONLY -- only if unknown */
    if ((impl == LPR_IMPL_LPRNG) &&
        (! strisdef(dest->docTitle))) {
        strupdate(& dest->docTitle, info->docTitle);
        changed = TRUE;
    }
    /* docRevision -- unknown */
    /* docComments -- unknown */
    /* docAuthor -- unknown */
    /* docType -- unknown */
    /* docCreator -- unknown */
    /* ownerName & ownerID -- only if certain of it */
    if ( (! strisdef(dest->ownerName)) && (strisdef(info->ownerName)) ) {
        strupdate(& dest->ownerName, info->ownerName);
        dest->ownerID = info->ownerID;
        changed = TRUE;
    }
    /* localHost -- if unknown, use current */
    if (! strisdef(dest->localHost)) {
        TRACE("localHost was undefined in update!");
        strupdatehostname(& dest->localHost);
        changed = TRUE;
    }
    /* localFile -- if unknown */
    if (!strisdef(dest->localFile)) {
        strupdate(& dest->localFile, info->localFile);
        changed = TRUE;
    }
    /* spoolHost */
    if ((! strisdef(dest->spoolHost)) ||
        (strcmp(dest->spoolHost, info->spoolHost) != 0)) {
        strupdate(& dest->spoolHost, info->spoolHost);
        changed = TRUE;
    }

    /* spoolFile */
    if ((! strisdef(dest->spoolFile)) ||
        (strcmp(dest->spoolFile, info->spoolFile) != 0)) {
        strupdate(& dest->spoolFile, info->spoolFile);
        changed = TRUE;
    }

    /* jobAttr -- DO NOT TOUCH */

    return changed;
}

/* ---------------------------------------------------------------------------
 * LprScanQueue_UpdateNonExistant()
 *
 * Updates a job's information based on the fact that it was NOT FOUND
 * during a queue scan.  Releases any locks we might have on the job
 * so that it can be freed normally.
 *
 * Uses procedures outlined in transport.h
 */
static int LprScanQueue_UpdateNonExistant(Job *job)
{
    int changed = FALSE;
    LprJobContext *jc;

    ASSERT(job);
    jc = (LprJobContext *)job->transportData;
    ASSERT(jc);

    /* if job not found, and not in production, assume complete */
    if (! (job->info->jobStatus & APS_JOB_PHASE_PRODUCTION)) {
        /* assume job has completed if we don't know better */
        if (! (job->info->jobStatus & APS_JOB_PHASE_ENDED)) {
            job->info->jobStatus = APS_JOB_COMPLETED;
            changed = TRUE;
        }
        /* release our locks */
        if (jc->locked) {
            jc->locked = FALSE;
            Aps_ReleaseHandle(JobGetHandleFromPtr(job)); /* release */
        }
    }
    return changed;
}

/* --------------------------------------------------------------------
 * LprJobControl()
 *
 * Performs a single specific operation / query on a job.
 *
 * Parameters: this          - transport object pointer
 *             job           - job object pointer
 *             operationCode - ID of operation to be performed
 *             argin         - inputs
 *             argout        - outputs
 *
 *     Return: Detailed Aps_Result code for the application.
 */
static Aps_Result LprJobControl(Transport *thisBase, Job *job,
    Aps_OperationID op, void *argin, void *argout)
{
    Aps_Result result;
    LprTransport  *lpr = (LprTransport *)thisBase;

    /* If we don't know the job ID we can't do much, so try to grab it */
    if (job->info->jobID == -1) {
        TRACE("jobID field was -1, try to update it");
        result = LprJobUpdate(thisBase, job);
        if (! Aps_Succeeded(result)) return result;
        if (job->info->jobID == -1) {
            ERROR("jobID is unknown, unable to perform operation");
            return APS_NOT_FOUND;
        }
    }

    /* Try to perform an operation */
    result = APS_NOT_SUPPORTED;
    switch (op) {
        /*** DELETE JOB PERMANENTLY ***/
        /*** CANCEL JOB BUT LEAVE IN QUEUE ***/
        /* lprm -P[printer] [jobid] */
        case APS_OP_JDELETE:
        case APS_OP_JCANCEL:
            if (lpr->config->impl != LPR_IMPL_MINIMAL) {
                /* lprm -P[printer] [jobid] */
                result = GetCommandOutput(lpr->config->cmd_lprm,
                    "-P%s %d", NULL,
                    APSCFG_LPR_TIMEOUT_LPRM_REMOVE, NULL, NULL, NULL,
                    job->info->printerName, job->info->jobID);
                FIXME("Unchecked output messages from lprm");
                return result;
            }
            break;

        /*** PAUSE JOB ***/
        case APS_OP_JPAUSE:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc hold [printer] [jobid] */
                result = GetCommandOutput(lpr->config->cmd_lpc,
                    "hold %s %d", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_PAUSE, NULL, NULL, NULL,
                    job->info->printerName, job->info->jobID);
                FIXME("Unchecked output messages from lpc");
                /* update on success */
                if (result == APS_SUCCESS)
                    result = LprJobUpdate(thisBase, job);
                return result;
            }
            break;

        /*** RESUME JOB ***/
        case APS_OP_JRESUME:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc release [printer] [jobid] */
                result = GetCommandOutput(lpr->config->cmd_lpc,
                    "release %s %d", NULL,
                    APSCFG_LPR_TIMEOUT_LPC_RESUME, NULL, NULL, NULL,
                    job->info->printerName, job->info->jobID);
                FIXME("Unchecked output messages from lpc");
                /* update on success */
                if (result == APS_SUCCESS)
                    result = LprJobUpdate(thisBase, job);
                return result;
            }
            break;

        /*** STORE JOB NOW ***/
        case APS_OP_JSTORENOW:
            /* not supported */
            break;

        /*** STORE JOB WHEN COMPLETE ***/
        case APS_OP_JSTOREWHENCOMPLETE:
            /* not supported */
            break;

        /*** UNKNOWN -- MAY BE FUTURE CODE -- MARK UNSUPPORTED ***/
        default:
            break;
    }
    return (result);
}

/* --------------------------------------------------------------------
 * LprQueueControl()
 *
 * Performs a single specific operation / query on a printer's queue.
 *
 * Parameters: this          - transport object pointer
 *             printer       - printer object pointer
 *             operationCode - ID of operation to be performed
 *             argin         - inputs
 *             argout        - outputs
 *
 *     Return: Detailed Aps_Result code for the application.
 */
static Aps_Result LprQueueControl(Transport *thisBase, Printer *printer,
    Aps_OperationID op, void *argin, void *argout)
{
    Aps_Result result;
    LprTransport  *lpr = (LprTransport *)thisBase;

    /* Try to perform an operation */
    result = APS_NOT_SUPPORTED;
    switch (op) {
        /*** PURGE QUEUE ***/
        case APS_OP_QPURGE:
            switch (lpr->config->impl) {
                case LPR_IMPL_MINIMAL:
                    /* not supported */
                    break;
                case LPR_IMPL_COMPLIANT:
                case LPR_IMPL_BSD:
                    /* lprm -P[printer] - */
                    result = GetCommandOutput(lpr->config->cmd_lprm,
                        "-P%s -", NULL,
                        APSCFG_LPR_TIMEOUT_LPRM_PURGE, NULL, NULL, NULL,
                        printer->name);
                    FIXME("Unchecked output messages from lprm");
                    return result;
                case LPR_IMPL_LPRNG:
                    /* lprm -P[printer] all */
                    result = GetCommandOutput(lpr->config->cmd_lprm,
                        "-P%s all", NULL,
                        APSCFG_LPR_TIMEOUT_LPRM_PURGE, NULL, NULL, NULL,
                        printer->name);
                    FIXME("Unchecked output messages from lprm");
                    return result;
            }
            break;

        /*** SUSPEND QUEUE ***/
        case APS_OP_QSUSPEND:
            if ((lpr->config->impl != LPR_IMPL_MINIMAL)&&
                (lpr->config->impl != LPR_IMPL_COMPLIANT)) {
                /* lpc stop [printer] */
                result = LprDoLPCCommand(thisBase, printer, "stop %s");
                return result;
            }
            break;

        /*** RESUME QUEUE ***/
        case APS_OP_QRESUME:
            if ((lpr->config->impl != LPR_IMPL_MINIMAL)&&
                (lpr->config->impl != LPR_IMPL_COMPLIANT)) {
                /* lpc start [printer] */
                result = LprDoLPCCommand(thisBase, printer, "start %s");
                return result;
            }
            break;

        /*** PAUSE ALL JOBS NOW! ***/
        case APS_OP_QPAUSEALLJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc hold [printer] all */
                result = LprDoLPCCommand(thisBase, printer, "hold %s all");
                return result;
            }
            break;

        /*** RESUME ALL JOBS NOW! ***/
        case APS_OP_QRESUMEALLJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc release [printer] all */
                result = LprDoLPCCommand(thisBase, printer, "release %s all");
                return result;
            }
            break;

        /*** PAUSE NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTOPAUSENEWJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc holdall [printer] */
                result = LprDoLPCCommand(thisBase, printer, "holdall %s");
                return result;
            }
            break;

        /*** DON'T PAUSE NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QNOAUTOPAUSENEWJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG) {
                /* lpc noholdall [printer] */
                result = LprDoLPCCommand(thisBase, printer, "noholdall %s");
                return result;
            }
            break;

        /*** DENY CREATION/QUEUEING OF NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTODENYNEWJOBS:
            if ((lpr->config->impl != LPR_IMPL_MINIMAL)&&
                (lpr->config->impl != LPR_IMPL_COMPLIANT)) {
                /* lpc disable [printer] */
                result = LprDoLPCCommand(thisBase, printer, "disable %s");
                return result;
            }
            break;

        /*** DON'T DENY CREATION/QUEUEING OF NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QNOAUTODENYNEWJOBS:
            if ((lpr->config->impl != LPR_IMPL_MINIMAL)&&
                (lpr->config->impl != LPR_IMPL_COMPLIANT)) {
                /* lpc enable [printer] */
                result = LprDoLPCCommand(thisBase, printer, "enable %s");
                return result;
            }
            break;

        /*** SILENTLY DISCARD NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTODISCARDNEWJOBS:
            /* not supported */
            break;

        /*** DON'T SILENTLY DISCARD NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QNOAUTODISCARDNEWJOBS:
            /* not supported */
            break;

        /*** IS QUEUE READY? ***/
        case APS_IS_QREADY:
            *((int *)argout) = (LprAnyGetQueueStatus(lpr, printer)
                & APS_PRINTER_READY) ? TRUE : FALSE;
            return APS_SUCCESS;

        /*** IS QUEUE SUSPENDED? ***/
        case APS_IS_QSUSPENDED:
            *((int *)argout) = (LprAnyGetQueueStatus(lpr, printer)
                & APS_PRINTER_SUSPENDED) ? TRUE : FALSE;
            return APS_SUCCESS;

        /*** IS QUEUE AUTOMATICALLY PAUSING NEW JOBS? ***/
        case APS_IS_QAUTOPAUSENEWJOBS:
            *((int *)argout) = (LprAnyGetQueueStatus(lpr, printer)
                & APS_PRINTER_PAUSE_NEW_JOBS) ? TRUE : FALSE;
            return APS_SUCCESS;

        /*** IS QUEUE AUTOMATICALLY DENYING NEW JOBS? ***/
        case APS_IS_QAUTODENYNEWJOBS:
            *((int *)argout) = (LprAnyGetQueueStatus(lpr, printer)
                & APS_PRINTER_DENY_NEW_JOBS) ? TRUE : FALSE;
            return APS_SUCCESS;

        /*** IS QUEUE AUTOMATICALLY DISCARDING NEW JOBS? ***/
        case APS_IS_QAUTODISCARDNEWJOBS:
            *((int *)argout) = (LprAnyGetQueueStatus(lpr, printer)
                & APS_PRINTER_DISCARD_NEW_JOBS) ? TRUE : FALSE;
            return APS_SUCCESS;

        /*** UNKNOWN -- MAY BE FUTURE CODE -- MARK UNSUPPORTED ***/
        default:
            break;
    }
    return (result);
}

/* --------------------------------------------------------------------
 * LprDoLPCCommand()
 *
 * Calls upon LPC to perform some task on behalf of the transport.
 *
 * Parameters: lpr      - lprtransport object pointer
 *             printer  - printer object pointer
 *             fmt      - argument format string
 *
 *     Return: Detailed Aps_Result code for the application.
 */
static Aps_Result LprDoLPCCommand(Transport *thisBase,
    Printer *printer, const char *fmt)
{
    Aps_Result result;
    LprTransport *lpr = (LprTransport *)thisBase;

    /* Do command normally */
    result = GetCommandOutput(lpr->config->cmd_lpc,
        fmt, NULL, APSCFG_LPR_TIMEOUT_LPC_DEFAULT, NULL, NULL, NULL,
        printer->name);

    /* If default printer and not 'lp' */
    /* LPRng and others sometimes fail to note the connection between
     * 'lp' and its alias */
    if (strcmp(printer->name, "lp") != 0) {
        int def;
        LprPrinterIsDefault(thisBase, printer, & def);
        if (def) {
            Aps_Result xresult;
            /* ignore result as it may be that the transport doesn't
             * understand these semantics */
            xresult = GetCommandOutput(lpr->config->cmd_lpc,
                fmt, NULL, APSCFG_LPR_TIMEOUT_LPC_DEFAULT, NULL, NULL, NULL,
                "lp");
        }
    }

    /* update on success */
    FIXME("Unchecked output messages from lpc");
    if (result == APS_SUCCESS) LprRebuildQueue(thisBase, printer);
    return result;
}

/* ---------------------------------------------------------------------------
 * LprJobIsOperationAvailable()
 *
 * Determines whether or not the specified operation can currently be
 * performed on a particular APS object.
 *
 * Parameters: thisBase          - A pointer to this transport
 *
 *             job               - A pointer to a Job object
 *
 *             operation         - The operation that the application is
 *                                 interested in.
 *
 *             anticipatedResult - APS_OPERATION_AVAILABLE if this operation
 *                                 is currently available, or the anticipated
 *                                 reason for failure if it is known that
 *                                 this operation cannot be performed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             APS_SUCCESS       - test is accurate
 *             APS_IGNORED       - test cannot be performed
 *             APS_NOT_FOUND     - operation ID not recognized
 */
static Aps_Result LprJobIsOperationAvailable(Transport *thisBase, Job *job,
    Aps_OperationID op, Aps_Result *anticipatedResult)
{
    Aps_Result result = APS_SUCCESS;
    LprTransport *lpr = (LprTransport *)thisBase;

    *anticipatedResult = APS_NOT_SUPPORTED;
    switch (op) {
        /*** DELETE JOB PERMANENTLY ***/
        /*** CANCEL JOB BUT LEAVE IN QUEUE ***/
        case APS_OP_JCANCEL:
        case APS_OP_JDELETE:
            if (lpr->config->impl != LPR_IMPL_MINIMAL)
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** PAUSE JOB ***/
        /*** RESUME JOB ***/
        case APS_OP_JPAUSE:
        case APS_OP_JRESUME:
            if ((lpr->config->impl == LPR_IMPL_BSD) ||
                (lpr->config->impl == LPR_IMPL_LPRNG))
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** STORE JOB NOW ***/
        /*** STORE JOB WHEN COMPLETE ***/
        case APS_OP_JSTORENOW:
        case APS_OP_JSTOREWHENCOMPLETE:
            break;

        /*** UNKNOWN -- MAY BE FUTURE CODE -- MARK NOT FOUND ***/
        default:
            result = APS_NOT_FOUND;
            break;
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * LprQueueIsOperationAvailable()
 *
 * Determines whether or not the specified operation can currently be
 * performed on a particular APS object.
 *
 * Parameters: thisBase          - A pointer to this transport
 *
 *             queue             - A pointer to a Queue object
 *
 *             operation         - The operation that the application is
 *                                 interested in.
 *
 *             anticipatedResult - APS_OPERATION_AVAILABLE if this operation
 *                                 is currently available, or the anticipated
 *                                 reason for failure if it is known that
 *                                 this operation cannot be performed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             APS_SUCCESS       - test is accurate
 *             APS_IGNORED       - test cannot be performed
 *             APS_NOT_FOUND     - operation ID not recognized
 */
static Aps_Result LprQueueIsOperationAvailable(Transport *thisBase,
    Printer *printer, Aps_OperationID op, Aps_Result *anticipatedResult)
{
    Aps_Result result = APS_SUCCESS;
    LprTransport *lpr = (LprTransport *)thisBase;

    *anticipatedResult = APS_NOT_SUPPORTED;
    switch (op) {
        /*** PURGE QUEUE ***/
        case APS_OP_QPURGE:
            if (lpr->config->impl != LPR_IMPL_MINIMAL)
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** SUSPEND QUEUE ***/
        /*** RESUME QUEUE ***/
        case APS_OP_QSUSPEND:
        case APS_OP_QRESUME:
            if ((lpr->config->impl == LPR_IMPL_BSD) ||
                (lpr->config->impl == LPR_IMPL_LPRNG))
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** PAUSE ALL JOBS NOW! ***/
        /*** RESUME ALL JOBS NOW! ***/
        case APS_OP_QPAUSEALLJOBS:
        case APS_OP_QRESUMEALLJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG)
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** PAUSE NEW JOBS AUTOMATICALLY ***/
        /*** DON'T PAUSE NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTOPAUSENEWJOBS:
        case APS_OP_QNOAUTOPAUSENEWJOBS:
            if (lpr->config->impl == LPR_IMPL_LPRNG)
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** DENY CREATION/QUEUEING OF NEW JOBS AUTOMATICALLY ***/
        /*** DON'T DENY CREATION/QUEUEING OF NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTODENYNEWJOBS:
        case APS_OP_QNOAUTODENYNEWJOBS:
            if ((lpr->config->impl == LPR_IMPL_BSD) ||
                (lpr->config->impl == LPR_IMPL_LPRNG))
                *anticipatedResult = APS_OPERATION_AVAILABLE;
            break;

        /*** SILENTLY DISCARD NEW JOBS AUTOMATICALLY ***/
        /*** DON'T SILENTLY DISCARD NEW JOBS AUTOMATICALLY ***/
        case APS_OP_QAUTODISCARDNEWJOBS:
        case APS_OP_QNOAUTODISCARDNEWJOBS:
            break;

        /*** IS QUEUE READY? ***/
        /*** IS QUEUE SUSPENDED? ***/
        /*** IS QUEUE AUTOMATICALLY PAUSING NEW JOBS? ***/
        /*** IS QUEUE AUTOMATICALLY DENYING NEW JOBS? ***/
        /*** IS QUEUE AUTOMATICALLY DISCARDING NEW JOBS? ***/
        case APS_IS_QREADY:
        case APS_IS_QSUSPENDED:
        case APS_IS_QAUTOPAUSENEWJOBS:
        case APS_IS_QAUTODENYNEWJOBS:
        case APS_IS_QAUTODISCARDNEWJOBS:
            if (lpr->config->impl != LPR_IMPL_MINIMAL) {
                *anticipatedResult = APS_OPERATION_AVAILABLE;
                if (lpr->config->impl == LPR_IMPL_COMPLIANT) {
                    result = APS_IGNORED;
                }
            }
            break;

        /*** UNKNOWN -- MAY BE FUTURE CODE -- MARK NOT FOUND ***/
        default:
            result = APS_NOT_FOUND;
            break;
    }
    return result;

}

/* ---------------------------------------------------------------------------
 * LprGetDefaultPrinterName()
 *
 * Obtains the name of the default printer.
 *
 * Parameters: thisBase   - A pointer to the abstract Transport structure for
 *                          the lpr transport.
 *
 *     Return: A pointer to a string containing the default printer for this
 *             transport, or NULL on failure (such as if this transport
 *             doesn't have any printers installed at this time).
 *             Free using Aps_ReleaseBuffer()
 */
static char *LprGetDefaultPrinterName(Transport * thisBase)
{
    LprTransport *this = (LprTransport *) thisBase;
    Aps_Printcap_Info *printcapEntry;
    int i;
    char *nameToReturn;

    /* Update the internal printer name cache. Fail if we don't have
     * a proper one.
     */
    LprUpdatePrintcapCache(thisBase);

    /* Loop through all known printers, looking for the first one labelled */
    /* as default. */
    for (i = 0; i < this->numberOfPrintcapEntries; ++i) {
        printcapEntry = this->apsPrintcapInfo[i];
        if (printcapEntry->defaultPrinter) {
            /* Found it. */
            nameToReturn = TrackMemDupString(NULL,
                printcapEntry->printerName, 0);
            return (nameToReturn);
        }
    }

    /*** HACK: Find better solution after release. ***/
    WARN("No printer was marked as default, selecting and marking first found");

    /* If we get to this point, then we didn't find any printer labelled as */
    /* default, so pick the first printer, if any. */
    if (this->numberOfPrintcapEntries > 0) {
        printcapEntry = this->apsPrintcapInfo[0];
        printcapEntry->defaultPrinter = TRUE;
        nameToReturn = TrackMemDupString(NULL,
            printcapEntry->printerName, 0);
        return (nameToReturn);
    }

    /* No printers. */
    return (NULL);
}

/*---------------------------------------------------------------------------
 * LprGetPPDFileName()
 *
 * Obtains the name of the PPD file associated with a particular printer,
 * if any.
 *
 * Parameters: this    - A pointer to the lpr transport managing this
 *                       the printer.
 *
 *             printer - A pointer to the printer being queried.
 *
 *    Return: A pointer to a newly allocated string containing the name
 *            of the PPD file, or NULL on failure. The string will be
 *            empty if no PPD file is associated with the printer.
 *            Free using Aps_ReleaseBuffer().
 * 
 */
static char *LprGetPPDFileName(Transport * this, Printer * printer)
{
    char *ppdFilename;
    char *rString;

    if (MetaRead(&ppdFilename, printer->name, "ppd") != APS_SUCCESS)
        return NULL;

    rString = TrackMemDupString(NULL, ppdFilename, 0);
    free(ppdFilename);
    return rString;
}

/* ---------------------------------------------------------------------------
 * LprSetPPDFileName()
 *
 * Changes the PPD file associated with a particular printer.
 *
 * Parameters: this    - A pointer to the lpr transport managing this
 *                       the printer.
 *
 *             printer - A pointer to the printer being queried.
 *
 *             name    - A string containing the name of the PPD file to be
 *                       associated with this printer, or an empty string
 *                       if no PPD file should be associated with the
 *                       printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result LprSetPPDFileName(Transport * this, Printer * printer,
                                    const char *name)
{
    return (MetaWrite(printer->name, "ppd", name));
}

/* ---------------------------------------------------------------------------
 * IsPrinterUnique()
 *
 * Checks if the printer name exists in the printcap file.
 *
 * Parameters: printerName    - name of the printer.
 *
 *             printcapInfo   - A pointer to array of pointers storing 
 *                              printcap entries.
 *
 *             numberOfEntry  - Number of printcap entries in the array.
 *
 *     Return: TRUE, if the printerName is new and FALSE otherwise.
 */
static BOOL IsPrinterNameUnique(const char *printerName,
                       Aps_Printcap_Info ** printcapInfo, int numberOfEntry)
{
    int i;

    for (i = 0; i < numberOfEntry; i++) {
        if (LprPrintcapTestByName(printcapInfo[i], printerName))
            return FALSE;
    }
    return TRUE;
}
/*---------------------------------------------------------------------------
 * AddPrinterEntryInPrintcap()
 *
 * Adds the printcap info for the new printer in the printcap array.
 *
 * Parameters: newEntry    - pointer to the structure storing the printcap 
 *                           info for the new printer. 
 *
 *             thisBase    - A pointer to the lpr transport layer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */

static Aps_Result AddPrinterEntryInPrintcap(Aps_Printcap_Info * newEntry,
                                            Transport * thisBase)
{
    LprTransport *this = (LprTransport *) thisBase;
    Aps_Printcap_Info **tempPrintcap;
    Aps_Result result;

    /* Used to check the return value of realloc */

    /*Check if the printer name to be added is there in printcap already */
    if (IsPrinterNameUnique(newEntry->printerName,
             this->apsPrintcapInfo, this->numberOfPrintcapEntries) == FALSE) {
        ERROR("A printer by the name of %s already exists",
            newEntry->printerName);
        return APS_ALREADY_EXISTS;
    }
    /* this printer name already exists in printcap */

    /* Allocate memory for 1 more pointer in the printcap array */
    tempPrintcap = (Aps_Printcap_Info **)
        realloc(this->apsPrintcapInfo,
         (this->numberOfPrintcapEntries + 1) * sizeof(Aps_Printcap_Info *));
    if (tempPrintcap == NULL)
        return APS_GENERIC_FAILURE;

    /* Add the new printer entry in the printcap Array */
    tempPrintcap[this->numberOfPrintcapEntries] = newEntry;
    result = WritePrinterEntriesInPrintcap(this->config->file_printcap,
        tempPrintcap, (this->numberOfPrintcapEntries + 1));
    if (result != APS_SUCCESS) return result;

    this->apsPrintcapInfo = tempPrintcap;
    /* Increment the printcap entry count by 1 */
    this->numberOfPrintcapEntries++;
    /* Update printcap timestamp. */
    LprNotifyPrintcapWasModified(thisBase, NULL, -1);

    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * GetPrinterRecPositionInArray()
 *
 * Finds the position of the printer rec in the printcap array for the 
 * printer name passed 
 *
 * Parameters: printerName - Name of the printer.
 *
 *             printcapInfo - the pointer to the printcap array storing the 
 *                            pointers to different printer records.
 *             numberOfEntry - Number of printers recoed in printcap array.
 *
 *     Return: index of the printer record in the princap file, if it exists
 *             -1 otherwise.
 */
static int GetPrinterRecPositionInArray(const char *printerName,
                       Aps_Printcap_Info ** printcapInfo, int numberOfEntry)
{
    int i;

    for (i = 0; i < numberOfEntry; i++) {
        if (LprPrintcapTestByName(printcapInfo[i], printerName)) return i;
    }
    return -1;
}

/*---------------------------------------------------------------------------
 * DeletePrinterEntryInPrintcap()
 *
 * Deleted the printer record from the printcap array for the printer name 
 * passed as argument.
 * 
 *
 * Parameters: printerName - Name of the printer.
 *
 *             thisBase    - the pointer to the transport layer.
 *
 *     Return: APS_NOT_FOUND, if the record for the printer to be deleted 
 *             is not found in the printcap array and APS_SUCCESS otherwise.
 */
static Aps_Result DeletePrinterEntryInPrintcap(const char *printerName,
                                               Transport * thisBase)
{
    Aps_Result result;
    LprTransport *this = (LprTransport *) thisBase;
    Aps_Printcap_Info **dupPrintcapArray;
    Aps_Printcap_Info *rec;
    int pos,       /* Position of printer record in the printcap array */
        i;         /* Temporary variable */

    /* Get the position of printer entry to be deleted in printcap, if exists */
    if ((pos = GetPrinterRecPositionInArray(printerName,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
        /* this printer name does not exist in printcap */
        return APS_NOT_FOUND;
    /* Save the pointer to be deleted */
    rec = this->apsPrintcapInfo[pos];
    /* Create the duplicate printcap array */
    dupPrintcapArray = (Aps_Printcap_Info **)
        malloc(sizeof(Aps_Printcap_Info *) *
               this->numberOfPrintcapEntries);
    for (i = 0; i < this->numberOfPrintcapEntries; i++) {
        dupPrintcapArray[i] = this->apsPrintcapInfo[i];
    }
    /* Rearrange all other valid entries in the printcap Array */
    for (i = pos; i < (this->numberOfPrintcapEntries - 1); i++)
        dupPrintcapArray[i] = dupPrintcapArray[i + 1];

    result = WritePrinterEntriesInPrintcap(this->config->file_printcap,
        dupPrintcapArray, (this->numberOfPrintcapEntries - 1));
    if (result != APS_SUCCESS) return result;

    /*free the memory occupied by the printcap array */
    /* make the last Aps_Printcap_Info pointer NULL in the printcap array */
    dupPrintcapArray[this->numberOfPrintcapEntries - 1] = NULL;
    free(this->apsPrintcapInfo);  /* Preblem is here */
    this->apsPrintcapInfo = dupPrintcapArray;

    /* Update printcap timestamp. */
    LprNotifyPrintcapWasModified(thisBase, NULL, -1);

    /* Delete the spool directory for this printer */
    if (!RemoveDirectory(rec->spoolDirectory, NULL))
        return APS_GENERIC_FAILURE;

    FreePrintcapInfoStructure(rec);      
    /* Decrement the printcap entry count by 1 */
    this->numberOfPrintcapEntries--;

    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * LprAddPrinter()
 *
 * Adds the entry for the new printer in the printcap file, Creates the 
 * Printer structure for the new printer and returns the handle to it if 
 * sought by the caller.
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             name     - the name of the printer.
 *             handle   - Handle to the newly added printer or NULL.
 *
 *     Return: APS_ACCESS_DENIED, if user does not have acces to printcap 
 *             file.
 *             APS_GENERIC_FAILURE, if there is a problem in creating the 
 *             directory or allocation of memory and APS_SUCCESS otherwise.
 */
Aps_Result LprAddPrinter(Transport * thisBase, const char *name,
                         Aps_PrinterHandle * handle)
{
    Aps_Printcap_Info *newPrintcapEntry;
    char *spoolDirName;
    LprTransport *this = (LprTransport *) thisBase;
    int i;
    int ret;

    /* So we have access to the printcap file.  Before going any further,
     * let's create the spool directory.
     */
    {
        int size;
        /* Does the spool's root exist?? */
        if ((ret = access(this->config->dir_spool, F_OK)) != 0) {
            if (mkdir(this->config->dir_spool, S_IRWXU | S_IRWXG | S_IROTH |
                S_IXOTH | S_ISVTX) != 0) return APS_ACCESS_DENIED;
        }
        /* Default name */
        if (! (spoolDirName = path_alloc(& size))) return APS_OUT_OF_MEMORY;
        sprintf(spoolDirName, "%s/%s", this->config->dir_spool, name);
        /* Avoid name clashes */
        for (i = 0; i < 37; i++) {
            /* If directory exists already, choose some other spool dir name */
            errno = 0;
            if (mkdir(spoolDirName, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH |
                S_ISVTX) == 0) {
                /* mkdir fails to set group write sometimes, hmm?? */
                chmod(spoolDirName, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH |
                   S_ISVTX);
                break;
            }
            /* Abort if error wasn't because of dupe directory */
            if ((errno != EEXIST) || (i == 36)) {
                free(spoolDirName);
                return APS_GENERIC_FAILURE;
            }
            /* Create new name and try again */
            sprintf(spoolDirName, "%s/%s%c", this->config->dir_spool, name,
                (i < 10) ? (i + 48) : (i + 55));
        }
    }


    /* Create the printcap structure, and initialize the minimal entries */
    if ( (newPrintcapEntry = (Aps_Printcap_Info *)
        malloc(sizeof(Aps_Printcap_Info))) ) {
        /* Init minimal fields */
        newPrintcapEntry->printerName = strdup(name);
        newPrintcapEntry->deviceName = strdup("/dev/null");
        newPrintcapEntry->spoolDirectory = strdup(spoolDirName);
        strcat(spoolDirName, "/filter");
        newPrintcapEntry->textFilter = strdup(spoolDirName);
        free(spoolDirName);
        spoolDirName = newPrintcapEntry->spoolDirectory; /* use as temp */
        if ((newPrintcapEntry->printerName) &&
           (newPrintcapEntry->deviceName) &&
           (newPrintcapEntry->spoolDirectory) &&
           (newPrintcapEntry->textFilter)) {
            LprFilterRH_TextOnlyCfg txtCfg; /* used below */

            newPrintcapEntry->defaultPrinter = 0;
            newPrintcapEntry->accountingFile = NULL;
            newPrintcapEntry->baudRate = 0;
            newPrintcapEntry->cifplotFilter = NULL;
            newPrintcapEntry->dftestDataFilter = NULL;
            newPrintcapEntry->isTTY = 0;
            newPrintcapEntry->formFeed = NULL;
            newPrintcapEntry->formFeedOnOpen = 0;
            newPrintcapEntry->isTTYBits = 0;
            newPrintcapEntry->graphDataFilter = NULL;
            newPrintcapEntry->headerPageLast = 0;
            newPrintcapEntry->ioctlSupport = 0;
            newPrintcapEntry->logFile = NULL;
            newPrintcapEntry->lockFile = NULL;
            newPrintcapEntry->maxFileSize = 0;
            newPrintcapEntry->nextDirectory = NULL;
            newPrintcapEntry->ditroffFilter = NULL;
            newPrintcapEntry->outputFilter = NULL;
            newPrintcapEntry->pricePerFoot = 0;
            newPrintcapEntry->pageLengthInLines = 0;
            newPrintcapEntry->pageWidthInChars = 0;
            newPrintcapEntry->pageWidthInPixels = 0;
            newPrintcapEntry->pageLengthInPixels = 0;
            newPrintcapEntry->fortranStyleFileFilter = NULL;
            newPrintcapEntry->restrictedGroup = NULL;
            newPrintcapEntry->remoteMachine = NULL;
            newPrintcapEntry->remotePrinterArgument = NULL;
            newPrintcapEntry->restrictToRemoteUsersWithLocalAcc = 0;
            newPrintcapEntry->openPrinterDeviceForReadWrite = 0;
            newPrintcapEntry->shortBanner = 0;
            newPrintcapEntry->suppressMultipleCopies = 0;
            newPrintcapEntry->supressFormFeed = 0;
            newPrintcapEntry->suppressBurstPageHeader = 0;
            newPrintcapEntry->statusFileName = NULL;
            newPrintcapEntry->troffDataFilter = NULL;
            newPrintcapEntry->trailingString = NULL;
            newPrintcapEntry->rasterImageFilter = NULL;

            /* Copy the filter launch script.  If we can't then we'll just
             * not use filters at all.  We will, of course, still create the
             * configs we need though.
             */
            if (APS_SUCCESS != CopyFileSpecial(
                NULL, this->config->file_masterfilter,
                spoolDirName, "filter",
                S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH,
                0, 0)) {
                /* On failure, fix printcap */
                free(newPrintcapEntry->textFilter);
                newPrintcapEntry->textFilter = NULL;
            }
            /* Create configs in spool directory before committing to printcap */
            LprFilterRH_txtcfgInit(& txtCfg); /* defaults are ok */
            if (APS_SUCCESS ==
                LprFilterRH_txtcfgWriteDefer(spoolDirName, & txtCfg)) {
                LprFilterRH_PostscriptCfg  psCfg;
                LprFilterRH_pscfgInit(& psCfg); /* defaults are ok */
                if (APS_SUCCESS ==
                    LprFilterRH_pscfgWriteDefer(spoolDirName, & psCfg)) {
                    LprFilterRH_GeneralCfg genCfg;
                    LprFilterRH_gencfgInit(& genCfg); /* defaults are ok */
                    if (APS_SUCCESS ==
                        LprFilterRH_gencfgWriteDefer(spoolDirName, & genCfg)) {
                        /* Add the new printer entry in the printcap array */
                        if ((ret = AddPrinterEntryInPrintcap(newPrintcapEntry,
                            thisBase)) == APS_SUCCESS) {
                            /* Commit all changes... */
                            LprFilterRH_txtcfgWriteCommit(spoolDirName, & txtCfg);
                            LprFilterRH_txtcfgRelease( & txtCfg);
                            LprFilterRH_pscfgWriteCommit(spoolDirName, & psCfg);
                            LprFilterRH_pscfgRelease( & psCfg);
                            LprFilterRH_gencfgWriteCommit(spoolDirName, & genCfg);
                            LprFilterRH_gencfgRelease( & genCfg);

                            /* Create an instance of the printer. */
                            ASSERT(handle != NULL);
                            *handle = PrinterGetHandleFromPtr((Printer*)
                                LprCreatePrinter((char *)name, thisBase));
                            /* Even if we failed to create an instance, we will not
                             * undo all the work we just did.  The printer may in
                             * fact have been created and we failed on some mundane
                             * condition like out-of-memory
                             */
                            if (*handle == NULL) return APS_GENERIC_FAILURE;
                            return APS_SUCCESS;
                        }
                        /* FAILED! Fallthrough! */
                        LprFilterRH_gencfgWriteAbort(spoolDirName, & genCfg);
                        LprFilterRH_gencfgRelease( & genCfg);
                    }
                    LprFilterRH_pscfgWriteAbort(spoolDirName, & psCfg);
                    LprFilterRH_pscfgRelease( & psCfg);
                }
                LprFilterRH_txtcfgWriteAbort(spoolDirName, & txtCfg);
                LprFilterRH_txtcfgRelease( & txtCfg);
            }
        }
        /* Free spool and printcap resources */
        RemoveDirectory(newPrintcapEntry->spoolDirectory, NULL); /* FIX ME! */
        if (newPrintcapEntry->printerName) free(newPrintcapEntry->printerName);
        if (newPrintcapEntry->deviceName) free(newPrintcapEntry->deviceName);
        if (newPrintcapEntry->spoolDirectory) free(newPrintcapEntry->spoolDirectory);
        if (newPrintcapEntry->textFilter) free(newPrintcapEntry->textFilter);
    } else {
        RemoveDirectory(spoolDirName, NULL); /* FIX ME! */
        free(spoolDirName);
    }
    free(newPrintcapEntry);
    return APS_GENERIC_FAILURE;
}

/*---------------------------------------------------------------------------
 * LprPrinterRemove()
 *
 * Deletes the printer entry from the printcap file for the handle passed a
 * an argument. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - Handle to be deleted.
 *
 *     Return: APS_ACCESS_DENIED if the user does not have access to the 
 *             printcap file.
 *             APS_GENERIC_FAILURE, if there is a problem in deletion 
 *             APS_SUCCESS otherwise.
 */
Aps_Result LprPrinterRemove(Transport * thisBase, Aps_PrinterHandle handle)
{
    Printer *printerObj;
    Aps_Result retValue;
/*    LprTransport *this = (LprTransport *) thisBase; */

    /* Find the printer handle from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Call DeletePrinterEntryInPrintcap function to delete the entry from 
     * the printcap array
     */
    if ((retValue = DeletePrinterEntryInPrintcap(printerObj->name,
                                                 thisBase)) != APS_SUCCESS)
        return (retValue);


    /* Free the memory associated with Printer object */
    Aps_ReleaseHandle(handle);

    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * LprPrinterGetMaxJobSize()
 *
 * Gets the Maximum Job Size set for the printer. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - handle of the printer to be set as deleted.
 *
 *     Return: APS_INVALID_HANDLE, if the printer handle is not valid,
 *             APS_NOT_FOUND, if the printer is not found in printcap
 *             APS_SUCCESS otherwise.
 */
static Aps_Result LprPrinterGetMaxJobSize(Transport * thisBase,
                               Aps_PrinterHandle handle, int *maxSize)
{
    LprTransport *this = (LprTransport *) thisBase;
    Printer *printerObj;
    int pos;  /* Position of the printer record in the printcap array */

    /* Find the printer pointer from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Get the position of printer entry to be deleted in printcap, 
     * if exists 
     */
    if ((pos = GetPrinterRecPositionInArray(printerObj->name,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
    /* this printer name does not exist in printcap */
        return APS_NOT_FOUND;
    if (this->apsPrintcapInfo[pos]->maxFileSize == 0)
        *maxSize = APS_NO_MAX_SIZE;
    else
        *maxSize = this->apsPrintcapInfo[pos]->maxFileSize;

    return APS_SUCCESS;
}
/*---------------------------------------------------------------------------
 * LprPrinterIsDefault()
 *
 * Checks if the printer,the handle for which is passed argument,is default. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - handle of the printer to be set as deleted.
 *
 *     Return: APS_INVALID_HANDLE, if the printer handle is not valid,
 *             APS_NOT_FOUND, if the printer is not found in printcap
 *             APS_SUCCESS otherwise.
 */
static Aps_Result LprPrinterIsDefault(Transport * thisBase,
                               Aps_PrinterHandle handle, int *isDefault)
{
    LprTransport *this = (LprTransport *) thisBase;
    Printer *printerObj;
    int pos;  /* Position of the printer record in the printcap array */

    /* Find the printer pointer from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Get the position of printer entry to be deleted in printcap, 
     * if exists 
     */
    if ((pos = GetPrinterRecPositionInArray(printerObj->name,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
        return APS_NOT_FOUND;
    /* this printer name does not exist in printcap */
    /* If this printer is already set as default,return TRUE */
    if (this->apsPrintcapInfo[pos]->defaultPrinter == TRUE)
        *isDefault = TRUE;
    else
        *isDefault = FALSE;

    return APS_SUCCESS;
}
/*---------------------------------------------------------------------------
 * LprPrinterSetAsDefault()
 *
 * Sets the printer, the handle for which is passed argument, as default. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - handle of the printer to be set as deleted.
 *
 *     Return: APS_NOT_FOUND, if the printer is not found in the printcap,
 *             APS_NO_CHANGE, if the printer is already set as 
 *             default printer  and   APS_SUCCESS otherwise.
 */
Aps_Result LprPrinterSetAsDefault(Transport * thisBase,
                                  Aps_PrinterHandle handle)
{
    Aps_Result result;
    LprTransport *this = (LprTransport *) thisBase;
    Printer *printerObj;
    Aps_Printcap_Info *rec;
    int recNo, currDefaultPrinter = -1, pos;
    /* The position of the printer set as default currently */

    /* Find the printer handle from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Get the position of printer entry to be deleted in printcap, 
     * if exists 
     */
    if ((pos = GetPrinterRecPositionInArray(printerObj->name,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
        return APS_NOT_FOUND;
    /* this printer name does not exist in printcap */
    rec = this->apsPrintcapInfo[pos];
    /* If this printer is already set as default , return */
    if (this->apsPrintcapInfo[pos]->defaultPrinter == TRUE)
        return APS_NO_CHANGE;
    /* Disable the other printer as default if set already */
    for (recNo = 0; recNo < this->numberOfPrintcapEntries; recNo++)
        if (this->apsPrintcapInfo[recNo]->defaultPrinter == TRUE) {
            this->apsPrintcapInfo[recNo]->defaultPrinter = FALSE;
            currDefaultPrinter = recNo;
        }
    /* Set the desired printer as default */
    this->apsPrintcapInfo[pos]->defaultPrinter = TRUE;
    result = WritePrinterEntriesInPrintcap(this->config->file_printcap,
        this->apsPrintcapInfo, this->numberOfPrintcapEntries);
    if (result != APS_SUCCESS) {
        this->apsPrintcapInfo[pos]->defaultPrinter = FALSE;
        if (currDefaultPrinter != -1)
            this->apsPrintcapInfo[currDefaultPrinter]->defaultPrinter = TRUE;
    }
    /* Update printcap timestamp. */
    LprNotifyPrintcapWasModified(thisBase, NULL, -1);
    return result;
}
/*---------------------------------------------------------------------------
 * LprPrinterSetMaxJobSize()
 *
 * Sets the max job size for the printer,the handle for which is 
 * passed argument. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - handle of the printer to be set as deleted.
 *             maxSize  - maximum job size.
 *
 *     Return: APS_ACCESS_DENIED if the user does not have access to printcap *             file.
 *             APS_NOT_FOUND, if the printer is not found in the printcap,
 *             APS_NO_CHANGE, if the printer is already set as 
 *             default printer  and   APS_SUCCESS otherwise.
 */
Aps_Result LprPrinterSetMaxJobSize(Transport * thisBase,
                                  Aps_PrinterHandle handle, int maxSize)
{
    LprTransport *this = (LprTransport *) thisBase;
    Printer *printerObj;
    Aps_Printcap_Info *rec;
    Aps_Result result;
    int oldMaxJobSize, pos;

    /* if the maxSize is negative discard it */
    if (maxSize < 0)
        return APS_INVALID_PARAM;
    /* Find the printer handle from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Get the position of printer entry to be deleted in printcap, 
     * if exists. 
     */
    if ((pos = GetPrinterRecPositionInArray(printerObj->name,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
        return APS_NOT_FOUND;
    /* this printer name does not exist in printcap */
    rec = this->apsPrintcapInfo[pos];
    /* If maxJobSize is already set to the desired value, return 
     * else set the maxJobSize to the maxSize.
     */
    oldMaxJobSize = rec->maxFileSize;
    if (rec->maxFileSize == maxSize)
        return APS_NO_CHANGE;
    else{
       if (maxSize == APS_NO_MAX_SIZE)
            rec->maxFileSize = 0;
       else
            rec->maxFileSize = maxSize;
    }

    /* Try to write changed entry */
    result = WritePrinterEntriesInPrintcap(this->config->file_printcap,
        this->apsPrintcapInfo, this->numberOfPrintcapEntries);
    if (result != APS_SUCCESS) {
        this->apsPrintcapInfo[pos]->maxFileSize = oldMaxJobSize;
    }
    /* Update printcap timestamp. */
    LprNotifyPrintcapWasModified(thisBase, NULL, -1);

    return result;
}
/*---------------------------------------------------------------------------
 * LprPrinterRename()
 *
 * Changes the name of the printer, the handle for which is passed 
 * argument, as default. 
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *             handle   - handle of the printer to be set as deleted.
 *
 *     Return: APS_ACCESS_DENIED if the user does not have access to printcap *             file.
 *             APS_NOT_FOUND, if the printer is not found in the printcap,
 *             APS_NO_CHANGE, if the new name is same as the
 *              current name and   APS_SUCCESS otherwise.
 */
Aps_Result LprPrinterRename(Transport * thisBase,
                            Aps_PrinterHandle handle, const char *newName)
{
    LprTransport *this = (LprTransport *) thisBase;
    Aps_Result result;
    Printer *printerObj;
    Aps_Printcap_Info *rec;
    char *newPrinterName, *oldName, *tmpPrinterName, *tmpOldName;
    int pos; /* The position of the printer set as default currently */

    /* Find the printer handle from the handle received */
    if ((printerObj = PrinterGetPtrFromHandle(handle)) == NULL)
        return APS_INVALID_HANDLE;

    /* Get the position of printer entry to be deleted in printcap, 
     * if exists 
     */
    if ((pos = GetPrinterRecPositionInArray(printerObj->name,
               this->apsPrintcapInfo, this->numberOfPrintcapEntries)) == -1)
        return APS_NOT_FOUND;

    /* this printer name does not exist in printcap */
    rec = this->apsPrintcapInfo[pos];

    /* Allocate the memory to change the name in Printer structure and in
     * printcap array 
     */
    if(((newPrinterName = (char *)malloc(strlen(newName)+1))== NULL)||
       ((tmpPrinterName = (char *)malloc(strlen(newName)+1))== NULL))
       return APS_GENERIC_FAILURE;

    /* Change the newName in the printcap array */
    oldName = rec->printerName;
    memset(newPrinterName, 0, (strlen(newName)+1));
    strcpy(newPrinterName, newName);
    rec->printerName = newPrinterName;

    /* Change the name in the Printer structure */
    tmpOldName = printerObj->name;
    memset(tmpPrinterName, 0, (strlen(newName)+1));
    strcpy(tmpPrinterName, newName);
    printerObj->name = tmpPrinterName;

    /* Write the new name in the printcap file */
    result = WritePrinterEntriesInPrintcap(this->config->file_printcap,
        this->apsPrintcapInfo, this->numberOfPrintcapEntries);
    if (result != APS_SUCCESS) {
        /* restore the previous name */
        rec->printerName = oldName;
        printerObj->name = tmpOldName;
        free(tmpPrinterName);
        free(newPrinterName);
    } else {
        free (oldName);
        free (tmpOldName);
    }

    /* Update printcap timestamp. */
    LprNotifyPrintcapWasModified(thisBase, NULL, -1);
    return result;
}
/*---------------------------------------------------------------------------
 * LprPrinterGetConfigFlags()
 *
 * Extracts the configuration flags for a specified printer.
 *
 * Parameters: thisBase - the pointer to the transport layer
 *             printer  - pointer to the printer object
 *             cfgFlags - the variable to get the configuration flags
 *
 *     Return: APS_ACCESS_DENIED, if the information is priviledged
 *             APS_NOT_FOUND, if the printer is not found in printcap
 *             APS_SUCCESS otherwise.
 */
static Aps_Result LprPrinterGetConfigFlags(
    Transport *thisBase, Printer *printer, long int *cfgFlags)
{
    Aps_Result result;
    LprFilterRH_TextOnlyCfg toCfg;
    LprFilterRH_GeneralCfg  genCfg;
    Aps_Printcap_Info *pi;

    *cfgFlags = 0;

    /* Locate printcap entry */
    {
        int pos;
        pos = GetPrinterRecPositionInArray(printer->name,
            ((LprTransport *)thisBase)->apsPrintcapInfo,
            ((LprTransport *)thisBase)->numberOfPrintcapEntries);
        if (pos == -1) return APS_NOT_FOUND;
        pi = ((LprTransport *)thisBase)->apsPrintcapInfo[pos];
    }

    /* Extract as much info as possible */
    LprFilterRH_txtcfgInit(& toCfg);
    result = LprFilterRH_txtcfgRead(pi->spoolDirectory, & toCfg);
    if (result == APS_SUCCESS) {
        if (toCfg.txtAddCR) *cfgFlags |= APS_CONFIG_ADD_CR;
        if (toCfg.txtSendEOF) *cfgFlags |= APS_CONFIG_EOF_AT_END;
        LprFilterRH_txtcfgRelease(& toCfg);
    }
    LprFilterRH_gencfgInit(& genCfg);
    result = LprFilterRH_gencfgRead(pi->spoolDirectory, & genCfg);
    if (result == APS_SUCCESS) {
        if (! genCfg.genASCIItoPS) *cfgFlags |= APS_CONFIG_TEXT_AS_TEXT;
    }
    LprFilterRH_gencfgRelease(& genCfg);
    if (pi->suppressBurstPageHeader) *cfgFlags |= APS_CONFIG_HEADER_PAGE;
    return APS_SUCCESS;
}

/*---------------------------------------------------------------------------
 * LprPrinterSetConfigFlags()
 *
 * Sets the configuration flags for a specified printer.
 *
 * Parameters: thisBase - the pointer to the transport layer
 *             printer  - pointer to the printer object
 *             flagMask - mask of flags which are affected
 *             flagSet  - flags to be set
 *
 *     Return: APS_INVALID_PARAM, if the flags are wrong (done by APS high-level)
 *             APS_ACCESS_DENIED, if the information is priviledged
 *             APS_NOT_FOUND, if the printer is not found in printcap
 *             APS_SUCCESS otherwise.
 */
static Aps_Result LprPrinterSetConfigFlags(
    Transport *thisBase, Printer *printer,
    long int flagMask, long int flagSet)
{
    Aps_Result result = APS_SUCCESS;
    Aps_Printcap_Info *pi;

    /* Locate printcap entry */
    {
        int pos;
        pos = GetPrinterRecPositionInArray(printer->name,
            ((LprTransport *)thisBase)->apsPrintcapInfo,
            ((LprTransport *)thisBase)->numberOfPrintcapEntries);
        if (pos == -1) return APS_NOT_FOUND;
        pi = ((LprTransport *)thisBase)->apsPrintcapInfo[pos];
    }

    /* Update as much info as possible until failure */
    /* Modify printcap first... */
    if (flagMask & APS_CONFIG_HEADER_PAGE) {
        BOOL sh = flagSet & APS_CONFIG_HEADER_PAGE;
        /* logical XOR would be useful below... */
        if ((pi->suppressBurstPageHeader && (!sh)) ||
            ((!pi->suppressBurstPageHeader) && sh)) {
            pi->suppressBurstPageHeader = sh;

            /* Update printcap -- RACE CONDITION */
            result = WritePrinterEntriesInPrintcap(
                ((LprTransport *)thisBase)->config->file_printcap,
                ((LprTransport *)thisBase)->apsPrintcapInfo,
                ((LprTransport *)thisBase)->numberOfPrintcapEntries);
            LprNotifyPrintcapWasModified(thisBase, NULL, -1);
            if (result != APS_SUCCESS) {
                pi->suppressBurstPageHeader =
                    ! (flagSet & APS_CONFIG_HEADER_PAGE);
                return result; /* BAIL OUT! */
            }
        }
    }
    /* Update textonly.cfg */
    if ( (flagMask & APS_CONFIG_ADD_CR) ||
         (flagMask & APS_CONFIG_EOF_AT_END) ) {
        LprFilterRH_TextOnlyCfg toCfg;
        LprFilterRH_txtcfgInit(& toCfg);
        result = LprFilterRH_txtcfgRead(pi->spoolDirectory, & toCfg);
        if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND)) {
            if (flagMask & APS_CONFIG_ADD_CR)
                toCfg.txtAddCR = flagSet & APS_CONFIG_ADD_CR;
            if (flagMask & APS_CONFIG_EOF_AT_END)
                toCfg.txtSendEOF = flagSet & APS_CONFIG_EOF_AT_END;
            result = LprFilterRH_txtcfgWrite(pi->spoolDirectory, & toCfg);
            LprFilterRH_txtcfgRelease(& toCfg);
        }
        if (result != APS_SUCCESS) return result; /* BAIL OUT! */
    }
    /* Update postscript.cfg */
    if ( (flagMask & APS_CONFIG_EOF_AT_END) ) {
        LprFilterRH_PostscriptCfg psCfg;
        LprFilterRH_pscfgInit(& psCfg);
        result = LprFilterRH_pscfgRead(pi->spoolDirectory, & psCfg);
        if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND)) {
            psCfg.psSendEOF = flagSet & APS_CONFIG_EOF_AT_END;
            result = LprFilterRH_pscfgWrite(pi->spoolDirectory, & psCfg);
            LprFilterRH_pscfgRelease(& psCfg);
        }
        if (result != APS_SUCCESS) return result; /* BAIL OUT! */
    }
    /* Update general.cfg */
    if ( (flagMask & APS_CONFIG_TEXT_AS_TEXT) ) {
        LprFilterRH_GeneralCfg genCfg;
        LprFilterRH_gencfgInit(& genCfg);
        result = LprFilterRH_gencfgRead(pi->spoolDirectory, & genCfg);
        if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND)) {
            genCfg.genASCIItoPS = ! (flagSet & APS_CONFIG_TEXT_AS_TEXT);
            result = LprFilterRH_gencfgWrite(pi->spoolDirectory, & genCfg);
        }
        LprFilterRH_gencfgRelease(& genCfg);
        if (result != APS_SUCCESS) return result; /* BAIL OUT! */
    }
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprPrinterSetConnectInfo()
 *
 * Changes the method and/or location used to connect to a particular printer.
 * The method can be via a local device, or via a network. In the case of a
 * local printer, the location is the name of the device that the printer is
 * connected to. In the case of a network printer, the location is the
 * network path to the printer.
 *
 * Parameters: printer        - A pointer to the printer object in question.
 *
 *             connectionType - An Aps_ConnectionType identifying the
 *                              mechanism (local / remote) used to
 *                              communicate with this printer.
 *
 *             location       - A string containing the device name or
 *                              network path for this printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 *
 * Expected network path types:
 *  LOCAL : <devicename>             (eg. /dev/lp0)
 *  SMB   : //<hostname>/<sharename>
 *  Unix  : //<hostname>/<queuename> (for symmetry)
 *
 * Paths must be of the form:
 *  w=<workgroup>\tu=<user>\tp=<password>\t<path>
 */
#define MAXOPTIONLENGTH 128
#define MAXPATHLENGTH APSCFG_MAX_NETWORK_PATH_SIZE

Aps_Result LprPrinterSetConnectInfo(
    Transport *thisBase, Printer *printer,
    Aps_ConnectionType connectionType, const char *location)
{
    Aps_Result result;
    Aps_Printcap_Info *pi;
    LprFilterRH_GeneralCfg genCfg;
    LprFilterRH_SMBNetworkCfg smbCfg;
    LprFilterRH_LPDNetworkCfg lpdCfg;
    char locUser[MAXOPTIONLENGTH],
         locPassword[MAXOPTIONLENGTH],
         locWorkgroup[MAXOPTIONLENGTH];
    char locPath[MAXPATHLENGTH];
    char locTempChar;

    /* Init */
    if ((! location)||(! *location)) return APS_INVALID_PARAM;

    /* Split location into locUser, locPassword, locWorkgroup and locPath */
    *locUser = '\0';
    *locPassword = '\0';
    *locWorkgroup = '\0';
    *locPath = '\0';
    result = APS_SUCCESS;
    while ((locTempChar = *location) && (location[1] == '=')) {
        /* locate end of field */
        const char *end = strchr(location + 2, '\t');
        int len;
        if (! end) return APS_NOT_FOUND;
        location += 2;
        len = end - location;
        if ((len + 1) > MAXOPTIONLENGTH) return APS_OUT_OF_MEMORY;

        /* setup effective location params */
        switch (locTempChar) {
            case 'u':
                strncpy(locUser, location, len);
                locUser[len] = '\0';
                break;
            case 'p':
                strncpy(locPassword, location, len);
                locPassword[len] = '\0';
                break;
            case 'w':
                strncpy(locWorkgroup, location, len);
                locWorkgroup[len] = '\0';
                break;
            default: /* ignore */
                break;
        }
        location = end + 1;
    }
    if ((strlen(location) + 1) > MAXPATHLENGTH) return APS_OUT_OF_MEMORY;
    strcpy(locPath, location);

    /* Locate printcap entry */
    {
        int pos;
        pos = GetPrinterRecPositionInArray(printer->name,
            ((LprTransport *)thisBase)->apsPrintcapInfo,
            ((LprTransport *)thisBase)->numberOfPrintcapEntries);
        if (pos == -1) return APS_NOT_FOUND;
        pi = ((LprTransport *)thisBase)->apsPrintcapInfo[pos];
    }

    /* Initialize the config file structures */
    result = LprFilterRH_lpdcfgInit(& lpdCfg);
    if (result == APS_SUCCESS) {
     result = LprFilterRH_smbcfgInit(& smbCfg);
     if (result == APS_SUCCESS) {
      result = LprFilterRH_gencfgInit(& genCfg);
      if (result == APS_SUCCESS) {
       Aps_ConnectionType prevConnectionType;
       /* Update the connection type in general.cfg */
       /* N.B. DEFERRED */
       result = LprFilterRH_gencfgRead(pi->spoolDirectory, & genCfg);
       if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND)) {
        prevConnectionType = genCfg.genPrinterType;
        genCfg.genPrinterType = connectionType;
        result = LprFilterRH_gencfgWriteDefer(pi->spoolDirectory, & genCfg);
       } else prevConnectionType = -1;
       if (result == APS_SUCCESS) {
        char *newLP, *newRM, *newRP;
        /* Prep. */
        newLP = pi->deviceName;
        newRM = pi->remoteMachine;
        newRP = pi->remotePrinterArgument;
        result = APS_SUCCESS;
        /* Setup stuff we need */
        switch (connectionType) {
         /*** LOCAL: <filesys device name> ***/
         case APS_CONNECT_LOCAL:
            newLP = strdup(locPath);
            newRM = NULL;
            newRP = NULL;
            break;
         /*** LPD  : //<hostname>/<queuename> ***/
         case APS_CONNECT_NETWORK_LPD:
            if ((*locPath == '/') || (locPath[1] == '/')) {
             const char *index, *rindex;
             size_t len;

             /* extract hostname */
             rindex = locPath + 2;
             index = strchr(rindex, '/');
             len = index - rindex;
             if (index && len) {
              /* Setup lp/rm/rh flags */
              newLP = NULL;
              if ( (newRM = malloc(len + 1)) ) {
               strncpy(newRM, rindex, len);
               newRM[len] = '\0';
               /* extract queuename */
               if (!(newRP = strdup(index + 1))) result = APS_OUT_OF_MEMORY;
              } else result = APS_OUT_OF_MEMORY;
             } else result = APS_NOT_FOUND;
#if (! APSCFG_LPR_LET_DAEMON_MANAGE_NETWORKS)
             /* Setup .config and .config.user */
             if (result == APS_SUCCESS) {
              char locQualifiedUser[MAXOPTIONLENGTH * 2];
              /* Do workgroup */
              if (* locWorkgroup) {
               snprintf(locQualifiedUser, MAXOPTIONLENGTH, "%s/%s",
                locWorkgroup, locUser);
              } else {
               strcpy(locQualifiedUser, locUser);
              }
              if ((newLP = strdup("/dev/null")) &&
                  strupdate(& lpdCfg.lpdUser, locQualifiedUser) &&
                  strupdate(& lpdCfg.lpdPassword, locPassword) &&
                  strupdate(& lpdCfg.lpdServer, newRM) &&
                  strupdate(& lpdCfg.lpdQueue, newRP)) {
               free(newRM); newRM = NULL;
               free(newRP); newRP = NULL;
              } else result = APS_OUT_OF_MEMORY;
             }
#endif
            } else result = APS_NOT_FOUND;
            break;
         /*** SMB  : //<hostname>/<path>/<share> ***/
         case APS_CONNECT_NETWORK_SMB:
            if ((*locPath == '/') || (locPath[1] == '/')) {
             char locQualifiedUser[MAXOPTIONLENGTH * 2];
             /* Do workgroup */
             if (* locWorkgroup) {
              snprintf(locQualifiedUser, MAXOPTIONLENGTH, "%s/%s",
               locWorkgroup, locUser);
             } else {
              strcpy(locQualifiedUser, locUser);
             }

             /* Setup .config and .config.user */
             newLP = NULL; newRM = NULL; newRP = NULL;

             if ((newLP = strdup("/dev/null")) &&
                 strupdate(& smbCfg.smbUser, locQualifiedUser) &&
                 strupdate(& smbCfg.smbPassword, locPassword) &&
                 strupdate(& smbCfg.smbHostIP, "") &&
                 strupdate(& smbCfg.smbShare, locPath)) {
              /* nothing else, really... */
             } else result = APS_OUT_OF_MEMORY;
            } else result = APS_NOT_FOUND;
            break;
         /*** HUNH?? ***/
         default:
            result = APS_INVALID_PARAM;
            break;
        }
        /* Okay, we're doing good! If we haven't failed yet then
         * we'll write the config back out safely */
        if (result == APS_SUCCESS) {
         switch (connectionType) {
          case APS_CONNECT_LOCAL: break;
          case APS_CONNECT_NETWORK_LPD:
           result = LprFilterRH_lpdcfgWriteDefer(pi->spoolDirectory, & lpdCfg);
           break;
          case APS_CONNECT_NETWORK_SMB:
           result = LprFilterRH_smbcfgWriteDefer(pi->spoolDirectory, & smbCfg);
           break;
          default: break; /* will never happen -- appease compiler warnings */
         }
        }
        /* Write back printcap, safely */
        if (result == APS_SUCCESS) {
         char *oldLP = pi->deviceName;
         char *oldRM = pi->remoteMachine;
         char *oldRP = pi->remotePrinterArgument;
         pi->deviceName = newLP;
         pi->remoteMachine = newRM;
         pi->remotePrinterArgument = newRP;
         result = WritePrinterEntriesInPrintcap(
          ((LprTransport *)thisBase)->config->file_printcap,
          ((LprTransport *)thisBase)->apsPrintcapInfo,
          ((LprTransport *)thisBase)->numberOfPrintcapEntries);
         /* Update printcap timestamp. */
         LprNotifyPrintcapWasModified(thisBase, NULL, -1);
         /* If succeeded, then commit changes */
         if (result == APS_SUCCESS) {
          if (oldLP) free(oldLP);
          if (oldRM) free(oldRM);
          if (oldRP) free(oldRP);
          switch (connectionType) {
           case APS_CONNECT_LOCAL: break;
           case APS_CONNECT_NETWORK_LPD:
            result = LprFilterRH_lpdcfgWriteCommit(pi->spoolDirectory, & lpdCfg);
            break;
           case APS_CONNECT_NETWORK_SMB:
            result = LprFilterRH_smbcfgWriteCommit(pi->spoolDirectory, & smbCfg);
            break;
           default: break; /* will never happen -- appease compiler warnings */
          }
          /* Now remove any unrelated configuration files */
          if ((connectionType == APS_CONNECT_LOCAL)
#if (APSCFG_LPR_LET_DAEMON_MANAGE_NETWORKS)
           || (connectionType == APS_CONNECT_NETWORK_LPD)
#endif
           ) {
           char configFilePath[MAXPATHLENGTH];
           snprintf(configFilePath, sizeof(configFilePath),
            "%s/%s", pi->spoolDirectory, ".config");
           unlink(configFilePath);
           snprintf(configFilePath, sizeof(configFilePath),
            "%s/%s", pi->spoolDirectory, ".config.user");
           unlink(configFilePath);
          }
         } else { /* Else abort */
          pi->deviceName = oldLP;
          pi->remoteMachine = oldRM;
          pi->remotePrinterArgument = oldRP;
          if (newLP) free(newLP);
          if (newRM) free(newRM);
          if (newRP) free(newRP);
          switch (connectionType) {
           case APS_CONNECT_LOCAL: break;
           case APS_CONNECT_NETWORK_LPD:
            result = LprFilterRH_lpdcfgWriteAbort(pi->spoolDirectory, & lpdCfg);
            break;
           case APS_CONNECT_NETWORK_SMB:
            result = LprFilterRH_smbcfgWriteAbort(pi->spoolDirectory, & smbCfg);
            break;
           default: break; /* will never happen -- appease compiler warnings */
          }
         }
        }
       }
       /* Commit changes to gencfg */
       if (result != APS_SUCCESS) {
        LprFilterRH_gencfgWriteAbort(pi->spoolDirectory, & genCfg);
       } else {
        LprFilterRH_gencfgWriteCommit(pi->spoolDirectory, & genCfg);
       }
       /* Fallthrough on earlier errors */
      }
      LprFilterRH_gencfgRelease(& genCfg);
     }
     LprFilterRH_smbcfgRelease(& smbCfg);
    }
    LprFilterRH_lpdcfgRelease(& lpdCfg);
    return result;
}

/* ---------------------------------------------------------------------------
 * LprPrinterGetConnectInfo()
 *
 * Obtains information on how a connection is established with a particular
 * printer. This includes the connection method, such as a printer connected
 * to a local device, or a network printer. It also includes a string
 * identifying the specific device or network name of the printer.
 *
 * Parameters: printer        - A pointer to the printer object in question.
 *
 *             connectionType - The address of an Aps_ConnectionType to
 *                              receive information on which type of
 *                              connection is being used.
 *
 *             location       - The address of a char * to receive a string
 *                              with the device/network name that this
 *                              printer is attached to. If this function
 *                              succeeds, it is the caller's responsibility
 *                              to dispose of this string by passing it to
 *                              Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 *
 * Returned paths:
 *  LOCAL : <devicename>             (eg. /dev/lp0)
 *  SMB   : //<hostname>/<sharename>
 *  Unix  : //<hostname>/<queuename> (for symmetry)
 *
 * Paths are prefixed with:
 *  u=<username>\t
 *  p=<password>\t
 *  w=<workgroup>\t
 */
Aps_Result LprPrinterGetConnectInfo(
    Transport *thisBase, Printer *printer,
    Aps_ConnectionType *connectionType, char **location)
{
    Aps_Printcap_Info *pi;
    Aps_Result result;

    /* Initialize */
    *connectionType = -1;
    *location = NULL;

    /* Locate printcap entry */
    {
        int pos;
        pos = GetPrinterRecPositionInArray(printer->name,
            ((LprTransport *)thisBase)->apsPrintcapInfo,
            ((LprTransport *)thisBase)->numberOfPrintcapEntries);
        if (pos == -1) return APS_NOT_FOUND;
        pi = ((LprTransport *)thisBase)->apsPrintcapInfo[pos];
    }

    /* Extract connection type from general.cfg */
    {
        LprFilterRH_GeneralCfg genCfg;
        LprFilterRH_gencfgInit(& genCfg);
        result = LprFilterRH_gencfgRead(pi->spoolDirectory, & genCfg);
        if ( (result == APS_SUCCESS) || (result == APS_NOT_FOUND) ) {
            *connectionType = genCfg.genPrinterType;
        } else return result; /* BAIL OUT! */
        LprFilterRH_gencfgRelease(& genCfg);
    }

    /* If APS_NOT_FOUND, we'll try each possibility, else we will
     * base our decision exclusively on what the filter told us
     * was true
     */

    /*** TRY LPD ***/
    if ( (*connectionType == APS_CONNECT_NETWORK_LPD) ||
        (result == APS_NOT_FOUND) ) {
        /* Check use of printcap (ie. daemon is doing the work)
         */
        if ((pi->remoteMachine) || (pi->remotePrinterArgument)) {
            char *host = pi->remoteMachine;
            char *queue = pi->remotePrinterArgument;

            if ((host) && (queue)) {
                *connectionType = APS_CONNECT_NETWORK_LPD;
                *location = TrackMemAlloc(NULL,
                    strlen(host) + strlen(queue) + 4, 0);
                if (! *location) return APS_OUT_OF_MEMORY;
                sprintf(*location, "//%s/%s", host, queue);
                return APS_SUCCESS;
            }
        } else {
        /* Try .config. We expect both remoteMachine and remotePrinterArgument
         * to be NULL.  Then try to extract information from .config file.
         */
            Aps_Result xresult;
            if ((! pi->remoteMachine) && (! pi->remotePrinterArgument)) {
                LprFilterRH_LPDNetworkCfg lpdCfg;
                LprFilterRH_lpdcfgInit(& lpdCfg);
                xresult = LprFilterRH_lpdcfgRead(pi->spoolDirectory, & lpdCfg);
                if (xresult == APS_SUCCESS) {
                    char locAll[MAXOPTIONLENGTH * 2 + MAXPATHLENGTH];
                    *locAll = '\0';
                    *connectionType = APS_CONNECT_NETWORK_LPD;
                    if (strisdef(lpdCfg.lpdUser)) {
                        snprintf(locAll, sizeof(locAll), "u=%s\t", lpdCfg.lpdUser);
                    }
                    if (strisdef(lpdCfg.lpdPassword)) {
                        snprintf(locAll + strlen(locAll),
                            sizeof(locAll), "p=%s\t", lpdCfg.lpdPassword);
                    }
                    snprintf(locAll + strlen(locAll),
                        sizeof(locAll), "//%s/%s", lpdCfg.lpdServer, lpdCfg.lpdQueue);
                    *location = strdup(locAll);
                    if (! *location) xresult = APS_OUT_OF_MEMORY;
                    LprFilterRH_lpdcfgRelease(& lpdCfg);
                    return xresult;
                }
            }
        }
        if (result != APS_NOT_FOUND) return APS_GENERIC_FAILURE;
    }

    /*** TRY SMB ***/
    if ( (*connectionType == APS_CONNECT_NETWORK_SMB) ||
        (result == APS_NOT_FOUND) ) {
        Aps_Result xresult;

        /* if SMB, then we expect both remoteMachine and remotePrinterArgument
         * to be NULL.  Then try to extract information from .config file.
         */
        if ((! pi->remoteMachine) && (! pi->remotePrinterArgument)) {
            LprFilterRH_SMBNetworkCfg smbCfg;
            LprFilterRH_smbcfgInit(& smbCfg);
            xresult = LprFilterRH_smbcfgRead(pi->spoolDirectory, & smbCfg);
            if (xresult == APS_SUCCESS) {
                char locAll[MAXOPTIONLENGTH * 2 + MAXPATHLENGTH];
                *locAll = '\0';
                *connectionType = APS_CONNECT_NETWORK_SMB;
                if (strisdef(smbCfg.smbUser)) {
                    snprintf(locAll, sizeof(locAll), "u=%s\t", smbCfg.smbUser);
                }
                if (strisdef(smbCfg.smbPassword)) {
                    snprintf(locAll + strlen(locAll),
                        sizeof(locAll), "p=%s\t", smbCfg.smbPassword);
                }
                snprintf(locAll + strlen(locAll),
                    sizeof(locAll), "%s", smbCfg.smbShare);
                *location = TrackMemDupString(NULL, locAll, 0);
                if (! *location) xresult = APS_OUT_OF_MEMORY;
                LprFilterRH_smbcfgRelease(& smbCfg);
                return xresult;
            }
        }
        if (result != APS_NOT_FOUND) return APS_GENERIC_FAILURE;
    }

    /*** ASSUME LOCAL ***/
    *connectionType = APS_CONNECT_LOCAL;
    *location = TrackMemDupString(NULL,
        (pi->deviceName)? pi->deviceName : "", 0);
    if (! *location) return APS_OUT_OF_MEMORY;
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprProbeModelForPrinter()
 *
 * Attempts to determine the model of a particular printer. Called if APS
 * isn't already configured with this information for this printer.
 *
 * Parameters: thisBase - Pointer to the base Transport for the LPR transport.
 *
 *             printer  - Pointer to a printer owned by this Transport.
 *
 *             model    - Receives a handle to a model object describing this
 *                        printer, or NULL on failure.
 *
 *     Return: APS_SUCCESS on success, APS_MODEL_UNKNOWN if this function
 *             was unable to determine the model type, or another Aps_Result
 *             on a more serious failure.
 */
static Aps_Result LprProbeModelForPrinter(Transport *thisBase,
                                          Printer *printer,
                                          Aps_ModelHandle *model)
{
    char *printerName = NULL;
    char *ifFilterName = NULL;
    int i = 0;
    Aps_Result result = APS_MODEL_UNKNOWN;
    LprTransport *this = (LprTransport *) thisBase;
    
    ASSERT(thisBase != NULL);
    ASSERT(printer != NULL);
    ASSERT(model != NULL);
    /* Initially, we have no model object for this printer. */
    *model = NULL;
    
    /* Printer* will give us the printer name. */
    printerName = printer->name; 

    /* We go through all printcap enteries to determine
     * the input fitername, the if entry. 
     */
    for(i = 0; i < this->numberOfPrintcapEntries; i++ ) {
        if( strcmp( printerName, 
              (this->apsPrintcapInfo[i]->printerName)) == 0 ) {
            ifFilterName = this->apsPrintcapInfo[i]->textFilter;
            break;
        }
    }
    if(ifFilterName) {
        char *gsdevice;
        char *resolution ;
        const char *fields[1] = {"gsdevice"};
        const char *value[1];
        

        /* Running the script to get gsdevice and resolution.
         */
        GetScriptOutput( printerName, ifFilterName,
                         &gsdevice, &resolution);
        if(gsdevice != NULL){
            value[0] = gsdevice;
            /* Searching printerdb for more information on this 
             * printer.
             */
            result = ModelDbSearch( model, fields, value, 1);
            if(result == APS_NOT_FOUND)
               result = APS_MODEL_UNKNOWN;
        }

        /* Release temporary memory. */
        if (gsdevice != NULL)
            free(gsdevice);
        if (resolution != NULL)
            free(resolution);
    }
 
    /* The LPR transport doesn't currently do any probing for model type;
     * user has to configure this information manually.
     */

    return result;
}

/* ---------------------------------------------------------------------------
 * LprPrinterAttrGetDefaults()
 *
 * Get default printer configuration attributes.
 * Should extract / guess as much meaningful information as possible
 * and assign to correct providers for processing.
 *
 * Parameters: thisBase - pointer to transport object
 *             printer  - pointer to printer to query
 *             attr     - attributes structure to update
 *
 * Result:     (not yet defined)
 */
static Aps_Result LprPrinterAttrGetDefaults(Transport *thisBase,
    Printer *printer, JobAttributes *attr)
{
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

/* ---------------------------------------------------------------------------
 * LprPrinterAttrSetDefaults()
 *
 * Set default printer configuration based on a set of attributes.
 * Should extract as much meaningful information from these attribs
 * (using the various providers) and update the cached context or
 * other configs based on that.
 *
 * Parameters: thisBase - pointer to transport object
 *             printer  - pointer to printer to query
 *             attr     - attributes structure to use
 *
 * Result:     APS_SUCCESS if the relevant attributes have been stored
 */
/*** Update string based on attribute value ***/
#define ATTRUPDATE_STRING(attrName, attrDef, attrStorage) \
    setting = NULL; \
    result = Aps_AttrGetSetting(jobAttr, (attrName), & setting); \
    if ((result != APS_SUCCESS) || (!setting)) setting = (attrDef); \
    if (setting) strupdate(& attrStorage, setting); \
    if (result == APS_SUCCESS) Aps_ReleaseBuffer(setting);

/*** Update integer based on attribute value ***/
#define ATTRUPDATE_INT(attrName, attrDef, attrStorage) \
    setting = NULL; \
    result = Aps_AttrGetSetting(jobAttr, (attrName), & setting); \
    if ((result != APS_SUCCESS) || (!setting)) setting = (attrDef); \
    if (setting) attrStorage = atoi(setting); \
    if (result == APS_SUCCESS) Aps_ReleaseBuffer(setting);

/*** Update pair of integers based on attribute value ***/
#define ATTRUPDATE_PAIR(attrName, attrDef, attrPre, attrStorage1, attrMid, attrStorage2, attrPost) \
    setting = NULL; \
    result = Aps_AttrGetSetting(jobAttr, (attrName), & setting); \
    if ((result != APS_SUCCESS) || (!setting)) setting = (attrDef); \
    if (setting) { \
        sscanf( setting, attrPre ## "%d" ## attrMid ## "%d" ## attrPost , \
            & attrStorage1, & attrStorage2); \
    } \
    if (result == APS_SUCCESS) Aps_ReleaseBuffer(setting);

static Aps_Result LprPrinterAttrSetDefaults(Transport *thisBase,
    Printer *printer, JobAttributes *attr)
{
    Aps_Result result;
    Aps_Printcap_Info *pi;
    LprFilterRH_GeneralCfg genCfg;
    LprFilterRH_PostscriptCfg psCfg;
    LprFilterRH_TextOnlyCfg txtCfg;
    Aps_JobAttrHandle jobAttr;

    ASSERT(thisBase && printer && attr);

    /* Locate printcap entry */
    {
        int pos;
        pos = GetPrinterRecPositionInArray(printer->name,
            ((LprTransport *)thisBase)->apsPrintcapInfo,
            ((LprTransport *)thisBase)->numberOfPrintcapEntries);
        if (pos == -1) return APS_NOT_FOUND;
        pi = ((LprTransport *)thisBase)->apsPrintcapInfo[pos];
    }
    jobAttr = JobAttrGetHandleFromPtr(attr);

    /* Initialize the config file structures and read current settings */
    result = LprFilterRH_txtcfgInit(& txtCfg);
    if (result == APS_SUCCESS) {
      result = LprFilterRH_txtcfgRead(pi->spoolDirectory, & txtCfg);
      if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND))
          result = LprFilterRH_pscfgInit(& psCfg);
      if (result == APS_SUCCESS) {
        result = LprFilterRH_pscfgRead(pi->spoolDirectory, & psCfg);
        if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND))
            result = LprFilterRH_gencfgInit(& genCfg);
        if (result == APS_SUCCESS) {
          result = LprFilterRH_gencfgRead(pi->spoolDirectory, & genCfg);
          if ((result == APS_SUCCESS) || (result == APS_NOT_FOUND)) {
            char *setting;
            /*** Update things to do with GS Attribute Provider ***/
            /* colorrendering */
            ATTRUPDATE_STRING("colorrendering", "", psCfg.psGSUniprintDrv);
            /* gsdevice */
            ATTRUPDATE_STRING("gsdevice", "POSTSCRIPT", psCfg.psGSDevice);
            /* gsoptions */
            ATTRUPDATE_STRING("gsoptions", "", psCfg.psGSOptions);
            /* n-up */
            ATTRUPDATE_INT("n-up", "1-up", psCfg.psNUPCount);
            /* *PageSize */
            ATTRUPDATE_STRING("*PageSize", "Letter", psCfg.psPaperSize);
            ATTRUPDATE_STRING("*PageSize", "Letter", genCfg.genPaperSize);
            /* *PaperDimension */
            /* *Resolution */
            ATTRUPDATE_PAIR("*Resolution", "0x0", "", psCfg.psXDPI, "x", psCfg.psYDPI, "");
            /* LeftMargin, TopMargin, RightMargin, BottomMargin */
            /* hmmm... should be split up perhaps??
             * FIXME -- best we can do for now */
            FIXME("Using attribute's Top/Left margins, Right/Bottom margins"
                  "ignored");
            ATTRUPDATE_INT("LeftMargin", "0", psCfg.psNUPHMargin);
            ATTRUPDATE_INT("TopMargin", "0", psCfg.psNUPVMargin);

            /*** Nothing else right now ***/

            /*** Update stored settings, safely ***/
            result = LprFilterRH_txtcfgWriteDefer(pi->spoolDirectory, & txtCfg);
            if (result == APS_SUCCESS) {
              result = LprFilterRH_pscfgWriteDefer(pi->spoolDirectory, & psCfg);
              if (result == APS_SUCCESS) {
                result = LprFilterRH_gencfgWriteDefer(pi->spoolDirectory, & genCfg);
                if (result == APS_SUCCESS) {
                  LprFilterRH_txtcfgWriteCommit(pi->spoolDirectory, & txtCfg);
                  LprFilterRH_pscfgWriteCommit(pi->spoolDirectory, & psCfg);
                  LprFilterRH_gencfgWriteCommit(pi->spoolDirectory, & genCfg);
                  /* SUCCESS!! */
                } else {
                /* Begin error fallthrough */
                LprFilterRH_txtcfgWriteAbort(pi->spoolDirectory, & txtCfg);
                LprFilterRH_pscfgWriteAbort(pi->spoolDirectory, & psCfg);
                }
              } else LprFilterRH_txtcfgWriteAbort(pi->spoolDirectory, & txtCfg);
            }
          }
          LprFilterRH_gencfgRelease(& genCfg);
        }
        LprFilterRH_pscfgRelease(& psCfg);
      }
      LprFilterRH_txtcfgRelease(& txtCfg);
    }
    return result;
}
#undef ATTRUPDATE_STRING
#undef ATTRUPDATE_INT
#undef ATTRUPDATE_PAIR

/* ---------------------------------------------------------------------------
 * LprPrinterLocateByName()
 *
 * Lookup a printer's printcap info data by name.  Returns NULL if it could
 * not be found.
 * This function is deliberately lightweight.
 *
 * Parameters: lpr      - pointer to LPR transport object
 *             name     - name of printer to search for
 *
 * Result:     Returns pointer to Aps_Printcap_Info structure if found,
 *             else NULL.
 */
Aps_Printcap_Info *LprPrintcapLocateByName(LprTransport *lpr,
    const char *name)
{
    int pos;
    pos = GetPrinterRecPositionInArray(name,
        lpr->apsPrintcapInfo, lpr->numberOfPrintcapEntries);
    if (pos == -1) return NULL;
    return lpr->apsPrintcapInfo[pos];
}

/* ---------------------------------------------------------------------------
 * LprPrintcapTestByName()
 *
 * Determine if the provided name matches the name of the printer spec'd
 * by the printcap structure.
 *
 * Parameters: pi       - pointer to Aps_Printcap_Info to test
 *             name     - name to match
 *
 * Result:     TRUE if matched, else FALSE.
 */
int LprPrintcapTestByName(Aps_Printcap_Info *pi, const char *name)
{
    return
        (strcmp(pi->printerName, name) == 0) ||             /* name */
        ((pi->defaultPrinter) && (strcmp("lp", name) == 0)); /* def alias */
}

/* ---------------------------------------------------------------------------
 * LprComparePrinterNames()
 *
 * Compare the names of two printers.
 *
 * Parameters: lpr      - pointer to LPR transport object
 *             name1    - name of printer to match
 *             name2    - 2nd name to match against
 *
 * Result:     TRUE if matched, else FALSE.
 */
int LprComparePrinterNames(LprTransport *lpr, const char *name1,
   const char *name2)
{
    Aps_Printcap_Info *pi;
    int pos;
    /* First check if names match superficially */
    if (strcmp(name1, name2) == 0) return TRUE;
    /* Now go get the printcap structure for one of them */
    pos = GetPrinterRecPositionInArray(name1,
        lpr->apsPrintcapInfo, lpr->numberOfPrintcapEntries);
    if (pos == -1) return FALSE;
    pi = lpr->apsPrintcapInfo[pos];
    /* Check if printcap matches name */
    return LprPrintcapTestByName(pi, name2);
}

#endif /* APSCFG_LPR_BUILD */
