/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: notificationsub.c
 *
 * Description: Implementation of the NotificationSub object, and instance
 *              of which exists for each subscription that the application
 *              registers in order to be notified when a particular event
 *              takes place.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "notificationsub.h"
#include "jobattributes.h"

DEBUG_CHANNEL_DEFAULT(object)

/* ---------------------------------------------------------------------------
 * NotificationGetPtrFromHandle()
 *
 * Obtains a pointer to a notification subscription instance given an
 * Aps_NotificationHandle.
 *
 * Parameters: handle - The handle received from the application.
 *
 *     Return: A NotificationSub pointer; NULL on failure.
 */
NotificationSub *NotificationGetPtrFromHandle(Aps_NotificationHandle handle)
{
    NotificationSub *notificationSub;

    if (handle == NULL)
        return NULL;

    notificationSub = (NotificationSub *) handle;

    /* Check to see that this looks like a valid Printer object. */
    if (notificationSub->baseClass.identifier != NOTIFICATION_HANDLE)
        return NULL;

    return notificationSub;
}

/* ---------------------------------------------------------------------------
 * NotificationGetHandleFromPtr()
 *
 * Obtains a handle that the client application can use to refer to a
 * notification subscription instance.
 *
 * Parameters: notificationSub - A pointer to a NotificationSub instance.
 *
 *     Return: The associated Aps_NotificationHandle.
 */
Aps_NotificationHandle NotificationGetHandleFromPtr(
                                          NotificationSub * notificationSub)
{
    ASSERT(notificationSub != NULL);

    return (Aps_NotificationHandle) notificationSub;
}

/* ---------------------------------------------------------------------------
 * NotificationCreate()
 *
 * Creates a new, empty notification subscription object.
 *
 * Parameters: notificationSub - A NotificationSub pointer to receive a
 *                               reference to the new notification
 *                               subscription object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result NotificationCreate(NotificationSub ** notificationSub)
{
    Aps_Result result;

    /* Attempt to allocate space for the new NotificationSub. */
    *notificationSub = (NotificationSub *)malloc(sizeof(NotificationSub));
    if (*notificationSub == NULL)
        return APS_OUT_OF_MEMORY;

    /* Attempt to initialize the base information common to all publically-
     * visible APS objects.
     */
    result = ObjectInitialize(&(*notificationSub)->baseClass,
        NOTIFICATION_HANDLE);
    if(result != APS_SUCCESS) {
        free(*notificationSub);
        return result;
    }

    /* Initialize member variables to their default values. */
    (*notificationSub)->associatedObject = NULL;
    (*notificationSub)->callbackFunction = NULL;
    (*notificationSub)->appData = NULL;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * NotificationDelete()
 *
 * Deallocates a notification subscription object.
 *
 * Parameters: notificationSub - A pointer to a NotificationSub instance.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
void NotificationDelete(NotificationSub * notificationSub)
{
    ASSERT(notificationSub != NULL);

    /* Detach this notification subscription from any associated target
     * object.
     */
    if (notificationSub->associatedObject != NULL) {
        NotificationDetach(notificationSub, notificationSub->associatedObject);
    }

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    notificationSub->baseClass.identifier = INVALID_HANDLE;

    /* Deallocate the memory used by the NotificationSub structure. */
    free(notificationSub);
}

/* ---------------------------------------------------------------------------
 * NotificationAttachCB()
 *
 * Attaches a callback function to be notified when the specified event
 * takes place in the specified object.
 *
 * Parameters: notificationSub - A pointer to a NotificationSub instance.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result NotificationAttachCB(NotificationSub * notificationSub,
                                ApsObject *associatedObject,
                                Aps_Event event,
                                Aps_NotificationCallback callbackFunction,
                                void *appData)
{
    ASSERT(notificationSub != NULL);
    ASSERT(associatedObject != NULL);

    /* The current implementation of notification subscriptions only supports
     * a single target object associated with each subscription, so we just
     * store the information passed to this function in simple member
     * variables of NotificationSub.
     */
    notificationSub->associatedObject = associatedObject;
    notificationSub->eventType = event;
    notificationSub->callbackFunction = callbackFunction;
    notificationSub->appData = appData;

    /* The current implementation of this function can never fail, but we
     * maintain a return code to allow for future implementations that
     * are more complex (such as would be the case if multiple target
     * objects could be associated with each notification subscription).
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * NotificationDetach()
 *
 * Detaches this notification subscription from the specified object,
 * deleting the subscription if this is the last object associated with the
 * subscription.
 *
 * Parameters: notificationSub  - A pointer to a NotificationSub instance.
 *
 *             associatedObject - The target object to be detached from this
 *                                notification subscription.
 *
 *     Return: None.
 */
void NotificationDetach(NotificationSub * notificationSub,
                        ApsObject *associatedObject)
{
    /* The current implementation assumes one target object per notification
     * subscription.
     */
    ASSERT(notificationSub->associatedObject == associatedObject);

    /* Remove this notification subscription from the associated target
     * object's list of notfication subscriptions.
     */
    ObjectRemoveNotificationSub(associatedObject, notificationSub);

    /* Mark this subscription as no longer having an associated target
     * object.
     */
    notificationSub->associatedObject = NULL;

    /* Since we currently never have more than object associated with each
     * notification subscription, delete the subscription object now.
     */
    NotificationDelete(notificationSub);
}

/* ---------------------------------------------------------------------------
 * NotificationMatchesEvent()
 *
 * Determines whether this notification subscription is set to respond to
 * the specified type of event.
 *
 * Parameters: notificationSub - A pointer to a NotificationSub instance.
 *
 *             event           - The type of event to test for.
 *
 *     Return: TRUE if this subscription handles this type of event.
 */
int NotificationMatchesEvent(NotificationSub * notificationSub,
                             Aps_Event event)
{
    ASSERT(notificationSub != NULL);

    /* Currently, each notification subscription only responds to a single
     * type of event, so we just check the specified event type against this
     * subscription's event type.
     */
    return(notificationSub->eventType == event);
}

/* ---------------------------------------------------------------------------
 * NotificationSend()
 *
 * Sends a notification to the subscriber by whatever means this subscription
 * calls for.
 *
 * Parameters: notificationSub - A pointer to a NotificationSub instance.
 *
 *             object          - The APS object on which this event has
 *                               taken place.
 *
 *             event           - The type of event that has occurred.
 *
 *             eventData       - Event specific data associated with this
 *                               event, if there is any.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
void NotificationSend(NotificationSub * notificationSub, ApsObject *object,
                      Aps_Event event, void *eventData)
{
    ASSERT(notificationSub != NULL);
    ASSERT(notificationSub->associatedObject == object);

    /* Currently, the only supported notification mechanism is callback
     * functions, so pass this call on to the application-provided callback.
     */
    ASSERT(notificationSub->callbackFunction != NULL);
    notificationSub->callbackFunction(ObjectGetHandleFromPtr(object),
                                      event,
                                      eventData,
                                      notificationSub->appData);
}
