/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: object.h
 *
 * Description: Common base for all APS object types.
 *
 */

#ifndef OBJECT_H
#define OBJECT_H

typedef enum HandleType_ HandleType;
typedef struct ApsObject_ ApsObject;

#include "aps.h"
#include "apsinternal.h"
#include "pointerarray.h"

/* Identifier used to distingush different types of handles from one   */
/* another, by examining the first byte of the structure pointed to by */
/* the handle.                                                         */
enum HandleType_ {
    INVALID_HANDLE,               /* Special value for deleted objects */
    PRINTER_HANDLE,
    JOB_HANDLE,
    JOB_ATTRIBUTES_HANDLE,
    NOTIFICATION_HANDLE,
    MODEL_HANDLE,
    QUEUE_HANDLE,
    FILTER_HANDLE
};

/* Base object itself. */
struct ApsObject_ {
    HandleType identifier;         /* marks the type of handle */
    int refCount;                  /* # of outstanding ref's, freed at 0 */

    /* For objects with property sets */
    PointerArray properties;       /* array of properties and values */

    /* For objects with notification */
    PointerArray notificationSubs; /* array of notification structures */

    /* For objects with filtering */
    Aps_FilterHandle  filterHandle; /* handle of attached filter */
    Aps_FilterOptions filterOpts;   /* filtering options */
};

/* Object property types. */
typedef enum {
    PROPERTY_EMPTY,
    PROPERTY_STRING,
    PROPERTY_STRING_ARRAY,
    PROPERTY_ATTRIBUTE,
} PropertyType;

/* Definitions for notification subscriptions. */
#include "notificationsub.h"

/* APS internal functions that operate on a generic APS object. */
Aps_Handle ObjectGetHandleFromPtr(ApsObject *object);
ApsObject *ObjectGetPtrFromHandle(Aps_Handle handle);
Aps_Result ObjectInitialize(ApsObject *object, HandleType type);
void ObjectSendEventNotification(ApsObject *object, Aps_Event event,
                                 void *eventData);
void ObjectRemoveNotificationSub(ApsObject *object,
                                 NotificationSub * notificationSub);

/* Definitions for properties, including attribute-type properties. */
#include "attribute.h"
#include "attrprovider.h"

Aps_Result ObjectGetPropertyAttribute(ApsObject *object, const char *name,
                                      Attribute **attribute);
Aps_Result ObjectGetOrAddPropertyAttribute(ApsObject *object, const char *name,
                                           Attribute **attribute,
                                           AttrProvider *provider);
Aps_Result ObjectGetProperties(ApsObject *object, TrackArray_PtrChar *names);
PropertyType ObjectGetPropertyType(ApsObject *object, const char *name);
void ObjectRemoveProperty(ApsObject *object, const char *name);
int ObjectDoesPropertyExist(ApsObject *object, const char *name);

#endif /* OBJECT_H */
