/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: ppdattrprovider.c
 *
 * Description: Implementation for job attributes that are based on
 *              information from a PPD file.
 *
 */

#include <stdio.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "ppdattrprovider.h"
#include "metaconfig.h"
#include "ppdmanager.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(attrprov)

/* Prototypes for implementation of attribute provider virtual functions. */
static void PPDAttrDestructor(AttrProvider * thisBase);
static Aps_Result PPDAttrCreateCopy(AttrProvider * thisBase,
                                    AttrProvider ** newProviderBase,
                                    Aps_Handle associatedObject);
static Aps_Result PPDAttrGetList(AttrProvider * thisBase,
                                 const char *group,
                                 TrackArray_PtrChar *attributeIDs);
static Aps_Result PPDAttrGetSubGroups(AttrProvider * thisBase,
                                      const char *group,
                                      TrackArray_PtrChar *subGroupNames);
static Aps_Result PPDAttrGetTranslatedName(AttrProvider * thisBase,
                                           const char *attributeID,
                                           char **translatedName);
static Aps_Result PPDAttrGetMainData(AttrProvider * thisBase,
                                     const char *attributeID,
                                     char **mainData);
static Aps_Result PPDAttrGetTranslatedData(AttrProvider * thisBase,
                                           const char *attributeID,
                                           char **translatedData);
static Aps_Result PPDAttrGetType(AttrProvider * thisBase,
                                 const char *attributeID,
                                 Aps_AttrType * attributeType);
static Aps_Result PPDAttrGetRange(AttrProvider * thisBase,
                                  const char *attributeID,
                                  double *minSetting,
                                  double *maxSetting);
static Aps_Result PPDAttrGetOptions(AttrProvider * thisBase,
                                    const char *attributeID,
                                    TrackArrayIndirect_AttrOption *options);
static Aps_Result PPDAttrGetSetting(AttrProvider * thisBase,
                                    const char *attributeID,
                                    char **setting);
static Aps_Result PPDAttrSetSetting(AttrProvider * thisBase,
                                    const char *attributeID,
                                    const char *setting);
static Aps_Result PPDAttrCheckConstraints(AttrProvider * thisBase,
                                          const char *attributeID,
                                          const char *setting,
                                          char **conflictingAttribute,
                                          char **conflictingSetting);
static Aps_Result PPDAttrWriteBlock(AttrProvider * thisBase,
                                    Aps_JobHandle job,
                                    Aps_BlockType blockType);
static Aps_Result PPDAttrResetToModelDefaults(AttrProvider * thisBase,
                                              Printer * printer);
static Aps_Result PPDAttrSaveAsPrinterDefaults(AttrProvider * thisBase,
                                               Printer * printer);
static Aps_Result PPDAttrSetToPrinterDefaults(AttrProvider * thisBase,
                                              Printer * printer);
static Aps_Result PPDAttrProvRemovePrinterDefaults(AttrProvider * this,
                                                   Printer * printer);

/* Virtual function pointer table for the PPD attribute provider. */
static AttrProviderVtbl PPDAttrProviderVtbl =
{
    PPDAttrDestructor,
    PPDAttrCreateCopy,
    PPDAttrGetList,
    PPDAttrGetSubGroups,
    PPDAttrGetTranslatedName,
    PPDAttrGetMainData,
    PPDAttrGetTranslatedData,
    PPDAttrGetType,
    PPDAttrGetRange,
    PPDAttrGetOptions,
    PPDAttrGetSetting,
    PPDAttrSetSetting,
    PPDAttrCheckConstraints,
    PPDAttrWriteBlock,
    PPDAttrResetToModelDefaults,
    PPDAttrSaveAsPrinterDefaults,
    PPDAttrSetToPrinterDefaults,
    PPDAttrProvRemovePrinterDefaults
};

/* ---------------------------------------------------------------------------
 * PPDAttrCreate()
 *
 * Create a new PPD Attribute Provider instance.
 *
 * Parameters: attrProvider     - The address of a PPDAttrProvider * to
 *                                receive a pointer to the new attribute
 *                                provider.
 *
 *             associatedObject - The APS object that this attribute provider
 *                                is providing attributes to.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result PPDAttrCreate(PPDAttrProvider ** attrProvider,
                         Aps_Handle associatedObject)
{
    Aps_Result result;

    /* Allocate memory for a PPD attribute provider. */
    *attrProvider = calloc(1, sizeof(PPDAttrProvider));
    if (*attrProvider == NULL) {
        return APS_OUT_OF_MEMORY;
    }
    /* Setup pointer to virtual function pointer table. */
    (*attrProvider)->baseClass.vtbl = &PPDAttrProviderVtbl;

    /* Call generic attribute provider initialization function. */
    result = AttrProvInitialize((AttrProvider *) * attrProvider,
                                associatedObject);
    if (result != APS_SUCCESS) {
        (*attrProvider)->baseClass.vtbl->Destructor(
                                             (AttrProvider *) attrProvider);
        return (result);
    }

    /* Create a new PPD manager. */
    (*attrProvider)->ppdManager = PPDCreate();
    if ((*attrProvider)->ppdManager == NULL) {
        (*attrProvider)->baseClass.vtbl->Destructor(
                                           (AttrProvider *) * attrProvider);
        return APS_OUT_OF_MEMORY;
    }
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PPDAttrCreateCopy()
 *
 * Creates a deep copy of an existing attribute provider that is semantically
 * identical to the original.
 *
 * Parameters: thisBase         - The existing attribute provider to copy.
 *
 *             newProviderBase  - A pointer to receive the new attribute
 *                                provider.
 *
 *             associatedObject - The associated job attributes or other
 *                                APS object for which this provider is
 *                                implementing 0 or more attributes.
 *
 *     Return: APS_SUCCESS on success, or APS_OUT_OF_MEMORY if insufficient
 *             memory was available to copy this attribute provider.
 */
Aps_Result PPDAttrCreateCopy(AttrProvider * thisBase,
                             AttrProvider ** newProviderBase,
                             Aps_Handle associatedObject)
{
    PPDAttrProvider *this = (PPDAttrProvider *)thisBase;
    PPDAttrProvider *newProvider;
    Aps_Result result;

    /* Allocate memory for a PPD attribute provider. */
    *newProviderBase = calloc(1, sizeof(PPDAttrProvider));
    if (*newProviderBase == NULL) {
        return APS_OUT_OF_MEMORY;
    }
    newProvider = (PPDAttrProvider *)*newProviderBase;
    
    /* Setup pointer to virtual function pointer table. */
    newProvider->baseClass.vtbl = &PPDAttrProviderVtbl;

    /* Call subclass initialization function. */
    result = AttrProvInitialize(*newProviderBase,
                                associatedObject);
    if (result != APS_SUCCESS) {
        AttrProvDelete(*newProviderBase);
        return result;
    }

    /* Create a copy of the PPD manager. */
    newProvider->ppdManager = PPDCreateCopy(this->ppdManager);
    if (newProvider->ppdManager == NULL) {
        return APS_OUT_OF_MEMORY;
    }

    /* We've successfully cloned ourselves. */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PPDAttrDestructor()
 *
 * Called to delete a PPD Attribute Provider instance.
 *
 * Parameters: thisBase - A pointer the the base AttrProvider instance for
 *                        this PPD Attribute Provider.
 *
 *     Return: void
 */
static void PPDAttrDestructor(AttrProvider * thisBase)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;

    ASSERT(this != NULL);

    /* Release the PPD Manager, if one exists. */
    if (this->ppdManager != NULL) {
        PPDDestroy(this->ppdManager);
    }
    
    /* Call base class destructor. */
    AttrProvDestructor(thisBase);
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetList()
 *
 * Obtains a list of individual attributes available from this attribute
 * provider. This can be a list of all attributes, or only the subset of
 * attributes specified by the group parameter.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be retrieved, or APS_GROUP_??? to
 *                             retrieve a predefined subset of attributes.
 *
 *             attributeIDs  - Pointer to an array that receives a list of
 *                             attribute IDs.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
static Aps_Result PPDAttrGetList(AttrProvider * thisBase,
                                 const char *group,
                                 TrackArray_PtrChar *attributeIDs)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    PPDPosition pos;
    char *value;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    if (group == APS_GROUP_ROOT_SETTINGS) {
        /* Begin iterating through all root-level UI keys. */
        if (!PPDGetUIsIterateStart(this->ppdManager, &pos, NULL))
            return APS_GENERIC_FAILURE;

        /* Loop until there are no more. */
        while (pos != NULL) {
            char *newString;
            value = NULL;
            if (!PPDGetUIsIterateNext(this->ppdManager, &pos, &value, NULL,
                                     NULL))
                return APS_GENERIC_FAILURE;
            newString = (char *)TrackMemDupString(*attributeIDs, value, 0);
            PPDDeleteString(this->ppdManager, &value);
            if (! TrackArrayAddLast_PtrChar(attributeIDs, newString))
                return APS_OUT_OF_MEMORY;
        }
        PPDIterateEnd(this->ppdManager, &pos);

    } else if (group == APS_GROUP_ALL_SETTINGS) {
        /* Begin iterating through all UI keys at any level. */
        if (!PPDGetAllUIsIterateStart(this->ppdManager, &pos))
            return APS_GENERIC_FAILURE;

        /* (FIXME?) Why is all this code duplicated?? ***/
        /* Loop until there are no more. */
        while (pos != NULL) {
            char *newString;
            value = NULL;
            if (!PPDGetUIsIterateNext(this->ppdManager, &pos, &value, NULL,
                                     NULL))
                return APS_GENERIC_FAILURE;
            newString = (char *)TrackMemDupString(*attributeIDs, value, 0);
            PPDDeleteString(this->ppdManager, &value);
            if (! TrackArrayAddLast_PtrChar(attributeIDs, newString))
                return APS_OUT_OF_MEMORY;
        }
        PPDIterateEnd(this->ppdManager, &pos);

    } else if (group == APS_GROUP_READ_ONLY_DATA) {
        FIXME("Not implemented: group == APS_GROUP_READ_ONLY_DATA");
        return APS_NOT_IMPLEMENTED;

    } else {
        FIXME("Not implemented: group did not match any in stacked if/else");
        return APS_NOT_IMPLEMENTED;
    }
    
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetSubGroups()
 *
 * Obtains a list of attribute groups belonging to another attribute group.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             group         - A string containing the name of the attributes
 *                             group to be searched for subgroups,
 *                             APS_GROUP_ROOT to obtain groups at the root
 *                             level, or APS_GROUP_ROOT_SETTINGS to obtain
 *                             root level groups or user-controllable
 *                             settings.
 *
 *             subGroupNames - Pointer to an array to receive a list of group
 *                             names.
 *
 *     Return: APS_SUCCESS on success, or a standard Aps_Result code.
 */
static Aps_Result PPDAttrGetSubGroups(AttrProvider * thisBase,
                                      const char *group,
                                      TrackArray_PtrChar *subGroupNames)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    
    if (group == APS_GROUP_ROOT_SETTINGS) {
        FIXME("Not implemented: group == APS_GROUP_ROOT_SETTINGS");
        return APS_NOT_IMPLEMENTED;

    } else if (group == APS_GROUP_ROOT) {
        FIXME("Not implemented: group == APS_GROUP_ROOT");
        return APS_NOT_IMPLEMENTED;

    } else if (group == APS_GROUP_ALL_SETTINGS) {
        FIXME("Not implemented: group == APS_GROUP_ALL_SETTINGS");
        return APS_NOT_IMPLEMENTED;

    } else if (group == APS_GROUP_READ_ONLY_DATA) {
        FIXME("Not implemented: group == APS_GROUP_READ_ONLY_DATA");
        return APS_NOT_IMPLEMENTED;

    } else {
        FIXME("Not implemented: group did not match any in stacked if/else");
        return APS_NOT_IMPLEMENTED;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetTranslatedName()
 *
 * Obtains the local translated name for the specified attribute. This is
 * the text that should be displayed to the user when presenting a list of
 * settings that can be altered.
 *
 * Parameters: thisBase       - A pointer the the base AttrProvider instance
 *                              for this PPD Attribute Provider.
 *
 *             atributeID     - A string identifying the attribute being
 *                              queried.
 *
 *             translatedName - A pointer to a string pointer that receives
 *                              a new buffer with the translated name for
 *                              this attribute. When you are finished with
 *                              this buffer, call Aps_ReleaseBuffer().
 *                              Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetTranslatedName(AttrProvider * thisBase,
                                           const char *attributeID,
                                           char **translatedName)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *optionKeyTrans = NULL;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    ASSERT(translatedName != NULL);

    /* Check whether the key corresponding to this attribute ID exists
     * in the PPD file.
     */
    if (!PPDDoesKeyExist(this->ppdManager, attributeID))
        return APS_NOT_FOUND;

    /* The PPD file format only provides translated names for UI keys. */
    if (PPDGetUI(this->ppdManager, attributeID, &optionKeyTrans, NULL)
        && optionKeyTrans != NULL) {
        /* Provide the caller with the translated name for this attribute. */
        *translatedName = TrackMemDupString(NULL, optionKeyTrans, 0);

        /* Tell the PPD manager that we're finished with this string. */
        PPDDeleteString(this->ppdManager, &optionKeyTrans);

        return (*translatedName) ? APS_SUCCESS : APS_OUT_OF_MEMORY;
    }

    /* This key exists, but no translated name is available. */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetMainData()
 *
 * Obtains the main (read-only) data provided by this attribute. The meaning
 * and contents of the main data string is dependant on the particular
 * attribute being queried.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying the attribute to
 *                             retrieve.
 *
 *             mainData      - A pointer to a char* to receive a new buffer
 *                             containing the main data stored in this
 *                             attribute.
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetMainData(AttrProvider * thisBase,
                                     const char *attributeID,
                                     char **mainData)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *value;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Obtain the non-translated value for this main key. */
    if (PPDGetKey(this->ppdManager, attributeID, &value, NULL)
        && value != NULL) {
        /* Allocate a string to pass back to the client. */
        *mainData = TrackMemDupString(NULL, value, 0);
        /* Cleanup temporary string from PPD manager. */
        PPDDeleteString(this->ppdManager, &value);

        return (*mainData) ? APS_SUCCESS : APS_OUT_OF_MEMORY;
    }

    /* If the key wasn't found in the PPD file, we'll proceed in searching */
    /* for this attribute ID in any other attribute providers.             */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetTranslatedData()
 *
 * Obtains the translated data string for a particular attribute.
 *
 * Parameters: thisBase       - A pointer the the base AttrProvider instance
 *                              for this PPD Attribute Provider.
 *
 *             attributeID    - A string uniquely identifying the attribute.
 *
 *             translatedData - A string pointer to receive a new buffer with
 *                              the translated data. When finished with this
 *                              buffer, you must call Aps_ReleaseBuffer().
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetTranslatedData(AttrProvider * thisBase,
                                           const char *attributeID,
                                           char **translatedData)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *valueTrans;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Obtain the translated value for this main key. */
    if (PPDGetKey(this->ppdManager, attributeID, NULL, &valueTrans)
        && valueTrans != NULL) {
        /* Allocate a string to pass back to the client. */
        *translatedData = TrackMemDupString(NULL, valueTrans, 0);
        /* Cleanup temporary string from PPD manager. */
        PPDDeleteString(this->ppdManager, &valueTrans);
        
        return (*translatedData) ? APS_SUCCESS : APS_OUT_OF_MEMORY;
    }

    /* If the key wasn't found in the PPD file, we'll proceed in searching */
    /* for this attribute ID in any other attribute providers.             */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetType()
 *
 * Obtains the type of a particular attribute. Attributes may be read-only
 * informational attributes, or may store one of a variety of types of user-
 * modifyable settings.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying an attribute.
 *
 *             attributeType - A pointer to an Aps_AttrType to receive the
 *                             type of the attribute.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetType(AttrProvider * thisBase,
                                 const char *attributeID,
                                 Aps_AttrType * attributeType)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    int ppdUIType;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Check whether the key corresponding to this attribute ID exists
     * in the PPD file.
     */
    if (!PPDDoesKeyExist(this->ppdManager, attributeID))
        return APS_NOT_FOUND;

    /* Check whether this key is one of the standard UI types. */
    if (!PPDGetUI(this->ppdManager, attributeID, NULL, &ppdUIType)) {
        /* There is no UI key by this name, so this attribute has no
         * associated setting.
         */
        *attributeType = APS_ATTR_READ_ONLY_DATA;
        return APS_SUCCESS;
    }

    /* Translate PPD UI type to corresponding attribute type. */
    switch (ppdUIType) {
        case PPD_UI_TYPE_BOOLEAN:
            *attributeType = APS_ATTR_SETTING_BOOL;
            return APS_SUCCESS;
        case PPD_UI_TYPE_PICK_ONE:
            *attributeType = APS_ATTR_SETTING_PICK_ONE;
            return APS_SUCCESS;
        case PPD_UI_TYPE_PICK_MANY:
            *attributeType = APS_ATTR_SETTING_PICK_MANY;
            return APS_SUCCESS;
    }

    /* If we reach this point, then we have a UI key of a non-standard
     * type.
     */
    return APS_INVALID_PPD;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetRange()
 *
 * Obtains the range of possible values for attributes with interger or
 * floating point user settings.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             minSetting    - A double to receive the minimum value.
 *
 *             maxSetting    - A double to receive the maximum value.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetRange(AttrProvider * thisBase,
                                  const char *attributeID,
                                  double *minSetting,
                                  double *maxSetting)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Check whether the specified main key exists in the PPD file. */
    if (PPDDoesKeyExist(this->ppdManager, attributeID)) {
        /* Since numerical range UI keys are not supported by PPD files,   */
        /* this query doesn't make sense for this attribute, so tell  this */
        /* to the client application.                                      */
        return APS_WRONG_TYPE;
    }

    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetOptions()
 *
 * Obtains a list of options (e.g. possible values for the user-setting) of
 * the specified attribute.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - A string uniquely identifying the attribute
 *                             being queried.
 *
 *             options       - Pointer to an of Aps_AttrOption instances
 *                             listing the available options.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetOptions(AttrProvider * thisBase,
                                    const char *attributeID,
                                    TrackArrayIndirect_AttrOption *optionArray)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    Aps_Result result;
    char *optionKey = NULL;
    char *optionKeyTranslated = NULL;
    char *value = NULL;
    Aps_AttrOption *option;
    PPDPosition pos;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Check whether the key corresponding to this attribute ID exists
     * in the PPD file.
     */
    if (!PPDDoesKeyExist(this->ppdManager, attributeID)) {
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Begin iterating through all root-level UI keys. */
    if (!PPDGetKeysIterateStart(this->ppdManager, &pos, attributeID)) {
        result = APS_GENERIC_FAILURE;
        goto cleanup;
    }

    /* Loop through all option keys corresponding to this main key. */
    while (pos != NULL) {
        /* Obtain the next option key for this main key. */
        if (!PPDGetKeysIterateNext(this->ppdManager, &pos, NULL, &optionKey,
                                   &optionKeyTranslated, &value, NULL))
            return APS_GENERIC_FAILURE;

        /* Add another entry to the array of possible options. */
        result = APS_OUT_OF_MEMORY;
        option = TrackArrayIndirectAddLastByRef_AttrOption(optionArray, NULL);
        if (! option) goto cleanup;

        /* Fill in this entry in the option array. */
        if (optionKey) {
            option->optionID = (char *)TrackMemDupString(option, optionKey, 0);
            if (! option->optionID) goto cleanup;
        } else option->optionID = NULL;
        if (optionKeyTranslated) {
            option->optionID = (char *)TrackMemDupString(option,
                optionKeyTranslated, 0);
            if (! option->translatedName) goto cleanup;
        } else option->translatedName = NULL;
        if (value) {
            option->value = (char *)TrackMemDupString(option, value, 0);
            if (! option->value) goto cleanup;
        } else option->value = NULL;

        /* Delete the strings returned by the PPD manager, in preparation
         * for next loop iteration.
         */
        PPDDeleteString(this->ppdManager, &optionKey);
        PPDDeleteString(this->ppdManager, &optionKeyTranslated);
        PPDDeleteString(this->ppdManager, &value);
    }
    PPDIterateEnd(this->ppdManager, &pos);

    /* If we reach this point, then we've successfully enumerated any
     * options for this attribute.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (optionKey != NULL)
        PPDDeleteString(this->ppdManager, &optionKey);
    if (optionKeyTranslated != NULL)
        PPDDeleteString(this->ppdManager, &optionKeyTranslated);
    if (value != NULL)
        PPDDeleteString(this->ppdManager, &value);

    return result;
}

/* ---------------------------------------------------------------------------
 * PPDAttrGetSetting()
 *
 * Obtains the current value of the user-setting for a particular attribute.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - The ID of attribute to retrieve.
 *
 *             setting       - A pointer to a char * to receive a new buffer
 *                             containing the current user-setting for this
 *                             attribute. When finished with this buffer,
 *                             deallocate it by calling Aps_ReleaseBuffer().
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrGetSetting(AttrProvider * thisBase,
                                    const char *attributeID,
                                    char **setting)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *value;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Obtain the current default for this key. */
    if (PPDGetDefaultKey(this->ppdManager, attributeID, &value)
        && value != NULL) {
        /* Allocate a string to pass back to the client. */
        *setting = TrackMemDupString(NULL, value, 0);
        /* Cleanup temporary string from PPD manager. */
        PPDDeleteString(this->ppdManager, &value);
        
        return (*setting) ? APS_SUCCESS : APS_OUT_OF_MEMORY;
    }

    /* If the key wasn't found in the PPD file, we'll proceed in searching */
    /* for this attribute ID in any other attribute providers.             */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrSetSetting()
 *
 * Changes the user-setting for a particular attribute.
 *
 * Parameters: thisBase      - A pointer the the base AttrProvider instance
 *                             for this PPD Attribute Provider.
 *
 *             attributeID   - A string containing the ID of the attribute
 *                             to be changed.
 *
 *             setting       - The new value for the user-setting.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrSetSetting(AttrProvider * thisBase,
                                    const char *attributeID,
                                    const char *setting)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Check whether the key corresponding to this attribute ID exists
     * in the PPD file.
     */
    if (!PPDDoesKeyExist(this->ppdManager, attributeID))
        return APS_NOT_FOUND;

    /* Attempt to change the current default for this key. */
    if (PPDSetDefaultKey(this->ppdManager, attributeID, setting))
        return APS_SUCCESS;

    /* If the key wasn't found in the PPD file, we'll proceed in searching */
    /* for this attribute ID in any other attribute providers.             */
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * PPDAttrCheckConstraints()
 *
 * Tests whether a proposed change to a particular attribute's user-setting
 * would violate any of the user-setting constraints.
 *
 * Parameters: thisBase             - A pointer the the base AttrProvider
 *                                    instance for this PPD Attribute
 *                                    Provider.
 *
 *             attributeID          - A string uniquely identifying the
 *                                    attribute whose user setting is
 *                                    being proposed for modification.
 *
 *             setting              - A string containing the new user
 *                                    setting for this attribute.
 *
 *             conflictingAttribute - If a conflict is found, this
 *                                    will be given a string buffer
 *                                    containing the ID of the first
 *                                    attribute found to conflict with
 *                                    the proposed change. Use
 *                                    Aps_ReleaseBuffer() to deallocate
 *                                    this buffer when finished with it.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *             conflictingSetting   - If a conflict is found, this will
 *                                    contain conflicting user-setting.
 *                                    Use Aps_ReleaseBuffer() to
 *                                    deallocate this buffer.
 *                                    Free with Aps_ReleaseBuffer()
 *
 *
 *     Return: APS_SUCCESS if no constraints would be violated by this
 *             change, APS_VIOLATES_CONSTRAINTS if there is is a conflict, or
 *             another Aps_Result code on failure.
 */
static Aps_Result PPDAttrCheckConstraints(AttrProvider * thisBase,
                                          const char *attributeID,
                                          const char *setting,
                                          char **conflictingAttribute,
                                          char **conflictingSetting)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *conflictingMainKey = NULL;
    char *conflictingValue = NULL;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);

    /* Ask the PPD manager whether it has any conflicts for this key. */
    if (PPDCheckForUIConstraints(this->ppdManager, attributeID, setting,
                                 &conflictingMainKey, &conflictingValue)
        && conflictingMainKey != NULL) {

        /* We've found one conflict, so we can stop searching for more. */
        /* Pass resulting strings back to client. */
        *conflictingAttribute = TrackMemDupString(NULL,
            conflictingMainKey, 0);
        PPDDeleteString(this->ppdManager, &conflictingMainKey);

        if (conflictingValue) {
            if (*conflictingAttribute) {
                *conflictingSetting = TrackMemDupString(NULL,
                    conflictingValue, 0);
                PPDDeleteString(this->ppdManager, &conflictingValue);
                if (*conflictingSetting) return APS_VIOLATES_CONSTRAINTS;
                TrackMemFree(*conflictingAttribute);
                *conflictingAttribute = NULL;
            }
        } else {
            if (*conflictingAttribute) return APS_VIOLATES_CONSTRAINTS;
        }
        return APS_OUT_OF_MEMORY;
    }

    /* No conflicts have been found, so continue in checking for conflicts */
    /* with any other attribute providers.                                 */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * PPDAttrWriteBlock()
 *
 * Writes a standard block of data into a job, based on the current state
 * of the attributes supplied by this attribute provider.
 *
 * Parameters: this                 - A pointer the the base AttrProvider
 *                                    instance for this Attribute Provider.
 *
 *             job                  - A handle to the job to write into.
 *
 *             blockType            - The type of data to be written.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrWriteBlock(AttrProvider * thisBase,
                                    Aps_JobHandle job,
                                    Aps_BlockType blockType)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *stringToOutput;
    Aps_Result result;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    ASSERT(job != NULL);

    /* The PPD manager provides some standard blocks of code in the form of
     * of a string in memory. If this is one of those blocks, we'll first
     * obtain the string, and then send it to the job.
     */
    switch(blockType) {
        case APS_BLOCK_FILE_HEADER:
            if (!PPDGetJCLHeader(this->ppdManager, &stringToOutput))
                return APS_SUCCESS;
            break;
        
        case APS_BLOCK_PS_PATCH:
            if (!PPDGetJobPatch(this->ppdManager, &stringToOutput))
                return APS_SUCCESS;
            break;
        
        case APS_BLOCK_PS_DEV_SETUP:
            if (!PPDGetPSSetup(this->ppdManager, &stringToOutput))
                return APS_SUCCESS;
            break;
        
        case APS_BLOCK_FILE_FOOTER:
            if (!PPDGetJCLFooter(this->ppdManager, &stringToOutput))
                return APS_SUCCESS;
            break;
        
        default:
            /* Some block type for which the PPD Attribute provider doesn't
             * have any data to contribute has been specified. Do nothing.
             */
            return APS_SUCCESS;
    }

    /* If the PPD Manager returned us a NULL string, then there is no data
     * to be written for this block type.
     */
    if (stringToOutput == NULL)
        return APS_SUCCESS;
    
    /* If we reach this point, then we have a string to be sent to the job. */
    result = Aps_JobWrite(job, stringToOutput, strlen(stringToOutput));

    /* Release the string provided by the PPD manager. */
    PPDDeleteString(this->ppdManager, &stringToOutput);
    
    return result;
}

/* ---------------------------------------------------------------------------
 * PPDAttrResetToModelDefaults()
 *
 * Resets these job attributes to the defaults for the specified model of
 * printer, based on information retrieved from the model database.
 *
 * Parameters: thisBase - Pointer to the attributes provider to operate on.
 *
 *             printer  - A pointer to the printer with which the default
 *                        attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrResetToModelDefaults(AttrProvider * thisBase,
                                              Printer * printer)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    Aps_Result result;
    char *ppdFilename = NULL;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    ASSERT(printer != NULL);

    /* If we don't have a model object for this printer, there is nothing
     * for us to do.
     */
    if (printer->model == NULL) {
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Look to see whether this is a PPD file associated with this model. */
    result = Aps_GetPropertyString(printer->model, "ppd", &ppdFilename);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If this field is empty, then there is no PPD file associated with this
     * model.
     */
    if (strlen(ppdFilename) == 0) {
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Select this PPD file into the PPD manager. */
    if (!PPDSetFilename(this->ppdManager, ppdFilename)) {
        result = APS_INVALID_PPD;
        goto cleanup;
    }

    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources. */
    if (ppdFilename != NULL) {
        Aps_ReleaseBuffer(ppdFilename);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * PPDAttrSaveAsPrinterDefaults()
 *
 * Saves the settings in these job attributes as the default for the specified
 * printer.
 *
 * Parameters: thisBase - Pointer to the attributes provider to operate on.
 *
 *             printer  - A pointer to the printer with which the default
 *                        attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrSaveAsPrinterDefaults(AttrProvider * thisBase,
                                               Printer * printer)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    char *currentPPDFilename = NULL;
    char *newPPDFilename = NULL;
    char *userConfigDir = NULL;
    char *baseFilename;
    Aps_Result result;
    int currentNameLen;
    int newLen;
    int suffixLen;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    ASSERT(printer != NULL);

    /* Obtain the name of the PPD file that is currently being used. */
    if (!PPDGetFilename(this->ppdManager, &currentPPDFilename)) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* If we are already using a customized PPD file, then simply it with a
     * new one containing all customized values.
     */
    if (PPDIsCustomPPD(this->ppdManager)) {
        if (PPDSaveToCustomPPD(this->ppdManager, currentPPDFilename)) {
            /* We successfully saved to the existing custom PPD filename, so
             * we're done!
             */
            result = APS_SUCCESS;
            goto cleanup;
        }
    }

    /* We were unable to save to a custom PPD file that is currently being
     * used, so we'll attempt to save to a new custom PPD file.
     */
    currentNameLen = strlen(currentPPDFilename);
    suffixLen = strlen(APSCFG_CONFIG_PATH_PPD_SEPARATOR)
                + strlen(printer->name)
                + strlen(APSCFG_CONFIG_PATH_PPD_SUFFIX);

    /* Next, try saving a custom PPD file in the same location as the current
     * PPD file, if it isn't a custom PPD.
     */
    if (!PPDIsCustomPPD(this->ppdManager)) {
        /* Generate filename for custom PPD file. */
        newPPDFilename = malloc(currentNameLen + suffixLen + 1);
        if (newPPDFilename == NULL) {
            result = APS_OUT_OF_MEMORY;
            goto cleanup;
        }

        strcpy(newPPDFilename, currentPPDFilename);

        /* Remove existing .ppd from end of filename, if any. */
        if (currentNameLen > 4 && stricmp(newPPDFilename + currentNameLen - 4,
                                          ".ppd") == 0) {
            newPPDFilename[currentNameLen - 4] = '\0';
        }

        /* Add standard suffix to filename that denotes it is a custom PPD
         * file.
         */
        strcat(newPPDFilename, APSCFG_CONFIG_PATH_PPD_SEPARATOR);
        strcat(newPPDFilename, printer->name);
        strcat(newPPDFilename, APSCFG_CONFIG_PATH_PPD_SUFFIX);

        /* Attempt to save custom settings into this file. */
        if (PPDSaveToCustomPPD(this->ppdManager, newPPDFilename)) {
            /* Save the name of the custom PPD file that is now being used
             * with this printer.
             */
            result = MetaWrite(printer->name, "ppd", newPPDFilename);
            if (result != APS_SUCCESS)
                goto cleanup;

            /* We've succeeded in saving the default settings for this
             * printer.
             */
            result = APS_SUCCESS;
            goto cleanup;
        }
    }

    /* Finally, attempt to save into this user's printer meta configuration
     * directory.
     */

    /* Obtain the location of the user's meta config directory. */
    userConfigDir = MetaGetUserDir();

    /* Allocate space to hold the overall filename. */
    if (newPPDFilename != NULL)
        free(newPPDFilename);
    newPPDFilename = malloc(strlen(userConfigDir) + currentNameLen
                            + suffixLen + 1);
    if (newPPDFilename == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Build new filename. */

    /* Copy base path (e.g. "/home/johndoe/.printers/"). */
    strcpy(newPPDFilename, userConfigDir);

    /* Append just the filename. (e.g. "laserjet.ppd") */
    baseFilename = strrchr(currentPPDFilename, '/');
    if (baseFilename == NULL)
        baseFilename = currentPPDFilename;
    strcpy(newPPDFilename, baseFilename);

    /* Ensure that this filename has the appropriate suffix. */
    newLen = strlen(newPPDFilename);
    if (newLen < suffixLen
        || strcmp(newPPDFilename + newLen - suffixLen,
                  APSCFG_CONFIG_PATH_PPD_SUFFIX) != 0) {
        /* Remove existing .ppd from end of filename, if any. */
        if (newLen > 4 && stricmp(newPPDFilename + newLen - 4, ".ppd") == 0) {
            newPPDFilename[newLen - 4] = '\0';
        }

        /* Add standard suffix to filename that denotes it is a custom PPD
         * file.
         */
        strcat(newPPDFilename, APSCFG_CONFIG_PATH_PPD_SEPARATOR);
        strcat(newPPDFilename, printer->name);
        strcat(newPPDFilename, APSCFG_CONFIG_PATH_PPD_SUFFIX);
    }

    /* Attempt to save custom settings into this file. */
    if (PPDSaveToCustomPPD(this->ppdManager, newPPDFilename)) {
        /* Save the name of the custom PPD file that is now being used with
         * this printer.
         */
        result = MetaWrite(printer->name, "ppd", newPPDFilename);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* We've succeeded in saving the default settings for this printer. */
        result = APS_SUCCESS;
        goto cleanup;
    }

    /* If we reach this point we were unable to save the settings into a
     * custom PPD file.
     */
    result = APS_GENERIC_FAILURE;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (currentPPDFilename != NULL) {
        PPDDeleteString(this->ppdManager, &currentPPDFilename);
    }

    if (newPPDFilename != NULL) {
        free(newPPDFilename);
    }

    if (userConfigDir != NULL) {
        free(userConfigDir);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * PPDAttrSetToPrinterDefaults()
 *
 * Initializes these job attributes based on the defaults for the specified
 * printer.
 *
 * Parameters: thisBase - Pointer to the attributes provider to operate on.
 *
 *             printer  - A pointer to the printer with which the default
 *                        attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrSetToPrinterDefaults(AttrProvider * thisBase,
                                               Printer * printer)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    Aps_Result result;
    char *ppdFilename = NULL;

    ASSERT(this != NULL);
    ASSERT(this->ppdManager != NULL);
    ASSERT(printer != NULL);

    /* Look to see whether this is a PPD file associated with this printer. */
    result = MetaRead(&ppdFilename, printer->name, "ppd");
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If this field is empty, then there is no PPD file associated with this
     * printer.
     */
    if (strlen(ppdFilename) == 0) {
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Select this PPD file into the PPD manager. */
    if (!PPDSetFilename(this->ppdManager, ppdFilename)) {
        result = APS_INVALID_PPD;
        goto cleanup;
    }

    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources. */
    if (ppdFilename != NULL) free(ppdFilename);

    return result;
}

/* ---------------------------------------------------------------------------
 * PPDAttrProvRemovePrinterDefaults()
 *
 * Removes any default settings associated with the specified printer that
 * are provided by this attribute provider. 
 *
 * Parameters: thisBase - Pointer to the attributes provider to operate on.
 *
 *             printer  - A pointer to the printer in question.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result PPDAttrProvRemovePrinterDefaults(AttrProvider * thisBase,
                                                   Printer * printer)
{
    PPDAttrProvider *this = (PPDAttrProvider *) thisBase;
    Aps_Result result;
    char *currentPPDFilename = NULL;

    /* If this printer has a custom PPD file associated with it, then remove
     * that custom PPD file.
     */
    if (PPDIsCustomPPD(this->ppdManager)) {
        /* Obtain the name of the PPD file that is currently being used. */
        if (!PPDGetFilename(this->ppdManager, &currentPPDFilename))
            return APS_OUT_OF_MEMORY;

        /* Remove the custom PPD file. */
        remove(currentPPDFilename);

        /* Release the temporary string holding the filename. */
        PPDDeleteString(this->ppdManager, &currentPPDFilename);
    }

    /* Remove the PPD filename key from this printer's meta configuration. */
    result = MetaRemove(printer->name, "ppd");
    if (result != APS_SUCCESS)
        return result;

    return APS_SUCCESS;
}
