/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printeraddremove.c
 *
 * Description: Printer installation and removal APIs.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_AddPrinter()
 *
 * Adds another printer to the set of available printers on this system.
 *
 * Parameters: name    - A string contining the name to give the new printer.
 *
 *             printer - The address of an Aps_PrinterHandle to receive a
 *                       handle to the newly added printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_AddPrinter(const char *name, Aps_PrinterHandle *printer)
{
    int transportCount;
    Transport **trans;
    Aps_Result result;
    char **printerNames;
    int numPrinters;
    int i;
    Aps_PrinterHandle newPrinter;

    if ((name == NULL) || (strlen(name) > MAX_PRINTER_NAME))
        return APS_INVALID_PARAM;

    /* Start with output parameter as NULL, in case of failure. */
    if (printer != NULL) /* Did caller supply output handle at all? */
        *printer = NULL;

    /* Check that no printer already exists with this name. */
    result = Aps_GetPrinters(&printerNames, &numPrinters);
    if (result != APS_SUCCESS)
        return result;

    for (i = 0; i < numPrinters; ++i) {
        if (strcmp(name, printerNames[i]) == 0)
            return APS_ALREADY_EXISTS;
    }

    /* Ask the primary (first) transport to add the new printer. */
    trans = TransGetAllTransports(&transportCount);
    /* no transports found??? */
    if (transportCount <= 0) return APS_GENERIC_FAILURE;

    /* try first available transport */
    if ((result = trans[0]->vtbl->AddPrinter(trans[0], name,
                                             &newPrinter)) != APS_SUCCESS)
       return result;

    /* Attempt to initialize the configuration for this new
     * printer to be labelled as a generic PostScript priter.
     * We ignore the result of this function; if we're unable
     * to set the model for this printer, we still want to
     * succeed in creating the printer.
     */
    Aps_PrinterSetModel(newPrinter, "Generic", "PostScript Printer");

    /* If this was the first & only printer, then flag it as the default
     * printer.
     */
    if (numPrinters == 0) {
        result = Aps_PrinterSetAsDefault(newPrinter);
        if (result != APS_SUCCESS) {
            Aps_ReleaseHandle(newPrinter);
            return result;
        }
    }

    /* Provide caller with handle to new printer, if required. Otherwise,
     * release this reference to the printer handle.
     */
    if (printer == NULL) {
        Aps_ReleaseHandle(newPrinter);
    } else {
        *printer = newPrinter;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterRemove()
 *
 * Removes a printer from the set of available printers. If this function
 * succeeds, the printer handle will have been released before it returns.
 *
 * Parameters: handle - A handle to the printer to be removed.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterRemove(Aps_PrinterHandle handle)
{
    Aps_Result result;
    Printer *printer;
    char *printerName = NULL;

    /* Check for reasonable parameter values. */
    if (handle == NULL)
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(handle))== NULL)
        return APS_INVALID_HANDLE;

    /* Store the name of this printer. */
    result = Aps_PrinterGetName(handle, &printerName);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Remove the default attributes information for this printer. */
    result = JobAttrRemovePrinterDefaults(printer->defaultJobAttributes,
                                          printer);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Ask the associated transport to remove this printer. */
    result = printer->transport->vtbl->PrinterRemove(printer->transport,
                                                     handle);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Remove any meta configuration information associated with this
     * printer.
     */
    result = MetaRemoveAll(printerName);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* The printer and associated resources are now removed. */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (printerName != NULL) {
        Aps_ReleaseBuffer(printerName);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterInstallLocalCopy()
 *
 * Adds a new printer to the current machine/user's list of regularly used
 * printers, given a printe rhandle opened by Aps_OpenRemotePrinter().
 *
 * Parameters: remotePrinter - A handle to the currently open printer.
 *
 *             newLocalCopy  - An optional pointer to a printer handle to
 *                             receive a handle to the newly created local
 *                             copy of this printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterInstallLocalCopy(Aps_PrinterHandle remotePrinter,
                                       Aps_PrinterHandle *newLocalCopy)
{
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

