/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerid.c
 *
 * Description: Printer identification APIs.
 */

#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetName()
 *
 * Obtains the name of an open printer.
 *
 * Parameters: printerHandle - A handle to a currently open printer.
 *
 *             name          - The address of a char * to receive a string
 *                             containing the name of this printer. If this
 *                             function succeeds, it is the caller's
 *                             responsibility to dispose of this string by
 *                             calling Aps_ReleaseBuffer();
 *                             Free with Aps_ReleaseBuffer()
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterGetName(Aps_PrinterHandle printerHandle, char **name)
{
    Printer *printer;

    /* Check for reasonable parameter values. */
    if (name == NULL)
        return APS_INVALID_PARAM;

    /* Obtain a pionter to the Printer structure for this handle. */
    printer = PrinterGetPtrFromHandle(printerHandle);
    if (printer == NULL)
        return APS_INVALID_HANDLE;

    /* Allocate memory to hold the printer name to be returned to the */
    /* caller.                                                        */
    ASSERT(printer->name != NULL);
    *name = TrackMemDupString(NULL, printer->name, 0);
    return (*name) ? APS_SUCCESS : APS_OUT_OF_MEMORY;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterRename()
 *
 * Changes the name of a printer.
 *
 * Parameters: printerHandle - A handle to a currently open printer.
 *
 *             newName - A string with the new name to give this printer.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterRename(Aps_PrinterHandle printerHandle, 
                             const char *newName)
{
    Aps_Result result;
    Printer *printer;
    char *originalName = NULL;

    /* Check for reasonable parameter values. */
    if ((printerHandle == NULL) || (newName == NULL))
        return APS_INVALID_PARAM;

    if ((printer = PrinterGetPtrFromHandle(printerHandle))== NULL)
        return APS_INVALID_HANDLE;

    /* Store the original name of this printer. */
    result = Aps_PrinterGetName(printerHandle, &originalName);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Ask the associated transport to rename this printer. */
    result = printer->transport->vtbl->PrinterRename(printer->transport,
                                                     printerHandle,
                                                     newName);
    if (result != APS_SUCCESS)
        goto cleanup;
        
    /* On success, also rename any meta configuration information
     * associated with this printer.
     */
    result = MetaRename(originalName, newName);
    if (result != APS_SUCCESS)
        goto cleanup;
            
    /* We've now successfully renamed this printer, so exit the
     * loop searching for the printer.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release any temporary resources allocated by this function. */
    if (originalName != NULL) {
        Aps_ReleaseBuffer(originalName);
    }

    return result;
}
