/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printermodelinfo.c
 *
 * Description: APIs for configuring the model information associated
 *              with a printer.
 *
 */

#include <stdio.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "printer.h"
#include "transport.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetModel()
 *
 * Obtains the printer manufacturer and model name for a particular printer.
 *
 * Parameters: printer      - A handle to the printer in question.
 *
 *             manufacturer - Receives the name of this printer's
 *                            manufacturer.
 *                            Free with Aps_ReleaseBuffer
 *
 *             model        - Receives the manufacturer's model name for this
 *                            printer.
 *                            Free with Aps_ReleaseBuffer
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterGetModel(Aps_PrinterHandle printer,
                               char **manufacturer,
                               char **model)
{
    Aps_ModelHandle modelHandle = NULL;
    Aps_Result result;

    /* Obtain a handle to this printer's model object. */
    result = Aps_PrinterGetModelHandle(printer, &modelHandle);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If the caller requested the manufacturer name for this printer, then
     * query for that information from the model object.
     */
    if (manufacturer != NULL) {
        result = Aps_GetPropertyString(modelHandle, "manufacturer",
                                       manufacturer);
        if (result != APS_SUCCESS)
            goto cleanup;
    }

    /* Likewise, if the caller requested the model name, obtain that
     * information from the model object.
     */
    if (model != NULL) {
        result = Aps_GetPropertyString(modelHandle, "model", model);
        if (result != APS_SUCCESS) {
            /* Release the manufacturer string if we've already obtained
             * it.
             */
            if (manufacturer != NULL) {
                Aps_ReleaseBuffer(*manufacturer);
                *manufacturer = NULL;
            }

            goto cleanup;
        }
    }

    /* If we reach this point, we've successfully obtained any information
     * that the caller requested.
     */
    result = APS_SUCCESS;

cleanup:
    /* If the modle handle was ever obtained, release our reference to it. */
    if (modelHandle != NULL) {
        Aps_ReleaseHandle(modelHandle);
    }

    return result;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetModel()
 *
 * Changes the printer manufacturer and model of a particular printer.
 *
 * Parameters: printer      - A handle to the printer in question.
 *
 *             manufacturer - The name of the new manufacturer to associate
 *                            with this printer.
 *
 *             model        - The name of the new model to associate with
 *                            this printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterSetModel(Aps_PrinterHandle printer,
                               const char *manufacturer,
                               const char *model)
{
    Printer *printerObject;
    Aps_ModelHandle newModel = NULL;
    Aps_Result result;

    /* Check for reasonable parameter values. */
    if (manufacturer == NULL || model == NULL) {
        result = APS_INVALID_PARAM;
        goto cleanup;
    }

    /* Obtain a pointer to the printer object that the caller has passed a
     * handle to.
     */
    printerObject = PrinterGetPtrFromHandle(printer);
    if (printerObject == NULL) {
        result = APS_INVALID_HANDLE;
        goto cleanup;
    }

    /* Attempt to obtain a handle to the manufacturer and model specified
     * by the caller. This function will fail if there is no known printer
     * by the specified name.
     */
    result = Aps_GetModel(manufacturer, model, &newModel);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Update the meta configuration information with the name of the
     * manufacturer and model that the caller has specified.
     */
    result = MetaWrite(printerObject->name, "manufacturer", manufacturer);
    if (result != APS_SUCCESS)
        goto cleanup;

    result = MetaWrite(printerObject->name, "model", model);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Replace the current model handle in this printer object with the new
     * model handle.
     */
    Aps_ReleaseHandle(printerObject->model);
    printerObject->model = newModel;

    /* Record the printer object's reference to the model object. */
    Aps_AddRef(newModel);
    
    /* Remove any old default job attributes for this printer. */
    result = JobAttrRemovePrinterDefaults(printerObject->defaultJobAttributes,
                                          printerObject);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Update the printer's default job attributes based on the newly selected
     * printer model, and save these settings for use by this printer in the
     * future.
     */
    result = JobAttrSetToModelDefaults(printerObject->defaultJobAttributes,
                                       printerObject);
    if (result != APS_SUCCESS)
        goto cleanup;

    result = JobAttrSaveAsPrinterDefaults(printerObject->defaultJobAttributes,
                                          printerObject);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* If we reach this point, we've successfully changed the model
     * associated with this printer.
     */
    result = APS_SUCCESS;

cleanup:
    /* Release this function's local reference to the model object. */
    if (newModel != NULL) {
        Aps_ReleaseHandle(newModel);
    }

    /* If we reach this point, we've successfully changed the manufacturer
     * and model associated with this printer.
     */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetModelHandle()
 *
 * Obtains a handle to a particular printer's model object. This allows access
 * to more extensive information about this printer's model, rather than just
 * its manufacturer and model name.
 *
 * Parameters: printer - A handle to a currently open printer.
 *
 *             model   - The address of an Aps_ModleHandle to receive a handle
 *                       to this printer's model object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result Aps_PrinterGetModelHandle(Aps_PrinterHandle printer,
                                     Aps_ModelHandle *model)
{
    Printer *printerObject;

    /* Check for reasonable parameter values. */
    if (model == NULL) {
        return APS_INVALID_PARAM;
    }

    /* Obtain a pointer to the printer object that the caller has passed a
     * handle to.
     */
    printerObject = PrinterGetPtrFromHandle(printer);
    if (printerObject == NULL)
        return APS_INVALID_HANDLE;

    /* Check whether this printer has an associated model handle. */
    if (printerObject->model == NULL) {
        return APS_MODEL_UNKNOWN;
    }

    /* Provide the caller with a handle to this model, adding another
     * reference to the model object's reference count.
     */
    *model = printerObject->model;
    Aps_AddRef(printerObject->model);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetPPDFileName()
 *
 * Obtains the name of the PPD file associated with a particular printer.
 *
 * Parameters: handle   - A handle to the printer being queried.
 *
 *             filename - The address of a string pointer to receive the
 *                        name of this printer's PPD file. On success, the
 *                        caller is reponsible for disposing of this string
 *                        using Aps_ReleaseBuffer().
 *                        Free with Aps_ReleaseBuffer
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterGetPPDFileName(Aps_PrinterHandle handle,
                                     char **filename)
{
    Printer *printer;
    Transport *transport;

    /* Check for reasonable parameter values. */
    if (filename == NULL)
        return APS_INVALID_PARAM;

    /* Obtain the printer and transport objects associated with the */
    /* specified printer handle. */
    printer = PrinterGetPtrFromHandle(handle);
    if (printer == NULL)
        return APS_INVALID_HANDLE;

    transport = (Transport *) printer->transport;
    ASSERT(transport != NULL);

    /* Pass this call on to the transport. */
    *filename = transport->vtbl->GetPPDFileName(transport, printer);
    if (*filename == NULL)
        return APS_GENERIC_FAILURE;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetPPDFileName()
 *
 * Changes the name of the PPD file associated with a particular printer.
 *
 * Parameters: handle   - A handle to the printer that the PPD file should
 *                        be associated with.
 *
 *             filename - The full path and filename of the PPD file for this
 *                        printer.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterSetPPDFileName(Aps_PrinterHandle handle,
                                     const char *filename)
{
    Printer *printer;
    Transport *transport;
    Aps_Result result;

    /* Check for reasonable parameter values. */
    if (filename == NULL)
        return APS_INVALID_PARAM;

    /* Obtain the printer and transport objects associated with the */
    /* specified printer handle. */
    printer = PrinterGetPtrFromHandle(handle);
    if (printer == NULL)
        return APS_INVALID_HANDLE;

    transport = (Transport *) printer->transport;
    ASSERT(transport != NULL);

    /* Pass this call on to the transport. */
    result = transport->vtbl->SetPPDFileName(transport, printer, filename);
    if (result != APS_SUCCESS)
        return result;
    
    /* We've successfully changed the PPD file associated with this printer. */
    return APS_SUCCESS;
}

