/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: queue.c
 *
 * Description: Low-level queue management operations.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"
#include "queue.h"

DEBUG_CHANNEL_DEFAULT(queue)

/* ---------------------------------------------------------------------------
 * QueueGetPtrFromHandle()
 *
 * Obtains a pointer to a Queue object, given an Aps_QueueHandle.
 *
 * Parameters: handle - The handle for which the corresponding Queue object
 *                      should be obtained.
 *
 *     Return: A pointer to a Queue object, or NULL on failure.
 */
Queue *QueueGetPtrFromHandle(Aps_QueueHandle handle)
{
    Queue *queue;

    if (handle == NULL)
        return (NULL);

    queue = (Queue *) handle;

    /* Check that this is actually a queue object, and not something else. */
    if (queue->baseClass.identifier != QUEUE_HANDLE)
        return (NULL);

    return (queue);
}

/* ---------------------------------------------------------------------------
 * QueueGetHandleFromPtr()
 *
 * Obtain the handle for a specified Queue object.
 *
 * Parameters: queue - The Queue for which the corresponding handle should be
 *                     obtained.
 *
 *     Return: An Aps_QueueHandle.
 */
Aps_QueueHandle QueueGetHandleFromPtr(Queue * queue)
{
    ASSERT(queue != NULL);

    return ((Aps_QueueHandle) queue);
}

/* --------------------------------------------------------------------
 * QueueCreate()
 *
 * Construct a Queue object (low-level)
 *
 * Parameters : printerHandle - handle of printer, or NULL for ALL
 * Result     : (inparam) queueHandle - new queue handle, or NULL on failure
 *
 * The returned handle must be freed by Aps_ReleaseHandle().
 * On failure *queueHandle set to NULL and detailed result returned.
 */
Aps_Result QueueCreate(Aps_PrinterHandle printerHandle,
    Aps_QueueHandle *queueHandle)
{
    Aps_Result result;
    Queue *queue;

    /* Allocate queue structure */
    queue = (Queue *)malloc(sizeof(Queue));
    if (queue) {
        /* Setup the baseclass */
        result = ObjectInitialize(& queue->baseClass, QUEUE_HANDLE);
        if (result == APS_SUCCESS) {
            /* Setup the handle */
            *queueHandle = QueueGetHandleFromPtr(queue);
            ASSERT(queueHandle);

            /* Setup a few things in the queue structure */
            queue->printerHandle = printerHandle;
            return APS_SUCCESS;
        }
        free(queue);
    } else result = APS_OUT_OF_MEMORY;
    *queueHandle = NULL;
    return result;
}

/* --------------------------------------------------------------------
 * QueueDelete()
 *
 * Destroy a Queue object (low-level)
 *
 * Parameters : queueHandle - handle of queue to delete
 * Result     : (inparam) queueHandle - new queue handle, or NULL on failure
 *
 * The returned handle must be freed by Aps_ReleaseHandle().
 * On failure *queueHandle set to NULL and detailed result returned.
 */
void QueueDelete(Queue *queue)
{
    /* Release locked objects */
    Aps_ReleaseHandle(queue->printerHandle);

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    queue->baseClass.identifier = INVALID_HANDLE;
    free(queue);
}

/* ---------------------------------------------------------------------------
 * QueueIsOperationAvailable()
 *
 * Determines whether or not the specified operation can currently be
 * performed on a particular APS object.
 *
 * Parameters: queue             - A pointer to a Queue object
 *
 *             operation         - The operation that the application is
 *                                 interested in.
 *
 *             anticipatedResult - APS_OPERATION_AVAILABLE if this operation
 *                                 is currently available, or the anticipated
 *                                 reason for failure if it is known that
 *                                 this operation cannot be performed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             APS_SUCCESS       - test is accurate
 *             APS_IGNORED       - test cannot be performed
 *             APS_NOT_FOUND     - operation ID not recognized
 */
Aps_Result QueueIsOperationAvailable(Queue *queue,
    Aps_OperationID operation, Aps_Result *anticipatedResult)
{
    Transport *trans;
    Printer   *printer;
    ASSERT(queue);

    /* If this queue is not global, then we will ask the Transport what
     * this queue is capable of.  Otherwise, we will ignore the request.
     */
    if (queue->printerHandle) {

        /* locate printer and transport */
        printer = PrinterGetPtrFromHandle(queue->printerHandle);
        ASSERT(printer);
        trans = printer->transport;
        ASSERT(trans);

        /* Ask the transport what we can do about this... */
        return trans->vtbl->QueueIsOperationAvailable(trans, printer,
            operation, anticipatedResult);
    /* If we're the global queue, we'll have to go through all attached
     * printers.  To do so, we cache the list of printers, open them
     * in turn, perform the desired operation, then release them.
     *
     * We will only return a result code if all transports agree that
     * the operation will succeed or fail.
     */
    } else {
        char     **printerNames;
        int        numNames;
        Aps_Result tempresult = APS_NUMRESULTS;
        Aps_Result temparesult = APS_OPERATION_AVAILABLE;
        Aps_Result xresult, xaresult;
        if (Aps_GetPrinters(& printerNames, & numNames) == APS_SUCCESS) {
            int i;
            for (i = 0; i < numNames; i++) {
                Aps_PrinterHandle printerHandle;
                xresult = Aps_OpenPrinter(printerNames[i], & printerHandle);
                if (xresult == APS_SUCCESS) {
                    /* locate printer and transport */
                    printer = PrinterGetPtrFromHandle(printerHandle);
                    ASSERT(printer);
                    trans = printer->transport;
                    ASSERT(trans);

                    xresult = trans->vtbl->QueueIsOperationAvailable(trans,
                        printer, operation, & xaresult);
                    Aps_ReleaseHandle(printerHandle);
                    if (Aps_Succeeded(xresult)) {
                        /* replace default with anything */
                        if (tempresult == APS_NUMRESULTS) {
                            TRACE("setting first time");
                            tempresult = xresult;
                            temparesult = xaresult;
                        /* replace ignored with anything */
                        } else if (tempresult == APS_IGNORED) {
                            TRACE("replacing APS_IGNORED");
                            tempresult = xresult;
                            temparesult = xaresult;
                        /* if results don't match, check for ignored */
                        } else if (tempresult != xresult) {
                            if (xresult != APS_IGNORED) {
                                /* mismatched return code */
                                TRACE("mismatched return, %s <-> %s",
                                    RESULTSTR(tempresult),
                                    RESULTSTR(xresult));
                                temparesult = APS_OPERATION_AVAILABLE;
                                tempresult = APS_IGNORED;
                                break;
                            }
                        /* if anticipated results don't match, check for
                         * consistency */
                        } else if (temparesult != xaresult) {
                            if ((Aps_Succeeded(temparesult) ? 1 : 0) ^
                                (Aps_Succeeded(xaresult) ? 1 : 0)) {
                                /* mismatched expected result code */
                                TRACE("mismatched expected return, %s <-> %s",
                                    RESULTSTR(tempresult),
                                    RESULTSTR(xresult));
                                temparesult = APS_OPERATION_AVAILABLE;
                                tempresult = APS_IGNORED;
                                break;
                            } else {
                                /* reduce to most generic term */
                                TRACE("reducing expected return, %s <-> %s",
                                    RESULTSTR(tempresult),
                                    RESULTSTR(xresult));
                                temparesult = Aps_Succeeded(temparesult) ?
                                    APS_OPERATION_AVAILABLE :
                                    APS_NOT_SUPPORTED;
                            }
                        }
                    }
                }
            }
            Aps_ReleaseBuffer(printerNames);
        }
        *anticipatedResult = temparesult;
        if (tempresult == APS_NUMRESULTS) tempresult = APS_NOT_FOUND;
        return tempresult;
    }
}
