/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: trackmem.c
 *
 * Description: Simple local implementation of memory tracking.
 *              Provides a primitive tracking mechanism for linked
 *              blocks of data.
 *
 * N.B:
 *   This module is NOT designed to be used as a standalone memory
 *   allocator, nor is it meant to provide the full functionality
 *   of a memory pool; it was decided that such functionality was
 *   not required for our immediate purposes...
 *   Remember this is designed strictly to handle simple cases
 *   of linked blocks.
 *
 * Limitations:
 *   - only one user at a time (no multithreading support)
 *   - somewhat memory expensive
 *   - requires a GOOD COMPILER capable of common subexpression
 *     elimination for optimum performance
 *   - tag space cannot be extended without reallocating the entire chunk
 *     (no extension block support yet)
 *   - destructors may only be attached to links
 *
 * Concept:
 *
 * Blocks are allocated as three chunks, Admin, User space and Tags.
 *   Admin space has a fixed size of sizeof(TrackMem_Block)
 *   User space varies from 0 to n bytes
 *   Tags space varies from 0 to n bytes
 *
 * These blocks are physically allocated as a contiguous block like this:
 *  .-------.
 *  | Tags  |  -- 0 to n*alignment size == n*8 on most systems
 *  +-------+
 *  | Admin |  -- sizeof(TrackMem_Block) == 16 bytes on 32bit systems
 *  +-------+
 *  |       |
 *  | User  |  -- 0 to x, arbitrary size
 *  |       |
 *  `-------'
 *
 * To improve compatibility, tag space size is kept to a multiple of the
 * native alignment size.
 *
 * For improved performance, the sizeof(tags) + sizeof(TrackMem_Block) may be
 * kept to a multiple of the native cache alignment size if possible so
 * that user space is on a cache alignment boundary.
 *
 * Tags must be preallocated before they are used.  This can be done at
 * creation time or extended later.
 *
 * All contents of a block have variable position.  This is especially a
 * concern for users of Tags in that they should NOT be externally
 * referenced for long.  All external references to Tags or User space are
 * invalid after either a Realloc() or TagRealloc() occurs.
 */

#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include "trackmem.h"

#define APS_ENVIRONMENT

#ifdef APS_ENVIRONMENT
#include "apsinternal.h"
DEBUG_CHANNEL_DEFAULT(mem)
#endif

/* define ASSERT */
#ifndef ASSERT
#include <assert.h>
#define ASSERT(x) assert(x)
#endif

/* define TINLINE and TINLINE_ON */
#ifdef APS_ENVIRONMENT
#if APSCFG_INLINE
#define TINLINE INLINE
#define TINLINE_ON
#else
#define TINLINE
#endif
#else
#ifndef INLINE
#define TINLINE
#else
#define TINLINE INLINE
#define TINLINE_ON
#endif
#endif

/***************
 *** OPTIONS ***
 ***************/

/* Note: All of the macros below must be defined for proper operation with
 *       the possible exception of memSize().  If not defined, then we
 *       will store the size of the allocation with each memory chunk.
 *       Though the effect is harmless, it can cause large quantities of
 *       memory to be wasted due to memory alignment overheads.
 *
 * Remarks:
 * - ELECTRIC FENCE
 *   + does not support malloc_usable_size() [returns 0]
 *
 * - DMALLOC
 *   + does not support malloc_usable_size() [returns "random" number]
 */

/*** Abstract from underlying memory allocation layer ***/
#ifndef APS_ENVIRONMENT /* for non-APS environment */
/* void *memAlloc(size_t size) */
#define memAlloc(size) ((void *)malloc(size))
/* void *memRealloc(void *mem, size_t size) */
#define memRealloc(mem, size) ((void *)realloc(mem, size))
/* void memFree(void *mem) */
#define memFree(mem) (free(mem))
/* size_t memSize(void *mem) */
#define memSize(mem) ((size_t)malloc_usable_size(mem))

/*** Alignment settings ***/
#define MEMALIGN_LOCAL   4 /* native alignment for adequate performance */
#define MEMALIGN_USER   16 /* guarantee alignment of user space to this
                            * assuming memAlloc() is also aligned to at least
                            * this */
#else
#define memAlloc(size) APSCFG_MALLOC_ALLOC(size)
#define memRealloc(mem, size) APSCFG_MALLOC_REALLOC(mem, size)
#define memFree(mem) APSCFG_MALLOC_FREE(mem)
#ifdef APSCFG_MALLOC_USABLE_SIZE
#define memSize(mem) APSCFG_MALLOC_USABLE_SIZE(mem)
#endif
#define MEMALIGN_LOCAL APSCFG_MALLOC_DATAALIGN
#define MEMALIGN_USER APSCFG_MALLOC_USERALIGN
#endif

/*** Magic number for debugging ***/
#define TRACKMEM_MAGIC 0x54726b4d /* ASCII: 'TrkM' */

/**************************
 *** PRIVATE STRUCTURES ***
 **************************/

typedef struct TrackMem_Tag_ TrackMem_Tag;

/*** Link a block of memory into a hierarchial list ***/
/* next : next block in this chain of children
 *        for the tail of a list, this is NULL
 * prev : previous block in this chain of children
 *        for the head of a list, this points to the parent or NULL (if none)
 *          isHead iff ((block->prev == NULL) || (block->prev->child == block))
 * child: pointer to the first child of the block
 * tags : pointer to the beginning of tag space
 *          isTagged iff (tags == block)
 *          for memRealloc() and memFree() purposes, tags contains the head
 *            of the actual block we got from memAlloc()
 * size : only present if we can't read this from the system
 */
typedef struct TrackMem_Block_ { /* LISTTREE NODE */
    struct TrackMem_Block_ *next;
    struct TrackMem_Block_ *prev;
    struct TrackMem_Block_ *child;
    TrackMem_Tag           *tags;
#ifndef memSize
    size_t                 size;
#endif
#ifdef DEBUG
    long int               magic;
#endif
} TrackMem_Block;

/*** Tag types ***/
/* These identify special tags associated with memory blocks.  Tags are
 * mobile blocks of memory attached to an allocation which contain private
 * user information.
 *
 * At this time there may only be a single tag of each type, so use them
 * judiciously.  If you are the sole user of an object, you may get around
 * this limitation by using tag types as unique ID fields.
 *
 * Public tag types are defined in memmanager.h
 *
 * TRACKMEM_TAG_END    : marks end of a tag list (length = 0)
 * TRACKMEM_TAG_FREE   : marks free space in a tag list (length = usable size)
 * TRACKMEM_TAG_LINK   : points to a link (if used, must be FIRST tag)
 * TRACKMEM_TAG_EXTEND : points to an taglist extension block
 *                      (replaces a TAG_END at the end of a block)
 */
enum {
    TRACKMEM_TAG_END        = 0x00,
    TRACKMEM_TAG_FREE       = TRACKMEM_TAG_BASE_PRIVATE + 0x01,
    TRACKMEM_TAG_IGNORE     = TRACKMEM_TAG_BASE_PRIVATE + 0x02,
    TRACKMEM_TAG_LINK       = TRACKMEM_TAG_BASE_PRIVATE + 0x03,
    TRACKMEM_TAG_EXTEND     = TRACKMEM_TAG_BASE_PRIVATE + 0x04
};

/*** Tag flags ***/
/* Internal flags are associated by name with the tags they modify.
 * The upper 4 bits are reserved for now.
 */
typedef enum {
    TRACKMEM_FLAG_MASK           = 0x0f,
    TRACKMEM_FLAG_MASK_RESERVED  = 0xf0,
    /* TRACKMEM_TAG_LINK */
        /* use memFree() */
    TRACKMEM_FLAG_LINK_DDEFAULT  = TRACKMEM_DESTRUCTOR_DEFAULT,
        /* no destructor */
    TRACKMEM_FLAG_LINK_DNONE     = TRACKMEM_DESTRUCTOR_NONE,
        /* type void func(void *link) */
    TRACKMEM_FLAG_LINK_DSIMPLE   = TRACKMEM_DESTRUCTOR_SIMPLE,
        /* type void func(void *link, TMEM void *ref) */
    TRACKMEM_FLAG_LINK_DSPECIAL  = TRACKMEM_DESTRUCTOR_SPECIAL
} TrackMem_TagFlag;

/*** Basic tag holder ***/
/* type  : type of flag   (TrackMem_TagType)
 * flags : flags as above (TrackMem_TagFlag)
 * length: length of user data (not including this structure)
 */
struct TrackMem_Tag_ {
    unsigned char      type;   /* usu.  8 bits */
    unsigned char      flags;  /* usu.  8 bits */
    unsigned short int length; /* usu. 16 bits */
};

/*** Links to other blocks of memory ***/
/* target : points to user data
 * size   : size of user block
 * funcXX : destructor functions -- specified using flags
 */
typedef struct TrackMem_TagLink_ {   /* TRACKMEM_TAG_LINK */
    TrackMem_Tag tag;
    void       *target; /* target pointer */
    size_t      size;   /* size of user block */
    union {
        TrackMem_LinkDestructorFunc        funcSimple;
        TrackMem_LinkSpecialDestructorFunc funcSpecial;
    } destructor;
} TrackMem_TagLink;

/*** Extension blocks in tag space (unsupported) ***/
/* seg    : pointer to next tag block
 * seglen : size of next tag block
 */
typedef struct TrackMem_TagExtend_ { /* TRACKMEM_TAG_EXTEND */
    TrackMem_Tag   tag;
    TrackMem_Tag  *seg;
    size_t        seglen;
} TrackMem_TagExtend;


/***********************
 *** INTERNAL PROTOS ***
 ***********************/

/** Internal support subroutines **/
static TINLINE void TrackMemFree_Recurse(TrackMem_Block *block);

/** Inlined operations without MACRO equivalents **/
static TINLINE TrackMem_Tag *TrackMem_TagSeek(const TrackMem_Block *block,
    TrackMem_TagType type);
static TINLINE void TrackMem_TagRemove(TrackMem_Tag *tag);
static TINLINE TrackMem_Block *TrackMem_BlockGetParent(
    const TrackMem_Block *block);
static TINLINE void TrackMem_BlockListLinkSequence(TMEM void *memparent,
    TrackMem_Block *child);
static TINLINE void TrackMem_BlockLinkFree(TrackMem_Block *block);
static TINLINE TrackMem_Block *TrackMem_BlockLinkInit(TMEM void *mem,
    size_t sizeTags, const void *link, size_t sizeLink, size_t sizeTotal);
static TINLINE TrackMem_Block *TrackMem_BlockInit(TMEM void *mem,
    size_t sizeTags, size_t sizeTotal);
static TINLINE size_t TrackMem_TagSpaceCompact(TrackMem_Block *block);
#ifdef DEBUG
static TINLINE void TrackMem_DebugValidate(const TMEM void *mem);
static TINLINE void TrackMem_DebugValidateBlock(
    const TrackMem_Block *block);
#endif

/** Inlined support operations **/
#ifdef TINLINE_ON
static TINLINE TrackMem_Tag *TrackMem_TagFirst(const TrackMem_Block *block);
static TINLINE TrackMem_Tag *TrackMem_TagSkip(const TrackMem_Tag *tag);
static TINLINE size_t TrackMem_TagSpaceSize(const TrackMem_Block *block);

static TINLINE TMEM void *TrackMem_BlockWrap(const TrackMem_Block *block);
static TINLINE TrackMem_Block *TrackMem_BlockUnwrap(const TMEM void *mem);
static TINLINE void *TrackMem_BlockData(const TrackMem_Block *block);
static TINLINE void TrackMem_BlockFree(TrackMem_Block *block);
static TINLINE void TrackMem_BlockListLink(TrackMem_Block *block,
    TMEM void *memparent);
static TINLINE void TrackMem_BlockListUnlink(TrackMem_Block *block);
static TINLINE void TrackMem_BlockListOrphan(TrackMem_Block *block);
static TINLINE int TrackMem_BlockIsHead(const TrackMem_Block *block);
static TINLINE int TrackMem_BlockIsTail(const TrackMem_Block *block);
static TINLINE int TrackMem_BlockIsTagged(const TrackMem_Block *block);
static TINLINE int TrackMem_BlockIsLink(const TrackMem_Block *block);
static TINLINE TrackMem_Block *TrackMem_BlockReinit(TMEM void *mem, size_t sizeTags,
    size_t sizeTotal);
static TINLINE void *TrackMem_BlockLinkData(const TrackMem_Block *block);
#endif


/********************************************
 *** INLINE SUPPORT OPERATIONS AND MACROS ***
 ********************************************/

/*** Make aligned size ***/
/* TrackMem_AlignLocal()
 *
 * Rounds size of chunk up to the next alignment marker for small-scale
 * allocations. eg. structure members and tags.
 */
#define TrackMem_AlignLocal(size) \
    ((size + (MEMALIGN_LOCAL - 1)) & (~(MEMALIGN_LOCAL - 1)))

/* TrackMem_AlignUser()
 *
 * Rounds size of chunk up to the next alignment marker for user allocations.
 * ie. Ensures that the size of negative-offset data does not shift the
 *     system's alignment guarantees for memory allocation.
 */
#define TrackMem_AlignUser(size) \
    ((size + (MEMALIGN_USER - 1)) & (~(MEMALIGN_USER - 1)))

/*** Compute storage to reserve for all tags ***/
/* TrackMem_CalcSizeTags(size)
 *
 * Returns the total size we'll need for all the tags including the
 * requisite END tag and for padding the admin structure up to fit within
 * the alignment specified by MEMALIGN_USER.
 *
 * Input sizeTags must NOT include the space required for the END tag.
 * Output sizeTags includes the space required for the END tag.
 */
#define TrackMem_CalcSizeTags(size) \
    ((size) ? (TrackMem_AlignUser(sizeof(TrackMem_Block) \
        + TrackMem_AlignLocal(size + sizeof(TrackMem_Tag))) \
        - sizeof(TrackMem_Block)) : 0)

/*** Compute total storage to allocate for an object ***/
/* TrackMem_CalcSizeTotal(size)
 *
 * Returns the size we'll need for the tags, admin and user space.
 *
 * Input sizeTags MUST include the space required for the END tag and
 * MUST be aligned correctly.
 */
#define TrackMem_CalcSizeTotal(sizeUser, sizeTags) \
    (sizeTags + sizeof(TrackMem_Block) + sizeUser)

/*** Index into tag space ***/
/* TrackMem_TagFirst()
 *
 * Returns pointer to first tag.
 * Assumes there are tags!  You should check first using
 * TrackMem_BlockIsTagged()
 */
#ifdef TINLINE_ON
static TINLINE TrackMem_Tag *TrackMem_TagFirst(const TrackMem_Block *block) {
    return block->tags;
}
#else
#define TrackMem_TagFirst(block) \
    (block->tags)
#endif

/*** Skip to the next tag in a tag list ***/
/* TrackMem_TagSkip()
 *
 * Returns pointer of next tag assuming there is one.
 * You shouldn't call this if the tag you're holding is END.
 */
#ifdef TINLINE_ON
static TINLINE TrackMem_Tag *TrackMem_TagSkip(const TrackMem_Tag *tag) {
    return ((TrackMem_Tag *)((char *)tag + sizeof(TrackMem_Tag)
        + (size_t)tag->length));
}
#else
#define TrackMem_TagSkip(tag) \
    ((TrackMem_Tag *)((char *)tag + sizeof(TrackMem_Tag) \
        + (size_t)tag->length))
#endif

/*** Determine if a block actually has tags ***/
/* TrackMem_BlockIsTagged()
 *
 * Returns non-zero if block has associated tags.
 */
#ifdef TINLINE_ON
static TINLINE int TrackMem_BlockIsTagged(const TrackMem_Block *block) {
    return ((void *)block->tags != (void *)block);
}
#else
#define TrackMem_BlockIsTagged(block) \
    ((void *)block->tags != (void *)block)
#endif

/*** Seek a tag in tag space ***/
/* TrackMem_TagSeek()
 *
 * Returns pointer to a tag matching the given type or NULL to indicate
 * that there is no such thing.
 */
static TINLINE TrackMem_Tag *TrackMem_TagSeek(const TrackMem_Block *block,
    TrackMem_TagType type) {
    TrackMem_Tag *tag;
    if (TrackMem_BlockIsTagged(block)) {
        tag = TrackMem_TagFirst(block);
        while (tag->type != type) {
            if (tag->type == TRACKMEM_TAG_END) return NULL;
            tag = TrackMem_TagSkip(tag);
        }
    } else tag = NULL;
    return tag;
}

/*** Remove a tag ***/
/* TrackMem_TagRemove()
 *
 * Removes a tag from tag space by marking it as FREE space.  Joins it
 * with the following block if it is also FREE.  Does not attempt to
 * merge with a previous FREE block.
 */
static TINLINE void TrackMem_TagRemove(TrackMem_Tag *tag) {
    TrackMem_Tag *nexttag = TrackMem_TagSkip(tag);
    tag->type = TRACKMEM_TAG_FREE;
    tag->flags = 0;
    if (nexttag->type == TRACKMEM_TAG_FREE)
        tag->length += nexttag->length + sizeof(TrackMem_Tag);
}

/*** Compute tag space size ***/
/* TrackMem_TagSpaceSize()
 *
 * Returns the total size of all tag space, including any space consumed
 * by the END tag.
 *
 * Returns 0 if there are no tags.
 */
#ifdef TINLINE_ON
static TINLINE size_t TrackMem_TagSpaceSize(const TrackMem_Block *block) {
    return ((size_t)((char *)block - (char *)block->tags));
}
#else
#define TrackMem_TagSpaceSize(block) \
    ((size_t)((char *)block - (char *)block->tags))
#endif

/*** Compact tag space ***/
/* TrackMem_TagSpaceCompact()
 *
 * Should only be called when the effective address of the memory block
 * reference has changed.  Doing this permits the following assertion to
 * be valid:
 * newspace = TrackMem_Realloc(oldspace, newsize);
 * if (newspace != oldspace) {
 *    // rebuild cached tag list
 * } else {
 *    // tags have not moved
 * }
 *
 * Returns the size of the largest tag that can be added including its
 * administrative info.
 *
 * TrackMemRealloc() users
 *   -- don't use unless effective address changed
 * TrackMemReallocTagSpace() users
 *   -- safe at all times if newtagspacesize != oldtagspacesize
 *      since we do not guarantee anything about the state of tag space.
 */
static TINLINE size_t TrackMem_TagSpaceCompact(TrackMem_Block *block) {
    size_t totalFree = 0;
    if (TrackMem_BlockIsTagged(block)) {
        char        *lastUsed;
        TrackMem_Tag *tag = TrackMem_TagFirst(block);
        lastUsed = (char *)tag;
        /* pack all tags at bottom and compute free space */
        while (tag->type != TRACKMEM_TAG_END) {
            TrackMem_Tag *nexttag = TrackMem_TagSkip(tag);
            size_t tagSize = tag->length + sizeof(TrackMem_Tag);
            if (tag->type == TRACKMEM_TAG_FREE) {
                /* if tag is FREE, add to count... */
                totalFree += tagSize;
            } else {
                if (totalFree != 0) {
                    /* if we encountered free space then we must move tags
                     * down the list to fill in gaps */
                    memmove(lastUsed, tag, tagSize);
                    lastUsed += tagSize;
                } else {
                    lastUsed = (char *)nexttag;
                }
            }
            tag = nexttag;
        }
        /* check position */
        ASSERT(((char *)tag + sizeof(TrackMem_Tag)) == (char *)block);
        /* add a new FREE tag to represent the entirety of free space */
        if (totalFree != 0) {
            tag = (TrackMem_Tag *)lastUsed;
            ASSERT(totalFree >= sizeof(TrackMem_Tag));
            /* add FREE */
            tag->type = TRACKMEM_TAG_FREE;
            tag->flags = 0;
            tag->length = totalFree - sizeof(TrackMem_Tag);
            tag = TrackMem_TagSkip(tag);
            /* add END */
            tag->type = TRACKMEM_TAG_END;
            tag->flags = 0;
            tag->length = 0;
            /* check position */
            ASSERT(((char *)tag + sizeof(TrackMem_Tag)) == (char *)block);
        }
    }
    return totalFree;
}

/*** Wrap a memory allocation to get storage marker ***/
/* TrackMem_BlockWrap()
 *
 * Returns a pointer to the storage marker to be returned to the user..
 * This may or may not be a pointer to the actual data space.
 */
#ifdef TINLINE_ON
static TINLINE TMEM void *TrackMem_BlockWrap(const TrackMem_Block *block) {
    return ((TMEM void *)((char *)block + sizeof(TrackMem_Block)));
}
#else
#define TrackMem_BlockWrap(block) \
    ((TMEM void *)((char *)block + sizeof(TrackMem_Block)))
#endif

/*** Validate a block ***/
/* TrackMem_DebugValidate() / TrackMem_DebugValidateBlock()
 *
 * Given a pointer to a block, causes an assertion if the provided
 * block reference appears to be inconsistent.  For memory refs, d
 * oes nothing if mem is NULL.
 */
#ifdef DEBUG
static TINLINE void TrackMem_DebugValidate(const TMEM void *mem) {
    if (mem) {
        TrackMem_Block *block =
            ((TrackMem_Block *)((char *)mem - sizeof(TrackMem_Block)));
        TrackMem_DebugValidateBlock(block);
    }
}
static TINLINE void TrackMem_DebugValidateBlock(
    const TrackMem_Block *block) {
    ASSERT(block != NULL);
    ASSERT(block->magic == TRACKMEM_MAGIC);
    ASSERT(block->child != block);
    ASSERT(block->prev != block);
    ASSERT(block->next != block);
    ASSERT((! block->child) || (block->child->prev == block));
}
#else
#define TrackMem_DebugValidate(mem)
#define TrackMem_DebugValidateBlock(block)
#endif

/*** Unwrap a memory allocation given storage marker ***/
/* TrackMem_BlockUnwrap()
 *
 * Given a pointer to the storage marker, locates and returns
 * a pointer to the block.
 */
#ifdef INLINE
static INLINE TrackMem_Block *TrackMem_BlockUnwrap(const TMEM void *mem) {
    ASSERT(mem != NULL);
    TrackMem_DebugValidate(mem);
    return ((TrackMem_Block *)((char *)mem - sizeof(TrackMem_Block)));
}
#else
#define TrackMem_BlockUnwrap(mem) \
    ((TrackMem_Block *)((char *)mem - sizeof(TrackMem_Block)))
#endif

/*** Get block size ***/
/* TrackMem_BlockSize()
 *
 * Returns the total allocate size of a block of memory (as was passed to
 * memAlloc() / memRealloc())
 */
#ifdef memSize
#define TrackMem_BlockSize(block) (memSize((block)->tags))
#else
#define TrackMem_BlockSize(block) ((block)->size)
#endif

/****************************************
 *** SUBROUTINES FOR ARBITRARY BLOCKS ***
 ****************************************/

/*** Get the actual data pointer associated with a given block ***/
/* TrackMem_BlockData()
 *
 * Returns a pointer to the user data associated with this block.
 * This is only really useful when LINKs may be involved.
 */
#ifdef TINLINE_ON
static TINLINE void *TrackMem_BlockData(const TrackMem_Block *block) {
    return ((void *)((TrackMem_BlockIsLink(block)) ?
        TrackMem_BlockLinkData(block) : TrackMem_BlockWrap(block)));
}
#else
#define TrackMem_BlockData(block) \
    ((void *)((TrackMem_BlockIsLink(block)) ? \
        TrackMem_BlockLinkData(block) : TrackMem_BlockWrap(block)))
#endif

/*** Free an entity completely ***/
/* TrackMem_BlockFree()
 *
 * Frees all storage associated with a block including LINKs and
 * EXTENDed tags. [not impl.]
 */
#ifdef TINLINE_ON
static TINLINE void TrackMem_BlockFree(TrackMem_Block *block) {
#ifdef DEBUG
    ASSERT(block->magic == TRACKMEM_MAGIC);
#endif
    if (TrackMem_BlockIsLink(block))
        TrackMem_BlockLinkFree(block);
    memFree(block->tags);
}
#else
#define TrackMem_BlockFree(block) { \
    if (TrackMem_BlockIsLink(block)) \
        TrackMem_BlockLinkFree(block); \
    memFree(block->tags); \
}
#endif

/*** Link an entity ***/
/* TrackMem_BlockListLink()
 *
 * Link a block to a parent's chain.
 */
#ifdef TINLINE_ON
static TINLINE void TrackMem_BlockListLink(TrackMem_Block *block,
    TMEM void *memparent) {
    if (memparent) {
        TrackMem_Block *parent = TrackMem_BlockUnwrap(memparent);
        if ( (block->next = parent->child) ) block->next->prev = block;
        block->prev = parent;
        parent->child = block;
    } else {
        block->prev = block->next = NULL;
    }
}
#else
#define TrackMem_BlockListLink(block, memparent) { \
    if (memparent) { \
        TrackMem_Block *parent = TrackMem_BlockUnwrap(memparent); \
        if ( (block->next = parent->child) ) block->next->prev = block; \
        block->prev = parent; \
        parent->child = block; \
    } else { \
        block->prev = block->next = NULL; \
    } \
}
#endif

/*** Link a list of blocks into a tree ***/
/* TrackMem_BlockListLinkSequence()
 *
 * Assign a linked list of blocks to a tree. Only next/prev
 * at ends are updated.  Children are linked in according to
 * the same rules as TrackMem_BlockListLink().
 */
static TINLINE void TrackMem_BlockListLinkSequence(TMEM void *memparent,
    TrackMem_Block *child)
{
    TrackMem_Block *firstchild = child;
    TrackMem_Block *lastchild;

    /* assumes these children have no valid parent and that they
     * have been correctly unlinked from any possible sponsors. */

    /* assumes child->prev is at head of list */
    if (memparent) {
         TrackMem_Block *parent = TrackMem_BlockUnwrap(memparent);
         /* only seeks to end of list if it has to link in old first child */
         if (parent->child) {
             do { lastchild = child; } while ( (child = child->next) );
             lastchild->next = parent->child;
             parent->child->prev = lastchild;
         }
         firstchild->prev = parent;
         parent->child = firstchild;
    } else {
         /* orphaned list */
         firstchild->prev = NULL;
         /* need not seek to end as it is already terminated */
    }
}

/*** Unlink an entity ***/
/* TrackMem_BlockListUnlink()
 *
 * Unlink a block from a parent's chain.
 * This is a very fast and inexpensive operation.
 *
 * Note: does not set block->next and block->prev to NULL.
 */
#ifdef TINLINE_ON
static TINLINE void TrackMem_BlockListUnlink(TrackMem_Block *block) {
    if (block->next) block->next->prev = block->prev;
    if (block->prev) {
        if (block->prev->child == block) /* parent? */
            block->prev->child = block->next;
        else block->prev->next = block->next;
    }
}
#else
#define TrackMem_BlockListUnlink(block) { \
    if (block->next) block->next->prev = block->prev; \
    if (block->prev) { \
        if (block->prev->child == block) /* parent? */ \
            block->prev->child = block->next; \
        else block->prev->next = block->next; \
    } \
}
#endif

/*** Unlink and orphan an entity ***/
/* TrackMem_BlockListOrphan()
 *
 * Unlink a block from a parent's chain and leave orphaned.
 * This is a very fast and inexpensive operation.
 *
 * Note: unlike BlockListUnlink() sets block->next and block->prev to NULL.
 */
#ifdef TINLINE_ON
static TINLINE void TrackMem_BlockListOrphan(TrackMem_Block *block) {
    TrackMem_BlockListUnlink(block);
    block->next = block->prev = NULL;
}
#else
#define TrackMem_BlockListOrphan(block) { \
    TrackMem_BlockListUnlink(block); \
    block->next = block->prev = NULL; \
}
#endif

/*** Determine if a node is head or tail ***/
/* TrackMem_BlockIs[Head/Tail]()
 *
 * Returns non-zero if block is a head or tail of a list.
 * [These are not used at the moment]
 */
#ifdef TINLINE_ON
static TINLINE int TrackMem_BlockIsHead(const TrackMem_Block *block) {
    return ((block->prev == NULL) || (block->prev->child == block));
}
static TINLINE int TrackMem_BlockIsTail(const TrackMem_Block *block) {
    return (block->next == NULL);
}
#else
#define TrackMem_BlockIsHead(block) \
    ((block->prev == NULL) || (block->prev->child == block))
#define TrackMem_BlockIsTail(block) \
    (block->next == NULL)
#endif

/*** Determine if a node represents a link ***/
/* TrackMem_BlockIsLink()
 *
 * Returns non-zero if block is a link.
 */
#ifdef TINLINE_ON
static TINLINE int TrackMem_BlockIsLink(const TrackMem_Block *block) {
    return (TrackMem_BlockIsTagged(block)) &&
        (TrackMem_TagFirst(block)->type == TRACKMEM_TAG_LINK);
}
#else
#define TrackMem_BlockIsLink(block) \
    ((TrackMem_BlockIsTagged(block)) && \
     (TrackMem_TagFirst(block)->type == TRACKMEM_TAG_LINK))
#endif

/*** Reinitialize a block after a realloc ***/
/* TrackMem_BlockReinit()
 *
 * Repairs a block pointer after a realloc of user space.
 *
 * SizeTags must reflect the total size of the tag space as returned
 * by TrackMem_TagSpaceSize() or TrackMem_CalcSizeTags().
 */
static TINLINE TrackMem_Block *TrackMem_BlockReinit(TMEM void *mem,
    size_t sizeTags, size_t sizeTotal) {
    TrackMem_Block *block;
    if (sizeTags) {
        block = (TrackMem_Block *)((char *)mem + sizeTags);
    } else {
        block = (TrackMem_Block *)mem;
    }
    block->tags = (TrackMem_Tag *)mem;
#ifndef memSize
    block->size = sizeTotal;
#else
#ifdef DEBUG
    block->magic = TRACKMEM_MAGIC;
#endif
    sizeTotal = sizeTotal; /* fool compiler into thinking this is useful */
#endif
    return block;
}

/*** Get parent ***/
/* TrackMem_BlockGetParent()
 *
 * Returns a pointer to the parent's block structure or NULL if there
 * is none.
 */
static TINLINE TrackMem_Block *TrackMem_BlockGetParent(
    const TrackMem_Block *block) {
    TrackMem_Block *pblock;
    while ( (pblock = block->prev) ) {
        if (pblock->child == block) return pblock;
        block = pblock;
    }
    return NULL;
}


/****************************************************************
 *** SPECIAL CODE FOR MANIPULATING BLOCKS WHICH ARE NOT LINKS ***
 ****************************************************************/

/*** Initialize newly-allocated entity (not a link) ***/
/* TrackMem_BlockInit()
 *
 * Initializes a blank block object.  In particular, sets up the
 * tag space and returns a pointer to the new block structure.
 *
 * sizeTags must be the value returned by TrackMem_CalcSizeTags().
 */
static TINLINE TrackMem_Block *TrackMem_BlockInit(TMEM void *mem,
    size_t sizeTags, size_t sizeTotal) {
    TrackMem_Block *block;
    if (sizeTags) {
        TrackMem_Tag *tag;
        ASSERT(sizeTags <= TRACKMEM_TAG_MAXTOTALSIZE);
        /* Setup TAG_FREE */
        tag = (TrackMem_Tag *)mem;
        tag->type = TRACKMEM_TAG_FREE;
        tag->flags = 0;
        tag->length = sizeTags - 2*sizeof(TrackMem_Tag);
        /* Setup TAG_END */
        /* This will still succeed even if sizeTags == sizeof(TrackMem_Tag) */
        tag = (TrackMem_Tag *)((char *)mem + sizeTags - sizeof(TrackMem_Tag));
        tag->type = TRACKMEM_TAG_END;
        tag->flags = 0;
        tag->length = 0;
        block = (TrackMem_Block *)((char *)mem + sizeTags);
    } else {
        block = (TrackMem_Block *)mem;
    }
    block->tags = (TrackMem_Tag *)mem;
#ifndef memSize
    block->size = sizeTotal;
#else
    sizeTotal = sizeTotal; /* fool compiler into thinking this is useful */
#endif
#ifdef DEBUG
    block->magic = TRACKMEM_MAGIC;
#endif
    return block;
}


/************************************************************
 *** SPECIAL CODE FOR MANIPULATING BLOCKS WHICH ARE LINKS ***
 ************************************************************/

/*** Get the data associated with a link ***/
/* TrackMem_BlockLinkData()
 *
 * Returns the target pointer of a link.
 * Assumes this is indeed a link
 */
#ifdef TINLINE_ON
static TINLINE void *TrackMem_BlockLinkData(const TrackMem_Block *block) {
    return ((TrackMem_TagLink *)TrackMem_TagFirst(block))->target;
}
#else
#define TrackMem_BlockLinkData(block) \
    (((TrackMem_TagLink *)TrackMem_TagFirst(block))->target)
#endif

/*** Free a link only ***/
/* TrackMem_BlockLinkFree()
 *
 * Calls the destructor attached to a link but does nothing else.
 * (ie. represents only part of the effective freeing operations)
 */
static TINLINE void TrackMem_BlockLinkFree(TrackMem_Block *block) {
    TrackMem_TagLink *tag = (TrackMem_TagLink *)TrackMem_TagFirst(block);
    switch (tag->tag.flags & TRACKMEM_FLAG_MASK) {
        default:
            ASSERT( /* give user a useful assertion message */
                (tag->tag.flags == TRACKMEM_FLAG_LINK_DNONE) ||
                (tag->tag.flags == TRACKMEM_FLAG_LINK_DDEFAULT) ||
                (tag->tag.flags == TRACKMEM_FLAG_LINK_DSIMPLE) ||
                (tag->tag.flags == TRACKMEM_FLAG_LINK_DSPECIAL));
            break;
        case TRACKMEM_FLAG_LINK_DNONE: /* no destructor */
            break;
        case TRACKMEM_FLAG_LINK_DDEFAULT: /* use memFree() */
            if (tag->target) memFree(tag->target);
            break;
        case TRACKMEM_FLAG_LINK_DSIMPLE:
            tag->destructor.funcSimple(tag->target);
            break;
        case TRACKMEM_FLAG_LINK_DSPECIAL:
            tag->destructor.funcSpecial(tag->target, TrackMem_BlockWrap(block));
            break;
    }
}

/*** Initialize newly-allocated LINK ***/
/* TrackMem_BlockLinkInit()
 *
 * Initializes a blank block object.  In particular, sets up the
 * link and the tag space then returns a pointer to the new block structure.
 *
 * sizeTags must be the value returned by TrackMem_CalcSizeTags()
 * with the size having had sizeof(TrackMem_TagLink) added to it first.
 */
static TINLINE TrackMem_Block *TrackMem_BlockLinkInit(TMEM void *mem,
    size_t sizeTags, const void *link, size_t sizeLink, size_t sizeTotal) {
    TrackMem_TagLink *tag;
    TrackMem_Block *block;
    ASSERT(sizeTags >= (sizeof(TrackMem_TagLink) + sizeof(TrackMem_Tag)));
    block = (TrackMem_Block *)((char *)mem + sizeLink);
    /* Setup TAG_LINK */
    tag = (TrackMem_TagLink *)mem;
    tag->tag.type = TRACKMEM_TAG_LINK;
    tag->tag.flags = TRACKMEM_FLAG_LINK_DDEFAULT;
    tag->tag.length = sizeof(TrackMem_TagLink) - sizeof(TrackMem_Tag);
    tag->target = (void *)link; /* discard const */
    tag->size = sizeLink;
    /* Setup TAG_FREE */
    tag++;
    tag->tag.type = TRACKMEM_TAG_FREE;
    tag->tag.flags = 0;
    tag->tag.length =
        sizeTags - 2*sizeof(TrackMem_Tag) - sizeof(TrackMem_TagLink);
    /* Setup TAG_END */
    /* This will still succeed even if sizeTags == sizeof(TrackMem_Tag) */
    tag = (TrackMem_TagLink *)((char *)mem + sizeTags - sizeof(TrackMem_Tag));
    tag->tag.type = TRACKMEM_TAG_END;
    tag->tag.flags = 0;
    tag->tag.length = 0;
    block->tags = (TrackMem_Tag *)mem;
#ifndef memSize
    block->size = sizeTotal;
#else
    sizeTotal = sizeTotal; /* fool compiler into thinking this is useful */
#endif
#ifdef DEBUG
    block->magic = TRACKMEM_MAGIC;
#endif
    return block;
}


/**********************************
 *** MEMORY TRACKING PRIMITIVES ***
 **********************************/

/* ---------------------------------------------------------------------------
 * TrackMemAlloc()
 *
 * Builds a hierarchy of dependant memory allocations from
 * a common source.  To create the initial hierarchial segment,
 * alloc with memparent = NULL.
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *              size      - size of user data we want to store here
 *              sizeTags  - size of tag space to preallocate
 *
 * Returns a pointer to a new memory block, or NULL on failure.
 */
TMEM void *TrackMemAlloc(TMEM void *memparent, size_t size, size_t sizeTags)
{
    TMEM void *mem;
    TrackMem_Block *block;

    ASSERT(sizeTags < TRACKMEM_TAG_MAXTOTALSIZE);
    sizeTags = TrackMem_CalcSizeTags(sizeTags);
    size = TrackMem_CalcSizeTotal(size, sizeTags);
    /* allocate block */
    mem = (TrackMem_Block *)memAlloc(size);
    if (mem) {
        /* init and link */
        block = TrackMem_BlockInit(mem, sizeTags, size);
        block->child = NULL;
        TrackMem_BlockListLink(block, memparent);
        mem = TrackMem_BlockWrap(block);
    }
    TrackMem_DebugValidate(mem);
    TrackMem_DebugValidate(memparent);
    return mem;
}

/* ---------------------------------------------------------------------------
 * TrackMemClone()
 *
 * Creates an exact duplicate of an existing TrackMem allocation including
 * the size and contents of its Tag and User spaces.  The new allocation
 * is then attached to another parent's tree.
 *
 * It is dangerous to perform this operation on LINKs as they may then
 * be freed twice if a destructor is selected.
 *
 * Note: This operation DOES NOT clone children!
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *              memsource - pointer to memory block to be cloned
 *
 * Returns a pointer to a new memory block, or NULL on failure.
 */
TMEM void *TrackMemClone(TMEM void *memparent, const TMEM void *memsource)
{
    TMEM void *mem = NULL;

    if (memsource) {
        TrackMem_Block *oldBlock, *newBlock;
        size_t sizeTags;
        size_t sizeTotal;
        size_t sizeUser;

        oldBlock = TrackMem_BlockUnwrap(memsource);
        sizeTags = TrackMem_TagSpaceSize(oldBlock);
        sizeTotal = TrackMem_BlockSize(oldBlock);
        sizeUser = sizeTotal - sizeTags - sizeof(TrackMem_Block);

        /* allocate new block */
        mem = (TrackMem_Block *)memAlloc(sizeTotal);
        if (mem) {
            /* init and copy data */
            newBlock = TrackMem_BlockInit(mem, sizeTags, sizeTotal);
            mem = TrackMem_BlockWrap(newBlock);
            if (sizeTags > 0)
                memcpy(newBlock->tags, oldBlock->tags, sizeTags);
            if (sizeUser > 0)
                memcpy(mem, memsource, sizeUser);

            /* link */
            newBlock->child = NULL;
            TrackMem_BlockListLink(newBlock, memparent);
        }
    }
    TrackMem_DebugValidate(mem);
    TrackMem_DebugValidate(memparent);
    return mem;
}

/* ---------------------------------------------------------------------------
 * TrackMemDupMemory()
 *
 * Allocate space for a duplicate of an existing memory object and copy.
 * This function does not assume that the memory source is an TrackMem
 * object so the duplicated block does not inherit any tags.
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *              memsource - pointer to a memory block to clone
 *              size      - size of user data to be cloned
 *              sizeTags  - size of tag space to preallocate
 *
 * Returns a pointer to a new memory block, or NULL on failure.
 */
TMEM void *TrackMemDupMemory(TMEM void *memparent, const void *memsource,
    size_t size, size_t sizeTags)
{
    TMEM void *mem = TrackMemAlloc(memparent, size, sizeTags);
    if (mem && memsource && size) {
        memcpy(mem, memsource, size);
    }
    TrackMem_DebugValidate(mem);
    TrackMem_DebugValidate(memparent);
    return mem;
}

/* ---------------------------------------------------------------------------
 * TrackMemDupString()
 *
 * Duplicate a string and link to a parent's tree.
 *
 * This function does not assume that the memory source is an TrackMem
 * object so the duplicated block does not inherit any tags.
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *              memsource - pointer to a memory block to clone
 *              size      - size of user data to be cloned
 *              sizeTags  - size of tag space to preallocate
 *
 * Returns a pointer to a new memory block, or NULL on failure.
 */
char *TrackMemDupString(TMEM void *memparent, const char *strsource,
    size_t sizeTags)
{
    return TrackMemDupMemory(memparent, strsource, strlen(strsource) + 1,
        sizeTags);
}

/* ---------------------------------------------------------------------------
 * TrackMemLink()
 *
 * Allocates a tracking identifier for a block of linked memory.
 * The contents of the target are not copied (as with TrackMemClone()),
 * but will be free'd by TrackMemFree() as if the block had been
 * allocated in the usual way.
 *
 * Restrictions:
 *   - the returned pointer is fictional and does not refer to any
 *     particular region of interest
 *   - TrackMemRealloc() will cause the link to be freed and copy
 *     the data... when this occurs, the new pointer void * will be
 *     valid, and the old one will not
 *   - when freed, searches for TRACKMEM_TAG_DESTRUCTOR which may
 *     contain a special destructor function as specified by
 *     TrackMemTagSetDestructor().
 *
 * Note: The link is absorbed if TrackMemRealloc gets called.
 *       (if this is not desirable, simply don't call TrackMemRealloc!!)
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *              target    - pointer to target of link (need not necessarily
 *                          be a memory block --> see Destructors)
 *              targetSize- size of target block (needed only to permit
 *                          absorption during TrackMemRealloc() and for
 *                          reading the data size)
 *              sizeTags  - size of tag space to preallocate
 *
 * Returns a pointer to a reference to be used for manipulating this link
 * or NULL on failure.  Note the returned pointer does NOT match that of
 * the target link!
 */
TMEM void *TrackMemLink(TMEM void *memparent, const void *target, size_t targetSize,
    size_t sizeTags)
{
    TMEM void *mem;
    TrackMem_Block *block;
    size_t size;

    ASSERT(sizeTags < TRACKMEM_TAG_MAXTOTALSIZE);
    sizeTags += sizeof(TrackMem_TagLink);
    sizeTags = TrackMem_CalcSizeTags(sizeTags);
    size = TrackMem_CalcSizeTotal(0, sizeTags);

    /* allocate block */
    mem = (TrackMem_Block *)memAlloc(size);
    if (mem) {
        /* init and link */
        block = TrackMem_BlockLinkInit(mem, sizeTags, target,
            targetSize, size);
        block->child = NULL;
        TrackMem_BlockListLink(block, memparent);
        mem = TrackMem_BlockWrap(block);
    }
    TrackMem_DebugValidate(mem);
    TrackMem_DebugValidate(memparent);
    return mem;
}

/* ---------------------------------------------------------------------------
 * TrackMemRealloc()
 *
 * Extends/shrinks an existing allocation.  Preserves all hierarchial and tag
 * information.  To extend the size of the taglist use
 * TrackMemReallocTagSpace().
 *
 * Note: LINKs are ABSORBED by this process.  That is, this function assumes
 *       that links point to accessible memory and copies the data to the new
 *       buffer after allocation.  It will then call the appropriate
 *       destructor for that link.  If this is undesirable, don't use this
 *       function on links!
 *
 * Parameters:  mem       - pointer to memory block to extend
 *              newSize   - desired new size
 *
 * Returns a pointer to the extended memory block or NULL on failure.
 * There are no guarantees that the original block is still valid if this
 * fails. ** FIXME!!! **
 */
TMEM void *TrackMemRealloc(TMEM void *mem, size_t newSize)
{
    TrackMem_Block *block, *oldBlock;
    void           *newMem, *oldMem;
    size_t          sizeTags, size;

    /* if no original allocation to extend, use Alloc */
    if (! mem) return mem = TrackMemAlloc(NULL, newSize, 0);

    /* locate old block and compute its size */
    oldBlock = TrackMem_BlockUnwrap(mem);
    sizeTags = TrackMem_TagSpaceSize(oldBlock);
    size = TrackMem_CalcSizeTotal(newSize, sizeTags);
    oldMem = oldBlock->tags;
    /* extend/shrink block */
    newMem = (TrackMem_Block *)memRealloc(oldMem, size);
    if (! newMem) return NULL; /* assume old block killed FIXME */
    block = TrackMem_BlockReinit(newMem, sizeTags, size);
    newMem = TrackMem_BlockWrap(block);

    /* if the old block was a link, then clone data first */
    if (TrackMem_BlockIsLink(block)) {
        TrackMem_TagLink *tag = (TrackMem_TagLink *)TrackMem_TagFirst(block);
        ASSERT(tag->tag.type == TRACKMEM_TAG_LINK);
        if ((tag->target) && (tag->size)) {
            /* copy data */
            memcpy(newMem, tag->target, (newSize < tag->size) ?
                newSize : tag->size);
        }
        /* kill old link and tag */
        TrackMem_BlockLinkFree(block);
        TrackMem_TagRemove((TrackMem_Tag *)tag);
    }

    /* only update links if we actually got a new block */
    if (oldBlock != block) {
        /* if linked to any siblings, adjust those links */
        if (block->next) block->next->prev = block;
        if (block->prev) {
            if (block->prev->child == oldBlock) /* parent? */
                block->prev->child = block;
            else block->prev->next = block;
        }
        if (block->child) block->child->prev = block;
        /* since the address changed, we can also compact the tags */
        TrackMem_TagSpaceCompact(block);
    }
    TrackMem_DebugValidate(newMem);
    return newMem;
}

/* ---------------------------------------------------------------------------
 * TrackMemReallocTagSpace()
 *
 * Extends/shrinks the current allocated size of tag space.  Note that
 * shrinking tag space is for internal purposes only and may cause
 * unpredictable behaviour if there are currently allocated tags in the
 * region which is discarded.
 *
 * Do not assume that if the user space pointer does not change that
 * pointers to tags have not become invalid.  Assume this function call
 * always invalidates all pointers and everything'll be just dandy.
 *
 * Note: LINKs are UNTOUCHED by this process.
 *
 * Parameters:  mem       - pointer to memory block to extend
 *              newSize   - desired new tag space size
 *
 * Returns a pointer to the extended memory block or NULL on failure.
 * There are no guarantees that the original block is still valid if this
 * fails. ** FIXME!!! **
 */
TMEM void *TrackMemReallocTagSpace(TMEM void *mem, size_t newSize)
{
    TrackMem_Block *newBlock, *oldBlock;
    void          *newMem, *oldMem;
    size_t         oldSize, oldTotal, newTotal;

    /* if no original allocation to extend, use Alloc */
    if (! mem) return mem = TrackMemAlloc(NULL, 0, newSize);

    ASSERT(newSize < TRACKMEM_TAG_MAXTOTALSIZE);
    newSize = TrackMem_CalcSizeTags(newSize);

    /* locate old block and compute its size */
    oldBlock = TrackMem_BlockUnwrap(mem);
    oldSize = TrackMem_TagSpaceSize(oldBlock);
    if (oldSize == newSize) return mem;
    oldTotal = TrackMem_BlockSize(oldBlock);
    newTotal = TrackMem_CalcSizeTotal(oldTotal -
        ((char *)mem - (char *)oldBlock->tags), newSize);
    oldMem = oldBlock->tags;
    /* extend/shrink block */
    if (newSize > oldSize) {
        /* if we're extending the block, realloc first, then shift data up */
        newMem = (TrackMem_Block *)memRealloc(oldMem, newTotal);
        if (! newMem) return NULL; /* assume old block killed */
        memmove((char *)newMem + newSize, (char *)newMem + oldSize,
            oldTotal - oldSize);
    } else {
        /* if we're shrinking the block, shift data down, then realloc */
        memmove((char *)oldMem + newSize, (char *)oldMem + oldSize,
            oldTotal - oldSize);
        newMem = (TrackMem_Block *)memRealloc(oldMem, newTotal);
        if (! newMem) return NULL; /* assume old block killed */
    }
    newBlock = TrackMem_BlockReinit(newMem, newSize, newTotal);
    newMem = TrackMem_BlockWrap(newBlock);

    /* clean up the tag space */
    if (newSize != 0) {
        TrackMem_Tag *tag;
        if (newSize > oldSize) {
            /* If we extended the block, change the old END tag to a FREE tag
             * then add a new END tag.  We need not coalesce the FREE chunks
             * as this will be done later. */
            size_t dist;
            dist = newSize - oldSize - sizeof(TrackMem_Tag);
            if (oldSize != 0) {
                tag = TrackMem_TagSeek(newBlock, TRACKMEM_TAG_END);
                ASSERT(tag);
                tag->length = dist;
            } else {
                tag = TrackMem_TagFirst(newBlock);
                if (dist == 0) {
                    tag->length = TRACKMEM_TAG_MAXTOTALSIZE; /* flag */
                } else {
                    tag->length = dist - sizeof(TrackMem_Tag);
                }
            }
        } else {
            /* If we shrank the block, locate the last valid tag, and replace
             * it by a FREE tag of correct length, then add a new END tag.
             * Note that this will safely handle conditions damaged tail tags.
             * Again, we needn't try to coalesce the FREE chunks as this will
             * be done later. */
            tag = TrackMem_TagFirst(newBlock);
            /* loops until (length of tag + size of its header + size of END
             * tag) exceeds or equals the remaining space */
            for(;;) {
                size_t dist;
                ASSERT(tag != TRACKMEM_TAG_END); /* should NEVER happen! */
                /* dist = remaining space required for all this plus END */
                dist = ((char *)newBlock - (char *)tag) - sizeof(TrackMem_Tag);
                /* if we only have room for the end tag, break now. */
                if (dist == 0) {
                    tag->length = TRACKMEM_TAG_MAXTOTALSIZE; /* flag */
                    break;
                }
                /* correct dist for space consumption of this tag...
                 * then check if the tag fits and if not replace with FREE */
                dist -= sizeof(TrackMem_Tag);
                if ((size_t)tag->length > dist) {
                    tag->length = dist;
                    break;
                }
                tag = TrackMem_TagSkip(tag);
            }
        }
        /* If we had room, add the FREE tag... */
        if (tag->length != TRACKMEM_TAG_MAXTOTALSIZE) {
            tag->type = TRACKMEM_TAG_FREE;
            tag->flags = 0;
            tag = TrackMem_TagSkip(tag);
        }
        /* Now add new END tag... */
        tag->type = TRACKMEM_TAG_END;
        tag->flags = 0;
        tag->length = 0;
        ASSERT(((char *)tag + sizeof(TrackMem_Tag)) == (char *)newBlock);
    } /* Otherwise we reduced tag space to nothing so there is nothing left
       * to do! */

    /* Note: We cannot assume the address of the block has changed simply
     *       because it has been displaced in the structure.  It is possible,
     *       however unlikely, that the allocated memory chunk was shifted
     *       so that the block appears at the same location from the user's
     *       perspective! */
    if (newBlock != oldBlock) {
        /* if linked to any siblings, adjust those links */
        if (newBlock->next) newBlock->next->prev = newBlock;
        if (newBlock->prev) {
            if (newBlock->prev->child == oldBlock) /* parent? */
                newBlock->prev->child = newBlock;
            else newBlock->prev->next = newBlock;
        }
        if (newBlock->child) newBlock->child->prev = newBlock;
    }
    /* now compact the tags */
    TrackMem_TagSpaceCompact(newBlock);
    TrackMem_DebugValidate(newMem);
    return newMem;
}

/* ---------------------------------------------------------------------------
 * TrackMemFree()
 *
 * Recursively frees an allocation and all of its children in one go.
 * This function traverses the tree of children in depth-first order;
 * that is, it will kill any children before attacking the parent. *8)
 *
 * Parameters:  mem       - pointer to memory block to be released
 */
void TrackMemFree(TMEM void *mem)
{
    /* silently ignore NULL pointers */
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        /* if linked to children, recurse to free all children first */
        if (block->child) {
            TrackMemFree_Recurse(block->child);
            block->child = NULL;    /* safer: in case of special destructor */
            TrackMem_DebugValidateBlock(block);
        }
        /* if linked to any siblings/parent, adjust those links */
        TrackMem_BlockListOrphan(block); /* safer: as above */
        /* free block control structure, data, tags, and links */
        TrackMem_BlockFree(block);
    }
}

/* ---------------------------------------------------------------------------
 * TrackMemFreeChildren()
 *
 * Recursively frees all the children of an allocation.
 * This function traverses the tree of children in depth-first order;
 * that is, it will kill any children before attacking the parent. *8)
 *
 * Parameters:  mem       - pointer to memory block whose children are to
 *                          be released
 */
void TrackMemFreeChildren(TMEM void *mem)
{
    /* silently ignore NULL pointers */
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        /* if linked to children, recurse to free all children */
        if (block->child) TrackMemFree_Recurse(block->child);
        block->child = NULL;
        TrackMem_DebugValidateBlock(block);
    }
}
static TINLINE void TrackMemFree_Recurse(TrackMem_Block *block)
{
    /* block->prev ignored */
    /* optimization: only recurse if there are children */
    while (block) {
        TrackMem_Block *prevblock;
        block->prev = NULL;
        if (block->child) {
            ASSERT(block->child != block);
            TrackMemFree_Recurse(block->child);
            block->child = NULL; /* safer: in case of special destructor */
            TrackMem_DebugValidateBlock(block);
        }
        /* locate next in list */
        prevblock = block;
        block = block->next;
        /* if linked to any siblings/parent, adjust those links */
        TrackMem_BlockListUnlink(prevblock);
        prevblock->next = NULL;
        /* free block control structure, data, tags, and links */
        TrackMem_BlockFree(prevblock);
    }
}

/* ---------------------------------------------------------------------------
 * TrackMemReassign()
 *
 * Unlink from the tree and reassign to some other parent or orphan (NULL
 * parent).  Preserved all hierarchial and tag information.
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *                          (may be NULL)
 *              mem       - pointer to memory block to be reassigned
 */
void TrackMemReassign(TMEM void *memparent, TMEM void *mem)
{
    /* silently ignore NULL pointers */
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        /* if linked to any siblings, adjust those links */
        TrackMem_BlockListUnlink(block);
        /* assign to some other parent */
        TrackMem_BlockListLink(block, memparent);
        TrackMem_DebugValidateBlock(block);
    }
    TrackMem_DebugValidate(memparent);
}

/* ---------------------------------------------------------------------------
 * TrackMemReassignChildren()
 *
 * Unlink all children from this allocation and add to the tree belonging
 * to some other parent, or orphan them all (if NULL).
 *
 * Parameters:  memparent - pointer to a memory block to use as parent
 *                          (may be NULL)
 *              mem       - pointer to memory block whose children are to be
 *                          reassigned
 */
void TrackMemReassignChildren(TMEM void *memparent, TMEM void *mem)
{
    /* silently ignore NULL pointers and NOP's */
    if ((mem) && (mem != memparent)) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        if (block->child) {
            /* if we have a new parent, relink the whole chain */
            if (memparent) {
                TrackMem_BlockListLinkSequence(block->child, memparent);
            /* harmless optimization: if we're to orphan them then we'll just
             * unlink them from their parent, though we could equally well
             * dissociate them from each other... */
            } else {
                block->child->prev = NULL;
            }
            block->child = NULL;
        }
    }
    TrackMem_DebugValidate(mem);
    TrackMem_DebugValidate(memparent);
}

/* ---------------------------------------------------------------------------
 * TrackMemUnlink()
 *
 * Removes a block of memory from a tracking tree.  Differs from
 * TrackMemReassign(NULL, mem) in that the target's storage pointer
 * is returned and the linking structure itself is unlinked and freed.
 *
 * The returned pointer may no longer be used by any TrackMem primitives
 * other than TrackMemLink.  Any children of this block are automatically
 * orphaned (as if TrackMemReassignChildren(NULL, mem) was called).
 *
 * Parameters:  mem       - pointer to memory block to be unlinked
 *
 * Returns a pointer to a new block of memory which can be freed using
 * the default process defined by memFree().
 */
void *TrackMemUnlink(TMEM void *mem)
{
    TrackMem_Block *block;
    if (! mem) return NULL;
    block = TrackMem_BlockUnwrap(mem);

   /* orphan any children */
    if (block->child) {
        block->child->prev = NULL;
        block->child = NULL;
    }

    /* unlink from parent */
    TrackMem_BlockListUnlink(block);
    /* if link, simply return linked object */
    if (TrackMem_BlockIsLink(block)) {
        TrackMem_TagLink *tag = (TrackMem_TagLink *)TrackMem_TagFirst(block);
        ASSERT(tag->tag.type == TRACKMEM_TAG_LINK);
        tag->tag.flags = (tag->tag.flags & TRACKMEM_FLAG_MASK_RESERVED) |
            TRACKMEM_FLAG_LINK_DNONE;
        mem = tag->target;
        TrackMem_BlockFree(block);
        return mem;
    } else {
        /* otherwise move data to head of structure, and use memRealloc()
         * to shrink size */
        size_t size = TrackMem_BlockSize(block)
            - ((char *)mem - (char *)block->tags);
        memmove(block->tags, mem, size);
        return memRealloc(mem, size);
    }
}

/* ---------------------------------------------------------------------------
 * TrackMemGetParent()
 *
 * Locate the parent of this block.
 *
 * Parameters:  mem       - pointer to memory block to examine
 *
 * Returns the address of the memory block functioning as the parent
 * of this block.
 */
TMEM void *TrackMemGetParent(const TMEM void *mem)
{
    /* silently ignore NULL pointers */
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        block = TrackMem_BlockGetParent(block);
        TrackMem_DebugValidateBlock(block);
        return TrackMem_BlockWrap(block);
    } else return NULL;
}

/* ---------------------------------------------------------------------------
 * TrackMemGetData()
 *
 * Safely get a pointer to the data this block represents.  Except in the case
 * of a LINK, the returned value should always equal the source input.  Only
 * use this if you're hanging onto a link from its reference container or
 * aren't sure.
 *
 * Parameters:  mem       - pointer to memory block to examine
 *
 * Returns a pointer to the contained data of this block.
 */
void *TrackMemGetData(const TMEM void *mem)
{
    /* silently ignore NULL pointers */
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        return TrackMem_BlockData(block);
    } else return NULL;
}

/* ---------------------------------------------------------------------------
 * TrackMemGetSize()
 *
 * Get size of a tracked allocation.  In the case of LINKs, this is the size
 * that was specified by the user upon creation; otherwise, the actual size
 * returned may differ (as in be larger) than the actual requested size.
 *
 * Parameters:  mem       - pointer to memory block to examine
 *
 * Returns a size_t representing the size of the allocation this block refers
 * to.
 */
size_t TrackMemGetSize(const TMEM void *mem)
{
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        if (TrackMem_BlockIsLink(block)) {
            /* if this is a link, then fetch the size from the tag */
            TrackMem_TagLink *tag = (TrackMem_TagLink *)TrackMem_TagFirst(block);
            ASSERT(tag->tag.type == TRACKMEM_TAG_LINK);
            return tag->size;
        } else {
            return TrackMem_BlockSize(block)
                - ((char *)mem - (char *)block->tags);
        }
    }
    return 0;
}

/* ---------------------------------------------------------------------------
 * TrackMemGetTagSpaceSize()
 *
 * Gets the amount of tag space that is currently associated with a given
 * memory block.  This is only useful during reallocation of tag space and
 * does not guarantee that a particular number or size of tags may actually
 * be stored here.
 *
 * Parameters:  mem       - pointer to memory block to examine
 *
 * Returns a size_t representing the effective size of currently allocated
 * tag space. A size of 0 indicates that no tag space was created.
 */
size_t TrackMemGetTagSpaceSize(const TMEM void *mem)
{
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        return TrackMem_TagSpaceSize(block);
    }
    return 0;
}

/* ---------------------------------------------------------------------------
 * TrackMemTagSet()
 *
 * Sets a tag for an object using pre-allocated storage.
 * Note that tag space location is volatile and becomes invalid if the
 * storage space is reallocated.  Therefore you should not hang onto refs
 * to tag storage after a resize.
 *
 * If a tag exists with the same type, TrackMemTagSet will remove the
 * old tag before adding the new one.
 *
 * If the operation fails due to insufficient tag space, it is guaranteed
 * that the old tag of the specified type (if any) has been removed
 * notwithstanding the partial success of other internal operations.
 * ie. Any old tags will ALWAYS be removed if found!!
 *
 * Note: To remove a tag, use TrackMemTagUnset() with the same tag type.
 *
 * Parameters:  mem       - pointer to memory block to be tagged
 *              tagType   - type of tag to add
 *              tagSize   - size of new tag
 *
 * Returns a pointer to the new tag space that has been created.
 */
void *TrackMemTagSet(TMEM void *mem, TrackMem_TagType tagType, size_t tagSize)
{
    void *memtag = NULL;

    /* setup */
    ASSERT(tagSize <= TRACKMEM_TAG_MAXTOTALSIZE);
    tagSize = TrackMem_AlignLocal(tagSize);
    if (mem) {
        TrackMem_Tag *tag;
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        if (! TrackMem_BlockIsTagged(block)) return NULL;
        /* search for existing tag */
        tag = TrackMem_TagSeek(block, tagType);
        if (tag) {
            if (tag->length == tagSize) {
                tag->flags = 0;
                TrackMem_DebugValidateBlock(block);
                return (void *)((char *)tag + sizeof(TrackMem_Tag));
            }
            TrackMem_TagRemove(tag);
        }
        /* search for some free space */
        tag = TrackMem_TagFirst(block);
        while (tag->type != TRACKMEM_TAG_END) {
            TrackMem_Tag *nexttag;
            if (tag->type == TRACKMEM_TAG_FREE) {
                /* if it's a FREE chunk, join it to the next FREE block... */
                /* Note: Here we don't actually move any of the blocks around
                 *       because we have guaranteed to the user that none of
                 *       the tags will move unless he explicitely does
                 *       something that can move/resize the buffer... */
                for (;;) {
                    nexttag = TrackMem_TagSkip(tag);
                    if (nexttag->type == TRACKMEM_TAG_FREE) {
                        tag->length += nexttag->length + sizeof(TrackMem_Tag);
                    } else break;
                    tag = nexttag;
                }
                /* if the FREE chunk large enough? */
                if (tag->length >= tagSize) {
                    size_t length = (size_t)tag->length - tagSize;
                    tag->type = tagType;
                    tag->flags = 0;
                    tag->length = tagSize;
                    memtag = (void *)((char *)tag + sizeof(TrackMem_Tag));
                    if (length >= sizeof(TrackMem_Tag)) {
                        tag = TrackMem_TagSkip(tag);
                        tag->type = TRACKMEM_TAG_FREE;
                        tag->flags = 0;
                        tag->length = length - sizeof(TrackMem_Tag);
                    }
                    break;
                }
                tag = nexttag;
            } else tag = TrackMem_TagSkip(tag);
        }
    }
    TrackMem_DebugValidate(mem);
    return memtag;
}

/* ---------------------------------------------------------------------------
 * TrackMemTagFind()
 *
 * Locate a tag's storage.
 *
 * Parameters:  mem       - pointer to memory block to be examined for a tag
 *              tagType   - type of tag to search for
 *              tagSize   - optional pointer to a size_t which will receive
 *                          the size of the tag's storage (ignored if NULL)
 *
 * Returns a pointer to the tag storage space.  Don't hang onto this for
 * long.
 */
void *TrackMemTagFind(const TMEM void *mem, TrackMem_TagType tagType, size_t *tagSize)
{
    void *memtag = NULL;
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        TrackMem_Tag *tag = TrackMem_TagSeek(block, tagType);
        if (tag) {
            if (tagSize) *tagSize = tag->length;
            memtag = (void *)((char *)tag + sizeof(TrackMem_Tag));
        }
    }
    TrackMem_DebugValidate(mem);
    return memtag;
}

/* ---------------------------------------------------------------------------
 * TrackMemTagSetLinkDestructor()
 *
 * Sets the destructor for a LINK object.  When the object is released, that
 * destructor will be called on the link.
 *
 * Parameters:  mem       - pointer to controlling memory reference for a
 *                          LINK
 *              dType     - public destructor type
 *              dFunc     - function to use, should be NULL for DEFAULT or
 *                          NONE types and must be correctly typed for
 *                          other types.
 *
 * Returns TRUE on success, else FALSE.
 *
 * Note: To change destructors, call this function again with a different
 *       type.
 */
int TrackMemTagSetLinkDestructor(TMEM void *mem, TrackMem_DestructorType dType,
    void *dFunc)
{
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        if (TrackMem_BlockIsLink(block)) {
            TrackMem_TagLink *tag = (TrackMem_TagLink *)TrackMem_TagFirst(block);
            switch (dType) {
                case TRACKMEM_DESTRUCTOR_DEFAULT:
                case TRACKMEM_DESTRUCTOR_NONE:
                    break;
                case TRACKMEM_DESTRUCTOR_SIMPLE:
                    ASSERT(dFunc != NULL);
                    tag->destructor.funcSimple =
                        (TrackMem_LinkDestructorFunc)dFunc;
                    break;
                case TRACKMEM_DESTRUCTOR_SPECIAL:
                    ASSERT(dFunc != NULL);
                    tag->destructor.funcSpecial =
                        (TrackMem_LinkSpecialDestructorFunc)dFunc;
                    break;
                default:
                    ASSERT( /* give user a useful assertion message */
                        (dType == TRACKMEM_DESTRUCTOR_NONE) ||
                        (dType == TRACKMEM_DESTRUCTOR_DEFAULT) ||
                        (dType == TRACKMEM_DESTRUCTOR_SIMPLE) ||
                        (dType == TRACKMEM_DESTRUCTOR_SPECIAL));
                    /* if we chose to ignore... */
                    dType = TRACKMEM_DESTRUCTOR_NONE;
                    break;
            }
            tag->tag.flags = (tag->tag.flags & TRACKMEM_FLAG_MASK_RESERVED) |
                dType; /* we can do this now as the public and private
                        * values are equal */
            TrackMem_DebugValidateBlock(block);
            return TRUE;
        }
#ifdef APS_ENVIRONMENT
        else ERROR("called on object which is not a link, %p", mem);
#endif
    }
    return FALSE;
}

/* ---------------------------------------------------------------------------
 * TrackMemTagUnset()
 *
 * Remove a tag that is no longer needed.  It is safe to call this function
 * to remove a tag which does not exist (however that may reflect bad
 * design on the user's part).
 *
 * Parameters:  mem       - pointer to memory block to be manipulated
 *              tagType   - type of tag to remove
 *
 * Note: This operation will not attempt to coalesce adjacent free chunks as
 *       that is adequately performed by the TrackMemTagSet() operation.
 */
void TrackMemTagUnset(TMEM void *mem, TrackMem_TagType tagType)
{
    if (mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(mem);
        TrackMem_Tag *tag = TrackMem_TagSeek(block, tagType);
        /* if found, free the tag */
        if (tag) {
            tag->type = TRACKMEM_TAG_FREE;
        }
        TrackMem_DebugValidateBlock(block);
    }
}

/* ---------------------------------------------------------------------------
 * TrackMemTagAdd()
 *
 * Add a tag to an object even if there is currently insufficient tag space.
 *
 * If necessary, this function will reallocate the existing tag space
 * using TrackMemReallocTagSpace().  As this behaviour is unpredictable,
 * all notes from TrackMemReallocTagSpace() regarding the invalidation of
 * Tag and User pointers are equally valid here.  BE CAREFUL!
 *
 * Do not assume that if the user space pointer does not change that
 * pointers to tags have not become invalid.  Assume this function call
 * always invalidates all pointers and everything'll be just dandy.
 *
 * To remind you of the danger (and to support the operation) the first
 * argument has an additional level of indirection.  As all arguments are
 * untyped, make sure you pass the right pointer.
 *
 * Also, see TrackMemTagSet() for more information about the setting of new
 * tags.
 *
 * Parameters:  mem       - pointer to memory block to be tagged
 *              tagType   - type of tag to add
 *              tagSize   - size of new tag
 *
 * Returns a pointer to the new tag space that has been created.
 *
 * Note : It's a Good Idea to preallocate tag space and use TrackMemTagSet()
 *        whenever possible to save yourself an unnecessary reallocation.
 */
void *TrackMemTagAdd(TMEM void **mem, TrackMem_TagType tagType, size_t tagSize)
{
    void *memtag = NULL;

    /* setup */
    ASSERT(tagSize <= TRACKMEM_TAG_MAXTOTALSIZE);
    tagSize = TrackMem_AlignLocal(tagSize);
    ASSERT(mem);
    if (*mem) {
        TrackMem_Block *block = TrackMem_BlockUnwrap(*mem);
        /* try to allocate space without jostling things about */
        memtag = TrackMemTagSet(*mem, tagType, tagSize);
        if (! memtag) {
            /* note: old tag has already been removed by TrackMemTagSet()
             * try to reallocate tags */

            /* compute the total free space remaining by compacting
             * the tags and get the total tags space size */
            size_t totSpace = TrackMemGetTagSpaceSize(*mem);
            size_t totFree = TrackMem_TagSpaceCompact(block);
            size_t totReq = tagSize + sizeof(TrackMem_Tag);

            /* if there appears to be enough room, try again in
             * compacted space */
            if (totFree >= totReq) {
                memtag = TrackMemTagSet(*mem, tagType, tagSize);
                if (memtag) goto done;
                /* hunh???? -- that really should have worked! */
                /* add more space in case it's some weird alignment issue */
                totReq += MEMALIGN_LOCAL;
            }
            *mem = TrackMemReallocTagSpace(*mem, totSpace + totReq - totFree);
            if (*mem) {
                /* try again! */
                memtag = TrackMemTagSet(*mem, tagType, tagSize);
            }
        }
    }
done:
    TrackMem_DebugValidate(*mem);
    return memtag;
}
