/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsfilter.C
 *
 * Description: Defines the ApsFilterObject class members.
 *
 */                         

#include "apscpp.h"
 
 
/*---------------------------------------------------------------------------
 * ApsFilterObject()
 *
 * This constructor takes Aps_FilterHandle type as argument.
 *
 * Parameters:
 *     filterHandle : filter handle.
 *
 *     Return: ApsResultObject.
 *
 */                      
ApsFilterObject::ApsFilterObject(Aps_FilterHandle filterHandle)
                         : ApsObject((Aps_Handle)filterHandle)
{
}

/*---------------------------------------------------------------------------
 * ApsFilterObject()
 *
 * Copy constructor.
 *
 * Parameters:
 *     apsFilter : Reference to ApsFilterObject object.
 *
 *     Return: ApsResultObject.
 *
 */                      
ApsFilterObject::ApsFilterObject(const ApsFilterObject &apsFilter)
{
    Aps_Handle apsHandle;
 
    apsHandle = apsFilter.GetHandle();
    SetHandle(apsHandle);
}      

/*----------------------------------------------------------------------------
 * JobByOwnerName ()
 *
 * Composes the filter rule for Job that compares job by owner name field.
 *
 * Parameters: filterMode   - one of Aps_FilterMode modes
 *             jobOwnerName - The name of the job owner.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByOwnerName(Aps_FilterMode filterMode, 
                                                     char *jobOwnerName)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle();
    return (ApsResultObject(Aps_FilterJobByOwnerName(filterHandle, filterMode, 
                                                     jobOwnerName)));
}

/*----------------------------------------------------------------------------
 * JobByHost ()
 *
 * Composes the filter rule for Job that compares job by host field.
 *
 * Parameters: filterMode   - one of Aps_FilterMode modes
 *             host         - The host name.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByHost(Aps_FilterMode filterMode, 
                                           char *host)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByHost(filterHandle, filterMode, host)));
}

/*----------------------------------------------------------------------------
 * JobByName ()
 *
 * Composes the filter rule for Job that compares Job by name.
 *
 * Parameters: filterMode   - one of Aps_FilterMode modes
 *             name         - The Job name.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByName(Aps_FilterMode filterMode, 
                                           char *name)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByName(filterHandle, filterMode, name)));
}

/*----------------------------------------------------------------------------
 * JobByFilename ()
 *
 * Composes the filter rule for Job that compares job by file name.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             fileName    - The file name.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByFilename(Aps_FilterMode filterMode, 
                                           char *fileName)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByFilename(filterHandle, filterMode, 
                                                                  fileName)));
}

/*----------------------------------------------------------------------------
 * JobByID ()
 *
 * Composes the filter rule for Job that compares Job by ID.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             id          - The Job ID.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByID(Aps_FilterMode filterMode, 
                                           int id)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByID(filterHandle, filterMode, id)));
}

/*----------------------------------------------------------------------------
 * JobByCreationTime ()
 *
 * Composes the filter rule for Job that compares job by creation time.
 *
 * Parameters: filterMode     - one of Aps_FilterMode modes
 *             creationTime   - The creation time.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByCreationTime(Aps_FilterMode filterMode, 
                                           time_t creationTime)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByCreationTime(filterHandle, 
                                                  filterMode, creationTime)));
}

/*----------------------------------------------------------------------------
 * JobByFormat ()
 *
 * Composes the filter rule for Job that compares job by format.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             format      - The job format.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByFormat(Aps_FilterMode filterMode, 
                                           char *format)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByFormat(filterHandle, 
                                                       filterMode, format)));
}

/*----------------------------------------------------------------------------
 * JobByOrder ()
 *
 * Composes the filter rule for Job that compares job by job order.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             order       - The job order.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByOrder(Aps_FilterMode filterMode, 
                                           int order)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByOrder(filterHandle, 
                                                    filterMode, order)));
}

/*----------------------------------------------------------------------------
 * JobByPriority ()
 *
 * Composes the filter rule for Job that compares job by job priority.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             priority    - The job priority.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByPriority(Aps_FilterMode filterMode, 
                                           int priority)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByPriority(filterHandle, 
                                                    filterMode, priority)));
}

/*----------------------------------------------------------------------------
 * JobByStatus ()
 *
 * Composes the filter rule for Job that compares job by status.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             jobStatus   - The job status.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByStatus(
                                           Aps_FilterMode filterMode,
                                           Aps_JobStatus jobStatus)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByStatus(filterHandle, filterMode, 
                                                  jobStatus)));
}

/*----------------------------------------------------------------------------
 * JobByPrinterName ()
 *
 * Composes the filter rule for Job that compares job by printer name.
 *
 * Parameters: filterMode       - one of Aps_FilterMode modes
 *             jobPrinterName   - The printer name of the job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByPrinterName(
                                           Aps_FilterMode filterMode, 
                                           char *jobPrinterName)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByPrinterName(filterHandle, filterMode, 
                                                       jobPrinterName)));
}

/*----------------------------------------------------------------------------
 * JobByOwnerID ()
 *
 * Composes the filter rule for Job that compares job by OwnerID.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             ownerID     - The ownerID of the job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByOwnerID(Aps_FilterMode filterMode, 
                                           int ownerID)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByOwnerID(filterHandle, 
                                                    filterMode, ownerID)));
}

/*----------------------------------------------------------------------------
 * JobByLocalHost ()
 *
 * Composes the filter rule for Job that compares job by local host.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             localHost   - The local host of the job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByLocalHost(Aps_FilterMode filterMode, 
                                           char *localHost)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByLocalHost(filterHandle, 
                                                    filterMode, localHost)));
}

/*----------------------------------------------------------------------------
 * JobByLocalFile ()
 *
 * Composes the filter rule for Job that compares job by local file.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             localFile   - The local file for the job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobByLocalFile(Aps_FilterMode filterMode, 
                                           char *localFile)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobByLocalFile(filterHandle, 
                                                    filterMode, localFile)));
}

/*----------------------------------------------------------------------------
 * JobBySpoolHost ()
 *
 * Composes the filter rule for Job that compares job by spooling host.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes
 *             spoolHost   - The spooling host for this job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobBySpoolHost(Aps_FilterMode filterMode, 
                                           char *spoolHost)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobBySpoolHost(filterHandle, 
                                                    filterMode, spoolHost)));
}

/*----------------------------------------------------------------------------
 * JobBySpoolFile ()
 *
 * Composes the filter rule for Job that compares job by spooling file.
 *
 * Parameters: filterMode  - one of Aps_FilterMode modes.
 *             spoolFile   - The spool file for this job.
 *
 *     Result: ApsResultObject.
 */
ApsResultObject ApsFilterObject::JobBySpoolFile(Aps_FilterMode filterMode, 
                                           char *spoolFile)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterJobBySpoolFile(filterHandle, 
                                                    filterMode, spoolFile)));
}
/*---------------------------------------------------------------------------
 * WithFunction()
 *
 * Composes a filter rule from the provided user callback function.
 *
 * Parameters:
 *     filterMode : Any one of APS_FILTER_XXX.
 *     matchFunction : Pointer to a function used as a step in the filter
 *                     comparison.
 *     userArg : An argument that will be passed to the function during 
 *               comparison.
 *
 *     Return: ApsResultObject.
 *
 */                      
ApsResultObject ApsFilterObject::WithFunction(
                                         Aps_FilterMode filterMode,
                                         Aps_FilterFunc matchFunction,
                                         void *userArg)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterWithFunction(&filterHandle, filterMode, 
                                             matchFunction, userArg)));
}

/*---------------------------------------------------------------------------
 * FilterAll()
 *
 * Composes a filter rule that always matches.
 *
 * Parameters:
 *     filterMode : Any one of APS_FILTER_XXX.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::FilterAll(Aps_FilterMode filterMode)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle();
    return (ApsResultObject(Aps_FilterAll(&filterHandle, filterMode)));
}

/*---------------------------------------------------------------------------
 * Merge()
 *
 * Compose a new step formed by merging in a second filter as an 
 * independent clause. The second filter will be evaluated as a single 
 * step, as always.
 *
 * Parameters:
 *     filterMode : Any one of APS_FILTER_XXX.
 *     srcHandle  : Handle of the filter to be merged in.  
 *                  May be NULL, in which case this operation is ignored.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::Merge(Aps_FilterMode filterMode,
                                             Aps_FilterHandle srcHandle)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle();
    return (ApsResultObject(Aps_FilterMerge(&filterHandle, 
                                                   filterMode, srcHandle)));
}

/*---------------------------------------------------------------------------
 * Clear()
 *
 * Clears a filter, but does not dispose of the handle.
 *
 * Parameters:   None.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::Clear()
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterClear(&filterHandle)));
}

/*---------------------------------------------------------------------------
 * Attach()
 *
 * Attach a Filter to an object. By default, many actions of the target object 
 * will be affected by this filter.
 *
 * Parameters:
 *    ObjHandle     : Handle of the object to attach the new filter to.
 *    filterOptions : Set of options (or'd together) used when attached.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::Attach(Aps_Handle objHandle, 
                                             Aps_FilterOptions filterOptions)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_AttachFilter(objHandle, filterHandle, 
                                                        filterOptions)));
}

/*---------------------------------------------------------------------------
 * Detach()
 *
 * Detach a Filter from an object. Future operations on this object will no 
 * longer be affected by the filter.
 *
 * Parameters:
 *    ObjHandle    : Handle of the object to attach the new filter to.
 *    filterOptions : Set of options (or'd together) used when attached.
 *
 *    Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::Detach(Aps_Handle objHandle, 
                                       Aps_FilterOptions *oldFilterOptions)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_DetachFilter(objHandle, &filterHandle, 
                                                       oldFilterOptions)));
}
/*---------------------------------------------------------------------------
 * Check()
 *
 * Checks to see if a specific object matches a filter.
 *
 * Parameters:
 *    filterOptions : Set of options (or'd together) as with 
 *                    Aps_AttachFilter().  APS_FILTER_FALLBACK is ignored here.
 *    objHandle     : Handle of the object to verify.
 *    matchesFilter : Set to TRUE if this object matches the filter, 
 *                    FALSE if not.
 *
 *    Return: ApsResultObject.
 *
 */
ApsResultObject ApsFilterObject::Check(Aps_FilterOptions filterOptions,
                                       Aps_Handle objHandle,
                                       int *matchesFilter)
{
    Aps_FilterHandle filterHandle = (Aps_FilterHandle)GetHandle(); 
    return (ApsResultObject(Aps_FilterCheck(filterHandle, filterOptions, 
                                                     objHandle, matchesFilter)));
}
