/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: debug.c
 *
 * Description: Debugging tools (see internal documentation)
 *
 */
#include "apsinternal.h"
#if (APSCFG_DEBUG_TRACE != 0)

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>

/*** Protos ***/
static int DebugParseChannels(const char *channels);
static void DebugSetFlags(int channelid, int classid, int flag);
static void DebugInitialize(void);

/*** Globals ***/
/* List of class/channel enable/disable bits */
unsigned char DebugChannelFlags[DEBUG_CHANNEL_INDEX_END];

/* TRUE if we must reinitialize */
int DebugChannelsInvalid = TRUE;

/* List of all debug channel names as def'd in debug.h */
static const char DebugChannelNames[DEBUG_CHANNEL_NAME_TOTSIZE] =
    DEBUG_CHANNEL_NAMES;

/* List of all debug class names as def'd in debug.h */
static const char DebugClassNames[DEBUG_CLASS_INDEX_END * 8 + 1] =
    DEBUG_CLASS_NAMES;

/* List of all result codes as defined in aps.h (IN THE SAME ORDER!) */
const char *DebugResultCodeNames[APS_NUMRESULTS - APS_SUCCESS] = {
    "APS_SUCCESS",
    "APS_NOT_IMPLEMENTED",
    "APS_NOT_SUPPORTED",
    "APS_INVALID_PARAM",
    "APS_MORE_DATA",
    "APS_OUT_OF_MEMORY",
    "APS_NOT_FOUND",
    "APS_ACCESS_DENIED",
    "APS_INVALID_HANDLE",
    "APS_GENERIC_FAILURE",
    "APS_DISK_FULL",
    "APS_INVALID_PWD",
    "APS_OUT_OF_SEQUENCE",
    "APS_VIOLATES_CONSTRAINTS",
    "APS_INVALID_PPD",
    "APS_WRONG_TYPE",
    "APS_ALREADY_EXISTS",
    "APS_OPERATION_AVAILABLE",
    "APS_NO_CHANGE",
    "APS_IGNORED",
    "APS_PARTIAL_SUCCESS",
    "APS_OPERATION_TIMEOUT",
    "APS_HAS_EXTENDED_SUPPORT",
    "APS_IO_ERROR",
    "APS_SUBPROGRAM_FAILED",
    "APS_FILTER_NOT_SUPPORTED",
    "APS_NO_MORE_DATA",
    "APS_MODEL_UNKNOWN"
};

/* ---------------------------------------------------------------------------
 * DebugPrintHeader()
 *
 * Output a header for a particular debug message to the user.  Each message
 * is identified by its class, channel, and location.
 *
 * Parameters:  channelIndex - index # of channel
 *              classIndex   - index # of class
 *              locFunction  - function name
 *              locFile      - file name
 *              locLine      - line #
 * Result:      true if message should really be displayed
 */
int DebugPrintHeader(int channelIndex, int classIndex,
    const char *locFunction, const char *locFile, int locLine)
{
    char locFileLine[3 + APSCFG_DEBUG_WIDTH_FILENAME +
        APSCFG_MAX_INT_STR_SIZE + 1];

    /* Check if we were properly initialized */
    if (DebugChannelsInvalid) {
        DebugInitialize();
    }
    /* Second-guessing... */
    if (DebugChannelFlags[channelIndex] & (1<<classIndex)) {
        int lenFuncMax,lenRFence;
        /* Limit filename and line number pair to 30 characters */
        sprintf(locFileLine, "(%.*s:%d)", APSCFG_DEBUG_WIDTH_FILENAME,
            locFile, locLine);
        /* Limit entire string to APSCFG_DEBUG_WIDTH_TOTAL chars */
        /* - 6 chars for fence
         * - 1 char  for space
         * - 8 chars for channel
         * - 1 char  for space
         * - 5 chars for class
         * - 3 chars for " : "
         * - x chars for locFunction
         * - 1 char  for space
         * - x chars for locFileLine
         * - 1 char  for space
         * - 2 chars for fence (minimum)
         */
        /* compute space for function and for right fence */
#if ((APSCFG_DEBUG_WIDTH_TOTAL - 28 - 3 - APSCFG_DEBUG_WIDTH_FILENAME - APSCFG_MAX_INT_STR_SIZE - 1) < 0)
#warning "APSCFG_DEBUG_WIDTH_TOTAL too small!"
#endif
        lenFuncMax = APSCFG_DEBUG_WIDTH_TOTAL - 28 - strlen(locFileLine);
        lenRFence = lenFuncMax - strlen(locFunction);
        if (lenRFence < 0) lenRFence = 0;

        fprintf(stderr, "====== %8.8s %5.5s = %.*s %s %.*s==\n",
            & DebugChannelNames[channelIndex << 3],
            & DebugClassNames[classIndex << 3],
            lenFuncMax, locFunction, locFileLine, lenRFence,
"============================================================================="
"============================================================================="
            );
        return TRUE;
    }
    return FALSE;
}

/* ---------------------------------------------------------------------------
 * DebugPrintMessage()
 *
 * Output a message associated with a particular debug header.  The contents
 * will be formatted and reflowed according to the following rules:
 *   - all text indented
 *   - long lines are reflowed after 79'th char
 *     (80'th left blank for compatibility with broken terminals which
 *      wrap lines prematurely)
 *   - tabs to spaces
 *   - control chars stripped
 *   - indents and tabs are 4 spaces
 *
 * Parameters:  printf format string and arguments
 * Result:      nothing
 */
void DebugPrintMessage(const char *fmt, ...)
{
    va_list va;
    va_start(va, fmt);

    /* Only if a message was supplied */
    if (fmt) {
        char line_in[APSCFG_DEBUG_MAX_MSG_SIZE];
        char line_out[APSCFG_DEBUG_MAX_MSG_SIZE];
        /* Format message */
        vsnprintf(line_in, sizeof(line_in), fmt, va);
        /* Reformat message */
        {
            char *char_in = line_in;
            char *char_out = line_out;
            int column = 0;
            char c;
            while ( (c = *(char_in++)) ) {
                /* newlines */
                if (c == '\n') {
                    if (column != 0) {
                        *(char_out++) = '\n';
                        column = 0;
                    }
                } else {
                    /* wrap if necessary */
                    if (column >= (APSCFG_DEBUG_WIDTH_TOTAL -
                        APSCFG_DEBUG_WIDTH_RINDENT)) {
                        *(char_out++) = '\n';
                        column = 0;
                    }
                    /* indent if necessary */
                    while (column++ < APSCFG_DEBUG_WIDTH_LINDENT)
                        *(char_out++) = ' ';
                    /* tabs */
                    if (c == '\t') {
                        int count = (column - 1 - APSCFG_DEBUG_WIDTH_LINDENT) %
                            APSCFG_DEBUG_WIDTH_SOFTTAB;
                        column += 3 - count;
                        if (column < (APSCFG_DEBUG_WIDTH_TOTAL -
                            APSCFG_DEBUG_WIDTH_RINDENT)) {
                            while (count++ < APSCFG_DEBUG_WIDTH_SOFTTAB)
                                *(char_out++) = ' ';
                        }
                    /* unprintable but consumes space? */
                    } else if (c < 32) {
                        *(char_out++) = ' ';
                    /* printable */
                    } else *(char_out++) = c;
                }
            }
            if (column != 0) *(char_out++) = '\n';
            *char_out = '\0';
        }
        fputs(line_out, stderr);
    }
    va_end(va);
}

/* ---------------------------------------------------------------------------
 * DebugParseChannels()
 *
 * Parse a list of channels and set the flags appropriately.  If the list is
 * NULL, simply clears the old settings and returns.
 *
 * Parameters:  channels  - list of channels
 * Result:      true on success, false if misformatted
 *
 * Note: This routine is sub-optimal.  The name search code could be folded
 *       although the number of parameters to be passed to the appropriate
 *       function might be large.
 */
static int DebugParseChannels(const char *channels)
{
    memset(DebugChannelFlags, 0, sizeof(DebugChannelFlags));
    if (channels) {
        int    channelid = -1;
        char   channelname[DEBUG_CHANNEL_NAME_SIZE];
        const char *search;
        size_t len;
        int    i;
        while (*channels) {
            /* locate next + or - */
            len = strcspn(channels, "+-");
            if ((len == 0)||(len > DEBUG_CHANNEL_NAME_SIZE))
                return FALSE;
            /* fetch channel name */
            for (i = 0; i < len; ++i) {
                char c = *(channels++);
                if (! isalpha(c)) return FALSE;  /* must be alphabetical! */
                channelname[i] = c;
            }
            while (i < DEBUG_CHANNEL_NAME_SIZE) channelname[i++] = ' ';
            /* search for name in list */
            search = DebugChannelNames;
            for (channelid = 0; channelid < DEBUG_CHANNEL_INDEX_END;
                ++channelid) {
                if (strnicmp(search, channelname, DEBUG_CHANNEL_NAME_SIZE)
                    == 0) break;
                search += DEBUG_CHANNEL_NAME_SIZE;
            }
            if (channelid == DEBUG_CHANNEL_INDEX_END) return FALSE;
            /* loop until we find a , */
            for (;;) {
                char classname[DEBUG_CLASS_NAME_SIZE];
                int  classid;
                char op = *(channels++);
                if (! op) return TRUE; /* end of string!! */
                len = strcspn(channels, ",+-");
                if (op == ',') break;
                if (len == 0) continue;
                /* fetch class name */
                for (i = 0; i < len; ++i) {
                    char c = *(channels++);
                    if (! isalpha(c)) return FALSE;  /* must be alphabetical! */
                    classname[i] = c;
                }
                while (i < DEBUG_CLASS_NAME_SIZE) classname[i++] = ' ';
                /* search for name in list */
                search = DebugClassNames;
                for (classid = 0; classid < DEBUG_CLASS_INDEX_END;
                    ++classid) {
                    if (strnicmp(search, classname, DEBUG_CLASS_NAME_SIZE)
                        == 0) break;
                    search += DEBUG_CLASS_NAME_SIZE;
                }
                if (classid == DEBUG_CLASS_INDEX_END) return FALSE;
                /* set flags  */
                DebugSetFlags(channelid, classid, (op == '+'));
            }
        }
    }
    return TRUE;
}

/* ---------------------------------------------------------------------------
 * DebugSetFlags()
 *
 * Sets up the flags for a given debug channel and class
 *
 * Parameters:  channelid - id of channel, may be all
 *              classid   - id of class, may be all
 *              flag      - true/false to enable/disable
 * Returns:     nothing
 */
static void DebugSetFlags(int channelid, int classid, int flag)
{
    int bits;
    int i;

    /* Handle all */
    if (classid == DEBUG_CLASS_INDEX_all) bits = 0xff;
    else bits = (1<<classid);

    for (i = 0; i < DEBUG_CHANNEL_INDEX_END; ++i) {
        if (channelid != DEBUG_CHANNEL_INDEX_all) i = channelid;
        if (flag) {
            DebugChannelFlags[i] |= bits;
        } else {
            DebugChannelFlags[i] &= ~bits;
        }
        if (channelid != DEBUG_CHANNEL_INDEX_all) break;
    }
}

/* ---------------------------------------------------------------------------
 * DebugInitialize()
 *
 * Initialize the internal debugging support.  Sets up:
 *   - flags for enables channels
 */
static void DebugInitialize(void) {
    const char *envvar = APSCFG_DEBUG_VAR_CHANNELS;
    const char *deftrace = APSCFG_DEBUG_DEF_CHANNELS;
    const char *apstrace;

    /* get debug options from environment */
    if ((envvar) && (*envvar)) {
        apstrace = getenv(APSCFG_DEBUG_VAR_CHANNELS);
    } else apstrace = NULL;
    /* else use defaults... */
    if (! apstrace) apstrace = deftrace;
    if (! DebugParseChannels(apstrace)) {
        if (apstrace == deftrace) {
            fputs("APS: Invalid default debug channels in configuration\n",
                stderr);
        } else {
            fprintf(stderr,
                "APS: Invalid debug channels provided in env var \"%s\"\n",
                APSCFG_DEBUG_VAR_CHANNELS);
        }
    }
    DebugChannelsInvalid = FALSE; /* don't try again */
}

#endif /* APSCFG_DEBUG_TRACE != 0 */
