/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: lprprefs.c
 *
 * Description: Preferences functions used by LprTransport.
 *
 * N.B. This is where C++ templates would REALLY come in handy!!! *sigh*
 */
#include "apsinternal.h"
#if (APSCFG_LPR_BUILD)

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include "aps.h"
#include "lprprefs.h"
#include "pwd.h"
#include "utils.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(lpr)

/* Text to appear at the beginning of each new configuration file */
static const char *PrefsHeaderComment =
 "# This configuration file provides settings for filter scripts installed\n"
 "# by default with APS.\n"
 "# This file was created using APS library functions developed by Corel\n"
 "# Corporation and distributed freely under GNU LGPL.\n\n";

/* Process:
 * ========
 *
 * To read a configuration file
 *   struct xxx  cfg;
 *   const char *dir;
 *   xxxcfgInit(& cfg);
 *   xxxcfgRead(dir, & cfg);
 *   <do something>
 *   xxxcfgRelease(& cfg);
 *
 * To write a new configuration file
 *   struct xxx  cfg;
 *   const char *dir;
 *   xxxcfgInit(& cfg);
 *   <make changes to defaults>
 *   xxxcfgWrite(dir, & cfg);
 *   xxxcfgRelease(& cfg);
 *
 * To write multiple changes in a fail-safe manner
 *   <setup cfg as usual>
 *   xxxcfgWriteDefer(dir, & cfg);
 *   if (success) {
 *      <do other things -- perhaps make more changes>
 *      if (success) xxxcfgWriteCommit(dir, & cfg);
 *      else xxxcfgWriteAbort(dir, & cfg);
 *   }
 *   xxxcfgRelease(& cfg);
 *
 * To modify an existing configuration in a fail-safe manner
 * ===NOT YET SUPPORTED===
 *   struct xxx  cfg;
 *   const char *dir;
 *   xxxcfgInit(& cfg);
 *   xxxcfgWriteDefer(dir, & cfg);
 *   if (success) {
 *      xxxcfgRead(dir, & cfg);
 *      <make changes>
 *      <do other things>
 *      if (success) xxxcfgWriteCommit(dir, & cfg);
 *      else xxxcfgWriteAbort(dir, & cfg);
 *   }
 *   xxxcfgRelease(& cfg);
 *
 * Expect changes to these functions in the near future to supported
 * extended uses.
 */


/*********************
 *** HELPER MACROS ***
 *********************/

/*** ALL ***/

/** Used to open a config file **/
#define CFG_OPENFILE( cfgDirectory, cfgFilename, cfgSuffix, cfgMode ) \
    cfg->private.fh = NULL; \
    cfg->private.fd = -1; \
    if ((strlen( (cfgDirectory) ) + \
         strlen( (cfgFilename) ) + \
         strlen( (cfgSuffix) ) + 2) < APSCFG_MAX_PATH_SIZE) { \
        char cfgPath[APSCFG_MAX_PATH_SIZE]; \
        sprintf(cfgPath, "%s/%s%s", \
            (cfgDirectory) , (cfgFilename), (cfgSuffix) ); \
        errno = 0; \
        cfg->private.fh = fopen(cfgPath, (cfgMode) ); \
        if (! cfg->private.fh) result = GetResultFromErrno(); \
    } else result = APS_NOT_FOUND;

/** Used to open a config file as FD **/
#define CFG_OPENFILEFD( cfgDirectory, cfgFilename, cfgSuffix, cfgMode ) \
    cfg->private.fh = NULL; \
    cfg->private.fd = -1; \
    if ((strlen( (cfgDirectory) ) + \
         strlen( (cfgFilename) ) + \
         strlen( (cfgSuffix) ) + 2) < APSCFG_MAX_PATH_SIZE) { \
        char cfgPath[APSCFG_MAX_PATH_SIZE]; \
        sprintf(cfgPath, "%s/%s%s", \
            (cfgDirectory) , (cfgFilename), (cfgSuffix) ); \
        errno = 0; \
        cfg->private.fd = open(cfgPath, (cfgMode) ); \
        if (cfg->private.fd == -1) result = GetResultFromErrno(); \
    } else result = APS_NOT_FOUND;

/*** READING ***/

/** Used to open a config file for reading **/
#define CFGREAD_OPEN( cfgDirectory, cfgFilename ) \
    CFG_OPENFILE( cfgDirectory, cfgFilename, "", "r" ); \
    if (cfg->private.fh) { \
        char cfgLine[APSCFG_MAX_CONFIG_LINE_SIZE]; \
        while (fgets(cfgLine, APSCFG_MAX_CONFIG_LINE_SIZE, \
            cfg->private.fh)) { \
            strtrim(cfgLine, (strncmp(cfgLine, "export ", 7) == 0) ? \
                cfgLine + 7 : cfgLine);

/** Used to close a config file for reading **/
#define CFGREAD_CLOSE() \
        } \
        fclose(cfg->private.fh); \
        cfg->private.fh = NULL; \
    }

/** Used to extract a string from a config file **/
#define CFGREAD_GETSTRING( cfgOption, cfgLength, cfgTarget ) \
            if (strncmp(cfgLine, (cfgOption) , (cfgLength) ) == 0) { \
                char *cfgValue = cfgLine + cfgLength; \
                strunquot(cfgValue, cfgValue, "''\"\""); \
                strupdate( (cfgTarget) , cfgValue); \
                continue; \
            }

/** Used to extract a boolean value from a config file **/
#define CFGREAD_GETBOOL( cfgOption, cfgLength, cfgTarget ) \
            if (strncmp(cfgLine, (cfgOption) , (cfgLength) ) == 0) { \
                char *cfgValue = cfgLine + cfgLength; \
                strunquot(cfgValue, cfgValue, "''\"\""); \
                *cfgTarget = ((stricmp(cfgValue, "yes") == 0) || \
                              (stricmp(cfgValue, "true") == 0) || \
                              (stricmp(cfgValue, "on") == 0) || \
                              (stricmp(cfgValue, "enable") == 0) || \
                              (stricmp(cfgValue, "enabled") == 0) || \
                              (atoi(cfgValue) == 1) ); \
                continue; \
            }

/** Used to extract an integer value from a config file **/
#define CFGREAD_GETINT( cfgOption, cfgLength, cfgTarget ) \
            if (strncmp(cfgLine, (cfgOption) , (cfgLength) ) == 0) { \
                char *cfgValue = cfgLine + cfgLength; \
                strunquot(cfgValue, cfgValue, "''\"\""); \
                *cfgTarget = atoi(cfgValue); \
                continue; \
            }

/** Used to extract a connected pair (tuple) of integers from a config file
 ** if not found, set to -1 **/
#define CFGREAD_GETPAIR( cfgOption, cfgLength, cfgPre, cfgMid, cfgPost, cfgTarget1, cfgTarget2 ) \
            if (strncmp(cfgLine, (cfgOption) , (cfgLength) ) == 0) { \
                char *cfgValue = cfgLine + cfgLength; \
                strunquot(cfgValue, cfgValue, "''\"\""); \
                sscanf(cfgValue, cfgPre ## "%d" ## cfgMid ## "%d" ## cfgPost, \
                    cfgTarget1, cfgTarget2); \
                continue; \
            }

/** Used to extract printer type from general.cfg, if not found sets -1 **/
#define CFGREAD_GETPRINTERTYPE( cfgOption, cfgLength, cfgTarget ) \
            if (strncmp(cfgLine, (cfgOption) , (cfgLength) ) == 0) { \
                char *cfgValue = cfgLine + cfgLength; \
                strunquot(cfgValue, cfgValue, "''\"\""); \
                if ((stricmp(cfgValue, "local") == 0) || (! *cfgValue)) \
                    *cfgTarget = APS_CONNECT_LOCAL; \
                else if ((stricmp(cfgValue, "unix") == 0) || \
                         (stricmp(cfgValue, "remote") == 0) || \
                         (stricmp(cfgValue, "lpd") == 0)) \
                    *cfgTarget = APS_CONNECT_NETWORK_LPD; \
                else if (stricmp(cfgValue, "smb") == 0) \
                    *cfgTarget = APS_CONNECT_NETWORK_SMB; \
                else *cfgTarget = -1; \
                continue; \
            }

/*** WRITING ***/
/** Used to open a config file for writing (deferred) **/
#define CFGWRITE_OPENDEFER( cfgDirectory, cfgFilename, cfgProtection ) \
    CFG_OPENFILEFD( cfgDirectory, cfgFilename, "", O_CREAT | O_RDWR ); \
    if (cfg->private.fd != -1) { \
        struct passwd *pw; \
        fchmod(cfg->private.fd, (cfgProtection) ); \
        pw = getpwnam("lp"); \
        fchown(cfg->private.fd, (pw) ? pw->pw_uid : 0, (pw) ? pw->pw_gid : 0); \
    }

/** Used to open a config file for writing **/
#define CFGWRITE_OPENCOMMIT( cfgDirectory, cfgFilename, cfgProtection ) \
    if (cfg->private.fd == -1) \
        CFGWRITE_OPENDEFER( cfgDirectory, cfgFilename, cfgProtection ); \
    if (cfg->private.fd != -1) { \
        ftruncate(cfg->private.fd, 0); \
        cfg->private.fh = fdopen(cfg->private.fd, "w"); \
        cfg->private.fd = -1;

/** Used to close a config file for writing **/
#define CFGWRITE_CLOSE() \
        if (cfg->private.fd != -1) { \
            close(cfg->private.fd); \
            cfg->private.fd = -1; \
        } \
        if (cfg->private.fh) { \
            fclose(cfg->private.fh); \
            cfg->private.fh = NULL; \
        } \
    }

/** Used to add plaintext to a config file **/
#define CFGWRITE_PUTTEXT( cfgText ) \
        fputs(cfgText, cfg->private.fh);

/** Used to put a string into a config file **/
#define CFGWRITE_PUTSTRING( cfgOption, cfgLength, cfgTarget ) \
        fprintf(cfg->private.fh, "%s\"%s\"\n", (cfgOption), (cfgTarget));

/** Used to put a boolean value into a config file **/
#define CFGWRITE_PUTBOOL( cfgOption, cfgLength, cfgTarget ) \
        fprintf(cfg->private.fh, "%s%s\n", (cfgOption), \
            ( (cfgTarget) )? "YES" : "NO");

/** Used to put an integer value into a config file **/
#define CFGWRITE_PUTINT( cfgOption, cfgLength, cfgTarget ) \
        fprintf(cfg->private.fh, "%s%d\n", (cfgOption), (cfgTarget));

/** Used to put a connected pair (tuple) of integers into a config file
 ** if not found, set to -1 **/
#define CFGWRITE_PUTPAIR( cfgOption, cfgLength, cfgPre, cfgMid, cfgPost, cfgTarget1, cfgTarget2 ) \
        fprintf(cfg->private.fh, "%s%s%d%s%d%s\n", (cfgOption), \
            (cfgPre), (cfgTarget1), (cfgMid), (cfgTarget2), (cfgPost));

/** Used to extract printer type from general.cfg, if not found sets -1 **/
#define CFGWRITE_PUTPRINTERTYPE( cfgOption, cfgLength, cfgTarget ) \
        { \
            const char *cfgTypename; \
            switch (cfgTarget) { \
                case APS_CONNECT_LOCAL: \
                    cfgTypename = "LOCAL"; break; \
                case APS_CONNECT_NETWORK_SMB: \
                    cfgTypename = "SMB"; break; \
                case APS_CONNECT_NETWORK_LPD: \
                    cfgTypename = "REMOTE"; break; \
                default: \
                    cfgTypename = ""; break; \
            } \
            fprintf(cfg->private.fh, "%s%s\n", (cfgOption), cfgTypename); \
        }

/************************
 *** HELPER FUNCTIONS ***
 ************************/

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxInit()
 *
 * Group of helper functions for initializing structure.
 * Call this *ONCE* to setup the structure after creation or Release.
 *
 * To RE-Initialize a structure, call Release then Init.
 *
 * Parameters: cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
Aps_Result LprFilterRH_txtcfgInit(LprFilterRH_TextOnlyCfg *cfg)
{
    cfg->private.fh = NULL;
    cfg->private.fd = -1;
    cfg->txtOptions = "";
    cfg->txtAddCR = FALSE;
    cfg->txtSendEOF = FALSE;
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_pscfgInit(LprFilterRH_PostscriptCfg *cfg)
{
    cfg->private.fh = NULL;
    cfg->private.fd = -1;
    cfg->psGSDevice = strdup("POSTSCRIPT");
    cfg->psGSOptions = "";
    cfg->psGSUniprintDrv = "";
    cfg->psPaperSize = strdup("letter");
    cfg->psXDPI = 0; cfg->psYDPI = 0;
    cfg->psReverseOrder = FALSE;
    cfg->psSendEOF = FALSE;
    cfg->psNUPCount = 1;
    cfg->psNUPHMargin = 0; cfg->psNUPVMargin = 0;
    return (cfg->psGSDevice && cfg->psPaperSize) ?
        APS_SUCCESS : APS_OUT_OF_MEMORY;
}
Aps_Result LprFilterRH_gencfgInit(LprFilterRH_GeneralCfg *cfg)
{
    cfg->private.fh = NULL;
    cfg->private.fd = -1;
    cfg->genDesired = strdup("printer");
    cfg->genPaperSize = strdup("letter");
    cfg->genPrinterType = APS_CONNECT_LOCAL;
    cfg->genASCIItoPS = FALSE;
    return (cfg->genDesired && cfg->genPaperSize) ?
        APS_SUCCESS : APS_OUT_OF_MEMORY;
}
Aps_Result LprFilterRH_smbcfgInit(LprFilterRH_SMBNetworkCfg *cfg)
{
    cfg->private.fh = NULL;
    cfg->private.fd = -1;
    cfg->smbHostIP = "";
    cfg->smbShare = "";
    cfg->smbUser = "";
    cfg->smbPassword = "";
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_lpdcfgInit(LprFilterRH_LPDNetworkCfg *cfg)
{
    cfg->private.fh = NULL;
    cfg->private.fd = -1;
    cfg->lpdServer = "";
    cfg->lpdQueue = "";
    cfg->lpdUser = "";
    cfg->lpdPassword = "";
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxRelease()
 *
 * Group of helper functions for releasing structure which were filled
 * in by xxxxRead() or created using strupdate().
 *
 * Parameters: cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
Aps_Result LprFilterRH_txtcfgRelease(LprFilterRH_TextOnlyCfg *cfg)
{
    strupdate(& cfg->txtOptions, NULL); /* always succeeds */
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_pscfgRelease(LprFilterRH_PostscriptCfg *cfg)
{
    strupdate(& cfg->psGSDevice, NULL); /* free string */
    strupdate(& cfg->psGSOptions, NULL); /* free string */
    strupdate(& cfg->psGSUniprintDrv, NULL); /* free string */
    strupdate(& cfg->psPaperSize, NULL); /* free string */
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_gencfgRelease(LprFilterRH_GeneralCfg *cfg)
{
    strupdate(& cfg->genDesired, NULL); /* free string */
    strupdate(& cfg->genPaperSize, NULL); /* free string */
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_smbcfgRelease(LprFilterRH_SMBNetworkCfg *cfg)
{
    strupdate(& cfg->smbHostIP, NULL); /* free string */
    strupdate(& cfg->smbShare, NULL); /* free string */
    strupdate(& cfg->smbUser, NULL); /* free string */
    strupdate(& cfg->smbPassword, NULL); /* free string */
    return APS_SUCCESS;
}
Aps_Result LprFilterRH_lpdcfgRelease(LprFilterRH_LPDNetworkCfg *cfg)
{
    strupdate(& cfg->lpdServer, NULL); /* free string */
    strupdate(& cfg->lpdQueue, NULL); /* free string */
    strupdate(& cfg->lpdUser, NULL); /* free string */
    strupdate(& cfg->lpdPassword, NULL); /* free string */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxRead()
 *
 * Group of helper functions for reading configuration files
 *
 * Parameters: spoolDir - Pointer to spool directory
 *             cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 *
 * N.B. Does not reinitialize... Strings alloc'd using strupdate().
 *      Call Release / Init first if the structure is dirty.
 */
Aps_Result LprFilterRH_txtcfgRead(const char *spoolDir,
    LprFilterRH_TextOnlyCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGREAD_OPEN(spoolDir, "textonly.cfg");
    CFGREAD_GETSTRING("TEXTONLYOPTIONS=", 16, & cfg->txtOptions);
    CFGREAD_GETBOOL("CRLFTRANS=", 10, & cfg->txtAddCR);
    CFGREAD_GETBOOL("TEXT_SEND_EOF=", 14, & cfg->txtSendEOF);
    CFGREAD_CLOSE();
    return result;
}
Aps_Result LprFilterRH_pscfgRead(const char *spoolDir,
    LprFilterRH_PostscriptCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGREAD_OPEN(spoolDir, "postscript.cfg");
    CFGREAD_GETSTRING("GSDEVICE=", 9, & cfg->psGSDevice);
    CFGREAD_GETSTRING("EXTRA_GS_OPTIONS=", 17, & cfg->psGSOptions);
    CFGREAD_GETSTRING("COLOR=", 6, & cfg->psGSUniprintDrv);
    CFGREAD_GETSTRING("PAPER_SIZE=", 11, & cfg->psPaperSize);
    CFGREAD_GETPAIR("RESOLUTION=", 11, "", "x", "", & cfg->psXDPI, & cfg->psYDPI);
    CFGREAD_GETBOOL("REVERSE_ORDER=", 14, & cfg->psReverseOrder);
    CFGREAD_GETBOOL("PS_SEND_EOF=", 12, & cfg->psSendEOF);
    CFGREAD_GETINT("NUP=", 4, & cfg->psNUPCount);
    CFGREAD_GETINT("RTLFTMAR=", 9, & cfg->psNUPHMargin);
    CFGREAD_GETINT("TOPBOTMAR=", 10, & cfg->psNUPVMargin);
    CFGREAD_CLOSE();
    return result;
}
Aps_Result LprFilterRH_gencfgRead(const char *spoolDir,
    LprFilterRH_GeneralCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGREAD_OPEN(spoolDir, "general.cfg");
    CFGREAD_GETPRINTERTYPE("PRINTER_TYPE=", 13, & cfg->genPrinterType);
    CFGREAD_GETSTRING("DESIRED_TO=", 11, & cfg->genDesired);
    CFGREAD_GETSTRING("PAPER_SIZE=", 11, & cfg->genPaperSize);
    CFGREAD_GETBOOL("ASCII_TO_PS=", 12, & cfg->genASCIItoPS);
    CFGREAD_CLOSE();
    return result;
}
Aps_Result LprFilterRH_smbcfgRead(const char *spoolDir,
    LprFilterRH_SMBNetworkCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGREAD_OPEN(spoolDir, ".config");
    CFGREAD_GETSTRING("hostip=", 7, & cfg->smbHostIP);
    CFGREAD_GETSTRING("share=", 6, & cfg->smbShare);
    CFGREAD_GETSTRING("user=", 5, & cfg->smbUser);
    CFGREAD_GETSTRING("password=", 9, & cfg->smbPassword);
    CFGREAD_CLOSE();
    if (result != APS_SUCCESS) { /* try to read unprotected .config.user */
        result = APS_SUCCESS; /* preset */
        CFGREAD_OPEN(spoolDir, ".config.user");
        CFGREAD_GETSTRING("hostip=", 7, & cfg->smbHostIP);
        CFGREAD_GETSTRING("share=", 6, & cfg->smbShare);
        CFGREAD_GETSTRING("user=", 5, & cfg->smbUser);
        CFGREAD_CLOSE();
        if (result == APS_SUCCESS) {
            strupdate(& cfg->smbPassword, "");
        }
    }
    return result;
}
Aps_Result LprFilterRH_lpdcfgRead(const char *spoolDir,
    LprFilterRH_LPDNetworkCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGREAD_OPEN(spoolDir, ".config");
    CFGREAD_GETSTRING("server=", 7, & cfg->lpdServer);
    CFGREAD_GETSTRING("queue=", 6, & cfg->lpdQueue);
    CFGREAD_GETSTRING("user=", 5, & cfg->lpdUser);
    CFGREAD_GETSTRING("password=", 9, & cfg->lpdPassword);
    CFGREAD_CLOSE();
    if (result != APS_SUCCESS) { /* try to read unprotected .config.user */
        result = APS_SUCCESS; /* preset */
        CFGREAD_OPEN(spoolDir, ".config.user");
        CFGREAD_GETSTRING("server=", 6, & cfg->lpdServer);
        CFGREAD_GETSTRING("queue=", 6, & cfg->lpdQueue);
        CFGREAD_GETSTRING("user=", 5, & cfg->lpdUser);
        CFGREAD_CLOSE();
        if (result == APS_SUCCESS) {
            strupdate(& cfg->lpdPassword, "");
        }
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxWrite()
 *
 * Group of helper functions for writing configuration files
 * These write the config file all in one go.
 *
 * Parameters: spoolDir - Pointer to spool directory
 *             cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
#define CFGWRITE_FUNCWRITE( cfgFuncBase, cfgStructName ) \
    Aps_Result cfgFuncBase ## Write (const char *spoolDir, cfgStructName *cfg) { \
        cfg->private.fh = NULL; \
        return cfgFuncBase ## WriteCommit (spoolDir, cfg); \
    }

CFGWRITE_FUNCWRITE( LprFilterRH_txtcfg, LprFilterRH_TextOnlyCfg )
CFGWRITE_FUNCWRITE( LprFilterRH_pscfg, LprFilterRH_PostscriptCfg )
CFGWRITE_FUNCWRITE( LprFilterRH_gencfg, LprFilterRH_GeneralCfg )
CFGWRITE_FUNCWRITE( LprFilterRH_smbcfg, LprFilterRH_SMBNetworkCfg )
CFGWRITE_FUNCWRITE( LprFilterRH_lpdcfg, LprFilterRH_LPDNetworkCfg )

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxWriteDefer()
 *
 * Group of helper functions for writing configuration files
 * These open the config file for writing but do not obliterate
 * the contents until WriteCommit is called.  Needless to say, the data must
 * still be valid once WriteCommit comes around.
 *
 * Parameters: spoolDir - Pointer to spool directory
 *             cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
#define CFGWRITE_FUNCWRITEDEFER( cfgFuncBase, cfgStructName, cfgFileName ) \
    Aps_Result cfgFuncBase ## WriteDefer (const char *spoolDir, cfgStructName *cfg) { \
        Aps_Result result = APS_SUCCESS; \
        CFGWRITE_OPENDEFER(spoolDir, (cfgFileName), S_IRUSR | S_IWUSR ); \
        return result; \
    }

CFGWRITE_FUNCWRITEDEFER( LprFilterRH_txtcfg, LprFilterRH_TextOnlyCfg, "textonly.cfg" )
CFGWRITE_FUNCWRITEDEFER( LprFilterRH_pscfg, LprFilterRH_PostscriptCfg, "postscript.cfg" )
CFGWRITE_FUNCWRITEDEFER( LprFilterRH_gencfg, LprFilterRH_GeneralCfg, "general.cfg" )
CFGWRITE_FUNCWRITEDEFER( LprFilterRH_smbcfg, LprFilterRH_SMBNetworkCfg, ".config" )
CFGWRITE_FUNCWRITEDEFER( LprFilterRH_lpdcfg, LprFilterRH_LPDNetworkCfg, ".config" )

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxWriteCommit()
 *
 * Group of helper functions for writing configuration files
 * Obliterates the contents of a file previously opened by WriteDefer.
 *
 * Parameters: spoolDir - Pointer to spool directory
 *             cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
Aps_Result LprFilterRH_txtcfgWriteCommit(const char *spoolDir,
    LprFilterRH_TextOnlyCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGWRITE_OPENCOMMIT(spoolDir, "textonly.cfg", S_IRUSR | S_IWUSR |
        S_IRGRP | S_IROTH);
    CFGWRITE_PUTTEXT(PrefsHeaderComment);
    CFGWRITE_PUTSTRING("TEXTONLYOPTIONS=", 16, cfg->txtOptions);
    CFGWRITE_PUTBOOL("CRLFTRANS=", 10, cfg->txtAddCR);
    CFGWRITE_PUTBOOL("TEXT_SEND_EOF=", 14, cfg->txtSendEOF);
    CFGWRITE_CLOSE();
    return result;
}
Aps_Result LprFilterRH_pscfgWriteCommit(const char *spoolDir,
    LprFilterRH_PostscriptCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGWRITE_OPENCOMMIT(spoolDir, "postscript.cfg", S_IRUSR | S_IWUSR |
        S_IRGRP | S_IROTH);
    CFGWRITE_PUTTEXT(PrefsHeaderComment);
    CFGWRITE_PUTSTRING("GSDEVICE=", 9, cfg->psGSDevice);
    CFGWRITE_PUTSTRING("EXTRA_GS_OPTIONS=", 17, cfg->psGSOptions);
    CFGWRITE_PUTSTRING("COLOR=", 6, cfg->psGSUniprintDrv);
    CFGWRITE_PUTSTRING("PAPER_SIZE=", 11, cfg->psPaperSize);
    CFGWRITE_PUTPAIR("RESOLUTION=", 11, "", "x", "", cfg->psXDPI, cfg->psYDPI);
    CFGWRITE_PUTBOOL("REVERSE_ORDER=", 14, cfg->psReverseOrder);
    CFGWRITE_PUTBOOL("PS_SEND_EOF=", 12, cfg->psSendEOF);
    CFGWRITE_PUTINT("NUP=", 4, cfg->psNUPCount);
    CFGWRITE_PUTINT("RTLFTMAR=", 9, cfg->psNUPHMargin);
    CFGWRITE_PUTINT("TOPBOTMAR=", 10, cfg->psNUPVMargin);
    CFGWRITE_CLOSE();
    return result;
}
Aps_Result LprFilterRH_gencfgWriteCommit(const char *spoolDir,
    LprFilterRH_GeneralCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGWRITE_OPENCOMMIT(spoolDir, "general.cfg", S_IRUSR | S_IWUSR |
        S_IRGRP | S_IROTH);
    CFGWRITE_PUTTEXT(PrefsHeaderComment);
    CFGWRITE_PUTPRINTERTYPE("PRINTER_TYPE=", 13, cfg->genPrinterType);
    CFGWRITE_PUTSTRING("DESIRED_TO=", 11, cfg->genDesired);
    CFGWRITE_PUTSTRING("PAPER_SIZE=", 11, cfg->genPaperSize);
    CFGWRITE_PUTBOOL("ASCII_TO_PS=", 12, cfg->genASCIItoPS);
    CFGWRITE_CLOSE();
    return result;
}
Aps_Result LprFilterRH_smbcfgWriteCommit(const char *spoolDir,
    LprFilterRH_SMBNetworkCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGWRITE_OPENCOMMIT(spoolDir, ".config", S_IRUSR | S_IWUSR | S_IRGRP
                                             | S_IROTH);
    CFGWRITE_PUTTEXT(PrefsHeaderComment);
    CFGWRITE_PUTSTRING("hostip=", 7, cfg->smbHostIP);
    CFGWRITE_PUTSTRING("share=", 6, cfg->smbShare);
    CFGWRITE_PUTSTRING("user=", 5, cfg->smbUser);
    CFGWRITE_PUTSTRING("password=", 9, cfg->smbPassword);
    CFGWRITE_CLOSE();
    CFGWRITE_OPENCOMMIT(spoolDir, ".config.user",
        S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
    CFGWRITE_PUTSTRING("hostip=", 7, cfg->smbHostIP);
    CFGWRITE_PUTSTRING("share=", 6, cfg->smbShare);
    CFGWRITE_PUTSTRING("user=", 5, cfg->smbUser);
    CFGWRITE_CLOSE();
    return result;
}
Aps_Result LprFilterRH_lpdcfgWriteCommit(const char *spoolDir,
    LprFilterRH_LPDNetworkCfg *cfg)
{
    Aps_Result result = APS_SUCCESS;
    CFGWRITE_OPENCOMMIT(spoolDir, ".config", S_IRUSR | S_IWUSR | S_IRGRP
                                             | S_IROTH);
    CFGWRITE_PUTTEXT(PrefsHeaderComment);
    CFGWRITE_PUTSTRING("server=", 7, cfg->lpdServer);
    CFGWRITE_PUTSTRING("queue=", 6, cfg->lpdQueue);
    CFGWRITE_PUTSTRING("user=", 5, cfg->lpdUser);
    CFGWRITE_PUTSTRING("password=", 9, cfg->lpdPassword);
    CFGWRITE_CLOSE();
    CFGWRITE_OPENCOMMIT(spoolDir, ".config.user",
        S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
    CFGWRITE_PUTSTRING("server=", 7, cfg->lpdServer);
    CFGWRITE_PUTSTRING("queue=", 6, cfg->lpdQueue);
    CFGWRITE_PUTSTRING("user=", 5, cfg->lpdUser);
    CFGWRITE_CLOSE();
    return result;
}

/* ---------------------------------------------------------------------------
 * LprFilterRH_xxxxWriteAbort()
 *
 * Group of helper functions for writing configuration files
 * These abort a deferred write.
 *
 * Parameters: spoolDir - Pointer to spool directory
 *             cfg      - Config structure
 *
 *     Return: APS_SUCCESS on success;
 *             APS_NOT_FOUND if not found;
 *             APS_ACCESS_DENIED if not accessible;
 */
#define CFGWRITE_FUNCABORT( cfgFuncBase, cfgStructName ) \
    Aps_Result cfgFuncBase ## WriteAbort (const char *spoolDir, cfgStructName *cfg) { \
        { CFGWRITE_CLOSE(); /* keep brackets!! */ \
        return APS_SUCCESS; \
    }

CFGWRITE_FUNCABORT( LprFilterRH_txtcfg, LprFilterRH_TextOnlyCfg )
CFGWRITE_FUNCABORT( LprFilterRH_pscfg, LprFilterRH_PostscriptCfg )
CFGWRITE_FUNCABORT( LprFilterRH_gencfg, LprFilterRH_GeneralCfg )
CFGWRITE_FUNCABORT( LprFilterRH_smbcfg, LprFilterRH_SMBNetworkCfg )
CFGWRITE_FUNCABORT( LprFilterRH_lpdcfg, LprFilterRH_LPDNetworkCfg )

#endif /* APSCFG_LPR_BUILD */
