/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerattr.c
 *
 * Description: APIs to access a printer's default attributes.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "printer.h"
#include "jobattributes.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_PrinterGetDefAttr()
 *
 * Obtains a copy of the default job attributes for the specified printer.
 *
 * Parameters: printerHandle  - Handle to the printer to obtain a copy of the
 *                              default job attributes from.
 *
 *             jobAttributes  - A pointer to an Aps_JobAttrHandle to receive
 *                              a handle to the copy of the default
 *                              job attributes.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterGetDefAttr(Aps_PrinterHandle printerHandle,
                                 Aps_JobAttrHandle *jobAttributes)
{
    Printer *printer;
    JobAttributes *newJobAttributes;

    /* Check parameter validity. */
    if (jobAttributes == NULL)
        return (APS_INVALID_PARAM);

    /* Obtain the Printer pointer corresponding to the specified handle. */
    printer = PrinterGetPtrFromHandle(printerHandle);
    if (printer == NULL)
        return (APS_INVALID_HANDLE);

    /* Create a copy of this printer's default job attributes. */
    newJobAttributes = JobAttrCreateCopy(printer->defaultJobAttributes);
    if (newJobAttributes == NULL)
        return (APS_OUT_OF_MEMORY);

    /* Provide the caller with a handle to this copy of the default job */
    /* attributes. */
    *jobAttributes = JobAttrGetHandleFromPtr(newJobAttributes);

    return (APS_SUCCESS);
}

/* ---------------------------------------------------------------------------
 * Aps_PrinterSetDefAttr()
 *
 * Changes the default job attributes for the specified printer.
 *
 * Parameters: printerHandle - The printer to be changed.
 *
 *             jobAttributes - The job attributes that should become the
 *                             default.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_PrinterSetDefAttr(Aps_PrinterHandle printerHandle,
                                 Aps_JobAttrHandle jobAttributes)
{
    Printer *printer;
    JobAttributes *callersAttributes;
    JobAttributes *newAttributes;
    Aps_Result result;

    /* Obtain the Printer pointer corresponding to the specified handle. */
    printer = PrinterGetPtrFromHandle(printerHandle);
    if (printer == NULL)
        return (APS_INVALID_HANDLE);

    /* Obtain the JobAttributes instance corresponding to the provided */
    /* handle. */
    callersAttributes = JobAttrGetPtrFromHandle(jobAttributes);
    if (callersAttributes == NULL)
        return (APS_INVALID_HANDLE);

    /* Attempt to create a copy of the caller's job attributes. */
    newAttributes = JobAttrCreateCopy(callersAttributes);
    if (newAttributes == NULL)
        return (APS_OUT_OF_MEMORY);

    /* Replace the default job attributes for this printer with a copy of */
    /* those provided by the caller. */
    Aps_ReleaseHandle(JobAttrGetHandleFromPtr(printer->defaultJobAttributes));
    printer->defaultJobAttributes = newAttributes;

    /*** UPDATE ANY META-DATA USED BY THE ATTRIBUTE PROVIDERS ***/
    /* Store the new job attributes as the default for this printer. */
    result = JobAttrSaveAsPrinterDefaults(printer->defaultJobAttributes,
                                          printer);

    /*** UPDATE TRANSPORT-SENSITIVE CONFIGURATION INFORMATION ***/
    /* Jeff Brown, March 6, 2000
     * Pass the message to save attributes down to the transport layer
     * so that the transport is informed about potential changes to its
     * context.  Ideally, it would be nice to be able to give a "hint"
     * as to which settings actually need to be updated.
     */
    result = printer->transport->vtbl->PrinterAttrSaveDefaults(
        printer->transport, printer, newAttributes);

    return (APS_SUCCESS);
}
