/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: resultcode.c
 *
 * Description: Result code support functionality.
 *
 */

#include <libintl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "aps.h"
#include "apsinternal.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(utils)

/* ---------------------------------------------------------------------------
 * Aps_GetResultText()
 *
 * Obtains a text string describing the meaning of any of the standard APS
 * result codes.
 *
 * Parameters: result - An Aps_Result code returned by a previous call to some
 *                      APS function.
 *
 *             text   - The string to receive the descriptive text.
 *
 *             size   - The size, in bytes, of the string pointed to by text.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_GetResultText(Aps_Result result, char *text, int size)
{
    char *sourceString;

    /* Check for parameter validity. */
    if(result >= APS_NUMRESULTS || text == NULL || size <= 0) {
        return (APS_INVALID_PARAM);
    }

    /* Obtain a pointer to the appropriate result text. We use a switch
     * statement instead of an array here to facilitate the of gettext.
     */
    switch(result) {
        case APS_SUCCESS:
            sourceString = gettext(
                "The operation has been successfully completed.");
            break;

        case APS_HAS_EXTENDED_SUPPORT:
            sourceString = gettext(
                "This printer has extended support for this feature "
                "that is not being provided here.");
            break;

        case APS_OPERATION_AVAILABLE:
            sourceString = gettext(
                "This operation is currently available.");
            break;

        case APS_NO_CHANGE:
            sourceString = gettext(
                "The operation succeeded but had no net effect.");
            break;

        case APS_IGNORED:
            sourceString = gettext(
                "The operation was ignored because the system was "
                "unable to get satisfactory results.");
            break;

        case APS_MORE_DATA:
            sourceString = gettext(
                "There was only enough space to return some of the available "
                "data.");
            break;

        case APS_PARTIAL_SUCCESS:
            sourceString = gettext(
                "The operation only partially succeeded: it may only "
                "have performed a subset of the required tasks.");
            break;

        case APS_FILTER_NOT_SUPPORTED:
            sourceString = gettext(
                "This operation does not support the attached filter; "
                "try again without.");
            break;

        case APS_NO_MORE_DATA:
            sourceString = gettext(
                "No additional data was available to satisfy the "
                "request.");
            break;

        case APS_NOT_FOUND:
            sourceString = gettext(
                "The specified item (file, printer, etc.) does not exist.");
            break;

        case APS_NOT_IMPLEMENTED:
            sourceString = gettext(
                "This functionality is not implemented yet.");
            break;

        case APS_NOT_SUPPORTED:
            sourceString = gettext(
                "The underlying printer transport does not support this "
                "operation.");
            break;
            
        case APS_INVALID_PARAM:
            sourceString = gettext(
                "A paramter was passed to an APS function with an invalid "
                "value.");
            break;

        case APS_OUT_OF_MEMORY:
            sourceString = gettext(
                "There was not enough memory available to complete the "
                "operation.");
            break;

        case APS_ACCESS_DENIED:
            sourceString = gettext(
                "This user does not have permission to access this "
                "resource.");
            break;

        case APS_INVALID_HANDLE:
            sourceString = gettext(
                "A specified handle is invalid or has been deleted.");
            break;

        case APS_GENERIC_FAILURE:
            sourceString = gettext(
                "The operation has failed, but the reason for failure could "
                "not be determined.");
            break;

        case APS_DISK_FULL:
            sourceString = gettext(
                "There was insufficient disk space to complete this "
                "operation.");
            break;

        case APS_INVALID_PWD:
            sourceString = gettext(
                "The operation was not completed, due to the specified "
                "password not being accepted.");
            break;
            
        case APS_OUT_OF_SEQUENCE:
            sourceString = gettext(
                "This operation cannot be performed at this time/state.");
            break;

        case APS_VIOLATES_CONSTRAINTS:
            sourceString = gettext(
                "This setting change violates one or more constraints on "
                "possible setting permutations.");
            break;

        case APS_INVALID_PPD:
            sourceString = gettext(
                "The specified file is not an existing valid PPD file.");
            break;

        case APS_WRONG_TYPE:
            sourceString = gettext(
                "The requested operation didn't match the correct data "
                "type.");
            break;

        case APS_ALREADY_EXISTS:
            sourceString = gettext(
                "An attempt was made to create a file or object under a name "
                "that already exists.");
            break;

        case APS_OPERATION_TIMEOUT:
            sourceString = gettext(
                "A timeout occured in communication with the underlying "
                "print transport.");
            break;

        case APS_IO_ERROR:
            sourceString = gettext(
                "An error occurred in communicating with the underlying "
                "print transport.");
            break;

        case APS_SUBPROGRAM_FAILED:
            sourceString = gettext(
                "A program in the underlying print system was unable to "
                "carry out this operation.");
            break;

        default:
            /* Unimplemented result code. */
            sourceString = "<invalid / unknown result>";
    }

    /* Check the size of the result text being requested, and return as much
     * of it as possible to the caller.
     */
    if (strlen(sourceString) >= (size_t)size) {
        /* There is not enough space for the enire string, so return partial
         * data.
         */
        strncpy(text, sourceString, size);
        text[size - 1] = '\0';
        return (APS_MORE_DATA);
    }
    else {
        /* We have enough space to return the entire string. */
        strcpy(text, sourceString);
        return (APS_SUCCESS);
    }
}

/* ---------------------------------------------------------------------------
 * Aps_Succeeded()
 *
 * Allows the application to distinguish between Aps_Result codes that
 * indicate an operation succeeded, and result codes that indicate a failure.
 *
 * Parameters: result - An Aps_Result code returned by a previous call to some
 *                      APS function.
 *
 *     Return: TRUE if the specified result code represents success, or FALSE
 *             if it represents a failure.
 */
int Aps_Succeeded(Aps_Result result)
{
    switch(result) {
        case APS_SUCCESS:
        case APS_HAS_EXTENDED_SUPPORT:
        case APS_OPERATION_AVAILABLE:
        case APS_NO_CHANGE:
        case APS_IGNORED:
        case APS_MORE_DATA:
        case APS_PARTIAL_SUCCESS:
            /* These are the result codes considered to be successful. */
            return TRUE;
            
        default:
            /* All other result codes are considered to represent a failure. */
            return FALSE;
    }
}

/* ---------------------------------------------------------------------------
 * GetResultFromErrno()
 *
 * Obtains the Aps_Result code corresponding to the current errno value.
 *
 * Parameters: None.
 *
 *     Return: An Aps_Result code; APS_GENERIC_FAILURE if there is no more
 *             specific result code for the current errno value.
 */
Aps_Result GetResultFromErrno(void)
{
    /* This is just a switch statement mapping common errno codes that have
     * corresponding Aps_Result codes. Since each C implementation can choose
     * which errno codes to implement, each case must be contained within an
     * #ifdef that checks whether or not that particular errno code is
     * provided by this implementation.
     */
    switch (errno) {
        case 0: /* no error! */
            return APS_SUCCESS;

#ifdef EACCES
        case EACCES:
            return APS_ACCESS_DENIED;
#endif
#ifdef EPERM
        case EPERM:
            return APS_ACCESS_DENIED;
#endif
#ifdef EROFS
        case EROFS:
            return APS_ACCESS_DENIED;
#endif
#ifdef ENOMEM
        case ENOMEM:
            return APS_OUT_OF_MEMORY;
#endif
#ifdef ENOSPC
        case ENOSPC:
            return APS_DISK_FULL;
#endif
#ifdef ENODEV
	case ENODEV:
            return APS_NOT_FOUND;
#endif
#ifdef ENOENT
	case ENOENT:
            return APS_NOT_FOUND;
#endif
#ifdef EINVAL
        case EINVAL:
            return APS_INVALID_PARAM;
#endif
#ifdef EIO
        case EIO:
            return APS_INVALID_PARAM;
#endif
        default:
            return APS_GENERIC_FAILURE;
    }
}

