/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsjobattr.C
 *
 * Description: Defines the ApsJobAttrObject class methods.
 *
 */                    

#include "apscpp.h"

/*---------------------------------------------------------------------------
 * ApsJobAttrObject()
 *
 * This constructor takes Aps_JobAttrHandle type as argument.
 *
 * Parameters:
 *     attrHandle : Job Attribute handle.
 *
 *     Return: ApsResultObject.
 *
 */                    
ApsJobAttrObject::ApsJobAttrObject(const Aps_JobAttrHandle attrHandle)
                       : ApsObject((Aps_Handle)attrHandle)
{
}
/*---------------------------------------------------------------------------
 * ApsJobAttrObject()
 *
 * Copy constructor.
 *
 * Parameters:
 *     jobAttr : Reference to ApsJobAttrObject object.
 *
 *     Return: ApsResultObject.
 *
 */                                                
ApsJobAttrObject::ApsJobAttrObject(const ApsJobAttrObject &jobAttr)
{
    Aps_Handle apsHandle;
 
    apsHandle = jobAttr.GetHandle();
    SetHandle(apsHandle);    
}

/*---------------------------------------------------------------------------
 * GetList()
 *
 * Obtains a list of individual attributes available in a particular job 
 * attributes object. This can be a list of attributes, or only the subset of 
 * attributes specified by the group parameter. 
 * This method internally calls Aps_AttrGetList Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     group : A string containing the name of the attributes group to be retrieved
 *     numAttributes : A pointer to an integer to receive the number of 
 *                     attributes that were found.
 *     attributeIDs : A char *** that receives a buffer containing an array of 
 *                    pointers to strings composing the attribute IDs.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetList(const char *group, 
                                    int *numAttributes, char ***attributeIDs)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetList(jobAttrHandle, group, 
                                            numAttributes, attributeIDs)));
}

/*---------------------------------------------------------------------------
 * GetSubGroups()
 *
 * Obtains a list of attribute sub-groups belonging to another attribute group.
 * This method internally calls Aps_AttrGetSubGroups Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     group : A string containing the name of the attributes group to be retrieved
 *     numAttributes : A pointer to an integer to receive the number of 
 *                     attributes that were found.
 *     subGroupNames : A char *** that receives a buffer containing an array of 
 *                     pointers to strings composing the list of group names. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetSubGroups(const char *group,
                                  int *numSubGroups, char ***subGroupNames)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetSubGroups(jobAttrHandle, group, 
                                          numSubGroups, subGroupNames)));
}

/*---------------------------------------------------------------------------
 * GetTranslatedName()
 *
 * Obtains the local translated name for the specified attribute.
 * This method internally calls Aps_AttrGetTranslatedName Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string identifying the attribute being queried.
 *     translatedName : A pointer to a string pointer that receives a new 
 *                      buffer with the translated name for this attribute.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetTranslatedName(const char *attributeID, 
                                                     char **translatedName)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetTranslatedName(jobAttrHandle, 
                                            attributeID, translatedName)));
}

/*---------------------------------------------------------------------------
 * GetMainData()
 *
 * Obtains the main data provided by this attribute.
 * This method internally calls Aps_AttrGetMainData Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     mainData : A pointer to a char* to receive a new buffer containing the 
 *                main data stored in this attribute.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetMainData(const char *attributeID, 
                                              char **mainData)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetMainData(jobAttrHandle, attributeID, 
                                                               mainData)));
}

/*---------------------------------------------------------------------------
 * GetTranslatedData()
 *
 * Obtains the translated data string for a particular attribute.
 * This method internally calls Aps_AttrGetTranslatedData Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     translatedData : A string pointer to receive a new buffer with the 
 *                      translated data. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetTranslatedData(const char *attributeID, 
                                                     char **translatedData)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetTranslatedData(jobAttrHandle, 
                                             attributeID, translatedData)));
}

/*---------------------------------------------------------------------------
 * GetType()
 *
 * Obtains the type of a particular attribute.
 * This method internally calls Aps_AttrGetType Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     attributeType : A pointer to an Aps_AttrType to receive the
 *                     type of the attribute.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetType(const char *attributeID, 
                                              Aps_AttrType *attributeType)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetType(jobAttrHandle, attributeID, 
                                                           attributeType)));
}

/*---------------------------------------------------------------------------
 * GetRange()
 *
 * Obtains the range of possible values for attributes with integer or 
 * floating point user settings.
 * This method internally calls Aps_AttrGetRange Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     minSetting : A double to receive the minimum value.
 *     maxSetting : A double to receive the maximum value.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetRange(const char *attributeID, 
                                            double *minSetting, 
                                            double *maxSetting) 
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetRange(jobAttrHandle, attributeID, 
                                                  minSetting, maxSetting)));
}

/*---------------------------------------------------------------------------
 * GetOptions()
 *
 * Obtains a list of optionsof the specified attributes.
 * This method internally calls Aps_AttrGetOptions Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     numOptions : A pointer to an integer to receive the number of available 
 *                  options.
 *     options : An Aps_AttrOption pointer to receive a pointer to a new array.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetOptions(const char*attributeID, 
                                              int *numOptions, 
                                              Aps_AttrOption ***option)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetOptions(jobAttrHandle, attributeID, 
                                                         numOptions, option)));
}

/*---------------------------------------------------------------------------
 * GetSetting()
 *
 * Obtains the current value of user setting for a particular attribute.
 * This method internally calls Aps_AttrGetSetting Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to retrieve.
 *     setting : A pointer to a char * to receive a new buffer containing the 
 *               current user-setting for this attribute. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::GetSetting(const char *attributeID, 
                                             char **setting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrGetSetting(jobAttrHandle, attributeID, 
                                                              setting)));
}

/*---------------------------------------------------------------------------
 * SetSetting()
 *
 * Changes the user setting for a particular attribute.
 * This method internally calls Aps_AttrSetSetting Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute to modify.
 *     setting : The new value for the user-setting.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::SetSetting(const char *attributeID, 
                                              const char *setting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrSetSetting(jobAttrHandle, attributeID, 
                                                              setting)));
}

/*---------------------------------------------------------------------------
 * CheckConstraints()
 *
 * Obtains the level of the postscript language that is supported.
 * This method internally calls Aps_AttrCheckConstraints Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     attributeID : A string uniquely identifying the attribute whose user 
 *                   setting is being proposed for modification.
 *     setting : A string containing the new user setting being proposed for 
 *               this attribute.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::CheckConstraints(const char *attributeID, 
                                                const char *setting,
                                                char **conflictingAttribute, 
                                                char **conflictingSetting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrCheckConstraints(jobAttrHandle, 
            attributeID,setting, conflictingAttribute, conflictingSetting)));
}

/*---------------------------------------------------------------------------
 * QuickGetPostScriptLevel()
 *
 * Obtains the level of the postscript language that is supported.
 * This method internally calls Aps_AttrQuickGetPostScriptLevel Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *     languageLevel : The address of an Aps_PostScriptSupport enumeration 
 *                     to be set to the language level supported by this 
 *                     device,if known.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetPostScriptLevel(
                                       Aps_PostScriptSupport  *languageLevel)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetPostScriptLevel(jobAttrHandle, 
                                                            languageLevel)));
}

/*---------------------------------------------------------------------------
 * QuickIsColorDevice()
 *
 * Determines whether or not this device is capable of printing in color.
 * This method internally calls Aps_AttrQuickIsColorDevice Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *    colorDevice : The address of an int to be set to TRUE if this is a 
 *                  color-capable device, FALSE if the device can only print 
 *                  grayscale.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickIsColorDevice(int *colorDevice)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickIsColorDevice(jobAttrHandle, 
                                                            colorDevice)));
}

/*---------------------------------------------------------------------------
 * QuickGetResOptions()
 *
 * Obtains a list of available resolution.
 * This method internally calls Aps_AttrQuickGetResOptions Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    resolutions : The address of a pointer to receive and array of 
 *                  Aps_Resolution pointers, consisting of one element for each 
 *                  available resolution. 
 *    numResolutions : An int to receive the number of resolutions that have 
 *                     been listed in the resolutions array.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetResOptions(
                                              Aps_Resolution ***resolutions,
                                              int *numResolutions)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetResOptions(jobAttrHandle, 
                                              resolutions,numResolutions))); 
}

/*---------------------------------------------------------------------------
 * QuickGetRes()
 *
 * Obtains the currently selected resolution that will be used for 
 * rasterization.
 * This method internally calls Aps_AttrQuickGetRes Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     resolution : The address of an Aps_Resolution structure to receive the 
 *                  current resolution setting.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetRes(Aps_Resolution *resolution)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetRes(jobAttrHandle, resolution)));
}

/*---------------------------------------------------------------------------
 * QuickSetRes()
 *
 * Sets the resolution that will be used for rasterization.
 * This method internally calls Aps_AttrQuickSetRes Aps library function and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    resolution : An Aps_Resolution value.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetRes(const Aps_Resolution *resolution)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetRes(jobAttrHandle, resolution)));
}

/*---------------------------------------------------------------------------
 * QuickGetMaxCopies()
 *
 * Obtains the maximum number of copies of a job that can be printed.
 * This method internally calls Aps_AttrQuickGetMaxCopies Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     maxCopies : The address of an int to receive the maximum number of 
 *                 copies.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetMaxCopies(int *maxCopies)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetMaxCopies(jobAttrHandle, 
                                                               maxCopies)));
}

/*---------------------------------------------------------------------------
 * QuickGetNumCopies()
 *
 * Obtains the number of copies of the entire job that will be printed.
 * This method internally calls Aps_AttrQuickGetNumCopies Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *     numCopies : The address of an int to receive the number of copies of 
 *                 the job to be printed.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetNumCopies(int *numCopies)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetNumCopies(jobAttrHandle, 
                                                           numCopies)));
}

/*---------------------------------------------------------------------------
 * QuickSetNumCopies()
 *
 * Changes the number of copies of the entire job that can be printed.
 * This method internally calls Aps_AttrQuickSetNumCopies Aps library fn. and
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    numCopies : The number of copies to be printed.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetNumCopies(int numCopies)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetNumCopies(jobAttrHandle, 
                                                            numCopies)));
}

/*---------------------------------------------------------------------------
 * QuickGetCollationOptions()
 *
 *  Obtains a list of standard collation options that are available.
 * This method internally calls Aps_AttrQuickGetCollationOptions Aps 
 * library function and returns the value returned by this function in the 
 * result data member of ApsResultObject.
 *
 * Parameters:
 *     collationOptions : The address of an Aps_CopyCollation * to receive an 
 *                      array listing collation options supported by this device.
 *     numOptions : An integer to receive the number of options in the 
 *                  collationOptions array.
 * 
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetCollationOptions(
                                   char ***collationOptions, int *numOptions)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetCollationOptions(jobAttrHandle, 
                                             collationOptions, numOptions)));
}

/*---------------------------------------------------------------------------
 * QuickGetCollation()
 *
 * Obtains the current setting for copy collation.
 * This method internally calls Aps_AttrQuickGetCollation Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     collationSetting : An Aps_CopyCollation enumeration identifying the 
 *                        current collation setting. 
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetCollation(char **collationSetting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetCollation(jobAttrHandle, 
                                                      collationSetting)));
}

/*---------------------------------------------------------------------------
 * QuickSetCollation()
 *
 * Changes the current setting for copy collation.
 * This method internally calls Aps_AttrQuickSetCollation Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     collationSetting : An Aps_CopyCollation enumeration identifying the 
 *                        current collation setting. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetCollation(const char *collationSetting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetCollation(jobAttrHandle, 
                                                         collationSetting)));
}

/*---------------------------------------------------------------------------
 * QuickGetPageSizeOptions()
 *
 * Obtains a list of available pre-defined page sizes.
 * This method internally calls Aps_AttrQuickGetPageSizeOptions Aps library 
 * function and returns the value returned by this function in the result data 
 * member of ApsResultObject.
 *
 * Parameters:
 *     pageSizes : The address of an Aps_PageSize ** to receive an array of 
 *                 pointers to Aps_PageSize structures, each of which will 
 *                 contain information on one of the page sizes that are 
 *                 available. 
 *     numPageSizes : An int to receive the number of available page sizes that 
 *                    have been listed in the pageSizes array.
 * 
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetPageSizeOptions(
                                                    Aps_PageSize ***pageSizes, 
                                                    int *numPageSizes)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetPageSizeOptions(jobAttrHandle, 
                                                  pageSizes, numPageSizes)));
}

/*---------------------------------------------------------------------------
 * QuickGetPageSize()
 *
 * Obtains the page size that is currently selected.
 * This method internally calls Aps_AttrQuickGetPageSize Aps 
 * library function and returns the value returned by this function in the 
 * result data member of ApsResultObject.
 *
 * Parameters:
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetPageSize(Aps_PageSize **pageSize)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetPageSize(jobAttrHandle, 
                                                                pageSize)));
}
/*---------------------------------------------------------------------------
 * QuickGetCustomPageSizeLimits()
 *
 * Obtains an Aps_CustomPageSizeLimits structure with information on the 
 * range of custom page sizes that are available, if custom page sizes are
 * supported by this printer.
 * This method internally calls Aps_AttrQuickGetCustomPageSizeLimits Aps 
 * library function and returns the value returned by this function in the 
 * result data member of ApsResultObject.
 *
 * Parameters:
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject 
ApsJobAttrObject::QuickGetCustomPageSizeLimits(
                              Aps_CustomPageSizeLimits **customPageSizeLimits)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetCustomPageSizeLimits(jobAttrHandle, 
                                                    customPageSizeLimits)));
}

/*---------------------------------------------------------------------------
 * QuickSetPredefinedPageSize()
 *
 * Sets the current page size to one of the pre-defined page sizes.
 * This method internally calls Aps_AttrQuickSetPredefinedPageSize Aps library 
 * function and returns the value returned by this function in the result data  
 * member of ApsResultObject.
 *
 * Parameters:
 *     pageSizeID : A string containing the ID of the page size to be selected. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetPredefinedPageSize(
                                                     const char *pageSizeID)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetPredefinedPageSize(jobAttrHandle, 
                                                               pageSizeID)));
}

/*---------------------------------------------------------------------------
 * QuickSetCustomPageSize()
 *
 * Selects a custom page size, rather than using one of the page sizesin the 
 * list of predefined page sizes.
 * This method internally calls Aps_AttrQuickSetCustomPageSize Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *   Aps_PageSize : The address of an Aps_PageSize structure containing the 
 *                  parameters that describe the page size to be used.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetCustomPageSize(
                                       const Aps_PageSize *customPageSize)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetCustomPageSize(jobAttrHandle, 
                                                           customPageSize)));
}

/*---------------------------------------------------------------------------
 * QuickGetInputSlotOptions()
 *
 * Obtains a list of available standard input slots that are available.
 * This method internally calls Aps_AttrQuickGetInputSlotOptions Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *    inputSlots : An Aps_InputSlot * to receive the address of an array 
 *                  listing the available input slots. 
 *    numInputSlots : An int to receive the number of available input slots 
 *                    that were placed in the array.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetInputSlotOptions(char ***inputSlots,
                                                         int *numInputSlots)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetInputSlotOptions(jobAttrHandle, 
                                                inputSlots, numInputSlots)));
}

/*---------------------------------------------------------------------------
 * QuickGetInputSlot()
 *
 * Obtains the currently selected input slot.
 * This method internally calls Aps_AttrQuickGetInputSlot Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *    inputSlot : An Aps_InputSlot to receive the input slot that is currently 
 *                selected. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetInputSlot(char **inputSlot)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetInputSlot(jobAttrHandle, 
                                                             inputSlot)));
}

/*---------------------------------------------------------------------------
 * QuickSetInputSlot()
 *
 * Changes the currently selected input slot.
 * This method internally calls Aps_AttrQuickSetInputSlot Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    inputSlot : An Aps_InputSlot identifying the input slot to use. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetInputSlot(const char *inputSlot)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetInputSlot(jobAttrHandle, 
                                                               inputSlot)));
}

/*---------------------------------------------------------------------------
 * QuickGetOutputBinOptions()
 *
 * Obtains a list of available standard output bins that are available.
 * This method internally calls Aps_AttrQuickGetOutputBinOptions Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *     outputBins : An Aps_OutputBin pointer to receive the address of an array 
 *                  listing the available output bins.
 *     numOutputBins : An int to receive the number of available output bins 
 *                     that were placed in the array.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetOutputBinOptions(char ***outputBins, 
                                                            int *numOutputBins)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetOutputBinOptions(jobAttrHandle, 
                                                outputBins, numOutputBins)));
}

/*---------------------------------------------------------------------------
 * QuickGetOutputBin()
 *
 * Obtains the currently selected output bin.
 * This method internally calls Aps_AttrQuickGetOutputBin Aps library function
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     outputBin : An Aps_OutputBin to receive the output bin that is currently 
 *                 selected. 
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetOutputBin(char **outputBin)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetOutputBin(jobAttrHandle, 
                                                               outputBin)));
}

/*---------------------------------------------------------------------------
 * QuickSetOutputBin()
 *
 * Changes the currently selected output bin.
 * This method internally calls Aps_AttrQuickSetOutputBin Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    outputBin : An Aps_OutputBin identifying the output bin to use.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickSetOutputBin(const char *outputBin)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetOutputBin(jobAttrHandle, 
                                                               outputBin)));
}

/*---------------------------------------------------------------------------
 * QuickGetDuplexOptions()
 *
 * Obtains a list of available settings for duplex (double-sided) printing.
 * This method internally calls Aps_AttrQuickGetDuplexOptions Aps library 
 * function and returns the value returned by this function in the result 
 * data member of ApsResultObject.
 *
 * Parameters:
 *     duplexSettings : The address of an Aps_Duplex pointer to receive a list
 *                      of the available duplexing settings. 
 *     numDuplexSettings : The number of elements in the array provided via 
 *                       duplexSettings.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetDuplexOptions(char ***duplexSettings,
                                                       int *numDuplexSettings)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetDuplexOptions(jobAttrHandle, 
                                        duplexSettings, numDuplexSettings)));
}

/*---------------------------------------------------------------------------
 * QuickGetDuplex()
 *
 * Obtains a list of available settings for duplex printing.
 * This method internally calls Aps_AttrQuickGetDuplex Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     duplexSetting : The address of an Aps_Duplex enumeration to receive the 
                       current value of the duplex setting. 
 *     Return: ApsResultObject.
 *
 */
 ApsResultObject ApsJobAttrObject::QuickGetDuplex(char **duplexSetting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetDuplex(jobAttrHandle, 
                                                           duplexSetting)));
}

/*---------------------------------------------------------------------------
 * QuickSetDuplex()
 *
 * Changes the current setting for duplex printing.
 * This method internally calls Aps_AttrQuickSetDuplex Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *    duplexSetting : The new duplex setting to use. 
 *
 *     Return: ApsResultObject.
 *
 */
 ApsResultObject ApsJobAttrObject::QuickSetDuplex(const char *duplexSetting)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickSetDuplex(jobAttrHandle, 
                                                            duplexSetting)));
}

/*---------------------------------------------------------------------------
 * QuickGetFonts()
 *
 * Obtains a list of fonts that are available on the device.
 * This method internally calls Aps_AttrQuickGetFonts Aps library function 
 * and returns the value returned by this function in the result data member 
 * of ApsResultObject.
 *
 * Parameters:
 *     fonts : The address of a pointer to receive an array of pointers to 
 *             Aps_FontInfo structures, with one element for each resident font.
 *     numFonts : An integer to receive the number of fonts in the array that 
 *             the fonts parameter passes back.
 *
 *     Return: ApsResultObject.
 *
 */
ApsResultObject ApsJobAttrObject::QuickGetFonts(Aps_FontInfo ***fonts, 
                                                                int *numFonts)
{
    Aps_JobAttrHandle jobAttrHandle;
 
    jobAttrHandle = (Aps_JobAttrHandle)GetHandle();
    return (ApsResultObject(Aps_AttrQuickGetFonts(jobAttrHandle, fonts, 
                                                                 numFonts)));
}
