/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apsprinter.C
 *
 * Description: Defines the ApsPrinterObject class object.
 *
 */            

#include "apscpp.h"

/*---------------------------------------------------------------------------
 * ApsPrinterObject ()
 *
 *  
 * This constructor takes local printer handle as argument.
 *
 * Parameters: 
 *    printerHandle : handle to some printer.
 *                
 *     Return: ApsResultObject.
 *             
 */      
ApsPrinterObject::ApsPrinterObject(const Aps_PrinterHandle printerHandle)
                          : ApsObject((Aps_Handle)printerHandle) 
{
}

/*---------------------------------------------------------------------------
 * ApsPrinterObject ()
 *
 * Copy constructor. 
 *
 * Parameters: 
 *    apsPrinter : reference to ApsPrinterObject object.
 *                
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsPrinterObject::ApsPrinterObject(const ApsPrinterObject &apsPrinter)
{
    Aps_Handle apsHandle;

    apsHandle = apsPrinter.GetHandle();
    SetHandle(apsHandle);
}

/*---------------------------------------------------------------------------
 * Constructor()
 *
 * This constructor takes local printer name as argument.
 *
 * Parameters: 
 *    name       : name of the printer connected locally.
 *                
 *     Return: ApsResultObject.
 *             
 */      
ApsPrinterObject::ApsPrinterObject(const char  *name)
{
    Aps_PrinterHandle printerHandle;
    Aps_Result ret;
       
    ret = Aps_OpenPrinter(name, &printerHandle);
    if (Aps_Succeeded(ret))
        SetHandle((Aps_Handle)printerHandle);
    else
        SetHandle((Aps_Handle)NULL);

}

/*---------------------------------------------------------------------------
 * Constructor()
 *
 * This constructor takes parameter for the remote printer.
 *
 * Parameters: 
 *    connectionType : An Aps_ConnectionType, which together with the 
 *                     location parameter uniquely specify the remote 
 *                     printer to connect to. 
 *    location       : A location string that uniquely identifies a
 *                     remote printer for a given connectionType.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsPrinterObject::ApsPrinterObject(Aps_ConnectionType connectionType, 
                                                        const char *location)
{
   Aps_PrinterHandle printerHandle;
   Aps_Result ret;
 
   ret = Aps_OpenRemotePrinter(connectionType, location, &printerHandle);
   if (Aps_Succeeded(ret))
       SetHandle((Aps_Handle)printerHandle);
   else
       SetHandle((Aps_Handle)NULL);
}
/*---------------------------------------------------------------------------
 * GetPrinters()
 *
 * Obtains a list of names of printers. 
 * This method internally calls Aps_GetPrinters Aps library function and 
 * returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters: 
 *    names : The address of a pointer to receive an array of string 
 *            pointers consisting of the names of all available printer.
 *    count : A pointer to an int to receive the count of the number of 
 *            elements filled in the array names.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetPrinters(char ***names, int *count)
{
    return (ApsResultObject(Aps_GetPrinters(names, count)));
}

/*------------------------------------------------------------------------------
 * GetRemotePrinters()
 *
 * Obtains a list of names of remote printers. 
 * This method internally calls Aps_GetRemotePrinters Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 * 
 *
 * Parameters: 
 *    connectionType : The type of printer connection to search for local 
 *                     printers on.
 *    subLocation    : An optional string identifying the server or group of 
 *                     printers within the specified connection type.
 *    remotePrinters : The address of a pointer to receive the address of a new 
 *                     array of Aps_RemotePrinter structures.
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetRemotePrinters(
                                             Aps_ConnectionType connectionType, 
                                             const char *subLocation, 
                                             Aps_RemotePrinter ***remotePrinters)
                              
{
    return (ApsResultObject(Aps_GetRemotePrinters(connectionType, subLocation, 
                                     remotePrinters)));
}

/*------------------------------------------------------------------------------
 * GetKnownManufacturers()
 *
 * Obtains a list of printer manufacturer names for which one or more printer 
 * names are known.                                        
 * This method internally calls Aps_GetKnownManufacturers Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 *
 * Parameters: 
 *    manufacturerNames : The address of a pointer to receive the address of an 
 *                        array of string pointers, each containing the name of 
 *                        a known manufacturer. 
 *    numManufacturers  : On success, the int at this address will receive the 
 *                        count of the number of printer manufacturers in the 
 *                        array provided to manufacturerNames.
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetKnownManufacturers(char *** manufacturerNames,
                                                    int *numManufacturers)
{
    return (ApsResultObject(Aps_GetKnownManufacturers(manufacturerNames, 
                                           numManufacturers)));
}

/*------------------------------------------------------------------------------
 * GetKnownModels()
 *
 * Obtains a list of printer models known for a particular manufacturer.
 * This method internally calls Aps_GetKnownModels Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 *
 * Parameters: 
 *    manufacturerName : Manufacturer name.
 *    modelNames : The address of a pointer to receive the address of an array 
 *                of string pointers, each containing the name of a known model. 
 *    numModels  : On success, the int at this address will receive the count of 
 *                 the number of printer models in the array provided to 
 *                 modelNames.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetKnownModels(const char *manufacturerName, 
                                             char ***modelNames,int *numModels)
{
    return (ApsResultObject(Aps_GetKnownModels(manufacturerName, modelNames, numModels)));
}

/*------------------------------------------------------------------------------
 * AddModel()
 *
 * Creates an empty entry in printer database for a new printer model.
 * This method internally calls Aps_AddModel Aps library function
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters: 
 *     manufacturer : A string with the name of the printer manufacturer for 
 *                    the new model.
 *     model : A string with the name of the specific printer model.
 *     modelHandle : The address of an Aps_ModelHandle to receive a handle 
 *                   to the new printer model, or NULL if not required.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::AddModel(const char *manufacturer, 
                                       const char *model,
                                       Aps_ModelHandle *modelHandle)
{
    return (ApsResultObject(Aps_AddModel(manufacturer, model, modelHandle)));
}

/*------------------------------------------------------------------------------
 * AddPrinter()
 *
 * Installs a new printer that may be printed to in the future.
 * This method internally calls Aps_AddPrinter Aps library function and returns 
 * the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     name : The name to give the new printer. 
 *     printer : The address of an Aps_PrinterHandle to receive a handle to 
 *               the newly created printer. 
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::AddPrinter(const char *name, 
                                         Aps_PrinterHandle *printer)
{

    return (ApsResultObject(Aps_AddPrinter(name, printer)));
}

/*------------------------------------------------------------------------------
 * DispatchJobToDefault()
 *
 * Spools the job to the default printer.
 * This method internally calls Aps_DispatchJob Aps library function and returns 
 * the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *   filename : The name of the file or other file system entity to print from.
 *   format   : A string describing the format (e.g. PostScript, 
 *              raw text, etc.) of the data being submitted. 
 *   jobAttributes : An optional handle to the job attributes to apply to this 
 *                   job.
 *   job : An optional pointer to receive an Aps_JobHandle to allow future 
 *         monitoring of this job, or NULL if not required. 
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::DispatchJobToDefault (const char *filename, 
                                       char *format,
                                       Aps_JobAttrHandle jobAttributes,
                                       Aps_JobHandle *job)
{
    return (ApsResultObject(Aps_DispatchJob(NULL, filename, 
                                     format, jobAttributes, job)));
}

/*------------------------------------------------------------------------------
 * Remove()
 *
 * This will remove the installed printer and frees the resources.
 * This method internally calls Aps_PrinterRemove Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     None.
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::Remove()
{
    Aps_Result resultCode;
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    resultCode = Aps_PrinterRemove(printerHandle);   
   /* If the printer is removed then release the handle and set the handle data
    * member as NULL.
    */
    if (Aps_Succeeded(resultCode))
	SetHandle(NULL);
    return (ApsResultObject(resultCode));
}

/*------------------------------------------------------------------------------
 * GetName()
 *
 * Gets the name of the printer with which the printer object is associated.
 * This method internally calls Aps_PrinterGetName Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    printerHandle : A handle to a currently open printer.
 *    name : The address of a character pointer to receive the location of a 
 *           new string with the name of this printer. 
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetName(char **name)
{
    Aps_PrinterHandle printerHandle;

    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetName(printerHandle, name)));
}

/*------------------------------------------------------------------------------
 * Rename()
 *
 * Changes the name of the printer.
 * This method internally calls Aps_PrinterRename Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    newName : A string containing the new name to assign to this printer.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::Rename(char *newName)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterRename(printerHandle, newName)));
}

/*------------------------------------------------------------------------------
 * IsDefault()
 *
 * Checks if the printer is a default printer.
 * This method internally calls Aps_PrinterIsDefault Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     isDefault : Set to TRUE if this is the default printer, FALSE if it isn't.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::IsDefault(bool *isDefault)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterIsDefault(printerHandle, (int *)isDefault)));
}
 
/*------------------------------------------------------------------------------
 * GetMaxJobSize()
 *
 * Returns the maximum job size in bytes that this printer will accept or 
 * APS_NO_MAX_SIZE if the printer is configured to accept the jobs of any size. 
 * This method internally calls Aps_PrinterGetMaxJobSize Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    maxSize : The maximum size of job, in bytes, that this printer will accept, *              or APS_NO_MAX_SIZE if the printer is configured to accept jobs 
 *              of any size.
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetMaxJobSize(int *maxSize)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetMaxJobSize(printerHandle, maxSize)));
}

/*------------------------------------------------------------------------------
 * SetMaxJobSize()
 *
 * Sets the maximum jobSize for the printer.
 * This method internally calls Aps_PrinterSetMaxJobSize Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    maxSize : The maximum size of job, in bytes, that this printer should 
 *              accept, or APS_NO_MAX_SIZE if this printer.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::SetMaxJobSize(int maxSize)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetMaxJobSize(printerHandle, maxSize)));
}

/*------------------------------------------------------------------------------
 * SetConfigFlags()
 *
 * This will set the config flags value passed as Input parameters for the printer.
 * This method internally calls Aps_PrinterSetConfigFlags Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     flagsToSet : Zero or more flags to turn on (set), joined together by the 
 *                  bitwise-or operator. 
 *     flagsToReset : Zero or more flags to turn off (reset), 
 *                    joined together by the bitwise-or operator.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::SetConfigFlags(long int flagsToSet, 
                                                             long flagsToReset)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetConfigFlags(printerHandle, 
                                                   flagsToSet, flagsToReset)));
}

/*-------------------------------------------------------------------------------
 * GetConfigFlags()
 *
 * This gets the currently configured flags value for the printer.
 * This method internally calls Aps_PrinterGetConfigFlags Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     configFlags : An int to receive the set of printer configuration flags 
 *                   that are currently set.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetConfigFlags(long int *configFlags)
{
    Aps_PrinterHandle printerHandle;

    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetConfigFlags(printerHandle, configFlags)));
}

/*-------------------------------------------------------------------------------
 * GetModel()
 *
 * This gets the printer model for the printer.
 * This method internally calls Aps_PrinterGetModel Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     manufacturer : The address of a string pointer to receive a buffer with 
 *                    the name of this printer's manufacturer.
 *     model :        The address of a string pointer to receive the name of 
 *                    the specific model.
 *
 *     Return: ApsResultObject.
 *             
 */      
ApsResultObject ApsPrinterObject::GetModel(char ** manufacturer, char **model)
{
    Aps_PrinterHandle printerHandle;

    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetModel(printerHandle, manufacturer, model)));
}
 
/*-------------------------------------------------------------------------------
 * SetModel()
 *
 * This sets the printer model for the printer.
 * This method internally calls Aps_PrinterSetModel Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     manufacturer : A string containing the name of this model's manufacturer.
 *     model : A string containing the name of the manufacturer's specific model.
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::SetModel(const char *manufacturer, 
                                                             const char *model)
{
    Aps_PrinterHandle printerHandle;

    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetModel(printerHandle, manufacturer, 
                                                                       model)));
}

/*-------------------------------------------------------------------------------
 * GetPPDFileName()
 *
 * This gets the PPD Filename for this printer.
 * This method internally calls Aps_PrinterGetPPDFileName Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    filename : The address of a pointer to receive the address of a new 
 *               string containing the name of the PPD file. 
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::GetPPDFileName(char **filename)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetPPDFileName(printerHandle, filename)));
}

/*-------------------------------------------------------------------------------
 * SetPPDFileName()
 *
 * This sets the PPD Filename for this printer.
 * This method internally calls Aps_PrinterSetPPDFileName Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    filename : A string containing the full path and filename of the PPD file 
 *               to be associated with this printer.
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::SetPPDFileName(const char *filename)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetPPDFileName(printerHandle, filename)));
}
/*-------------------------------------------------------------------------------
 * GetModelHandle()
 *
 * Obtains a handle to a particular printer's model object.
 * This method internally calls Aps_PrinterGetModelHandle Aps library function 
 * and returns the value returned by this function in the result data member of
 * ApsResultObject.
 *
 * Parameters:
 *    model : The address of an Aps_ModelHandle to receive a handle to this 
 *            printer's model object.
 *
 *     Return: ApsResultObject.
 *
 */      
ApsResultObject ApsPrinterObject::GetModelHandle(Aps_ModelHandle *model)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetModelHandle(printerHandle, model)));
}
/*-------------------------------------------------------------------------------
 * GetConnectInfo()
 *
 * Gets the Connect Info for this printer.
 * This method internally calls Aps_PrinterGetConnectInfo Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *     connectionType : The address of an Aps_ConnectionType to receive 
 *                      information on which type of connection is being used.
 *     location : The address of a char * to receive a string with the 
 *                device/network name that this printer is attached to. 
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::GetConnectInfo(Aps_ConnectionType *connectionType,
                                    char **location)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetConnectInfo(printerHandle, connectionType, location)));
}

/*-------------------------------------------------------------------------------
 * SetConnectInfo()
 *
 * This sets the Connect Info for this printer.
 * This method internally calls Aps_PrinterSetConnectInfo Aps library function and
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *  connectionType : An Aps_ConnectionType identifying the  mechanism 
 *                      (local / remote) used to communicate with this printer.
 *  location : A string containing the device name or network path for this printer.
 *
 *     Return: ApsResultObject.
 *             
 */
 ApsResultObject ApsPrinterObject::SetConnectInfo(Aps_ConnectionType 
                                      connectionType, const char *location)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetConnectInfo(printerHandle, connectionType, 
                                                    location)));
}

/*-------------------------------------------------------------------------------
 * GetDefAttr()
 *
 * This gets the printerHandle to the  'default attribute' for this printer. 
 * This method internally calls Aps_PrinterGetDefAttr Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    jobAttributes : The address of an Aps_JobAttrHandle that will receive a 
 *                    printerHandle to the job attributes copy on success. 
 * 
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::GetDefAttr(Aps_JobAttrHandle *jobAttributes)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterGetDefAttr(printerHandle, jobAttributes)));
}

/*-------------------------------------------------------------------------------
 * SetDefAttr()
 *
 * This sets the default job attributes for the printer.
 * This method internally calls Aps_PrinterSetDefAttr Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    jobAttributes : The default job attributes to apply to future jobs sent 
 *                    to this printer.
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::SetDefAttr(Aps_JobAttrHandle jobHandle)
{
    Aps_PrinterHandle printerHandle;

    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterSetDefAttr(printerHandle, jobHandle)));
}

/*-------------------------------------------------------------------------------
 * DispatchJob()
 *
 * This dispatches the job to the printer.
 * This method internally calls Aps_DispatchJob Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    filename : The name of the file or other file system entity to print from.
 *    format   : A string describing the format (e.g. PostScript, raw text, 
 *               etc.) of the data being submitted. 
 *    jobAttributes : handle to the job attributes.
 *    job : An optional pointer to receive an Aps_JobHandle to allow future 
 *          monitoring of this job, or NULL if not required.
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::DispatchJob(char *filename, 
				char *format, Aps_JobAttrHandle jobAttributes,
                                Aps_JobHandle *job)
{
    Aps_PrinterHandle printerHandle;
     
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_DispatchJob(printerHandle, filename, format, 
                                 jobAttributes, job)));
}

/*-------------------------------------------------------------------------------
 * StartJob()
 *
 * Begins a new job on the device, providing the caller with an Aps_JobHandle that
 * can be used to manipulate job settings, or to write data to be printed.
 * This method internally calls Aps_StartJob Aps library function and 
 * returns the value returned by this function in the result data member of 
 * ApsResultObject.
 *
 * Parameters: 
 *    format : A string describing the format (e.g. PostScript, raw text, etc.) 
 *             of the data being submitted.
 *    job :   A pointer to an Aps_JobHandle to receive the new job handle
 *
 *     Return: ApsResultObject.
 *             
 */
ApsResultObject ApsPrinterObject::StartJob(char *format, Aps_JobHandle *job)
{
    Aps_PrinterHandle printerHandle;
    
    printerHandle = (Aps_PrinterHandle)GetHandle();
    return (ApsResultObject(Aps_PrinterStartJob(printerHandle, format,job)));
}
