/* APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: apscpp.h
 *
 * Description: The declaration of the classes is done in this file to 
 *              implement the C++ interface for APS library.
 *
 */                   

#ifndef APSCPP_INTERFACE
#define APSCPP_INTERFACE
#include "aps.h"

/*-------------------------------------------------------------------------------- 
 * Class name : ApsResultObject 
 * Description: The ApsResultObject encapsulates the methods to operate on 
 *              Aps_Result type data member.
 */
class ApsResultObject{
    Aps_Result result;
 
public :
    ApsResultObject(Aps_Result value){result = value;}
    bool IsSucceded();
    ApsResultObject GetResultText(char *text, int size);
    void SetResultCode(Aps_Result code){result = code;}
    bool operator==(Aps_Result val){return result==val;}
    Aps_Result GetResultCode(){ return (Aps_Result)result;}
};             

/*-------------------------------------------------------------------------------- 
 * Class name : ApsObject  
 * Description: The ApsObject class acts as the base class for ApsPrinterObject,
 *             ApsQueueObject, ApsJobAttrObject, ApsJobObject, ApsFilterObject. 
 *             It encapsulates some general methods and the handle data member. 
 *             The derived class methods will operate on this handle. 
 */
class ApsObject{
    Aps_Handle handle;
 
public :
    ApsObject(){ handle = NULL;}
    ApsObject(const Aps_Handle apsHandle) { handle = apsHandle;}
    ApsObject(const ApsObject  &apsObj);
    Aps_Handle GetHandle()const { return handle;}
    ApsResultObject SetHandle(Aps_Handle apsHandle);
    ApsResultObject ReleaseBuffer(void *buffer);
    ~ApsObject();
 
protected :
    ApsResultObject ReleaseHandle();
    ApsResultObject AddRef();
    ApsResultObject IsOperationAvailable(Aps_OperationID operationID,
                                       Aps_Result *anticipatedResult);
};         

/*-------------------------------------------------------------------------------- 
 * Class name : ApsFilterObject  
 * Description: The ApsFilterObject class encapsulates the methods to operate on
 *              Aps_FilterHandle handle type.
 */
class ApsFilterObject: public ApsObject {

public :
    ApsFilterObject();
    ApsFilterObject(Aps_FilterHandle filterHandle);
    ApsFilterObject(const ApsFilterObject &apsFilter);
    ApsResultObject JobByOwnerName(Aps_FilterMode filterMode,
                                   char *jobOwnerName);
    ApsResultObject JobByHost(Aps_FilterMode filterMode, char *host);
    ApsResultObject JobByName(Aps_FilterMode filterMode, char *name);
    ApsResultObject JobByFilename(Aps_FilterMode filterMode, char *fileName) ;
    ApsResultObject JobByID(Aps_FilterMode filterMode, int id);
    ApsResultObject JobByCreationTime(Aps_FilterMode filterMode, 
                                                         time_t creationTime);
    ApsResultObject JobByFormat(Aps_FilterMode filterMode, char *format);
    ApsResultObject JobByOrder(Aps_FilterMode filterMode, int order);
    ApsResultObject JobByPriority(Aps_FilterMode filterMode, int priority);
    ApsResultObject JobByOwnerID(Aps_FilterMode filterMode, int ownerID);
    ApsResultObject JobByLocalHost(Aps_FilterMode filterMode, char *localHost);
    ApsResultObject JobByLocalFile(Aps_FilterMode filterMode, char *localFile);
    ApsResultObject JobBySpoolHost(Aps_FilterMode filterMode, char *spoolHost);
    ApsResultObject JobBySpoolFile(Aps_FilterMode filterMode, char *spoolFile);
    ApsResultObject JobByStatus(Aps_FilterMode filterMode,
                                   Aps_JobStatus jobStatus);
    ApsResultObject JobByPrinterName(Aps_FilterMode filterMode,
                                   char *jobPrinterName);
    ApsResultObject WithFunction(Aps_FilterMode filterMode,Aps_FilterFunc
                                matchFunction, void *userArg);
    ApsResultObject FilterAll(Aps_FilterMode filterMode);
    ApsResultObject Merge(Aps_FilterMode filterMode,Aps_FilterHandle srcHandle);
    ApsResultObject Clear();
    ApsResultObject Attach(Aps_Handle objHandle, Aps_FilterOptions filterOptions);
    ApsResultObject Detach(Aps_Handle objHandle, 
                                        Aps_FilterOptions *oldFilterOptions);
    ApsResultObject Check(Aps_FilterOptions filterOptions,
                                        Aps_Handle objHandle,int *matchesFilter);
    ~ApsFilterObject();                                 
};  

/*-------------------------------------------------------------------------------- 
 * Class name : ApsJobObject  
 * Description: The ApsJobObject class encapsulates the methods to operate on 
 *              Aps_JobHandle handle type.
 */
class ApsJobObject: public ApsObject {
public :
    ApsJobObject();
    ApsJobObject(Aps_JobHandle handle);
    ApsJobObject(const ApsJobObject &);
    ApsResultObject WriteJob(void *data, int size);
    ApsResultObject WriteBlock(Aps_BlockType blockType);
    ApsResultObject GetFileDescriptor(int *fileDescriptor);
    ApsResultObject End();
    ApsResultObject Abort();
    ApsResultObject Update();
    ApsResultObject GetStatus(Aps_JobStatus *status);
    ApsResultObject IssueCommand(Aps_OperationID operationCode, 
                                 void *reserved = NULL);
    ApsResultObject IssueCommand(Aps_OperationID operationCode);
    ApsResultObject MakeQuickJobInfo(Aps_QuickJobInfo **jobInfo);
    ~ApsJobObject();
};        

/*-------------------------------------------------------------------------------- 
 * Class name : ApsJobAttrObject  
 * Description: The ApsJobAttrObject class encapsulates the methods to operate 
 *              on Aps_JobAttrHandle handle type.
 */
class ApsJobAttrObject : public ApsObject{
public :
    ApsJobAttrObject();
    ApsJobAttrObject(const Aps_JobAttrHandle attrHandle);
    ApsJobAttrObject(const ApsJobAttrObject &jobAttr);
    ApsResultObject GetList(const char *group, int *numAttributes,
                                               char ***attributeIDs);
    ApsResultObject GetSubGroups(const char *group, int *numSubGroups,
                                 char ***subGroupNames);
    ApsResultObject GetTranslatedName(const char *attributeID,
                                     char **translatedName);
    ApsResultObject GetMainData(const char *attributeID, char **mainData);
    ApsResultObject GetTranslatedData(const char *attributeID,
                                      char **translatedData);
    ApsResultObject GetType(const char *attributeID, Aps_AttrType *attributeType);
    ApsResultObject GetRange(const char *attributeID,
                        double *minSetting, double *maxSetting);
    ApsResultObject GetOptions(const char*attributeID,
                          int *numOptions, Aps_AttrOption ***option);
    ApsResultObject GetSetting(const char *attributeID, char **setting);
    ApsResultObject SetSetting(const char *attributeID, const char *setting);
    ApsResultObject CheckConstraints(const char *attributeID, const char *setting,
                             char **conflictingAttribute, char **conflictingSetting);
    ApsResultObject QuickGetPostScriptLevel( Aps_PostScriptSupport  *languageLevel);
    ApsResultObject QuickIsColorDevice(int *colorDevice);
    ApsResultObject QuickGetResOptions(Aps_Resolution ***resolutions,
                              int *numResolutions);
    ApsResultObject QuickGetRes(Aps_Resolution *resolution);
    ApsResultObject QuickSetRes(const Aps_Resolution *resolution);
    ApsResultObject QuickGetMaxCopies(int *maxCopies);
    ApsResultObject QuickGetNumCopies(int *numCopies);
    ApsResultObject QuickSetNumCopies(int numCopies);
    ApsResultObject QuickGetCollationOptions(char ***collationOptions,
                                                        int *numOptions);
    ApsResultObject QuickGetCollation(char **collationSetting);   
    ApsResultObject QuickSetCollation(const char *collationSetting);
    ApsResultObject QuickGetPageSizeOptions(Aps_PageSize ***pageSizes,
                                             int *numPageSizes);
    ApsResultObject QuickGetCustomPageSizeLimits(
                        Aps_CustomPageSizeLimits **customPageSizeLimits);
    ApsResultObject QuickGetPageSize(Aps_PageSize **pageSize);
    ApsResultObject QuickSetPredefinedPageSize(const char *pageSizeID);
    ApsResultObject QuickSetCustomPageSize( const Aps_PageSize *customPageSize);
    ApsResultObject QuickGetInputSlotOptions(char ***inputSlots, 
                                                      int *numInputSlots);
    ApsResultObject QuickGetInputSlot(char **inputSlot);
    ApsResultObject QuickSetInputSlot(const char *inputSlot);
    ApsResultObject QuickGetOutputBinOptions(char ***outputBins,
                                    int *numOutputBins);
    ApsResultObject QuickGetOutputBin(char **outputBin);
    ApsResultObject QuickSetOutputBin(const char *outputBin);
    ApsResultObject QuickGetDuplexOptions(char ***duplexSettings,
                                                int *numDuplexSettings);
    ApsResultObject QuickGetDuplex(char **duplexSetting);
    ApsResultObject QuickSetDuplex(const char *duplexSetting);
    ApsResultObject QuickGetFonts(Aps_FontInfo ***fonts, int *numFonts);
    ~ApsJobAttrObject();
};                                               

/*-------------------------------------------------------------------------------- 
 * Class name : ApsPrinterObject  
 * Description: The ApsPrinterObject class encapsulates the methods to operate 
 *              on Aps_PrinterHandle handle type.
 */
class ApsPrinterObject: public ApsObject {
public :
 
    ApsPrinterObject();
        // Opens the default printer and initializes the handle.
    ApsPrinterObject(const Aps_PrinterHandle printerHandle);
        //Creates the object on the printerHandle.
    ApsPrinterObject(const ApsPrinterObject &apsPrinter);
    ApsPrinterObject(const char  *name);
       //Creates the printer object for the name passed as argument.
    ApsPrinterObject(Aps_ConnectionType connectionType, const char *location);
        /* This is meant for remote printer. This calls the Aps_OpenRemotePrinter
         * APS library function internally.
         */
    static ApsResultObject GetPrinters(char ***names, int *count);
    static ApsResultObject GetRemotePrinters(Aps_ConnectionType connectionType,
                 const char *subLocation, Aps_RemotePrinter ***remotePrinters);
    static ApsResultObject GetKnownManufacturers(char *** manufacturerNames,
                                               int *numManufacturers);
    static ApsResultObject GetKnownModels(const char *manufacturerName,
                                      char ***modelNames, int *numModels);
    static ApsResultObject AddModel(const char *manufacturer, const char *model,
                           Aps_ModelHandle *modelHandle);
    static ApsResultObject AddPrinter(const char *name, 
                                                Aps_PrinterHandle *printer);
    static ApsResultObject DispatchJobToDefault (const char *filename, char *format, 
                               Aps_JobAttrHandle jobAttributes, Aps_JobHandle *job);
    ApsResultObject Remove();
    ApsResultObject GetName(char **name);
    ApsResultObject Rename(char *newName);
    ApsResultObject IsDefault(bool *isDefault);
    ApsResultObject GetMaxJobSize(int *maxSize);
    ApsResultObject SetMaxJobSize(int maxSize);
    ApsResultObject GetConfigFlags(long int *configFlags);        
    ApsResultObject SetConfigFlags(long int flagsToSet, long flagsToReset);
    ApsResultObject GetModel(char ** manufacturer, char **model);
    ApsResultObject SetModel(const char *manufacturer, const char *model);
    ApsResultObject GetPPDFileName(char **filename);
    ApsResultObject SetPPDFileName(const char *filename);
    ApsResultObject GetConnectInfo(Aps_ConnectionType *connectionType,
                                   char **location);
    ApsResultObject GetModelHandle(Aps_ModelHandle *model);
    ApsResultObject SetConnectInfo(Aps_ConnectionType connectionType,
                                   const char *location);
    ApsResultObject GetDefAttr(Aps_JobAttrHandle *jobAttributes);
    ApsResultObject SetDefAttr(Aps_JobAttrHandle jobHandle);
    ApsResultObject DispatchJob(char *filename, char *format,
                             Aps_JobAttrHandle jobAttributes, Aps_JobHandle *job);
    ApsResultObject StartJob(char *format, Aps_JobHandle *job);
    ~ApsPrinterObject();
};                                                         

/*-------------------------------------------------------------------------------- 
 * Class name : ApsQueueObject  
 * Description: The ApsQueueObject class encapsulates the methods to operate on 
 *              ApsQueueHandle handle type.
 */
class ApsQueueObject : public ApsObject {
public :
    ApsQueueObject();  // Default constructor
    ApsQueueObject(Aps_QueueHandle queueHanle);
    ApsQueueObject(const ApsQueueObject &);
    ApsResultObject IssueCommand(Aps_OperationID operationCode, void *reserved);
    ApsResultObject IssueQuery(Aps_OperationID operationCode, void *reserved);
    ApsResultObject GetNumberOfJobs(int *numberOfJobs);
    ApsResultObject MakeQuickPrinterQInfoArray(
                    Aps_QuickPrinterQInfo ***printerQueueInfo, int *numEntries);
    ApsResultObject OpenGlobalQueue();
    ~ApsQueueObject();
};                                
#endif
