/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: cupstransport.c
 *
 * Description: Implementation specific to the CUPS transport abstraction
 *              layer.
 *
 */
#include "apsinternal.h"
#if (APSCFG_CUPS_BUILD)

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include "aps.h"
#include "cupstransport.h"
#include "cupsprinter.h"
#include "job.h"
#include <dlfcn.h>
#include <stdlib.h>

DEBUG_CHANNEL_DEFAULT(cups)

/* C++-like virtual functions for the Cups transport. */
static int CupsIsPrinterKnown(Transport * thisBase, const char *name);
static Printer *CupsCreatePrinterInstance(Transport * thisBase, char *name);
static char *CupsGetDefaultPrinterName(Transport * thisBase);
static char *CupsGetPPDFileName(Transport * thisBase, Printer * printer);
static Aps_Result CupsSetPPDFileName(Transport * thisBase, Printer * printer,
                                     char *name);
/* Printer object manipulation
 */
static Aps_Result CupsGetPrinters(Transport * thisBase,
    TrackArray_PtrChar *names);

/* Printer configuration and browsing
 */
/* Job dispatch
 */
static Aps_Result CupsDispatchJob(Transport * thisBase,
                                  Aps_PrinterHandle handle,
                                  char *filename, char *format,
                                  Aps_JobAttrHandle jobAttributes,
                                  Aps_JobHandle * job);
static Aps_Result CupsAbortJob(Aps_JobHandle handle);

/* Queue handling
 */
static Aps_Result CupsPurgeQueue(Aps_PrinterHandle handle, int all);

static int CupsGetLibraryFnPtrs(CupsTransport * thisBase);
static TransportVtbl cupsTransportVtbl = {
    CupsIsPrinterKnown,
    CupsCreatePrinterInstance,
    CupsDispatchJob,
    CupsGetPrinters,
    CupsGetDefaultPrinterName,
    CupsPurgeQueue,
    CupsGetPPDFileName,
    CupsSetPPDFileName
};

/* ---------------------------------------------------------------------------
 * CupsGetLibraryFnPtrs ()
 *
 * Gets the function pointers for the CUPS library functions.
 *
 * Parameters:  thisBase - pointer to CupsTransport structure.
 *
 *     Return: TRUE if the printer is known to this transport,
 *             FALSE if it is not.
 */
static int CupsGetLibraryFnPtrs(CupsTransport * thisBase)
{
    /*
     * Check whether it is already loaded.
     */
    if (thisBase->CupsInterface.pSharedObject != NULL)
        return TRUE;

    /*
     * Try to load it.
     */
    thisBase->CupsInterface.pSharedObject = dlopen("libcups.so.1", RTLD_LAZY);
    if (thisBase->CupsInterface.pSharedObject == NULL)
        return FALSE;

    /*
     * Obtain pointers to functions that we will require.
     */
    thisBase->CupsInterface.cupsGetPrinters =
        dlsym(thisBase->CupsInterface.pSharedObject, "cupsGetPrinters");
    thisBase->CupsInterface.cupsPrintFile =
        dlsym(thisBase->CupsInterface.pSharedObject, "cupsPrintFile");

    thisBase->CupsInterface.cupsGetDefault =
        dlsym(thisBase->CupsInterface.pSharedObject, "cupsGetDefault");
    thisBase->CupsInterface.cupsGetPPD =
        dlsym(thisBase->CupsInterface.pSharedObject, "cupsGetPPD");
    /*
     * If we failed to obtain any of the required function pointers, then
     * we discard the shared library and tell the caller that it cannot be
     * used.
     */
    if ((thisBase->CupsInterface.cupsGetPrinters == NULL) ||
        (thisBase->CupsInterface.cupsPrintFile == NULL) ||
        (thisBase->CupsInterface.cupsGetPPD == NULL) ||
        (thisBase->CupsInterface.cupsGetDefault == NULL)) {
        dlclose(thisBase->CupsInterface.pSharedObject);
        thisBase->CupsInterface.pSharedObject = NULL;
        return FALSE;
    }
    return TRUE;
}

/* ---------------------------------------------------------------------------
 * CupsIsPrinterKnown()
 *
 * Determines whether or not the specified printer is recognized by this
 * transport.
 *
 * Parameters: thisBase - Pointer to the Transport object for this
 *                        transport;
 *
 *             name     - The name of the printer to match.
 *
 *     Return: TRUE when the printer is known to this transport,
 *             FALSE when it is not,
 *             APS_GENERIC_FAILURE when no printer is attached.
 */
static int CupsIsPrinterKnown(Transport * thisBase, const char *name)
{
    int numberOfPrinters = 0, i = 0, found = FALSE;
    char **names;
    CupsTransport *this = (CupsTransport *) thisBase;

    if (this->CupsInterface.pSharedObject == NULL)
        if (CupsGetLibraryFnPtrs(this) != TRUE)
            return (FALSE);
    /* Get the count of number of printers */
    numberOfPrinters = this->CupsInterface.cupsGetPrinters(&names);
    if (numberOfPrinters == 0)
        return (APS_GENERIC_FAILURE);

    i = 0;
    /* Check if the printer is in the list of CUPS printer */
    while (i < numberOfPrinters) {
        if (strcmp(name, names[i]) == 0)
            return (TRUE);
        i++;
    }
    /* Free the memory occupied by names */
    if (numberOfPrinters > 0) {
        for (i = 0; i < numberOfPrinters; i++)
            free(names[i]);
        free(names);
    }
    return (FALSE);
}

/* ---------------------------------------------------------------------------
 * CupsCreatePrinterInstance()
 *
 * Opens a printer accessible through this transport.
 *
 * Parameters: thisBase - Pointer to the Transport object for this
 *                        transport.
 *
 *             name      - The name of the printer to open.
 *
 *     Return: Pointer to a new Printer, or NULL on failure.
 */
static Printer *CupsCreatePrinterInstance(Transport * thisBase, char *name)
{
    /* Create a new CupsPrinter instance associated with this transport. */
    return ((Printer *) CupsCreatePrinter(name, thisBase));
}

/* ---------------------------------------------------------------------------
 * CupsCreateTransport()
 *
 * Creates a new instance of the cups transport.
 *
 * Parameters: None.
 *
 *     Return: A pointer to CupsTransport object if CUPS daemon runnning and
 *             NULL otherwise.
 */
Transport *CupsCreateTransport()
{
    CupsTransport *transport = (CupsTransport *) calloc(1,
                                                        sizeof

                                                        (CupsTransport));
    if (!IsCupsRunning())
        return ((Transport *) NULL);
    transport->CupsInterface.pSharedObject = NULL;
    if (CupsGetLibraryFnPtrs(transport) != TRUE)
        return ((Transport *) NULL);
    transport->baseClass.vtbl = &cupsTransportVtbl;
    return ((Transport *) transport);
}

/* ---------------------------------------------------------------------------
 * CupsGetPrinters()
 *
 * Gets the name of the printers attached to CUPS.
 *
 * Parameters: thisBase - Pointer to the Transport object for this
 *                        transport.
 *            names    - pointer  to the array of printer names.
 *
 * Return:    APS_SUCCESS when the function returns the list of printer names
 *            APS_GENERIC_FAILURE when no printer is attached to CUPS,
 *            APS_OUT_OF_MEMORY on out of memory conditions.
 *
 * N.B. It is the caller's responsibility to free the array.
 */
Aps_Result CupsGetPrinters(Transport * thisBase, TrackArray_PtrChar *names)
{
    char **printerNames;        /* pointing to printer name array */
    int printerCount,           /* Count of printers in CUPS */
     i;                         /* Temporary loop counter */
    Aps_Result result;          /* Temporary result code holder */
    CupsTransport *this = (CupsTransport *) thisBase;

    /* if the Shared Object pointer to CUPS is NULL, get the CUPS 
     * Library function pointers 
     */
    if (this->CupsInterface.pSharedObject == NULL)
        if (CupsGetLibraryFnPtrs(this) != TRUE)
            return (APS_GENERIC_FAILURE);

    /* Obtain the list of all printers available through CUPS. */
    FIXME("Memory leak here...");
    printerCount = this->CupsInterface.cupsGetPrinters(&printerNames);
    if (printerCount == 0)      /* error */
        return (APS_GENERIC_FAILURE);

    /* Loop through all known printer names. */
    for (i = 0; i < printerCount; i++) {
        char *newString;
        ASSERT(printerNames[i] != NULL);
        newString = (char *)TrackMemDupString(*names, printerNames[i]);
        if (! newString) return APS_OUT_OF_MEMORY;
        if (! TrackArrayAddLast_PtrChar(names, newString))
            return APS_OUT_OF_MEMORY;
    }

    /* If we get to this point, then we successfully added the names of */
    /* the available printers (if any) to the array */
    return (APS_SUCCESS);
}

/* Job dispatch
 */

/* ---------------------------------------------------------------------------
 * CupsDispatchJob()
 *
 * Dispatches a print job to the CUPS printer.
 *
 * Parameters: thisBase      - Pointer to the Transport object for this
 *                             transport.
 *             handle        - printer handle.
 *             filename      - file to be printed.
 *             format        - type of the file (postscript etc.).
 *             settings      - job settings handle.
 *             job           - job handle.
 *
 *     Return: APS_INVALID_HANDLE when return value of PrinterGetPtrFromHandle
 *             is NULL.
 *             APS_GENERIC_FAILURE when return value of CupsGetLibraryFnPtr is 
 *             not TRUE.
 *             APS_SUCCESS otherwise.
 */
Aps_Result CupsDispatchJob(Transport * thisBase, Aps_PrinterHandle handle,
                           char *filename, char *format,
                           Aps_JobAttrHandle jobAttributes,
                           Aps_JobHandle * job)
{
    Printer *printerHandle;
    char *printerName, *pName = NULL;
    int noOfOptions = 0, jobID = 0;
    cups_option_t *options;

    CupsTransport *this = (CupsTransport *) thisBase;

    if ((printerHandle = PrinterGetPtrFromHandle(handle)) == NULL) {
        return (APS_INVALID_HANDLE);
    } else {
        /* Valid Printer Name, route the job request to the printer */
        if (this->CupsInterface.pSharedObject == NULL)
            if (CupsGetLibraryFnPtrs(this) != TRUE)
                return (APS_GENERIC_FAILURE);
        if (jobAttributes != NULL) {
            /*
               jobID = this->CupsInterface.cupsPrintFile(printerHandle->name, 
                       filename, 
               "cups", jobSettings->usedProperties, 
               (cups_option_t *)jobSettings->properties);
             */
	  /********* To be modified here for job Attributes ************/
            jobID =
                this->CupsInterface.cupsPrintFile(printerHandle->name,
                                                  filename, "cups", 0, NULL);
        } else {
            /* Settings is null */

            jobID =
                this->CupsInterface.cupsPrintFile(printerHandle->name,
                                                  filename, "cups", 0, NULL);
        }
    }
    /* The job should point to the Job structure , to be included later 
     * Currently set to NULL
     */
    job = NULL;
    return (APS_SUCCESS);
}

/* ---------------------------------------------------------------------------
 * CupsGetDefaultPrinterName()
 *
 * Obtains the name of the default printer.
 *
 * Parameters: thisBase  - Pointer to the Transport object for this
 *                         transport.
 *
 *     Return: A pointer to a string containing the default printer for this
 *             transport, or NULL on failure (such as if this transport
 *             doesn't have any printers installed at this time).
 *             Free with Aps_ReleaseBuffer()
 */
static char *CupsGetDefaultPrinterName(Transport * thisBase)
{
    CupsTransport *this = (CupsTransport *) thisBase;
    char *nameToReturn, *name;

    if (this->CupsInterface.pSharedObject == NULL)
        if (CupsGetLibraryFnPtrs(this) != TRUE)
            return (FALSE);

    nameToReturn = (char *)this->CupsInterface.cupsGetDefault();
    name = TrackMemDupString(NULL, nameToReturn, 0);
    return name;
}

/* ---------------------------------------------------------------------------
 * CupsPurgeQueue()
 *
 *
 * Parmaters: handle - Printer handle.
 *            all    - TRUE to remove all jobs from the queue or FALSE to only
 *                     remove those jobs owned by the current user.
 *
 * Return:    A standard Aps_Result code.
 * 
 */
Aps_Result CupsPurgeQueue(Aps_PrinterHandle handle, int all)
{
    FIXME("Not implemented");
    return (APS_NOT_IMPLEMENTED);
}

/* ---------------------------------------------------------------------------
 * CupsGetPPDFileName()
 *
 * Obtains the name of the PPD file associated with a particular printer,
 * if any.
 *
 * Parmaters: this    - A pointer to the CUPS transport managing this
 *                      printer.
 *
 *            printer - A pointer to the structure of a printer being queried.
 *
 * Return:    A pointer to a newly allocated string containing the name
 *            of the PPD file, or NULL on failure. The string will be
 *            empty if no PPD file is associated with the printer.It is 
 *            callers' responsibility to free the memory pointed by the 
 *            filename pointer.
 *            Free with Aps_ReleaseBuffer()
 */
static char *CupsGetPPDFileName(Transport * thisBase, Printer * printer)
{
    const char *filename;
    char *newBuffer;

    CupsTransport *this = (CupsTransport *) thisBase;

    if (this->CupsInterface.pSharedObject == NULL)
        if (CupsGetLibraryFnPtrs(this) != TRUE)
            return (NULL);
    filename = this->CupsInterface.cupsGetPPD(printer->name);
    newBuffer = TrackMemDupString(NULL, filename, 0);
    return (newBuffer);
}

/* ---------------------------------------------------------------------------
 * CupsSetPPDFileName()
 *
 * Changes the PPD file associated with a particular printer.
 *
 * Parmaters: this    - A pointer to the CUPS transport managing this
 *                      printer.
 *
 *            printer - A pointer to the structure of printer being queried.
 *
 *            name    - A string containing the name of the PPD file to be
 *                      associated with this printer, or an empty string
 *                      if no PPD file should be associated with the
 *                      printer.
 *
 * Return:    A standard Aps_Result code.
 * 
 */
static Aps_Result CupsSetPPDFileName(Transport * thisBase, Printer * printer,
                                     char *name)
{
    FIXME("Not implemented");
    return (APS_NOT_IMPLEMENTED);
}

#endif /* APSCFG_CUPS_BUILD */
