/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: gsattrprovider.c
 *
 * Description: An attribute provider that gives access to GhostScript command
 *              line options.
 *
 */

#include <string.h>
#include <stdio.h>
 
#include "aps.h"
#include "apsinternal.h"
#include "jobattributes.h"
#include "gsattrprovider.h"
#include "metaconfig.h"
#include "object.h"

DEBUG_CHANNEL_DEFAULT(attrprov)

/* Prototypes for implementation of virtual functions. */
static Aps_Result GSAttrResetToModelDefaults(AttrProvider * this,
                                             Printer * printer);
static Aps_Result GSAttrSetToPrinterDefaults(AttrProvider * this,
                                             Printer * printer);

/* Virtual function pointer table for the generic attribute provider. */
static AttrProviderVtbl GSAttrProviderVtbl =
{
    AttrProvDestructor,
    AttrProvCreateCopy,
    AttrProvGetList,
    AttrProvGetSubGroups,
    AttrProvGetTranslatedName,
    AttrProvGetMainData,
    AttrProvGetTranslatedData,
    AttrProvGetType,
    AttrProvGetRange,
    AttrProvGetOptions,
    AttrProvGetSetting,
    AttrProvSetSetting,
    AttrProvCheckConstraints,
    AttrProvWriteBlock,
    GSAttrResetToModelDefaults,
    AttrProvSaveAsPrinterDefaults,
    GSAttrSetToPrinterDefaults,
    AttrProvRemovePrinterDefaults
};

/* Prototypes for private functions local to this module. */
static Aps_Result GSAttrSetFromModel(AttrProvider * this,
                                     Aps_ModelHandle model,
                                     const char *attributeID,
                                     const char *optionsField);
static Aps_Result GSAttrAddPageSize(AttrProvider * this,
                                    const char *pageSizeName,
                                    double width,
                                    double height);
static Aps_Result GSAttrReadPrinterDefault(AttrProvider * this,
                                           Printer * printer,
                                           const char *attributeID);

/* ---------------------------------------------------------------------------
 * GSAttrCreate()
 *
 * Create a new GhostScript Attribute Provider instance.
 *
 * Parameters: attrProvider     - The address of a GSAttrProvider * to
 *                                receive a pointer to the new attribute
 *                                provider.
 *
 *             associatedObject - The APS object that this attribute provider
 *                                is providing attributes to.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result GSAttrCreate(GSAttrProvider ** attrProvider,
                        Aps_Handle associatedObject)
{
    Aps_Result result;

    /* Allocate memory for a GS attribute provider. */
    *attrProvider = calloc(1, sizeof(GSAttrProvider));
    if (*attrProvider == NULL) {
        return APS_OUT_OF_MEMORY;
    }

    /* Setup pointer to virtual function pointer table. */
    (*attrProvider)->baseClass.vtbl = &GSAttrProviderVtbl;

    /* Call super-class initialization function. */
    result = AttrProvInitialize((AttrProvider *) * attrProvider,
                                associatedObject);
    if (result != APS_SUCCESS) {
        (*attrProvider)->baseClass.vtbl->Destructor(
                                             (AttrProvider *) attrProvider);
        return (result);
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * GSAttrResetToModelDefaults()
 *
 * Resets these job attributes to the defaults for the specified model of
 * printer, based on information retrieved from the model database.
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer with which the default
 *                       attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result GSAttrResetToModelDefaults(AttrProvider * this,
                                             Printer * printer)
{
    Aps_ModelHandle model;
    Aps_Result result;
    Attribute *attribute;
    const char *marginAttributes[] = {"LeftMargin", "RightMargin",
                                      "TopMargin", "BottomMargin"};
    int i;

    ASSERT(this != NULL);
    ASSERT(printer != NULL);

    /* Obtain the model object describing this type of printer. */
    model = printer->model;
    if (model == NULL)
        return APS_SUCCESS; /* No model, so nothing to do. */

    /* Setup the resolution attribute with the defaults for this model. */
    result = GSAttrSetFromModel(this, model, "*Resolution", "resolutions");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Setup the color rendering attribute with the defaults for this model. */
    result = GSAttrSetFromModel(this, model, "colorrendering",
                                "colorrendering");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Setup the GhostScript driver attribute with the driver associated with
     * this model.
     */
    result = GSAttrSetFromModel(this, model, "gsdevice", "gsdevice");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Setup a GhostScript options attribute with an empty setting value. */
    result = ObjectGetOrAddPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        "gsoptions", &attribute, this);
    if (result != APS_SUCCESS)
        return result;

    attribute->type = APS_ATTR_SETTING_STRING;

    result = AttrSetSetting(attribute, "");
    if (result != APS_SUCCESS)
        return result;

    /* Setup margin attributes. */
    for (i = 0; i < 4; ++i) {
        result = ObjectGetOrAddPropertyAttribute(
            ObjectGetPtrFromHandle(this->associatedObject),
            marginAttributes[i], &attribute, this);
        if (result != APS_SUCCESS)
            return result;

        attribute->type = APS_ATTR_SETTING_FLOAT;
        attribute->minSetting = 0;
        attribute->maxSetting = 72;

        result = AttrSetSetting(attribute, "0");
        if (result != APS_SUCCESS)
            return result;
    }

    /* Setup n-up attribute. */
    result = ObjectGetOrAddPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        "n-up", &attribute, this);
    if (result != APS_SUCCESS)
        return result;

    attribute->type = APS_ATTR_SETTING_PICK_ONE;

    result = AttrSetSetting(attribute, "1-up");
    if (result != APS_SUCCESS)
        return result;

    AttrRemoveAllOptions(attribute);

    result = AttrAddOption(attribute, "1-up", "1-up (Full page)", NULL);
    if (result != APS_SUCCESS)
        return result;
    
    result = AttrAddOption(attribute, "2-up", "2-up", NULL);
    if (result != APS_SUCCESS)
        return result;

    result = AttrAddOption(attribute, "4-up", "4-up", NULL);
    if (result != APS_SUCCESS)
        return result;

    result = AttrAddOption(attribute, "8-up", "8-up", NULL);
    if (result != APS_SUCCESS)
        return result;

    /* Setup standard page sizes. */
    result = ObjectGetOrAddPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        "*PageSize", &attribute, this);
    if (result != APS_SUCCESS)
        return result;
    attribute->type = APS_ATTR_SETTING_PICK_ONE;
    AttrRemoveAllOptions(attribute);
    result = AttrSetSetting(attribute, "Letter");
    if (result != APS_SUCCESS)
        return result;

    result = ObjectGetOrAddPropertyAttribute(
        ObjectGetPtrFromHandle(this->associatedObject),
        "*PaperDimension", &attribute, this);
    if (result != APS_SUCCESS)
        return result;
    attribute->type = APS_ATTR_READ_ONLY_DATA;
    AttrRemoveAllOptions(attribute);

    if (result != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Letter", 612, 792))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Legal", 612, 1008))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Statement", 396, 612))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Executive", 522, 756))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Slide", 527.76, 792))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "US Fanfold", 1071.36, 792))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "German Fanfold", 612, 864))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Folio", 612, 936))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Tabloid", 792, 1224))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Broad Sheet", 1296, 1728))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Quarto", 609.45, 779.53))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A0", 2383.937, 3370.394))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A1", 1683.78, 2383.937))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A2", 1190.551, 1683.78))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A3", 841.89, 1190.551))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A4", 595.276, 842.89))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A5", 419.528, 595.276))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "A6", 297.638, 419.528))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "B1", 2004.094, 2834.646))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "B4", 708.661, 1000.63))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "B5", 498.898, 708.661))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "C3", 918.425, 1298.268))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "C4", 649.134, 918.425))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "C5", 459.213, 649.134))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "C6", 323.15, 459.213))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "#9 Envelope", 639.36, 279.36))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "#10 Envelope", 684, 297.36))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "#11 Envelope", 747.36, 324))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "#12 Envelope", 792, 342))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "#14 Envelope", 828, 360))
        != APS_SUCCESS)
        return result;
    if ((result = GSAttrAddPageSize(this, "Monarch Envelope", 540, 279.36))
        != APS_SUCCESS);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * GSAttrSetFromModel()
 *
 * Sets up an individual attribute based on information obtained from the
 * provided model object.
 *
 * Parameters: this         - Pointer to the attributes provider to operate
 *                            on.
 *
 *             model        - A handle to the model object to initialize the
 *                            attribute from.
 *
 *             attributeID  - A string uniquely identifying the attribute
 *                            to be initialized.
 *
 *             optionsField - A string uniquely identifying the field in the
 *                            model object listing the valid options for
 *                            this attribute with this printer.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result GSAttrSetFromModel(AttrProvider * this,
                                     Aps_ModelHandle model,
                                     const char *attributeID,
                                     const char *optionsField)
{
    ApsObject *associatedObject;
    Attribute *attribute;
    char **modelFieldList = NULL;
    char *optionID;
    char *translatedName;
    int numElements;
    int i;
    Aps_Result result;
    PropertyType modelFieldType;
    char *modelFieldValue = NULL;

    /* Check whether the specified field exists in this model object. */
    if (!ObjectDoesPropertyExist(ObjectGetPtrFromHandle(model),
                                 optionsField)) {
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider. This APS object will
     * typically be a JobAttributes object.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* Obtain a pointer to the attribute object to be modified. */
    result = ObjectGetOrAddPropertyAttribute(associatedObject, attributeID,
                                             &attribute, this);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Check the type of the specified model object field. */
    modelFieldType = ObjectGetPropertyType(ObjectGetPtrFromHandle(model),
                                           optionsField);
    if (modelFieldType == PROPERTY_STRING) {
        /* Special-case handling for simple string fields. */

        /* Obtain the value of this simple string field. */
        result = Aps_GetPropertyString(model, optionsField, &modelFieldValue);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* Make this the current setting of the attributes. */
        result = AttrSetSetting(attribute, modelFieldValue);
        if (result != APS_SUCCESS)
            goto cleanup;

        /* Flag this attribute type as a string. */
        attribute->type = APS_ATTR_SETTING_STRING;

        /* We're done! */
        result = APS_SUCCESS;
        goto cleanup;

    } else if (modelFieldType != PROPERTY_STRING_ARRAY) {
        /* This should never be possible - only string and string array
         * properties can be read from the model database. */
        ASSERT(FALSE);
        result = APS_NOT_FOUND;
        goto cleanup;
    }

    /* Obtain the value of the corresponding field in the model object. */
    result = Aps_GetPropertyStrArray(model, optionsField, &modelFieldList,
                                     &numElements);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Remove all options that are currently listed in this attribute. */
    AttrRemoveAllOptions(attribute);

    /* Set the attribute type. */
    attribute->type = APS_ATTR_SETTING_PICK_ONE;
    
    /* Loop through the list of possible settings obtained from the model
     * object.
     */
    for (i = 0; i < numElements; ++i) {
        /* Extract the ID and translated name strings from the next
         * element in the list of settings.
         */
        optionID = modelFieldList[i];
        
        translatedName = strchr(optionID, ';');
        if (translatedName != NULL) {
            *translatedName = '\0';
            ++translatedName;
        }
        
        /* Attempt to add this option to the attribute. */
        result = AttrAddOption(attribute, optionID, translatedName, NULL);
        if (result != APS_SUCCESS)
            goto cleanup;
    }

    /* Set the current setting for this attribute to be the first in the
     * list of options.
     */
    if (attribute->numOptions > 0) {
        result = AttrSetSetting(attribute, attribute->options[0].optionID);
        if (result != APS_SUCCESS)
            goto cleanup;
    }
    
    /* If we reach this point, we've successfully reset this attribute
     * based on the defaults provided by the specified model object.
     */
    result = APS_SUCCESS;
    
cleanup:
    /* Release any temporary resources allocated by this function. */
    if (modelFieldList != NULL)
        Aps_ReleaseBuffer(modelFieldList);

    if (modelFieldValue != NULL)
        Aps_ReleaseBuffer(modelFieldValue);

    return result;
}

/* ---------------------------------------------------------------------------
 * GSAttrAddPageSize()
 *
 * Adds a page size to the list of available page sizes.
 *
 * Parameters: this         - Pointer to the attributes provider to operate
 *                            on.
 *
 *             pageSizeName - The name of the page size to add.
 *
 *             width        - The width of this page size, in points.
 *
 *             height       - The height of this page size, in points.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result GSAttrAddPageSize(AttrProvider * this,
                                    const char *pageSizeName,
                                    double width,
                                    double height)
{
    ApsObject *associatedObject;
    Attribute *attribute;
    char dimensions[30];
    Aps_Result result;

    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider. This APS object will
     * typically be a JobAttributes object.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* Add this page size to the *PageSize attribute. */
    result = ObjectGetOrAddPropertyAttribute(associatedObject,
                                             "*PageSize",
                                             &attribute, this);
    if (result != APS_SUCCESS)
        return result;

    result = AttrAddOption(attribute, pageSizeName, NULL, NULL);
    if (result != APS_SUCCESS)
        return result;

    /* Store the dimensions of this page in the *PaperDimension read-only
     * attribute.
     */
    result = ObjectGetOrAddPropertyAttribute(associatedObject,
                                             "*PaperDimension",
                                             &attribute, this);
    if (result != APS_SUCCESS)
        return result;

    sprintf(dimensions, "%f %f", width, height);
    result = AttrAddOption(attribute, pageSizeName, NULL, dimensions);
    if (result != APS_SUCCESS)
        return result;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * GSAttrSetToPrinterDefaults()
 *
 * Initializes these job attributes based on the defaults for the specified
 * printer.
 *
 * Parameters: this    - Pointer to the attributes provider to operate on.
 *
 *             printer - A pointer to the printer with which the default
 *                       attributes should be associated.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result GSAttrSetToPrinterDefaults(AttrProvider * this,
                                             Printer * printer)
{
    Aps_Result result;

    /* Read the default setting saved for the resolution attribute. */
    result = GSAttrReadPrinterDefault(this, printer, "*Resolution");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the resolution attribute. */
    result = GSAttrReadPrinterDefault(this, printer, "colorrendering");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the GhostScript driver. */
    result = GSAttrReadPrinterDefault(this, printer, "gsdevice");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the GhostScript options. */
    result = GSAttrReadPrinterDefault(this, printer, "gsoptions");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the margin attributes. */
    result = GSAttrReadPrinterDefault(this, printer, "LeftMargin");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    result = GSAttrReadPrinterDefault(this, printer, "RightMargin");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    result = GSAttrReadPrinterDefault(this, printer, "TopMargin");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    result = GSAttrReadPrinterDefault(this, printer, "BottomMargin");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the resolution attribute. */
    result = GSAttrReadPrinterDefault(this, printer, "*PageSize");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    /* Read the default setting saved for the n-up attribute. */
    result = GSAttrReadPrinterDefault(this, printer, "n-up");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        return result;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * GSAttrReadPrinterDefault()
 *
 * Set the specified attribute to the default setting for this printer, if
 * a printer-specific default has been configured.
 *
 * Parameters: this        - Pointer to the attributes provider to operate on.
 *
 *             printer     - A pointer to the printer with which these
 *                           attributes are associated.
 *
 *             attributeID - A string uniquely identifying the attribute to
 *                           set.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
static Aps_Result GSAttrReadPrinterDefault(AttrProvider * this,
                                           Printer * printer,
                                           const char *attributeID)
{
    ApsObject *associatedObject;
    Attribute *attribute;
    char *defaultSetting;
    Aps_Result result;
    int i;
    
    /* Obtain a pointer to the APS object containing the attributes
     * implemented by this attribute provider. This APS object will
     * typically be a JobAttributes object.
     */
    associatedObject = ObjectGetPtrFromHandle(this->associatedObject);
    ASSERT(associatedObject != NULL);

    /* If the specified attribute doesn't exist, then there is nothing that
     * we need to do.
     */
    if (!ObjectDoesPropertyExist(associatedObject, attributeID))
        return APS_SUCCESS; /* Intentional; we gracefully skip this one. */

    /* Obtain a pointer to the attribute object to be modified. */
    result = ObjectGetOrAddPropertyAttribute(associatedObject, attributeID,
                                             &attribute, this);
    if (result != APS_SUCCESS)
        return result;

    /* Look for the default setting in this printer's meta configuration. */
    result = MetaRead(&defaultSetting, printer->name, attributeID);
    if (result != APS_SUCCESS)
        return result;

    /* Validate the current setting for this attribute. */
    switch (attribute->type) {
        case APS_ATTR_SETTING_PICK_ONE:
            /* Ensure that the specified setting matches one of the valid
             * options for this attribute.
             */
            for (i = 0; i < attribute->numOptions; ++i) {
                if (strcmp(attribute->options[i].optionID, defaultSetting) == 0)
                    break;
            }
    
            /* If we iterated through all options without finding one that
             * matches the default read from the configuration file, then the
             * configuration file contains an invalid setting for this
             * attribute, so ignore it.
             */
            if (i == attribute->numOptions) {
                free(defaultSetting);
                return APS_SUCCESS; /* We gracefully ignore this setting. */
            }
            break;
        default:
    }

    /* Attempt to change the current setting for this attribute. */
    result = AttrSetSetting(attribute, defaultSetting);
    free(defaultSetting);
    if (result != APS_SUCCESS)
        return result;
        
    return APS_SUCCESS;
}

