/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *        File: jobcontrol.c
 *
 * Description: APIs for active job control (eg. pause / resume)
 *              Covers jobs in APS_JOB_PHASE_PENDING / WORKING / ON_HOLD.
 *              [some operations may work in any phase but may be
 *               deferred]
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"
#include "filter.h"

DEBUG_CHANNEL_DEFAULT(job)

/* Local protos */
static Aps_Result CallJobControl(Aps_JobHandle jobHandle,
    Aps_OperationID operationCode, void *argin, void *argout);

/* --------------------------------------------------------------------
 * Aps_JobIssueCommand()
 *
 * Performs a single specific operation on a job.
 *
 * Parameters: jobHandle     - Handle of job on which to operate
 *
 *             operationCode - ID of operation to be performed
 *
 *             reserved      - RESERVED -- leave as NULL
 *
 *     Return: Detailed Aps_Result code for the application.
 */
Aps_Result Aps_JobIssueCommand(
    Aps_JobHandle jobHandle, Aps_OperationID operationCode,
    void *reserved)
{
    /* Check parameters */
    if (reserved) return APS_INVALID_PARAM; /* for now */

    /* Dispatch operation */
    return CallJobControl(jobHandle, operationCode, NULL, NULL);
}

/*** This operation is not currently provided but might be in the
 *** near future.  We'll just ask the preprocess to not include
 *** it at this time.
 ***/
#if 0
/* --------------------------------------------------------------------
 * Aps_JobIssueQuery()
 *
 * Performs a single specific verification request on a job.
 *
 * Parameters: jobHandle   - Handle of job on which to operate
 *
 *             queryCode   - ID of query command to be performed
 *
 *             output      - Result of query.
 *
 *     Return: Detailed Aps_Result code for the application.
 */
Aps_Result Aps_JobIssueQuery(
    Aps_JobHandle jobHandle, Aps_OperationID queryCode,
    void *output)
{
    /* Check parameters */
    if (! output) return APS_INVALID_PARAM;

    /* Dispatch operation */
    return CallJobControl(jobHandle, queryCode, NULL, output);
}
#endif

/* ---------------------------------------------------------------------------
 * CallJobControl()
 *
 * Low-level sub for Job control / queries.  Passes the message
 * back to the transport for processing.
 *
 * Parameters : jobHandle      - handle of job to access
 *              operationCode  - one of Aps_OperationID
 *              argin          - input arguments
 *              argout         - output arguments
 *
 * Returns    : A standard Aps_Result code indicating success / failure.
 */
static Aps_Result CallJobControl(Aps_JobHandle jobHandle,
    Aps_OperationID operationCode, void *argin, void *argout)
{
    Job *job;
    job = JobGetPtrFromHandle(jobHandle);
    if (! job) return APS_INVALID_HANDLE;

    return job->transport->vtbl->JobControl(job->transport, job,
        operationCode, argin, argout);
}
