/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: lprdetect.c
 *
 * Description: Detection code for LPR implementations.
 *
 */
#include "apsinternal.h"
#if (APSCFG_LPR_BUILD)

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include "aps.h"
#include "lprdetect.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(lpr)

static Aps_Result LprBuildImplInfo(LprImplInfo *info);

/* --------------------------------------------------------------------
 * LprCreateImplInfo
 *
 * Build an LprImplInfo structure telling us about LPR configuration
 * and implementation type.
 *
 * Usually this will only be built once, so if a nasty user chooses
 * to switch things around in the middle, we're up the creek!
 *
 * Returns: (value)    Aps_Result indicating failure
 *          (in param) Pointer to new implementation info structure
 *                     NULL on failure (like if LPR is not installed)
 *
 * Warning: may fail if not root.
 */
Aps_Result LprCreateImplInfo(LprImplInfo **info) {
    Aps_Result result;

    /*** Initialize our structure ***/
    if (!(*info = calloc(1, sizeof(LprImplInfo))))
        return APS_OUT_OF_MEMORY;

    /*** Build it ***/
    result = LprBuildImplInfo(*info);
    if (result != APS_SUCCESS) {
        LprFreeImplInfo(*info);
        *info = NULL;
    }
    return result;
}

/* --------------------------------------------------------------------
 * LprFreeImplInfo
 *
 * Free LprImplInfo structure that was built by LprCreateImplInfo.
 *
 * Parameters: info    pointer to structure to be freed (NULL is okay)
 *
 * Returns: (value)    Aps_Result indicating failure
 *
 * This call always succeeds.
 */
Aps_Result LprFreeImplInfo(LprImplInfo *info) {
    if (info) {
        if (info->cmd_lpr) free(info->cmd_lpr);
        if (info->cmd_lprm) free(info->cmd_lprm);
        if (info->cmd_lpq) free(info->cmd_lpq);
        if (info->cmd_lpc) free(info->cmd_lpc);
        if (info->cmd_lpd) free(info->cmd_lpd);
        if (info->file_printcap) free(info->file_printcap);
        if (info->file_masterfilter) free(info->file_masterfilter);
        if (info->dir_spool) free(info->dir_spool);
        if (info->dir_filters) free(info->dir_filters);
    }
    return APS_SUCCESS;
}

/*** Build the LprImplInfo.
 * Performs appropriate tests.
 */
static Aps_Result LprBuildImplInfo(LprImplInfo *info) {
    Aps_Result result;
    char **cmdOutput;
    int    cmdNumLines;
    int    cmdResult;

    /*** These are not detected yet ***/
    if (!(info->dir_spool = strdup(APSCFG_LPR_PATH_SPOOL_DIR)))
        return APS_OUT_OF_MEMORY;
    if (!(info->dir_filters = strdup(APSCFG_LPR_PATH_FILTER_DIR)))
        return APS_OUT_OF_MEMORY;
    if (!(info->file_masterfilter = strdup(APSCFG_LPR_PATH_MASTERFILTER_FILE)))
        return APS_OUT_OF_MEMORY;

    /*** Try to find printcap ***/
    result = FindFile(APSCFG_LPR_PATH_PRINTCAP_SEARCHFILE,
        APSCFG_LPR_PATH_PRINTCAP_SEARCHDIR, & info->file_printcap);
    if (result != APS_SUCCESS) {
        info->file_printcap = strdup(APSCFG_LPR_PATH_PRINTCAP_DEFFILE);
        if (! info->file_printcap) return APS_OUT_OF_MEMORY;
    }

    /*** First, locate the binaries we need ***/
    result = FindProgram(APSCFG_LPR_PATH_LPR_SEARCHPGM,
        NULL, & info->cmd_lpr);
    if (result != APS_SUCCESS) return result;

    result = FindProgram(APSCFG_LPR_PATH_LPQ_SEARCHPGM,
        NULL, & info->cmd_lpq);
    if (result != APS_SUCCESS) return result;

    result = FindProgram(APSCFG_LPR_PATH_LPRM_SEARCHPGM,
        NULL, & info->cmd_lprm);
    if (result != APS_SUCCESS) return result;

    result = FindProgram(APSCFG_LPR_PATH_LPC_SEARCHPGM,
        NULL, & info->cmd_lpc);
    if (result != APS_SUCCESS) return result;

    /* If we can't find LPD, it's not a big deal */
    result = FindProgram(APSCFG_LPR_PATH_LPD_SEARCHPGM,
        NULL, & info->cmd_lpd);
    if (result != APS_SUCCESS) info->cmd_lpd = NULL;

    /*** Now, probe to find the implementation ***/

    /* Returned on first line:
     *  lpr     -- lpr: . is a directory
     *  lprng   -- LPRng Version LPRng-3.5.1 Copyright 1988-1997
     */
    result = GetCommandOutput(info->cmd_lpr, "-V .", NULL,
        APSCFG_LPR_TIMEOUT_LPR_VERSION, &cmdOutput, &cmdNumLines, &cmdResult);

    if (result == APS_SUCCESS) {
        const char *vinfo;
        int line;

        /* Scan all lines of output */
        result = APS_PARTIAL_SUCCESS; /* preset flag */
        for (line = 0; line < cmdNumLines; line++) {
            if ( (vinfo = strstr(cmdOutput[line], "LPRng")) ) {
                /*** Assume it's LPRng [would be strange otherwise] ***/
                /* Locate version code, if present
                 */
                if ( (vinfo = strstr(cmdOutput[line], "Version LPRng-")) ) {
                    int vM = 0, vm = 0, vl = 0;
                    vinfo += 14; /* skip to number */
                    sscanf(vinfo, "%d.%d.%d", &vM, &vm, &vl);
                    info->version = (vM << 24) | (vm << 16) | (vl << 8);
                } else info->version = 0;
                /* Add more rigourous LPRng tests here...
                 */
                info->impl = LPR_IMPL_LPRNG;
                result = APS_SUCCESS; /* flag */
                break;
            }
        }
    }

    Aps_ReleaseBuffer(cmdOutput);

    if (result == APS_PARTIAL_SUCCESS) {
        /*** Assume it's BSD [safe but not secure assumption] ***/
        /* We really don't know right now. Each version of Berkeley
         * LPR is bound to have its own side-effects which may be
         * difficult to pin down. If any other transports come to
         * light, then we'll test them individually before making
         * this assumption.
         */
        /* Add more rigourous LPR tests here...
         */
        info->impl = LPR_IMPL_BSD;
        info->version = 0;
        result = APS_SUCCESS;
    }

    if (result != APS_SUCCESS) {
        /*** Assume it's NOT BSD because LPR should not fail this
         * test.  But it 'looks' like LPR is installed, so we'll
         * just return LPR_COMPLIANT because we don't know
         * otherwise
         */
        /* Add more rigourous LPR compliance tests here...
         */
        info->impl = LPR_IMPL_COMPLIANT;
        info->version = 0;
        result = APS_SUCCESS;
    }
    return result;
}

#endif /* APSCFG_LPR_BUILD */
