/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: lprtransportsup.c
 *
 * Description: Support code for lprtransport.c.  Keeps from bogging down
 *              the source file with external interface junk.
 *
 */
#include "apsinternal.h"
#if (APSCFG_LPR_BUILD)

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>

#include "aps.h"
#include "lprtransport.h"
#include "lprprinter.h"
#include "lprdetect.h"
#include "job.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(lpr)

/**************************************************
 *** SUPPORT CODE FOR ANY IMPLEMENTATION OF LPR ***
 **************************************************/

static Aps_Result LprAnyPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array);

/* --------------------------------------------------------------------
 * LprAnyPostEntry()
 *
 * Posts (adds) a completed job entry to a table. Assumes there is nothing
 * left to be done to the entry.
 *
 * Parameters : info      - new Aps_QueueInfo
 *              array     - array to which the job will be added
 *
 * Result     : A standard APS result code indicating success or failure.
 */
static Aps_Result LprAnyPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array)
{
    Aps_QuickJobInfo *package;
    int   packageSize;
    void *storage;

    /* Add package to array */
    packageSize = QuickJobInfoComputePackageSize(info);
    storage = (void *)TrackMemAlloc(*array, packageSize, 0);
    package = TrackArrayIndirectAddLastByRef_QuickJobInfo(array, NULL);
    if (! package) {
        ERROR("Couldn't post job queue entry: entry will be dropped...");
        return APS_OUT_OF_MEMORY;
    }

    QuickJobInfoFillInPackage(package, storage, info);
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprAnyGetQueueStatus()
 *
 * Obtains queue status in one shot without a job info.
 *
 * Parameters : lpr       - transport object pointer
 *              printer   - printer object pointer
 *
 * Result     : Aps_PrinterStatus, APS_PRINTER_UNKNOWN on failure.
 */
Aps_PrinterStatus LprAnyGetQueueStatus(LprTransport *lpr, Printer *printer)
{
    Aps_QuickJobInfo info;
    Aps_PrinterStatus status;
    Aps_Result result;

    /* Initialize information for function call */
    QuickJobInfoInit(& info, 0);

    /* Setup printerName and printerHandle */
    info.printerHandle = PrinterGetHandleFromPtr(printer);
    info.printerStatus = APS_PRINTER_UNKNOWN;
    if (strupdate(& info.printerName, printer->name)) {
        /* Get queue status */
        switch (lpr->config->impl) {
            case LPR_IMPL_MINIMAL:
                result = APS_NOT_SUPPORTED;
                break;
            case LPR_IMPL_COMPLIANT:
            case LPR_IMPL_BSD:
                result = LprBSDGetQueueStatus(lpr, & info);
                break;
            case LPR_IMPL_LPRNG:
                result = LprNGGetQueueStatus(lpr, & info);
                break;
            default:
                result = APS_GENERIC_FAILURE;
        }
    } else result = APS_OUT_OF_MEMORY;

    /* Cleanup */
    status = info.printerStatus;
    QuickJobInfoCleanup(& info);

    /* Return dummy status */
    if (result != APS_SUCCESS) status = APS_PRINTER_READY;
    return status;
}

/*************************************
 *** SUPPORT CODE FOR BERKELEY LPR ***
 *************************************/

static Aps_Result LprBSDPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array);
static Aps_Result LprBSDParseLine1(Aps_QuickJobInfo *info,
    char *buffer, char *index);
static Aps_Result LprBSDParseLine2(Aps_QuickJobInfo *info, char *buffer);
static Aps_Result LprBSDParseQueueLines(Aps_QuickJobInfo *info,
    char **bufLine, int bufNumLines, TrackArrayIndirect_QuickJobInfo *array);

/* --------------------------------------------------------------------
 * LprBSDReadQueue()
 *
 * Read queue and sent to LprBSDParseQueueLines to build a table of entries
 *
 * Parameters : lpr       - transport pointer
 *              protoinfo - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus
 *                            protoinfo->jobID (optional)
 *              array     - array to which job table will be appended
 *
 * Result     : A standard APS result code indicating success or failure.
 */
Aps_Result LprBSDReadQueue(LprTransport *lpr,
    Aps_QuickJobInfo *protoinfo, TrackArrayIndirect_QuickJobInfo *array)
{
    char   *commandArgs;
    int     commandResult = 0;
    char  **bufLine;
    int     bufNumLines;
    Aps_Result result;

    ASSERT(lpr && protoinfo && array);

    /* If we have a JOB ID we'll filter out all those which don't
     * match it to save time
     */
    if (protoinfo->jobID != -1) {
        const char cmdArgFormat[] = " -l -P%s %6d";

        commandArgs = alloca(
            strlen(cmdArgFormat) - /* 1x %s, 1x %6d */ 5 + 1 /* null */
            + strlen(protoinfo->printerName)
            + 6 /* number */);
        sprintf(commandArgs, cmdArgFormat, protoinfo->printerName,
            protoinfo->jobID);

    /* Otherwise we will return all entries in that printer's queue */
    } else {
        const char cmdArgFormat[] = " -l -P%s";

        commandArgs = alloca(
            strlen(cmdArgFormat) - /* 1x %s, */ 2 + 1 /* null */
            + strlen(protoinfo->printerName));
        sprintf(commandArgs, cmdArgFormat, protoinfo->printerName);
    }

    /** CALL LPQ TO LOCATE THE PRINT JOB
     *
     * lpq -l -P<printer> [job #]
     *
     * <printer> is ready and printing
     * Warning: <printer> is down:
     * Warning: <printer> queue is turned off
     * Warning: no daemon present
     *
     * <owner>: <rank>                      [job ###<hostname>]
     *          <name>                      <size> bytes
     *          <name>                      <size> bytes
     *
     * <rank> is one of: "active", "1st", "2nd", "3rd", "4th", "#th".
     */
    result = GetCommandOutput( lpr->config->cmd_lpq, commandArgs, NULL,
        APSCFG_LPR_TIMEOUT_LPQ_READ, &bufLine, &bufNumLines, &commandResult);

    if (result != APS_SUCCESS) {
        /* We were unable to read all/part of the queue so we'll abort here
         */
        result = APS_SUBPROGRAM_FAILED;
    } else {
        /* We have the output from LPQ, try to parse it */
        result = LprBSDParseQueueLines(protoinfo, bufLine, bufNumLines, array);
    }

    if (bufLine) Aps_ReleaseBuffer(bufLine);
    return result;
}

/* --------------------------------------------------------------------
 * LprBSDParseQueueLines()
 *
 * Parses output from LPR's lpq command to produce a table of jobs
 * in the printer queue.  The table will only contain 'known' data and
 * requires additional processing to be useful to an application.
 *
 * Parameters : lpr       - transport pointer
 *              info      - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus
 *              bufLine   - array of char * containing lines to parse
 *              bufNumLines - # of lines in buffer
 *              array     - array to which jobs will be appended
 *
 * Result     : A standard APS result code indicating success or failure.
 */
static Aps_Result LprBSDParseQueueLines(Aps_QuickJobInfo *info,
    char **bufLine, int bufNumLines, TrackArrayIndirect_QuickJobInfo *array)
{
    int  line;
    Aps_Result result = APS_SUCCESS;

    /* Scan through all the lines...
     * 1. if line is the beginning of an entry...
     *    a. flush out old entry
     *    b. initialize storage for new entry
     *    c. extract owner name
     *    d. parse rank
     *    e. extract job number and hostname
     * 2. if an entry was started previously
     *    a. search backwards for "bytes"
     *    b. extract size before bytes
     *    c. presume remaining chars are part of filename
     *    d. add size to stored size and concatenate filenames to form
     *       a list delimited by '|' chars.
     * 3. if a line fails flush out any current entry to callback and skip
     */

    info->jobSize = -1; /* mark size as invalid */
    info->jobID   = -1; /* mark ID as invalid */

    for (line = 0; line < bufNumLines; line++) {
        char  buffer[256]; /* working buffer */
        char *index = buffer;
        char  c;

        /* strip out useless characters */
        strdelimit(buffer, bufLine[line], " \t", ' ');
        strtrim(buffer, buffer);

        /* 1. */
        /* looks for an entry ending with ':' */
        while ( (c = *(index++)) ) {
            if ((c == ' ')||(c == ':')) break;  /* too far! */
        }
        /*
         * if new entry... DO FIRST LINE
         */
        if ((c == ':') && (*index == ' ')) {
            /* 1a. & 1b. -- flush old & create new */
            if (info->jobSize != -1) {
                result = LprBSDPostEntry(info, array);
                if (result != APS_SUCCESS) {
                    ERROR("Failed to post entry, aborting on %s",
                        RESULTSTR(result));
                    break;
                }
            }
            result = QuickJobInfoRecycle(info); /* reuse */
            if (result != APS_SUCCESS) {
                ERROR("Failed to recycled temporary QuickJobInfo, aborting "
                      "on %s", RESULTSTR(result));
                break;
            }
            info->jobSize = -1; /* mark size as invalid */
            info->jobID   = -1; /* mark ID as invalid */

            /* go parse first line */
            result = LprBSDParseLine1(info, buffer, index);
        /*
         * if entry in progress... DO NEXT LINE
         */
        } else if (info->jobID != -1) {
            result = LprBSDParseLine2(info, buffer);
        /*
         * otherwise, throw away line
         */
        } else result = APS_IGNORED;

        /* 3. */
        /* test for errors -- if last line ignored, flush */
        if (result == APS_IGNORED) {
            if (info->jobSize != -1) {
                result = LprBSDPostEntry(info, array);
                if (result != APS_SUCCESS)
                    ERROR("Failed to post entry, aborting on %s",
                        RESULTSTR(result));
                info->jobSize = -1;
            } else result = APS_SUCCESS;
        }
        if (result != APS_SUCCESS) break;
    }
    /* post any remaining lines */
    if ((result == APS_SUCCESS) && (info->jobSize != -1)) {
        result = LprBSDPostEntry(info, array);
        if (result != APS_SUCCESS)
            ERROR("Failed to post entry, aborting on %s", RESULTSTR(result));
    }
    return result;
}


/* --------------------------------------------------------------------
 * LprBSDParseLine1()
 *
 * Parse first line of an entry from LPR's lpq command.
 * <owner>: <rank>                      [job ###<hostname>]
 *
 * Parameters : info      - Aps_QueueInfo to be updated
 *              buffer    - points to head of line
 *              index     - points to ' ' after colon
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if the line contained errors and was ignored;
 *              APS_SUCCESS if all went well.
 */
static Aps_Result LprBSDParseLine1(Aps_QuickJobInfo *info,
    char *buffer, char *index)
{
    struct passwd *pwd;
    char *line = index;

    /* 1c. -- get owner */
    index[-1] = 0; /* replace ':' with a NULL */
    strupdate(& info->ownerName, buffer);
    /* get user id, assumption: user is local (Berkeley LPR does
     * not directly support remote queueing) */
    pwd = getpwnam(info->ownerName);
    if (pwd) info->ownerID = pwd->pw_uid;

    /* 1d. -- get order */
    ++index; /* skip to next field */
    /* we know index points to '\0' or some valid field
     * if we can't match it, we just bail out. */
    if (strncmp(index, "active", 6) == 0) {
        info->jobOrder = 0;
        index += 6; /* skip to char after string */
    } else {
        char *endOfNumber;
        int number;
        char c, d;
        /* smarter than atoi() */
        number = (int) strtoul(index, & endOfNumber, 10);
        /* search for 'st', 'nd', 'rd', 'th' but not picky */
        index = endOfNumber;
        if ( (number > 0) &&
            (c = *(index++)) && (d = *(index++)) ) {
            if ( ((c == 's') && (d == 't')) ||
                ((c == 'n') && (d == 'd')) ||
                ((c == 'r') && (d == 'd')) ||
                ((c == 't') && (d == 'h')) ) {
            } else {
                WARN("Expected 1st, 2nd, 3rd, #th rank field: aborting\n"
                     "\"%s\"", line);
                return APS_IGNORED;
            }
        } else {
            WARN("Expected 1st, 2nd, 3rd, #th rank field: aborting\n"
                 "\"%s\"", line);
            return APS_IGNORED;
        }
        info->jobOrder = number;
    }

    /* 1e. -- get job id & hostname */
    /* next field looks like [job ###hostname] */
    if (strncmp(index, " [job ", 6) == 0) {
        char *endOfNumber;
        int number;
        index += 6; /* skip to # */
        /* extract number */
        number = (int) strtoul(index, & endOfNumber, 10);
        index = endOfNumber;
        if (number >= 0) {
            char c;
            /* extract hostname */
            while ( (c = *(index++)) ) {
                if (c == ']') { /* end of hostname */
                index[-1] = 0; /* over ] */
                strupdate(& info->jobHost, endOfNumber);
                break;
                }
            }
            if (c != ']') {
                WARN("Malformed id/host field of the form [job ###hostname] :"
                     " aborting\n\"%s\"", line);
                return APS_IGNORED;
            }
        }
        info->jobID = number;
    } else {
        WARN("Expected id/host field of the form [job ###hostname] :"
             " ignoring\n\"%s\"", line);
    }
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprBSDParseLine2()
 *
 * Parse second and subsequent lines of an entry from LPR's lpq command.
 *          <name>                      <size> bytes
 *
 * Parameters : info      - Aps_QueueInfo to be updated
 *              buffer    - points to head of line
 *              index     - points to ' ' after colon
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if the line contained errors and was ignored;
 *              APS_SUCCESS if all went well.
 */
static Aps_Result LprBSDParseLine2(Aps_QuickJobInfo *info, char *buffer)
{
    int   sizeBytes;
    char *temp = NULL;
    char *index = NULL;

    /* 2. -- check initial delimiter */
    if (*buffer != ' ') return APS_IGNORED;

    /* 2a. -- search for last occurance of "bytes" */
    /* don't issue warning here as we aren't sure if anything was
     * supposed to be here */
    index = buffer;
    while ( (temp = strstr(index + 1, "bytes")) ) index = temp;
    if (! index) return APS_IGNORED;

    /* 2b. -- extract size in bytes */
    if (*(--index) != ' ') {
        WARN("Expected field of the form ### bytes: ignoring\n"
             "\"%s\"", buffer);
        return APS_IGNORED; /* no delimiter? */
    }
    *index = '\0';
    index = strrchr(buffer, ' '); /* find last delimiter */

    if (! index) {
        WARN("Expected ### bytes field to be last on line: ignoring\n"
             "\"%s\"", buffer);
        return APS_IGNORED;
    }
    *(index++) = '\0'; /* split string at delimiter for filename */
    sizeBytes = (int) strtoul(index, & temp, 10);
    if ((sizeBytes < 0) || (*temp != '\0')) {
        WARN("Expected sensible filename and size fields: ignoring\n"
             "\"%s\"", buffer);
        return APS_IGNORED;
    }

    /* 2c. -- extract filename */
    /* no work to do! buffer+1 points to it right now */

    /* 2d. -- collect data */
    if (! strupdatejoin(& info->localFile, "|", buffer + 1))
        return APS_OUT_OF_MEMORY;
    if (info->jobSize == -1) info->jobSize = 0;
    info->jobSize += sizeBytes;

    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprBSDPostEntry()
 *
 * Posts (adds) a completed job entry to a table.  First updates any
 * uncertain fields in case they were omitted or otherwise not
 * supplied.
 *
 * Parameters : info      - new Aps_QueueInfo
 *              array     - array to which the job will be added
 *
 * Result     : A standard APS result code indicating success or failure.
 */
static Aps_Result LprBSDPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array)
{
    /* Before posting entry, fill in the blanks */
    FIXME("Unchecked allocations");
    strupdatehostname(& info->localHost);

    /* Assumptions:
     *   Berkeley LPR only used for local printing so set
     *   jobName = jobFilename = localFile
     *   spoolHost = localHost
     *   spoolFile = "" (short of checking the control data)
     */
    strupdate(& info->jobFilename, info->localFile);
    strupdate(& info->jobName, info->jobFilename);
    strupdate(& info->spoolHost, info->localHost);

    /* Set jobStatus to something meaningful */
    info->jobStatus = APS_JOB_UNKNOWN;
    if (info->jobOrder == 0) info->jobStatus = APS_JOB_PRINTING;
    else if (info->jobOrder > 0) info->jobStatus = APS_JOB_PENDING;

    /* Go post the entry */
    return LprAnyPostEntry(info, array);
}

/* --------------------------------------------------------------------
 * LprBSDGetQueueStatus()
 *
 * Obtains status of printer queue from LPR.
 *
 * Parameters : lpr       - transport pointer
 *              info      - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus (will be corrected)
 *
 * Result     : A standard APS result code indicating success or failure.
 */
Aps_Result LprBSDGetQueueStatus(LprTransport *lpr, Aps_QuickJobInfo *info)
{
    const char cmdArgFormat[] = " status %s";
    Aps_Result result;
    char   *commandArgs;
    int     commandResult = 0;
    char  **bufLine;
    int     bufNumLines;
    int     printerNameLen;

    /* Set status to UNKNOWN first */
    info->printerStatus = APS_PRINTER_UNKNOWN;

    /* Build command arguments */
    printerNameLen = strlen(info->printerName);
    commandArgs = alloca(
        strlen(cmdArgFormat) - /* 1x %s, */ 2 + 1 /* null */
        + printerNameLen);
        sprintf(commandArgs, cmdArgFormat, info->printerName);
    if (! commandArgs) return APS_OUT_OF_MEMORY;

    /** CALL LPC TO GET QUEUE / PRINTER STATUS
     *
     * lpc status [printer]
     *
     * <printer>:
     *         queuing is enabled   / queuing is disabled
     *         printing is enabled  / printing is disabled
     *         no entries           / [#] entry(ies) in spool area
     *         no daemon present    / lp is ready and printing
     */
    result = GetCommandOutput( lpr->config->cmd_lpc, commandArgs, NULL,
        APSCFG_LPR_TIMEOUT_LPC_STATUS, &bufLine, &bufNumLines,
        &commandResult);

    /* We were unable to read all/part of the information */
    if ((result != APS_SUCCESS)||(! bufLine)) result = APS_SUBPROGRAM_FAILED;
    else {
    /* Otherwise, scan through all the lines...
     * 1. if line is the beginning of a printer entry
     *    a. skip to next line
     *    b. search for "queu" and "enable" / "disable"     (DENYNEWJOBS)
     *    c. search for "print" and "enable" / "disable"    (SUSPEND)
     *    d. search for "no" / [#] and "entr"               (NUMJOBS)
     *    e. search for "no daemon", "ready" and "printing" (ACTIVE)
     *    f. end.
     * 2. otherwise skip to next line
     */
        /* 1. */
        int i;
        for (i = 0; i < bufNumLines; i++) {
            char *line = bufLine[i];
            /* fixme, trouble with aliases */
            if (strncmp(line, info->printerName, printerNameLen) == 0) {
                /* matched name, check for ':' */
                if (line[printerNameLen] == ':') break;
            }
        }
        /* 1a. */
        while (++i < bufNumLines) {
            char *line = bufLine[i];

            /* 1b. */
            if (strstr(line, "queu")) {
                if (strstr(line, "disable")) {
                    info->printerStatus |= APS_PRINTER_DENY_NEW_JOBS;
                    continue;
                }
                if (strstr(line, "enable")) {
                    info->printerStatus &= ~APS_PRINTER_DENY_NEW_JOBS;
                    continue;
                }
            }
            /* 1c. */
            if (strstr(line, "print")) {
                if (strstr(line, "disable")) {
                    info->printerStatus |= APS_PRINTER_SUSPENDED;
                    continue;
                }
                if (strstr(line, "enable")) {
                    info->printerStatus &= ~APS_PRINTER_SUSPENDED;
                    continue;
                }
            }
            /* 1d. */
            if (strstr(line, "entr")) {
                /* number of entries is discarded */
                continue;
            }
            /* 1e. */
            if (strstr(line, "no daemon")) {
                info->printerStatus |= APS_PRINTER_IDLE;
                continue;
            }
            if (strstr(line, "ready")) {
                info->printerStatus |= APS_PRINTER_READY;
                continue;
            }
            if (strstr(line, "printing")) {
                info->printerStatus |= APS_PRINTER_ACTIVE;
                continue;
            }
            break;
        }
    }

    /* release buffer */
    if (bufLine) Aps_ReleaseBuffer(bufLine);
    if (result != APS_SUCCESS) return result;

    return (info->printerStatus != APS_PRINTER_UNKNOWN) ?
        APS_SUCCESS : APS_IGNORED;
}

/******************************
 *** SUPPORT CODE FOR LPRNG ***
 ******************************/

static Aps_Result LprNGPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array);
static Aps_Result LprNGParseQueueLines(
    LprTransport *lpr, Aps_QuickJobInfo *info,
    char **bufLine, int bufNumLines, TrackArrayIndirect_QuickJobInfo *array);
static Aps_Result LprNGParseQualifiedName(char **index,
    char **qName, char **qHost, int *qNumber);
static Aps_Result LprNGParseJobStatus(Aps_QuickJobInfo *info,
    const char *buffer);
static Aps_Result LprNGParseJobSize(Aps_QuickJobInfo *info,
    const char *buffer);
static Aps_Result LprNGParseJobTime(Aps_QuickJobInfo *info,
    const char *buffer);
static Aps_Result LprNGParseCFileTag(Aps_QuickJobInfo *info,
    const char *buffer);
static Aps_Result LprNGParseHFileTag(Aps_QuickJobInfo *info,
    const char *buffer);

/* --------------------------------------------------------------------
 * LprNGReadQueue()
 *
 * Read queue and sent to LprNGParseQueueLines to build a table of entries
 *
 * Parameters : lpr       - transport pointer
 *              protoinfo - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus
 *                            protoinfo->jobID (optional)
 *              array     - array to which job table will be appended
 *
 * Result     : A standard APS result code indicating success or failure.
 */
Aps_Result LprNGReadQueue(LprTransport *lpr,
    Aps_QuickJobInfo *protoinfo, TrackArrayIndirect_QuickJobInfo *array)
{
    char   *commandArgs;
    int     commandResult = 0;
    char  **bufLine;
    int     bufNumLines;
    Aps_Result result;

    ASSERT(lpr && protoinfo && array);

    /* If we have a JOB ID we'll filter out all those which don't
     * match it to save time
     */
    if (protoinfo->jobID != -1) {
        const char cmdArgFormat[] = " -v -P%s %6d";

        commandArgs = alloca(
            strlen(cmdArgFormat) - /* 1x %s, 1x %6d */ 5 + 1 /* null */
            + strlen(protoinfo->printerName)
            + 6 /* number */);
        sprintf(commandArgs, cmdArgFormat, protoinfo->printerName,
            protoinfo->jobID);

    /* Otherwise we will return all entries in that printer's queue */
    } else {
        const char cmdArgFormat[] = " -v -P%s";

        commandArgs = alloca(
            strlen(cmdArgFormat) - /* 1x %s, */ 2 + 1 /* null */
            + strlen(protoinfo->printerName));
        sprintf(commandArgs, cmdArgFormat, protoinfo->printerName);
    }

    /** CALL LPQ TO LOCATE THE PRINT JOB
     *
     * lpq -v -P[printer] {[job #]}
     *
     * Printer: [printer]@[spoolHost]
     *  Printing: yes
     *  Aborted: yes
     *  Spooling: yes
     *  Queue: 4 printable jobs          / no printable jobs in queue
     *                                   / 1 printable job
     *   Holding: 1 held jobs in queue
     *  Server: pid [pid] active, Unspooler: pid [pid] active
     *  Server: no server active
     *  Status: <some status string>
     *  Filter_status: <????>
     *  Job: [ownerName]@[jobHost]+[job #] status= [status]
     *  Job: [ownerName]@[jobHost]+[job #] size= [size bytes]
     *  Job: [ownerName]@[jobHost]+[job #] time= [hh]:[mm]:[ss]
     *  Job: [ownerName]@[jobHost]+[job #] CONTROL=
     *  - <series of control tag lines here all prefixed with '-'>
     *  Job: [ownerName]@[jobHost]+[job #] HOLDFILE=
     *  - <series of holdfile tag lines here all prefixed with '-'>
     *
     * [status] one of:
     *   redirect->[host] : (APS_JOB_QUEUEING) redirecting to <host>
     *   [order]          : (order = #) job is #th in line to be printed
     *   active           : (order = 0) job is pending and server is up
     *   stalled(#sec)    : (order = 0) job stalled for # secs
     *   hold             : (APS_JOB_PAUSED) job has been held
     *   class            : (APS_JOB_PAUSED) job held by class restrictions
     *   done             : (APS_JOB_COMPLETED) job has finished
     *   ERROR: <string>  : (APS_JOB_FAILED) job stopped on an error
     *   remove           : (APS_JOB_DISCARDED) job has been removed
     */
    result = GetCommandOutput( lpr->config->cmd_lpq, commandArgs, NULL,
        APSCFG_LPR_TIMEOUT_LPQ_READ, &bufLine, &bufNumLines, &commandResult);

    if (result != APS_SUCCESS) {
        /* We were unable to read all/part of the queue so we'll abort here
         */
        result = APS_SUBPROGRAM_FAILED;
    } else {
        /* We have the output from LPQ, try to parse it */
        result = LprNGParseQueueLines(lpr, protoinfo, bufLine,
            bufNumLines, array);
    }

    if (bufLine) Aps_ReleaseBuffer(bufLine);
    return result;
}

/* --------------------------------------------------------------------
 * LprNGParseQueueLines()
 *
 * Parses output from LPRng's lpq command to produce a table of jobs
 * in the printer queue.  The table will only contain 'known' data and
 * requires additional processing to be useful to an application.
 *
 * Parameters : lpr       - transport pointer
 *              info      - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus
 *              bufLine   - array of char * containing lines to parse
 *              bufNumLines - # of lines in buffer
 *              array     - array to which jobs will be appended
 *
 * Result     : A standard APS result code indicating success or failure.
 */
/* enum to tell us what we know about a job so far */
typedef enum LprNGParseStatus_ {
    LPRNG_PS_NOTHING  = 0,
    LPRNG_PS_PRINTER  = 0x01, /* have printer (printerName & spoolHost) */
    LPRNG_PS_JOB      = 0x02, /* have job (jobID, jobHost, ownerName) */
    LPRNG_PS_JSTATUS  = 0x04, /* have job status (jobStatus) */
    LPRNG_PS_JSIZE    = 0x08, /* have job size (jobSize) */
    LPRNG_PS_JTIME    = 0x10, /* have job time (jobCreationTime) */
    LPRNG_PS_JCONTROL = 0x20, /* have job control file */
    LPRNG_PS_JHOLD    = 0x40, /* have job hold file */
    LPRNG_PS_REQ      =       /* requirements for a complete job */
        LPRNG_PS_PRINTER | LPRNG_PS_JOB | LPRNG_PS_JSTATUS |
        LPRNG_PS_JSIZE | LPRNG_PS_JCONTROL
} LprNGParseStatus;
/* enum to tell us what we're doing at the moment */
typedef enum LprNGParseMode_ {
    LPRNG_PM_PRINTER,  /* searching for printer */
    LPRNG_PM_JOB,      /* searching for job */
    LPRNG_PM_JCONTROL, /* working on job control file */
    LPRNG_PM_JHOLD     /* working on job hold file */
} LprNGParseMode;

static Aps_Result LprNGParseQueueLines(
    LprTransport *lpr, Aps_QuickJobInfo *info,
    char **bufLine, int bufNumLines, TrackArrayIndirect_QuickJobInfo *array)
{
    int line;
    Aps_Result result = APS_SUCCESS;
    LprNGParseMode   pMode   = LPRNG_PM_PRINTER;
    LprNGParseStatus pStatus = LPRNG_PS_NOTHING;

    /* Scan through all the lines...
     * 1. if line does not start with ' '
     *    a. flush out old entry and clear flag
     *    b. if line starts with "Printer: "
     *       i. compare name with desired printer's name
     *       ii. set a flag indicating that we have found a printer
     *           entry and store spool host
     * 2. if we don't have the printer yet, ignore line
     * 3. if entry begins with " Job: "
     *    a. extract job owner's name, host and job #
     *    b. if not working on an entry, or if the job #'s don't match
     *       then flush out the old entry, set mode flag and store new info
     *    c. if info is "status= " call LprNGParseJobStatus(),
     *       if info is "size= " call LprNGParseJobSize(),
     *       if info is "time= " parse locally assumes today,
     *       if info is "CONTROL= " set mode flag for control file
     *       if info is "HOLDFILE= " set mode flag for hold file
     *       otherwise ignore line
     * 4. if entry begins with " - "
     *    a. if not working on an entry's control or hold file, ignore
     *    b. if working on control file, call LprNGParseCFileTag()
     *       if working on hold file, call LprNGParseHFileTag()
     */

    for (line = 0; line < bufNumLines; line++) {
        char  buffer[256]; /* working buffer */
        char *index = buffer;

        /* strip out useless characters */
        strdelimit(buffer, bufLine[line], " \t", ' ');
        strtrim(buffer, buffer);

        /* 1. -- handle lines not starting with ' ' */
        if (*index != ' ') {
            /* 1a. -- flush old entry if necessary */
            if ((pStatus & LPRNG_PS_REQ) == LPRNG_PS_REQ) {
                result = LprNGPostEntry(info, array);
            }
            pStatus = LPRNG_PS_NOTHING;
            pMode = LPRNG_PM_PRINTER;
            if (result != APS_SUCCESS) {
                ERROR("Failed to post entry, aborting on: %s",
                    RESULTSTR(result));
                break; /* uhoh! */
            }

            /* 1b. -- handle "Printer:" */
            if (strncmp(index, "Printer: ", 9) == 0) {
                char *printerName = NULL;
                char *spoolHost = NULL;

                index += 9; /* skip to name */
                result = LprNGParseQualifiedName(& index,
                    & printerName, & spoolHost, NULL);
                if (result == APS_IGNORED) continue; /* ignore line */
                if (result != APS_SUCCESS) {
                    ERROR("Failed to find qualified printer name, aborting\n");
                    break; /* uhoh! */
                }
                ASSERT(printerName && spoolHost);

                if (LprComparePrinterNames(lpr, printerName,
                    info->printerName)) {
                    /* 1c. */
                    strupdate(& info->spoolHost, NULL);
                    info->spoolHost = spoolHost;
                    pStatus = LPRNG_PS_PRINTER;
                    pMode = LPRNG_PM_JOB;
                } else strupdate(& spoolHost, NULL);
                continue; /* finished here */
            }
        }

        /* 2. -- if no printer, skip */
        if (! (pStatus & LPRNG_PS_PRINTER)) continue; /* ignore */

        /* 3. -- handle "Job:" */
        index++; /* we know buffer starts with ' ' */
        if (strncmp(index, "Job: ", 5) == 0) {
            int   jobID;
            char *ownerName = NULL;
            char *jobHost = NULL;
            int   newEntry = FALSE;

            /* 3a. -- get owner's name and compare */
            index += 5; /* skip to ownerName */
            pMode = LPRNG_PM_JOB; /* cancel holdfile / control modes */
            result = LprNGParseQualifiedName(& index,
                & ownerName, & jobHost, & jobID);
            if (result == APS_SUCCESS) {
                ASSERT(ownerName && jobHost);
                if ((! (pStatus & LPRNG_PS_JOB)) || (info->jobID != jobID))
                    newEntry = TRUE;
            } else jobID = -1;

            /* 3b. -- flush entry on trouble or new */
            if ((result != APS_SUCCESS)||(newEntry)) {
                LprNGParseStatus oldStatus = pStatus;

                /* flush */
                pStatus = LPRNG_PS_PRINTER;
                if ((oldStatus & LPRNG_PS_REQ) == LPRNG_PS_REQ) {
                    result = LprNGPostEntry(info, array);
                    if (result != APS_SUCCESS) {
                        ERROR("Failed to post entry, aborting on: %s",
                            RESULTSTR(result));
                        break; /* uhoh! */
                    }
                }
            }
            /* 3b. -- store new information */
            if (newEntry) {
                char *spoolHost = info->spoolHost;

                /* recycle but keep spoolHost */
                info->spoolHost = NULL;
                QuickJobInfoRecycle(info);
                info->spoolHost = spoolHost;
                /* store ID, owner and jobhost */
                info->jobID = jobID;
                strupdate(& info->ownerName, NULL);
                strupdate(& info->jobHost, NULL);
                info->ownerName = ownerName;
                info->jobHost = jobHost;
                ownerName = NULL;
                jobHost = NULL;
                /* fix internal status */
                pStatus |= LPRNG_PS_JOB;
            }
            /* free resources */
            if (ownerName) strupdate(& ownerName, NULL);
            if (jobHost) strupdate(& jobHost, NULL);

            /* if entry is not valid, ignore... */
            if (jobID == -1) continue;

            /* 3c. -- handle information fields */
            if (strncmp(index, " status= ", 9) == 0) {
                index += 9;
                result = LprNGParseJobStatus(info, index);
                if (result == APS_SUCCESS) pStatus |= LPRNG_PS_JSTATUS;
            } else if (strncmp(index, " size= ", 7) == 0) {
                index += 7;
                result = LprNGParseJobSize(info, index);
                if (result == APS_SUCCESS) pStatus |= LPRNG_PS_JSIZE;
            } else if (strncmp(index, " time= ", 7) == 0) {
                index += 7;
                result = LprNGParseJobTime(info, index);
                if (result == APS_SUCCESS) pStatus |= LPRNG_PS_JTIME;
            } else if (strncmp(index, " CONTROL=", 9) == 0) {
                pMode = LPRNG_PM_JCONTROL;
            } else if (strncmp(index, " HOLDFILE=", 10) == 0) {
                pMode = LPRNG_PM_JHOLD;
            } else result = APS_IGNORED; /* mark ignored */

            /* left this way to ease future enhancements... */
            if ((result == APS_SUCCESS) || (result == APS_IGNORED)) continue;
            break;
        }
        /* 4. -- handle lines starting with '-' */
        if (strncmp(index, "- ", 2) == 0) {
            index += 2;

            if (pMode == LPRNG_PM_JCONTROL) {
                /* 1a. */
                pStatus |= LPRNG_PS_JCONTROL;
                result = LprNGParseCFileTag(info, index);
            } else if (pMode == LPRNG_PM_JHOLD) {
                /* 1b. */
                pStatus |= LPRNG_PS_JHOLD;
                result = LprNGParseHFileTag(info, index);
            } else result = APS_IGNORED; /* mark ignored */
        }
        /* left this way to ease future enhancements... */
        if ((result == APS_SUCCESS) || (result == APS_IGNORED)) continue;
        break;
    }

    /* fix result codes */
    if (result == APS_IGNORED) result = APS_SUCCESS;
    /* flush last entry */
    if (result == APS_SUCCESS) {
        if ((pStatus & LPRNG_PS_REQ) == LPRNG_PS_REQ) {
            result = LprNGPostEntry(info, array);
        }
    }
    return result;
}

/* --------------------------------------------------------------------
 * LprNGParseQualifiedName()
 *
 * Parse a qualified name as returned by LPRng.
 * [name]@[host]+[#]
 *
 * Parameters : index  - points to first char of name
 *                       (will be updated to point to char following
 *                        last field)
 *              qName   - char ** will be updated with name field or ""
 *              qHost   - char ** will be updated with host field or ""
 *              qNumber - int * will be filled in with # or -1.
 *                        if NULL, assumes none is expected and
 *                        discards it if found
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if the line contained errors and was ignored;
 *              APS_SUCCESS if all went well.
 *
 * Remark : segment of original buffer containing qualified name may
 *          be trashed.
 */
static Aps_Result LprNGParseQualifiedName(char **index,
    char **qName, char **qHost, int *qNumber)
{
    char *temp;
    char c;
    ASSERT(index && qName && qHost);

    /* search for first '@', ' ', or '+' */
    temp = *index;
    while ( (c = *(temp++)) ) {
        if ((c == ' ')||(c == '@')||(c == '+')) break;
    }
    if (! c) {
        WARN("Expected ' ', '@', '+' or '\\0' in qualified name, got\n"
             "\"%s\"", *index);
        return APS_IGNORED;
    }

    /* extract name -- REQUIRE */
    temp[-1] = 0; /* kill space, '@' or '+' */
    if (! strupdate(qName, *index)) return APS_OUT_OF_MEMORY;
    *index = temp;

    /* look for host or number */
    strupdate(qHost, "");
    if (c == '@') {
        /* hostname follows, search for first ' ' or '+' */
        while ( (c = *(temp++)) ) {
            if ((c == ' ')||(c == '+')) break;
        }
        if (c != '\0') temp[-1] = 0; /* kill ' ' or '+' */
        if (! strupdate(qHost, *index)) return APS_OUT_OF_MEMORY;
        *index = temp;
    }

    /* look for number -- OPTIONAL */
    if (qNumber) {
        if (c == '+') {
            *qNumber = strtoul(*index, & temp, 10);
            if (*index == temp) {
                WARN("Expected numeric string, got\n\"%s\"", *index);
                return APS_IGNORED; /* invalid number! */
            }
            *index = temp;
        } else *qNumber = -1;
    }

    /* all went well */
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprNGParseJobStatus()
 *
 * Parses the status string produced by LPRng's lpq command.
 * Updates into with the supplied information.
 *
 * Parameters : info   - Aps_QueueInfo to update
 *              buffer - string containing string to be parsed
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if not recognized;
 *              APS_SUCCESS if updated.
 */
static Aps_Result LprNGParseJobStatus(Aps_QuickJobInfo *info,
    const char *buffer)
{
    ASSERT(info && buffer);

    /* reset status */
    info->jobStatus = APS_JOB_UNKNOWN;
    info->jobOrder = -1;

    /* get status */
    if (strncmp(buffer, "redirect", 8) == 0) {
        info->jobStatus = APS_JOB_QUEUEING;
    } else if (strncmp(buffer, "active", 6) == 0) {
        info->jobStatus = APS_JOB_PRINTING;
        info->jobOrder = 0;
    } else if (strncmp(buffer, "stalled", 7) == 0) {
        info->jobStatus = APS_JOB_PRINTING;
        info->jobOrder = 0;
    } else if (strncmp(buffer, "hold", 4) == 0) {
        info->jobStatus = APS_JOB_PAUSED | APS_JOB_PENDING;
    } else if (strncmp(buffer, "class", 5) == 0) {
        info->jobStatus = APS_JOB_PAUSED | APS_JOB_PENDING;
    } else if (strncmp(buffer, "done", 4) == 0) {
        info->jobStatus = APS_JOB_COMPLETED;
    } else if (strncmp(buffer, "ERROR", 5) == 0) {
        info->jobStatus = APS_JOB_FAILED;
    } else if (strncmp(buffer, "remove", 6) == 0) {
        info->jobStatus = APS_JOB_DISCARDED;
    } else { /* else get order in queue */
        char *end;
        int   order = strtoul(buffer, & end, 10);
        if ((buffer != end) && (! *end)) {
            info->jobStatus = APS_JOB_PENDING;
            info->jobOrder = order;
        } else return APS_IGNORED;
    }
    return APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprNGParseJobSize()
 *
 * Parses the size string produced by LPRng's lpq command.
 * Updates into with the supplied information.
 *
 * Parameters : info   - Aps_QueueInfo to update
 *              buffer - string containing string to be parsed
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if not recognized;
 *              APS_SUCCESS if updated.
 */
static Aps_Result LprNGParseJobSize(Aps_QuickJobInfo *info,
    const char *buffer)
{
    char *end;
    ASSERT(info && buffer);

    /* extract size */
    info->jobSize = (size_t)strtoul(buffer, & end, 10);
    return (end == buffer) ? APS_IGNORED : APS_SUCCESS;
}

/* --------------------------------------------------------------------
 * LprNGParseJobTime()
 *
 * Parses the time string produced by LPRng's lpq command.
 * Updates into with the supplied information.
 *
 * Parameters : info   - Aps_QueueInfo to update
 *              buffer - string containing string to be parsed
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if not recognized;
 *              APS_SUCCESS if updated.
 *
 * Caveats    : LPRng does not indicate the date, only the time of
 *              job creation.  We must therefore make the assumption that
 *              if the time appears to be in the future, it must have
 *              actually been yesterday.  Perhaps a better heuristic
 *              will be forthcoming later on...
 */
static Aps_Result LprNGParseJobTime(Aps_QuickJobInfo *info,
    const char *buffer)
{
    char *end;
    int hour,min,sec;
    time_t diff;
    time_t now;
    struct tm *tm;

    /* get today's time */
    now = time(NULL);
    tm = localtime(& now); /* today's time */
    if (tm) {
        /* extract components from string */
        hour = strtoul(buffer, & end, 10);
        if (*(end++) == ':') {
            min = strtoul(end, & end, 10);
            if (*(end++) == ':') {
                sec = strtoul(end, & end, 10);

                /* avoid playing with day/month/year data simply by
                 * using the difference in seconds to create a new
                 * time_t based on the one we got earlier */
                /* compute difference in seconds -- shift if in the past */
                diff = (tm->tm_hour - hour)*3600 + (tm->tm_min - min)*60 +
                    (tm->tm_sec - sec);
                /* deals with both signed and unsigned arithmentic */
                if ((diff < 0)||(diff > 86400)) diff += 86400;

                /* subtract difference from now to get a time_t */
                info->jobCreationTime = now - diff;
                return APS_SUCCESS;
            }
        }
    }
    return APS_IGNORED;
}

/* --------------------------------------------------------------------
 * LprNGParseCFileTag()
 *
 * Parses a single line from one of LPRng's spool C-files (control files).
 * Updates into with the supplied information.
 *
 * Parameters : info   - Aps_QueueInfo to update
 *              buffer - string containing tag to be parsed
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if not recognized or used;
 *              APS_SUCCESS if updated or accepted;
 *              APS_NOT_FOUND if no valid tag was located.
 */
static Aps_Result LprNGParseCFileTag(Aps_QuickJobInfo *info,
    const char *buffer)
{
    char tag;
    ASSERT(info && buffer);

    /* Expects buffer to point to the identifier char, and for the
     * string to be NULL-terminated and without extraneous control
     * characters or whitespace
     *
     * Each line looks like: <tag char><value>
     */

    /* Locate option tag */
    tag = *(buffer++);
    switch (tag) {
        /* A[id] */
        case 'A': /* ignored -- job identifier */
        /* I[indent] */
        case 'I': /* ignored -- indent, in characters */
        /* L[user name] */
        case 'L': /* ignored -- name to appear on banner page */
        /* M[eMail address] */
        case 'M': /* ignored -- address for confirmation email */
        /* U[file name] */
        case 'U': /* ignored -- name of file to unlink() when done */
        /* W[width] */
        case 'W': /* ignored -- page width in characters */
        /* Z[options] */
        case 'Z': /* ignored -- options for use by output filters */
        /* #[font file] */
        case '1': case '2': case '3': case '4': /* ignored -- troff fonts */
            return APS_IGNORED;

        /* C[class] */
        case 'C': { /* class/priority -- A = highest, Z = lowest */
                char class = *(buffer++);
                if ((! *buffer) && (class >= 'A') && (class <= 'Z')) {
                    int pri = 'A' - (int)class;
                    info->jobPriority = pri;
                    return APS_SUCCESS;
                }
            /* strange -- perhaps different format */
            } return APS_IGNORED;

        /* H[hostname] */
        case 'H': /* hostname -- where job was CREATED */
            return (strupdate(& info->jobHost, buffer) ?
                APS_SUCCESS : APS_OUT_OF_MEMORY);

        /* J[job name] */
        case 'J': /* job name from source */
            return (strupdate(& info->jobName, buffer) ?
                APS_SUCCESS : APS_OUT_OF_MEMORY);

        /* P[user name] */
        case 'P': /* login name of person who invoked lpr */
            info->ownerID = -1;
            return (strupdate(& info->ownerName, buffer) ?
                APS_SUCCESS : APS_OUT_OF_MEMORY);

        /* N[file name] */
        case 'N': /* file name of original source file */
            return (strupdatejoin(& info->jobFilename, "|", buffer) ?
                APS_SUCCESS : APS_OUT_OF_MEMORY);

        /* T[title] */
        case 'T': /* document title */
            return (strupdate(& info->docTitle, buffer) ?
                APS_SUCCESS : APS_OUT_OF_MEMORY);

        /* f[file name] -- formatted */
        /* p[file name] -- format with 'pr' */
        /* r[file name] -- formatted with FORTRAN ctrl chars */
        case 'f': case 'p': case 'r':
            return ((strupdatejoin(& info->spoolFile, "|", buffer) &&
                     strupdate(& info->jobFormat, APS_FORMAT_TEXT)) ?
                APS_SUCCESS: APS_OUT_OF_MEMORY);

        /* c[file name] -- 'cifplot' output */
        /* d[file name] -- 'dvi' output */
        /* g[file name] -- 'plot' output graph */
        /* t[file name] -- 'troff' output */
        case 'c': case 'd': case 'g': case 't':
            return ((strupdatejoin(& info->spoolFile, "|", buffer) &&
                     strupdate(& info->jobFormat, "")) ?
                APS_SUCCESS: APS_OUT_OF_MEMORY);

        /* l[file name] -- binary */
        /* v[file name] -- raster image */
        case 'l': case 'v':
            return ((strupdatejoin(& info->spoolFile, "|", buffer) &&
                     strupdate(& info->jobFormat, APS_FORMAT_NATIVE)) ?
                APS_SUCCESS: APS_OUT_OF_MEMORY);

        /* NULL? */
        case '\0':
            return APS_NOT_FOUND;

        /* Unknown tag... we'll accept them in case new ones are created */
        default:
            return APS_IGNORED;
    }
}

/* --------------------------------------------------------------------
 * LprNGParseHFileTag()
 *
 * Parses a single line from one of LPRng's spool H-files (hold files).
 * Updates into with the supplied information.
 *
 * Parameters : info   - Aps_QueueInfo to update
 *              buffer - string containing tag to be parsed
 *
 * Result     : A standard APS result code indicating success or failure.
 *              APS_IGNORED if not recognized or unused;
 *              APS_SUCCESS if updated or accepted;
 *              APS_NOT_FOUND if no valid tag was located.
 */
static Aps_Result LprNGParseHFileTag(Aps_QuickJobInfo *info,
    const char *buffer)
{
    const char *tag;
    const char *value;

    ASSERT(info && buffer);

    /* Expects buffer to point to the identifier string, and for the
     * string to be NULL-terminated and without extraneous control
     * characters or whitespace
     *
     * Each line looks like: <tagstring> <value>
     */

    /* Locate option tag */
    tag = buffer;
    /* Locate value */
    value = strchr(buffer, ' ');
    /* ensure that the delimiter was found, and that the next char
     * is valid */
    if ((! value) || (! *(++value))) return APS_NOT_FOUND;

    /*** Test for options (case-sensitive) ***/
    /* This format has been retained for ease of maintenance, not
     * code elegance. If the tag names were used for anything else,
     * we would create a structure mapping keys to enumerated
     * constants instead.
     */

#if 0  /*** vvv  NOT USED FOR NOW  vvv ***/
    /* active_time : time job became active (secs since Jan 1, 1970) */
    if (strncmp(tag, "active_time", 11) == 0) {
    } else
    /* attempt : ignored -- attempt # ?? */
    if (strncmp(tag, "attempt", 7) == 0) {
    } else
    /* copies : ignored -- number of copies left */
    if (strncmp(tag, "copies", 6) == 0) {
    } else
    /* copy_done : ignored -- time copy finished (secs since Jan 1, 1970) */
    if (strncmp(tag, "copy_done", 9) == 0) {
    } else
    /* done : ignored -- time job finished (secs since Jan 1, 1970) */
    if (strncmp(tag, "done", 4) == 0) {
    } else
    /* end : ignored -- ?? */
    if (strncmp(tag, "end", 3) == 0) {
    } else
    /* error : ignored -- ?? */
    if (strncmp(tag, "error", 5) == 0) {
    } else
    /* hold : time job was held (secs since Jan 1, 1970) */
    if (strncmp(tag, "hold", 4) == 0) {
    } else
    /* ident : ignored -- ?? */
    if (strncmp(tag, "ident", 5) == 0) {
    } else
    /* priority : ignored -- bumped by topq */
    if (strncmp(tag, "priority", 8) == 0) {
    } else
    /* receiver : ignored -- ?? */
    if (strncmp(tag, "receiver", 8) == 0) {
    } else
    /* redirect : ignored -- ?? */
    if (strncmp(tag, "redirect", 8) == 0) {
    } else
    /* remove : time job was removed (secs since Jan 1, 1970) */
    if (strncmp(tag, "remove", 6) == 0) {
    } else
    /* route : ignored -- ?? */
    if (strncmp(tag, "route", 5) == 0) {
    } else
    /* routed : time job was routed (secs since Jan 1, 1970) */
    if (strncmp(value, "routed", 6) == 0) {
    } else
    /* sequence : sequence # */
    if (strncmp(value, "sequence", 8) == 0) {
    } else
    /* server : ignored -- ?? */
    if (strncmp(value, "server", 6) == 0) {
    } else
    /* status : ignored -- ?? */
    if (strncmp(value, "status", 6) == 0) {
    } else
    /* subserver : ignored -- ?? */
    if (strncmp(value, "subserver", 9) == 0) {
    } else
#endif /*** ^^^  NOT USED FOR NOW  ^^^ ***/
    ;
    return APS_IGNORED;
}

/* --------------------------------------------------------------------
 * LprNGPostEntry()
 *
 * Posts (adds) a completed job entry to a table.  First updates any
 * uncertain fields in case they were omitted or otherwise not
 * supplied.
 *
 * Parameters : info      - new Aps_QueueInfo
 *              array     - array to which the job will be added
 *
 * Result     : A standard APS result code indicating success or failure.
 */
static Aps_Result LprNGPostEntry(Aps_QuickJobInfo *info,
    TrackArrayIndirect_QuickJobInfo *array)
{

    /* Before posting entry, fill in the blanks */
    strupdatehostname(& info->localHost);

    /* Assumptions:
     *  if localHost and spoolHost are the same,
     *   then localFile = jobFilename
     */
    if (strcmp(info->localHost, info->spoolHost) == 0) {
        strupdate(& info->localFile, info->jobFilename);
    }

    /* Go post the entry */
    return LprAnyPostEntry(info, array);
}

/* --------------------------------------------------------------------
 * LprNGGetQueueStatus()
 *
 * Obtains status of printer queue from LPRng.
 *
 * Parameters : lpr       - transport pointer
 *              info      - prototype Aps_QueueInfo w/
 *                            protoinfo->printerHangle
 *                            protoinfo->printerName
 *                            protoinfo->printerStatus (will be corrected)
 *
 * Result     : A standard APS result code indicating success or failure.
 */
Aps_Result LprNGGetQueueStatus(LprTransport *lpr, Aps_QuickJobInfo *info)
{
    const char cmdArgFormat[] = " status %s";
    Aps_Result result;
    char   *commandArgs;
    int     commandResult = 0;
    char  **bufLine;
    int     bufNumLines;
    int     printerNameLen;

    /* Set status to UNKNOWN first */
    info->printerStatus = APS_PRINTER_UNKNOWN;

    /* Build command arguments */
    printerNameLen = strlen(info->printerName);
    commandArgs = alloca(
        strlen(cmdArgFormat) - /* 1x %s, */ 2 + 1 /* null */
        + printerNameLen);
        sprintf(commandArgs, cmdArgFormat, info->printerName);
    if (! commandArgs) return APS_OUT_OF_MEMORY;

    /** CALL LPC TO GET QUEUE / PRINTER STATUS
     *
     * lpc status [printer]
     *
     *  Printer           Printing Spooling Jobs  Server   Slave Redirect
     *    Status
     * <printer@host>      enabled  enabled   #     pid#    pid#  none
     *
     * <printer@host>      disabled enabled   #     pid#    pid#  none
     *    (holdall)
     */
    result = GetCommandOutput( lpr->config->cmd_lpc, commandArgs, NULL,
        APSCFG_LPR_TIMEOUT_LPC_STATUS, &bufLine, &bufNumLines,
        &commandResult);

    /* We were unable to read all/part of the information */
    if ((result != APS_SUCCESS)||(! bufLine)) result = APS_SUBPROGRAM_FAILED;
    else {
    /* Otherwise, scan through all the lines...
     * 1. if line contains <printer>@<host>
     *    a. if entry does not match the one we want, skip it
     *    b. parse next "enable"/"disable" field            (SUSPEND)
     *    c. parse next "enable"/"disable" field            (DENYNEWJOBS)
     *    d. parse next [#] field                           (NUMJOBS)
     *    e. search for holdall flag
     *    f. end.
     * 2. otherwise skip to next line and try again
     */
        /* 1. */
        int i;
        for (i = 0; i < bufNumLines; i++) {
            /* fixme, trouble with aliases */
            if ((strncmp(bufLine[i], info->printerName, printerNameLen)
                == 0) && (bufLine[i][printerNameLen] == '@')) {
                char  buffer[256]; /* working buffer */
                char *index = buffer;

                /* strip out useless characters */
                strdelimit(buffer, bufLine[i], " \t", ' ');
                strtrim(buffer, buffer);

                /* 1b. skip host name and locate first enable/disable field */
                index = strchr(buffer + printerNameLen, ' ');
                if (! index) break;
                if (strncmp(index, " disabled", 9) == 0) {
                    info->printerStatus |= APS_PRINTER_SUSPENDED;
                    index += 9;
                } else if (strncmp(index, " enabled", 8) == 0) {
                    info->printerStatus &= ~APS_PRINTER_SUSPENDED;
                    index += 8;
                } else continue; /* skip */
                /* 1c. locate second enable/disable field */
                if (strncmp(index, " disabled", 9) == 0) {
                    info->printerStatus |= APS_PRINTER_DENY_NEW_JOBS;
                    index += 9;
                } else if (strncmp(index, " enabled", 8) == 0) {
                    info->printerStatus &= ~APS_PRINTER_DENY_NEW_JOBS;
                    index += 8;
                } else continue; /* skip */
                /* 1d. number of jobs is discarded */
                /* 1e. hold all */
                if (strstr(index, "(holdall)")) {
                    info->printerStatus |= APS_PRINTER_PAUSE_NEW_JOBS;
                } else {
                    info->printerStatus &= ~APS_PRINTER_PAUSE_NEW_JOBS;
                }
                /* 1f. end */
                break;
            }
        }
    }

    /* release buffer */
    if (bufLine) Aps_ReleaseBuffer(bufLine);
    if (result != APS_SUCCESS) return result;

    return (info->printerStatus != APS_PRINTER_UNKNOWN) ?
        APS_SUCCESS : APS_IGNORED;
}

#endif /* APSCFG_LPR_BUILD */
