/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: object.c
 *
 * Description: Operations that can be performed on any type of object.
 *
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "printer.h"
#include "jobattributes.h"
#include "job.h"
#include "queue.h"
#include "filter.h"
#include "model.h"
#include "object.h"
#include "attribute.h"

DEBUG_CHANNEL_DEFAULT(object)

/* Internal information stored for each generic property. */
typedef struct {
    PropertyType type;
    char *name;                          /* TrackMem obj */
    union {
        char               *string;      /* TrackMem obj */
        TrackArray_PtrChar  stringArray; /* TrackArray obj */
        Attribute          *attribute;
    } value;
} Property;

/* Private helper function prototypes. */
static void ObjectDestructor(ApsObject *object);
static Property *ObjectFindProperty(ApsObject *object, const char *name);
static Property *ObjectFindOrCreateProperty(ApsObject *object,
                                            const char *name);
static void ObjectDeleteProperty(Property *property);
                                            
/* ---------------------------------------------------------------------------
 * ObjectGetHandleFromPtr()
 *
 * Obtains a handle to a generic APS object given its ApsObject pointer.
 *
 * Parameters: object - A pointer to the object's base ApsObject.
 *
 *             type   - An APS HandleType identifying the type of object.
 *
 *     Return: A handle that the application can use to refer to this APS
 *             object.
 */
Aps_Handle ObjectGetHandleFromPtr(ApsObject *object)
{
    ASSERT(object != NULL);

    return (Aps_Handle)object;
}

/* ---------------------------------------------------------------------------
 * ObjectGetPtrFromHandle()
 *
 * Obtains the ApsObject from a handle, performing sanity checks to ensure
 * that this appears to be a valid handle.
 *
 * Parameters: handle - The handle for which the ApsObject should be returned.
 *
 *     Return: A pointer to the ApsObject, or NULL if the handle was deemed
 *             to be invalid.
 */
ApsObject *ObjectGetPtrFromHandle(Aps_Handle handle)
{
    ApsObject *object;

    /* If the handle is NULL, then deem it to be invalid. */
    if (handle == NULL) {
        return NULL;
    }

    /* If the handle is non-NULL, then obtain the ApsObject handle pointer */
    /* for this handle.                                                    */
    object = (ApsObject *)handle;

    if (   (object->identifier != PRINTER_HANDLE)
        && (object->identifier != JOB_HANDLE)
        && (object->identifier != JOB_ATTRIBUTES_HANDLE)
        && (object->identifier != MODEL_HANDLE)
        && (object->identifier != NOTIFICATION_HANDLE)
        && (object->identifier != FILTER_HANDLE)
        && (object->identifier != QUEUE_HANDLE) ) {
        return NULL;
    }

    /* As far as we can tell this is a valid object, so return its pointer */
    /* to the caller.                                                      */
    return object;
}

/* ---------------------------------------------------------------------------
 * ObjectInitialize()
 *
 * Initializes the base class information stored for any APS object type.
 *
 * Parameters: object - A pointer to the object's base ApsObject.
 *
 *             type   - An APS HandleType identifying the type of object.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result ObjectInitialize(ApsObject *object, HandleType type)
{
    /* Initialize the object type identifier to handles to this object
     * to later be identified as the appropriate type of handle.
     */
    object->identifier = type;

    /* We always initialize the reference count of any new object to 1,
     * as the caller immediately receives a reference to this object.
     */
    object->refCount = 1;

    /* We don't create an associated notification subscription array until
     * the client application actually subscribes to receive notifications.
     */
     object->notificationSubs = NULL;

    /* Likewise, we don't create an array of properties until a property
     * is set.
     */
    object->properties = NULL;

    /* We now set the default filter for the object to NONE. */
    object->filterHandle = NULL;
    object->filterOpts = 0;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ObjectDestructor()
 *
 * Called when any type of APS object is being destroyed. Performs cleanup
 * required for any object type, but does not release memory of the ApsObject
 * itself.
 *
 * Parameters: object - A pointer to the object's base ApsObject.
 *
 *     Return: None.
 */
static void ObjectDestructor(ApsObject *object)
{
    Property *property;

    ASSERT(object != NULL);

    /* If this object has any generic properties, clean them up. */
    if (object->properties != NULL) {
        while (PtrArrGetSize(object->properties) > 0) {
            /* Obtain a pointer to the next property. */
            property = (Property *)PtrArrGetAt(object->properties, 0);
            
            /* Remove this property from this object. */
            ObjectRemoveProperty(object, property->name);
        }

        /* Finally, delete the array of propery pointers. */
        PtrArrDelete(object->properties);
        object->properties = NULL;
    }

    /* Cleanup notification subscription information, if it was ever
     * created.
     */
    if (object->notificationSubs != NULL) {
        /* Detach all notification subscriptions from this object, deleting
         * the notification subscription if this is the last object
         * associated with this subscription.
         */
        while(PtrArrGetSize(object->notificationSubs) != 0) {
            NotificationSub *notificationSub = (NotificationSub *)
                PtrArrGetAt(object->notificationSubs, 0);
            NotificationDetach(notificationSub, object);
        }

        /* Delete the array of pointers to notification subscriptions. */
        PtrArrDelete(object->notificationSubs);
        object->notificationSubs = NULL;
    }

    /* Detach any filters if any had been previously attached */
    if (object->filterHandle != NULL) {
        Aps_ReleaseHandle(object->filterHandle);
    }
}

/* ---------------------------------------------------------------------------
 * Aps_AddRef()
 *
 * Adds another reference (lock) to an object. The object will only be
 * destroyed when all references to it have been released.
 *
 * Parameters: handle - The handle for which a reference should be added.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_AddRef(Aps_Handle handle)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    TRACE("(%p) objtype = %d", handle, object ? object->identifier : 0);

    /* If we were unable to obtain an ApsObject for this handle, then it is */
    /* an invalid handle.                                                   */
    if (object == NULL) {
        return APS_INVALID_HANDLE;
    }

    /* Otherwise, increment this object's reference count. */
    ASSERT(object->refCount >= 1);
    object->refCount++;
    ASSERT(object->refCount >= 1);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_ReleaseHandle()
 *
 * Releases any type of APS handle, deallocating asociated memory if this
 * was the last reference to this "object".
 *
 * Parameters: handle - The handle for which a reference should be removed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_ReleaseHandle(Aps_Handle handle)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    TRACE("(%p) objtype = %d", handle, object ? object->identifier : 0);

    /* If we were unable to obtain an ApsObject for this handle, then it is */
    /* an invalid handle.                                                   */
    if (object == NULL) {
        return APS_INVALID_HANDLE;
    }

    /* Remove this reference. If there are other references to this object, */
    /* then return without deleting the object.                             */
    if (--object->refCount >= 1) {
        return APS_SUCCESS;
    }

    /* This was the last reference to this object, so we will delete it. */
    

    /* Call the base object destuctor to perform cleanup common to all object
     * types.
     */
    ObjectDestructor(object);

    /* Pass the deletion request on to the object itself. */
    switch (object->identifier) {
        case PRINTER_HANDLE:
            PrinterDelete(PrinterGetPtrFromHandle(handle));
            break;

        case JOB_HANDLE:
            JobDelete(JobGetPtrFromHandle(handle));
            break;

        case JOB_ATTRIBUTES_HANDLE:
            JobAttrDelete(JobAttrGetPtrFromHandle(handle));
            break;

        case NOTIFICATION_HANDLE:
            NotificationDelete(NotificationGetPtrFromHandle(handle));
            break;

        case MODEL_HANDLE:
            ModelDelete(handle);
            break;

        case FILTER_HANDLE:
            FilterDelete(FilterGetPtrFromHandle(handle));
            break;

        case QUEUE_HANDLE:
            QueueDelete(QueueGetPtrFromHandle(handle));
            break;

        default:
            /* If this were an invalid handle, it should have been caught */
            /* in ObjectGetPtrFromHandle(), so there is a logic error or  */
            /* an unimplemented object type in this function.             */
            ASSERT(FALSE);
            return APS_GENERIC_FAILURE;
    }

    /* If we get to this point, then the handle was successfully released. */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_SubscribeToNotificationCB()
 *
 * Allows the application to be notified via a callback function when some
 * event associated with the specified object takes place, such as a
 * particular change in the object's state.
 *
 * Parameters: handle             - A handle to any type of APS object
 *                                  supporting notifications.
 *
 *             event              - The type of event that the application
 *                                  wishes to be notified of.
 *
 *             callbackFunction   - A pointer to the callback function to be
 *                                  called whenever the specified event
 *                                  occurs.
 *
 *             appData            - An option pointer to application-defined
 *                                  data to be passed back to the callback
 *                                  function whenever the even toccurs.
 *
 *             notificationHandle - An option pointer to an
 *                                  Aps_NotificationHandle that the application
 *                                  can use to prematurely cancel notification.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_SubscribeToNotificationCB(Aps_Handle handle,
                                         Aps_Event event,
                                         Aps_NotificationCallback
                                             callbackFunction,
                                         void *appData,
                                         Aps_NotificationHandle *
                                             notificationHandle)
{
    Aps_Result result;
    NotificationSub *notificationSub;
    ApsObject *object = ObjectGetPtrFromHandle(handle);

    /* Check for reasonable parameter values. */
    if (callbackFunction == NULL) {
        return APS_INVALID_PARAM;
    }

    /* If we were unable to obtain an ApsObject for this handle, then it is */
    /* an invalid handle.                                                   */
    if (object == NULL) {
        return APS_INVALID_HANDLE;
    }

    /* Attempt to create a pointer array to track notification subscriptions,
     * if one doen't already exist in this object.
     */
    if (object->notificationSubs == NULL) {
        result = PtrArrCreate(&object->notificationSubs);
        if (result != APS_SUCCESS)
            return result;
    }

    /* Attempt to create a new notification subscription object. */
    result = NotificationCreate(&notificationSub);
    if (result != APS_SUCCESS)
        return result;

    /* Attach the parent object to the new notification subscription. */
    result = NotificationAttachCB(notificationSub, object, event,
                                  callbackFunction, appData);
    if (result != APS_SUCCESS) {
        NotificationDelete(notificationSub);
        return result;
    }

    /* Attempt to add this notification subscription to the parent object. */
    result = PtrArrAdd(object->notificationSubs, notificationSub);
    if (result != APS_SUCCESS) {
        NotificationDelete(notificationSub);
        return result;
    }

    /* Provide the caller with a handle to this notification subscription,
     * to permit it to cancel the subscription prior to the deletion of
     * the parent object.
     */
    *notificationHandle = NotificationGetHandleFromPtr(notificationSub);

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ObjectSendEventNotification()
 *
 * Notifies any subscribers that wish to be notified of a particular type of
 * event that the event has taken place.
 *
 * Parameters: object    - A handle to the APS object that the event is
 *                         associated with.
 *
 *             event     - An Aps_Event identifying the type of event that
 *                         has taken place.
 *
 *             eventData - A pointer to event-specific data to pass to the
 *                         subscriber, or NULL if no additional data is
 *                         provided with this type of event.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
void ObjectSendEventNotification(ApsObject *object, Aps_Event event,
                                 void *eventData)
{
    int i;
    int numSubscriptions;

    ASSERT(object != NULL);

    /* If no notification subscriptions have been registered for this object
     * then return immediately without doing anything.
     */
    if (object->notificationSubs == NULL)
        return;

    /* Determine the number of notification subscriptions that this object
     * has.
     */
    numSubscriptions = PtrArrGetSize(object->notificationSubs);

    /* Loop through all notification subscriptions that this object has. */
    for(i = 0; i < numSubscriptions; ++i) {
        NotificationSub *notificationSub = (NotificationSub *)
            PtrArrGetAt(object->notificationSubs, i);
        
        /* If this subscription is for this event type, send a
         * notification to the subscriber.
         */
        if (NotificationMatchesEvent(notificationSub, event)) {
            NotificationSend(notificationSub, object, event, eventData);
        }
    }
}

/* ---------------------------------------------------------------------------
 * ObjectRemoveNotificationSub()
 *
 * Removes the specified notification subscription from this object's array
 * of notification subscriptions. Does not delete the notification
 * scription object itself. This function is called from NotificationDetach().
 *
 * Parameters: object    - A handle to the APS object that the event is
 *                         associated with.
 *
 *             event     - An Aps_Event identifying the type of event that
 *                         has taken place.
 *
 *             eventData - A pointer to event-specific data to pass to the
 *                         subscriber, or NULL if no additional data is
 *                         provided with this type of event.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
void ObjectRemoveNotificationSub(ApsObject * object,
                                 NotificationSub * notificationSub)
{
    int i;
    int numSubscriptions;

    ASSERT(object != NULL);

    /* We should never have the situation where we're attempting to remove
     * a notification subscription for an object that never had any
     * subscriptions.
     */
    ASSERT(object->notificationSubs != NULL);

    /* Determine the number of notification subscriptions that this object
     * has.
     */
    numSubscriptions = PtrArrGetSize(object->notificationSubs);

    /* Loop through all notification subscriptions that this object has. */
    for(i = 0; i < numSubscriptions; ++i) {
        NotificationSub *currentNotificationSub = (NotificationSub *)
            PtrArrGetAt(object->notificationSubs, i);
        
        /* If this matches the subscription to be removed, then remove it
         * from the array, and exit.
         */
        if (currentNotificationSub == notificationSub) {
            PtrArrRemoveAt(object->notificationSubs, i, 1);
            return;
        }
    }

    /* If we reach this point, then we were asked to remove a notification
     * subscription that is not listed in this object.
     */
    ASSERT(FALSE);
}

/* ---------------------------------------------------------------------------
 * Aps_IsOperationAvailable()
 *
 * Determines whether or not the specified operation can currently be
 * performed on a particular APS object.
 *
 * Parameters: handle            - A handle to any type of APS object.
 *
 *             operation         - The operation that the application is
 *                                 interested in.
 *
 *             anticipatedResult - APS_OPERATION_AVAILABLE if this operation
 *                                 is currently available, or the anticipated
 *                                 reason for failure if it is known that
 *                                 this operation cannot be performed.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 *             APS_SUCCESS       - test is accurate
 *             APS_IGNORED       - test cannot be performed
 *             APS_NOT_FOUND     - operation ID not recognized
 */
Aps_Result Aps_IsOperationAvailable(Aps_Handle handle,
                                    Aps_OperationID operation,
                                    Aps_Result *anticipatedResult)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);

    /* Check for reasonable parameter values. */
    if (! object) return APS_INVALID_HANDLE;
    if (anticipatedResult == NULL) {
        return APS_INVALID_PARAM;
    }
    /* Default to supported */
    *anticipatedResult = APS_OPERATION_AVAILABLE;

    /* Dispatch call to appropriate object handler */
    switch (object->identifier) {
        case INVALID_HANDLE:  /* should not get here */
            return APS_INVALID_HANDLE;
        case JOB_HANDLE:
            return JobIsOperationAvailable((Job *)object,
                operation, anticipatedResult);
        case QUEUE_HANDLE:
            return QueueIsOperationAvailable((Queue *)object,
                operation, anticipatedResult);
        default:
    }
    return APS_NOT_FOUND;
}

/* ---------------------------------------------------------------------------
 * Aps_GetPropertyString()
 *
 * Obtains the current value of the specified property of this object, in the
 * form of a character string.
 *
 * Parameters: handle       - A handle to any type of APS object.
 *
 *             propertyName - A string uniquely identifying the property.
 *
 *             value        - The address of a char * to receive a new
 *                            buffer containing this property's value.
 *                            Free with Aps_ReleaseBuffer()
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_GetPropertyString(Aps_Handle handle, const char *propertyName,
                                 char **value)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    Property *property;

    /* Check for reasonable parameter values. */
    if (propertyName == NULL || *propertyName == '\0' || value == NULL)
        return APS_INVALID_PARAM;

    /* Initialize the output parameter to NULL in case of failure. */
    *value = NULL;

    /* If we were unable to obtain an ApsObject for this handle, then it is
     * an invalid handle.
     */
    if (object == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain a pointer to the specified property, if it exists. */
    property = ObjectFindProperty(object, propertyName);
    if (property == NULL)
        return APS_NOT_FOUND;

    /* Check for a property type mismatch. */
    if (property->type != PROPERTY_STRING)
        return APS_WRONG_TYPE;

    /* Attempt to allocate a buffer to pass this property's value back to
     * the caller.
     */
    *value = (char *)TrackMemDupString(NULL, property->value.string, 0);
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_SetPropertyString()
 *
 * Changes the current value of the specified property of this object, given
 * a new value in the form of a string.
 *
 * Parameters: handle       - A handle to any type of APS object.
 *
 *             propertyName - A string uniquely identifying the property.
 *
 *             value        - A string with the new value for this property.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_SetPropertyString(Aps_Handle handle, const char *propertyName,
                                 const char *value)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    Property *property;

    /* Check for reasonable parameter values. */
    if (propertyName == NULL || *propertyName == '\0' || value == NULL)
        return APS_INVALID_PARAM;

    /* If we were unable to obtain an ApsObject for this handle, then it is
     * an invalid handle.
     */
    if (object == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain a pointer to the specified property, attempting to create it if
     * it doesn't already exist.
     */
    property = ObjectFindOrCreateProperty(object, propertyName);
    if (property == NULL)
        return APS_OUT_OF_MEMORY;

    /* Initialize this property if it has no value yet. */
    if (property->type == PROPERTY_EMPTY) {
        property->value.string = NULL;
    }

    /* Check for a property type mismatch. */
    if (property->type != PROPERTY_STRING && property->type != PROPERTY_EMPTY)
        return APS_WRONG_TYPE;

    /* Ensure that the string is large enough to hold the new value. */
    property->value.string = (char *)TrackMemRealloc(property->value.string,
        strlen(value) + 1);
    if (! property->value.string) return APS_OUT_OF_MEMORY;

    /* Copy the contents of the value string into this property. */
    strcpy(property->value.string, value);

    /* We can now safely mark this property as containing a string value. */
    property->type = PROPERTY_STRING;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_GetPropertyStrArray()
 *
 * Obtains the current value of the specified property of this object, in the
 * form of an array of character strings.
 *
 * Parameters: handle       - A handle to any type of APS object.
 *
 *             propertyName - A string uniquely identifying the property.
 *
 *             value        - The address of a char ** to receive an array
 *                            of string pointers.
 *                            Free with Aps_ReleaseBuffer()
 *
 *             numElements  - The number of elements in the returned value
 *                            array.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_GetPropertyStrArray(Aps_Handle handle, const char *propertyName,
                                  char ***value, int *numElements)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    Property *property;
    TrackArray_PtrChar stringArrayClone;
    int i;

    /* Check for reasonable parameter values. */
    if (propertyName == NULL || strlen(propertyName) == 0 || value == NULL
        || numElements == NULL)
        return APS_INVALID_PARAM;

    /* Initialize the output parameters to NULL in case of failure. */
    *value = NULL;
    *numElements = 0;

    /* If we were unable to obtain an ApsObject for this handle, then it is
     * an invalid handle.
     */
    if (object == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain a pointer to the specified property, if it exists. */
    property = ObjectFindProperty(object, propertyName);
    if (property == NULL)
        return APS_NOT_FOUND;

    /* Check for a property type mismatch. */
    if (property->type != PROPERTY_STRING_ARRAY)
        return APS_WRONG_TYPE;

    /* Attempt to create a new buffer array to hold this proprerty's value. */
    *numElements = TrackArrayGetSize_PtrChar(property->value.stringArray);
    stringArrayClone = TrackArrayNew_PtrChar(NULL, *numElements);
    if (! stringArrayClone) return APS_OUT_OF_MEMORY;

    /* Copy the individual strings into the new array. */
    for (i = 0; i < *numElements; ++i) {
        stringArrayClone[i] =
            (char *)TrackMemDupString(stringArrayClone,
            property->value.stringArray[i], 0);
        if (! stringArrayClone[i]) {
            *numElements = 0;
            TrackArrayDelete_PtrChar(stringArrayClone);
            return APS_OUT_OF_MEMORY;
        }
    }
    *value = stringArrayClone;
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_SetPropertyStrArray()
 *
 * Changes the current value of the specified property of this object, given
 * a new value in the form of an array of character strings.
 *
 * Parameters: handle       - A handle to any type of APS object.
 *
 *             propertyName - A string uniquely identifying the property.
 *
 *             value        - The address of an array of string pointers.
 *
 *             numElements  - The number of string pointers in the array.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_SetPropertyStrArray(Aps_Handle handle, const char *propertyName,
                                   const char **value, int numElements)
{
    ApsObject *object = ObjectGetPtrFromHandle(handle);
    Property *property;
    TrackArray_PtrChar stringArray;
    int i;

    /* Check for reasonable parameter values. */
    if (propertyName == NULL || strlen(propertyName) == 0
        || (value == NULL && numElements > 0) || numElements < 0)
        return APS_INVALID_PARAM;

    /* If we were unable to obtain an ApsObject for this handle, then it is
     * an invalid handle.
     */
    if (object == NULL)
        return APS_INVALID_HANDLE;

    /* Obtain a pointer to the specified property, attempting to create it if
     * it doesn't already exist.
     */
    property = ObjectFindOrCreateProperty(object, propertyName);
    if (property == NULL)
        return APS_OUT_OF_MEMORY;

    /* Check for a property type mismatch. */
    if (property->type != PROPERTY_STRING_ARRAY
        && property->type != PROPERTY_EMPTY)
        return APS_WRONG_TYPE;

    /* Attempt to create a new buffer array to hold this proprerty's value. */
    stringArray = TrackArrayNew_PtrChar(property, numElements);
    if (! stringArray) return APS_OUT_OF_MEMORY;

    /* Copy the individual strings from the caller into the new array. */
    for (i = 0; i < numElements; ++i) {
        stringArray[i] = (char *)TrackMemDupString(stringArray, value[i], 0);
        if (! stringArray[i]) goto error;
    }

    /* Throw away this property's current value, if it has one. */
    if (property->type == PROPERTY_STRING_ARRAY) {
        TrackArrayDelete_PtrChar(property->value.stringArray);
    }

    /* We can now safely mark this property as containing a string value,
     * and store the new string array in the Property structure.
     */
    property->value.stringArray = stringArray;
    property->type = PROPERTY_STRING_ARRAY;

    return APS_SUCCESS;

error:
    TrackArrayDelete_PtrChar(stringArray);
    return APS_OUT_OF_MEMORY;
}

/* ---------------------------------------------------------------------------
 * ObjectGetPropertyAttribute()
 *
 * Obtains a pointer to the Attribute structure storing information on a
 * full-fledged "attribute" type of property.
 *
 * Parameters: object    - A pointer to some object's base ApsObject.
 *
 *             name      - A string uniquely identifying the property.
 *
 *             attribute - Receives a pointer to the Attribute structure.
 *
 *     Return: APS_SUCCESS on success, APS_NOT_FOUND if no such attribute
 *             exists, or another Aps_Result code on failure.
 */
Aps_Result ObjectGetPropertyAttribute(ApsObject *object, const char *name,
                                      Attribute **attribute)
{
    Property *property;

    /* Obtain a pointer to the requested Property, if it exists. */
    property = ObjectFindProperty(object, name);
    if (property == NULL)
        return APS_NOT_FOUND;
        
    /* Check that this is an Attribute-type property. */
    if (property->type != PROPERTY_ATTRIBUTE)
        return APS_WRONG_TYPE;    

    /* Provide the caller with a pointer to our Attribute instance for this
     * property.
     */
    *attribute = property->value.attribute;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ObjectGetOrAddPropertyAttribute()
 *
 * Obtains a pointer to the Attribute structure storing information on a
 * full-fledged "attribute" type of property, creating the attribute property
 * if it doesn't already exist.
 *
 * Parameters: object    - A pointer to some object's base ApsObject.
 *
 *             name      - A string uniquely identifying the property.
 *
 *             attribute - Receives a pointer to the Attribute structure.
 *
 *             provider  - Pointer to the provider to be associated with
 *                         the attribute.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result ObjectGetOrAddPropertyAttribute(ApsObject *object, const char *name,
                                           Attribute **attribute,
                                           AttrProvider *provider)
{
    Property *property;
    Aps_Result result;

    /* Obtain a pointer to the requested Property, if it exists. */
    property = ObjectFindProperty(object, name);
    if (property == NULL) {
        /* Create a new attribute. */
        result = AttrCreate(attribute, provider);
        if (result != APS_SUCCESS)
            return result;

        /* Create a new property to store this Attribute. */
        property = ObjectFindOrCreateProperty(object, name);
        if (property == NULL) {
            AttrDelete(*attribute);
            return APS_OUT_OF_MEMORY;
        }

        /* Setup this new property with the appropriate type and value
         * information
         */
        property->type = PROPERTY_ATTRIBUTE;
        property->value.attribute = *attribute;
    }

    /* We assume that if this function is called when the attribute already
     * exists, that the existing attribute will be associated with the same
     * provider.
     */
    ASSERT(property->value.attribute->provider == provider);
    
    /* Check that this is an Attribute-type property. */
    if (property->type != PROPERTY_ATTRIBUTE)
        return APS_WRONG_TYPE;    

    /* Provide the caller with a pointer to our Attribute instance for this
     * property.
     */
    *attribute = property->value.attribute;

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ObjectGetProperties()
 *
 * Obtains the names of all properties stored in this object.  Names are
 * appended to an existing array.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             names  - Ptr to a TrackArray of PtrChar to receive pointers
 *                      to the names of all properties.
 *
 *     Return: APS_SUCCESS on success, or another Aps_Result code on failure.
 */
Aps_Result ObjectGetProperties(ApsObject *object, TrackArray_PtrChar *names)
{
    int i;
    Property *property;

    ASSERT(object != NULL);
    ASSERT(names != NULL);

    /* If this object doesn't have any properties, then do nothing.
     */
    if (object->properties == NULL)
        return APS_SUCCESS;

    /* Loop through all properties, adding them to the array. */
    for (i = 0; i < PtrArrGetSize(object->properties); ++i) {
        char *newString;
        property = (Property *)PtrArrGetAt(object->properties, i);
        ASSERT(property->name);
        newString = (char *)TrackMemDupString(*names, property->name, 0);
        if (! newString) return APS_OUT_OF_MEMORY;
        if (! TrackArrayAddLast_PtrChar(names, newString))
            return APS_OUT_OF_MEMORY;
    }

    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * ObjectGetPropertyType()
 *
 * Determines what type of information is stored for the specified
 * property. This function should only be called for those properties that
 * are known to exist.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             name   - A string uniquely identifying the property.
 *
 *     Return: The type of the specified property.
 */
PropertyType ObjectGetPropertyType(ApsObject *object, const char *name)
{
    Property *property;

    ASSERT(object != NULL);

    /* Search for the specified property. */
    property = ObjectFindProperty(object, name);
    if (property == NULL) {
        /* The specified property doesn't exist in this object! */
        ASSERT(FALSE);
        return PROPERTY_EMPTY;
    }

    /* Return the type of this property to the caller. */
    return property->type;
}

/* ---------------------------------------------------------------------------
 * ObjectRemoveProperty()
 *
 * Removes the specified property from this object, if it exists.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             name   - A string uniquely identifying the property.
 *
 *     Return: None.
 */
void ObjectRemoveProperty(ApsObject *object, const char *name)
{
    int i;
    Property *property;

    ASSERT(object != NULL);
    ASSERT(name != NULL);

    /* If this object doesn't have any properties, there is no search to be
     * performed.
     */
    if (object->properties == NULL)
        return;

    /* Loop through all properties, looking for one with a matching name.
     */
    for (i = 0; i < PtrArrGetSize(object->properties); ++i) {
        property = (Property *)PtrArrGetAt(object->properties, i);
        if (strcmp(property->name, name) == 0) {
            /* We have found the property in question, so delete it. */
            ObjectDeleteProperty(property);

            /* Remove this item from the property array. */
            PtrArrRemoveAt(object->properties, i, 1);
            
            /* We're finished, no need to continue searching. */
            return;
        }
    }

    /* If we reach this point, no properties of this object match the
     * specified name - this property doesn't exist, so there was nothing
     * for us to do.
     */
}

/* ---------------------------------------------------------------------------
 * ObjectDeleteProperty()
 *
 * Deletes a property structure, releasing any associated memory.
 *
 * Parameters: property - A pointer to the property to be released.
 *
 *     Return: None.
 */
static void ObjectDeleteProperty(Property *property)
{
    /* Release the property's value. */
    switch (property->type) {
        case PROPERTY_EMPTY:
            /* This property has no value, so there is nothing for
             * us to do.
             */
            break;

        case PROPERTY_STRING:
            ASSERT(property->value.string);
            TrackMemFree(property->value.string);
            break;

        case PROPERTY_STRING_ARRAY:
            /* Delete the array and the strings it contains. */
            ASSERT(property->value.stringArray);
            TrackArrayDelete_PtrChar(property->value.stringArray);
            break;

        case PROPERTY_ATTRIBUTE:
            /* Delete all information associated with a full-fledged
             * "attribute" property.
             */
            ASSERT(property->value.attribute);
            AttrDelete(property->value.attribute);
            break;

        default:
            /* An unimplemented property type! */
            ASSERT(FALSE);
    }

    /* Delete the Property structure itself, its name, and any associated
     * resources... */
    TrackMemFree((void *)property);
}

/* ---------------------------------------------------------------------------
 * ObjectDoesPropertyExist()
 *
 * Determines whether or not this object contains a property with the
 * specified name.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             name   - A string uniquely identifying the property.
 *
 *     Return: TRUE if the property is available, FALSE if not.
 */
int ObjectDoesPropertyExist(ApsObject *object, const char *name)
{
    /* Search for the specified property in this object. */
    return ObjectFindProperty(object, name) != NULL;
}

/* ---------------------------------------------------------------------------
 * ObjectFindProperty()
 *
 * Searches for a property of this object with the specified name.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             name   - A string uniquely identifying the property.
 *
 *     Return: A pointer to the Property structure for the specified
 *             property, or NULL if this property does not exist.
 */
static Property *ObjectFindProperty(ApsObject *object, const char *name)
{
    int i;
    Property *property;

    ASSERT(object != NULL);
    ASSERT(name != NULL);

    /* If this object doesn't have any properties, there is no search to be
     * performed.
     */
    if (object->properties == NULL)
        return NULL;

    /* Loop through all properties, looking for one with a matching name. The
     * current implementation stores properties as a simple unsorted array. If
     * at some point in the future there are situations where a single object
     * has a lot of properties, some benefit might be realized by storing
     * the properties in some easier to search structure.
     */
    for (i = 0; i < PtrArrGetSize(object->properties); ++i) {
        property = (Property *)PtrArrGetAt(object->properties, i);
        if (strcmp(property->name, name) == 0) {
            /* We have found the property in question, so return it to the
             * caller.
             */
            return property;
        }
    }

    /* If we reach this point, no properties of this object match the
     * specified name - this property doesn't exist.
     */
    return NULL;
}

/* ---------------------------------------------------------------------------
 * ObjectFindOrCreateProperty()
 *
 * Obtains a pointer to a Property structure for this object's property of the
 * specified name, creating the property if it doesn't already exist.
 *
 * Parameters: object - A pointer to some object's base ApsObject.
 *
 *             name   - A string uniquely identifying the property.
 *
 *     Return: A pointer to the Property structure for the specified
 *             property, or NULL on memory allocation failure.
 */
static Property *ObjectFindOrCreateProperty(ApsObject *object,
                                            const char *name)
{
    Property *property;

    ASSERT(object != NULL);
    ASSERT(name != NULL);

    /* First, check whether this property already exists. */
    property = ObjectFindProperty(object, name);

    /* If the property already exists, return it to the caller. */
    if (property != NULL)
        return property;

    /* If we get to this point, the property doesn't already exists, so we'll
     * attempt to allocate space for a new one.
     */
     
    /* First, ensure that the pointer array of properties exists. */
    if (object->properties == NULL) {
        if (PtrArrCreate(&object->properties) != APS_SUCCESS)
            return NULL;
    }

    /* Allocate the Property structure itself. */
    property = (Property *)TrackMemAlloc(NULL, sizeof(Property), 0);
    if (property == NULL)
        return NULL;

    /* Allocate space for the name of this property, and store the name for
     * future identification of this property.
     */
    property->name = TrackMemDupString(property, name, 0);
    if (property->name == NULL) {
       free(property);
       return NULL;
    }

    /* Flag this property as initially having no value. */
    property->type = PROPERTY_EMPTY;

    /* Attempt to add this new property to the array of properties. */
    if (PtrArrAdd(object->properties, property) != APS_SUCCESS) {
        TrackMemFree((void *)property);
        return NULL;
    }

    return property;
}
