/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printerenuminst.c
 *
 * Description: Printer enumeration and instatiation APIs.
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * Aps_OpenDefaultPrinter()
 *
 * Obtains a handle to the printer that has been set as the default, allowing
 * various operations to be performed on that printer, such as sending jobs.
 *
 * Parameters: handle - A pointer to an Aps_PrinterHandle to receive a handle to
 *                      the opened printer.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_OpenDefaultPrinter(Aps_PrinterHandle * handle)
{
    int numTransports;
    int i;
    Transport **transports;
    char *defaultPrinterName;
    Aps_Result openResult;

    /* Obtain an array of the available low-level transports. */
    transports = TransGetAllTransports(&numTransports);

    /* Loop through all the transports, looking for the first one that has a */
    /* default printer available. */
    for (i = 0; i < numTransports; ++i) {
        defaultPrinterName = transports[i]->vtbl->
            GetDefaultPrinterName(transports[i]);
        if (defaultPrinterName != NULL) {
            /* We've obtained the name of a default printer, so attempt to */
            /* open it. */
            openResult = Aps_OpenPrinter(defaultPrinterName, handle);
            Aps_ReleaseBuffer(defaultPrinterName);
            if (openResult == APS_SUCCESS)
                return (APS_SUCCESS);
        }
    }

    /* If we reached this point, no transport was able to provide us with */
    /* a default printer that we could open. */
    return (APS_NOT_FOUND);
}

/* ---------------------------------------------------------------------------
 * Aps_GetPrinters()
 *
 * Obtains the names of all currently installed printers.
 *
 * Parameters: names - A char *** to receive a buffer containing an array of
 *                     string pointers with the names of the available
 *                     printers. If no printers were found, this will be NULL.
 *                     Otherwise, it is the callers responsibility to pass it
 *                     to Aps_ReleaseBuffer().
 *
 *             count - The number of installed printers. This is the number
 *                     of elements in the names array.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_GetPrinters(char ***names, int *count)
{
    int transportCount = 0;
    Transport **trans;
    int i;
    Aps_Result result;

    /* Safeguard against obviously invalid parameters. */
    if (names == NULL || count == NULL)
        return APS_INVALID_PARAM;
        
    /* Reduce unpredictability by ensuring that the names array and printer */
    /* count are always initialized, even on failure.                       */
    *names = NULL;
    *count = 0;
    
    /* Obtain a list of all transports. */
    trans = TransGetAllTransports(&transportCount);

    /* Create a new buffer array to contain the list of printer names. */
    *names = TrackArrayNew_PtrChar(NULL, 0);
    if (! *names) return APS_OUT_OF_MEMORY;

    /* Loop through all transports, */
    for (i = 0; i < transportCount; ++i) {
        ASSERT(trans[i] != NULL);

        /* Add to the array the names of all printers available from this */
        /* transport.                                                     */
        result = trans[i]->vtbl->GetPrinters(trans[i], names);
        
        /* On failure, discard the name array and propogate reason for */
        /* failure back to the caller.                                 */
        if (result != APS_SUCCESS) {
            TrackArrayDelete_PtrChar(*names);
            *names = NULL;
            return result;
        }
    }
    
    /* If no printers were found, then we don't give the caller a buffer. */
    /* Otherwise, we give a pointer to a contiguous buffer with the array */
    /* of pointers and the printer name strings themselves.               */
    if ((*count = TrackArrayGetSize(*names)) == 0) {
        TrackArrayDelete_PtrChar(*names);
        *names = NULL;
    }
    
    /* If we reach this point, we have either successfully determined that */
    /* there are no printers installed, or we have successfully obtained   */
    /* a list of all available printers.                                   */
    return APS_SUCCESS;
}

/* ---------------------------------------------------------------------------
 * Aps_OpenPrinter()
 *
 * Obtains a handle to an installed printer, allowing various operations to
 * be performed on that printer, such as sending jobs, manipulating the
 * queue or changing settings.
 *
 * Parameters: name   - A string that identifies the printer to be opened.
 *
 *             handle - A pointer to an Aps_PrinterHandle to receive a handle
 *                      to the specified printer.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_OpenPrinter(const char *name, Aps_PrinterHandle * handle)
{
    Aps_Result result;
    Printer *printer;

    /* Check for reasonable parameter values. */
    if (name == NULL || handle == NULL)
        return (APS_INVALID_PARAM);

    /* Obtain a pointer to the appropriate printer from the transport */
    /* abstraction layer. */
    result = TransGetPrinter((char *)name, &printer);
    if (result != APS_SUCCESS)
        return (result);

    /* Obtain a handle for this printer. */
    *handle = PrinterGetHandleFromPtr(printer);

    return (APS_SUCCESS);
}

/* ---------------------------------------------------------------------------
 * Aps_OpenRemotePrinter()
 *
 * Obtains a handle to a remote printer that is not installed on this
 * machine's/user's list of regularly used printers. A typical use for this
 * function is to allow one-time printing to a network printer without
 * requiring the printer to be "installed" locally.
 *
 * Parameters: connectionType - An APS_ConnectionType, which together with the
 *                              location parameter uniquely identify a remote
 *                              printer.
 *
 *             location       - A location string specifying a specific
 *                              resource available through the specified
 *                              connection type.
 *
 *             printer        - A pointer to an Aps_PrinterHandle to receive a
 *                              handle to the specified printer.
 *
 *     Return: A standard APS_RESULT code indicating success or reason for
 *             failure.
 */
Aps_Result Aps_OpenRemotePrinter(Aps_ConnectionType connectionType,
                                 const char *location,
                                 Aps_PrinterHandle *printer)
{
    /* Check for reasonable parameter values. */
    if(location == NULL || printer == NULL) {
        return APS_INVALID_PARAM;
    }
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

