/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: queueaccess.c
 *
 * Description: Queue access APIs (for scanning active jobs and getting
 *              particulars)
 *
 */

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "job.h"
#include "queue.h"

DEBUG_CHANNEL_DEFAULT(queue)

/* --------------------------------------------------------------------
 * Aps_PrinterOpenQueue()
 *
 * Obtains a handle to an abstract queue representing this printer.
 *
 * Parameters : printerHandle - handle of printer for which a queue should
 *                              be opened
 *
 * Result     : (inparam) queueHandle - new queue handle, or NULL on failure
 *
 *
 * The returned handle must be freed by Aps_ReleaseHandle().
 * On failure *queueHandle set to NULL and detailed result returned.
 */
Aps_Result Aps_PrinterOpenQueue(Aps_PrinterHandle printerHandle,
    Aps_QueueHandle *queueHandle)
{
    Aps_Result result;

    /* Check params */
    if (! queueHandle) return APS_INVALID_PARAM;
    *queueHandle = NULL;

    /* Get a lock on the printer */
    result = Aps_AddRef(printerHandle);
    if (result != APS_SUCCESS) return APS_INVALID_HANDLE;

    /* Create queue object */
    result = QueueCreate(printerHandle, queueHandle);
    if (result == APS_SUCCESS) return result;

    Aps_ReleaseHandle(printerHandle);
    return result;
}

/* --------------------------------------------------------------------
 * Aps_OpenGlobalQueue()
 *
 * Obtains a handle to an abstract queue representing all printers.
 *
 * Result     : (inparam) queueHandle - new queue handle, or NULL on failure
 *
 * The returned handle must be freed by Aps_ReleaseHandle().
 * On failure *queueHandle set to NULL and detailed result returned.
 */
Aps_Result Aps_OpenGlobalQueue(Aps_QueueHandle *queueHandle)
{
    /* Check params */
    if (! queueHandle) return APS_INVALID_PARAM;
    *queueHandle = NULL;

    /* Create queue object */
    return QueueCreate(NULL, queueHandle);
}

/* --------------------------------------------------------------------
 * Aps_QueueGetNumberOfJobs()
 *
 * Gets the number of jobs matching the filter attached to the queue.
 *
 * Parameters : queueHandle - handle of queue to query
 * Result     : (inparam) numberOfJobs - # of jobs matching filter
 */
Aps_Result Aps_QueueGetNumberOfJobs(Aps_QueueHandle queueHandle,
    int *numberOfJobs)
{
    Aps_Result    result;
    Aps_JobHandle iterator = NULL;

    /* Check params */
    if (! numberOfJobs) return APS_INVALID_PARAM;

    *numberOfJobs = 0;
    while (Aps_Succeeded(
        result = Aps_QueueIterateJobs(queueHandle, & iterator)))
        (*numberOfJobs)++;

    if (result == APS_NO_MORE_DATA) result = APS_SUCCESS;
    return result;
}

/* --------------------------------------------------------------------
 * Aps_QueueMakeQuickPrinterQInfoArray()
 *
 * Gets the number of jobs matching the filter attached to the queue
 * and various other criteria.  Builds an array of Aps_QuickPrinterQInfo
 * structures.
 *
 * Parameters : queueHandle - handle of queue to query
 * Result     : (inparam) printerQueueInfo - array of Aps_QuickPrinterQInfo
 *              Free with Aps_ReleaseBuffer()
 */
Aps_Result Aps_QueueMakeQuickPrinterQInfoArray(Aps_QueueHandle queueHandle,
    Aps_QuickPrinterQInfo ***printerQueueInfo, int *numElements)
{
    Aps_Result             result;
    Aps_JobHandle          iterator = NULL;
    Aps_PrinterHandle      lastPrinter = NULL;
    Aps_QuickPrinterQInfo *lastInfo = NULL;
    Queue                 *queue;

    /* Check params */
    if (! (printerQueueInfo && numElements)) return APS_INVALID_PARAM;
    queue = QueueGetPtrFromHandle(queueHandle);
    if (! queue) return APS_INVALID_HANDLE;

    /* Create array */
    *numElements = 0;
    *printerQueueInfo = TrackArrayIndirectNew_QuickPrinterQInfo(NULL, 0);
    if (! *printerQueueInfo) return APS_OUT_OF_MEMORY;

    /* Iterate through all jobs */
    while (Aps_Succeeded(result =
        QueueIterateAllJobs(queueHandle, & iterator))) {
        int  match;
        Job *job = JobGetPtrFromHandle(iterator);
        Aps_JobStatus jobStatus = job->info->jobStatus;

        /* If this is a new printer, setup the fields as req'd */
        if ((! lastPrinter) || (lastPrinter != job->info->printerHandle)) {
            lastInfo = TrackArrayIndirectAddLastByRef_QuickPrinterQInfo(
                printerQueueInfo, NULL);
            if (! lastInfo) { result = APS_OUT_OF_MEMORY; break; }

            /* Version 0 */
            lastInfo->version = 0;
            lastPrinter = lastInfo->printerHandle = job->info->printerHandle;
            lastInfo->printerStatus = job->info->printerStatus;
            /* printerName */
            lastInfo->printerName = (char *)TrackMemDupString(lastInfo,
                job->info->printerName, 0);
            if (!lastInfo->printerName) { result = APS_OUT_OF_MEMORY; break; }
            lastInfo->numJobsTotal = 0;
            lastInfo->numJobsMatch = 0;
            lastInfo->numJobsProduction = 0;
            lastInfo->numJobsPending = 0;
            lastInfo->numJobsWorking = 0;
            lastInfo->numJobsOnHold = 0;
            lastInfo->numJobsEnded = 0;
            /* Version 1, here */
        }
        /* Update counters */
        /* total */
        lastInfo->numJobsTotal++;
        result = QueueCheckFilter(queue, iterator, & match);
        if (! Aps_Succeeded(result)) break;
        if (match) {
            /* match */
            lastInfo->numJobsMatch++;
            /* production */
            if (jobStatus & APS_JOB_PHASE_PRODUCTION)
                lastInfo->numJobsProduction++;
            /* pending */
            if (jobStatus & APS_JOB_PHASE_PENDING)
                lastInfo->numJobsPending++;
            /* working */
            if (jobStatus & APS_JOB_PHASE_WORKING)
                lastInfo->numJobsWorking++;
            /* on hold */
            if (jobStatus & APS_JOB_PHASE_ON_HOLD)
                lastInfo->numJobsOnHold++;
            /* ended */
            if (jobStatus & APS_JOB_PHASE_ENDED)
                lastInfo->numJobsEnded++;
        }
    }
    if (result == APS_NO_MORE_DATA) {
        result = APS_SUCCESS;
        *numElements =
            TrackArrayIndirectGetSize_QuickPrinterQInfo(*printerQueueInfo);
    } else {
        TrackArrayIndirectDelete_QuickPrinterQInfo(*printerQueueInfo);
        *printerQueueInfo = NULL;
    }
    return result;
}

/* --------------------------------------------------------------------
 * Aps_QueueIterateJobs()
 *
 * Iterates through all jobs in a given queue and returns only those
 * matching a filter (if supported). See QueueIterateAllJobs()
 *
 * Parameters : queueHandle - handle of queue to query
 *              jobHandle   - handle of last job examined
 * Result     : (inparam) jobHandle - handle of next job in queue
 */
Aps_Result Aps_QueueIterateJobs(Aps_QueueHandle queueHandle,
    Aps_JobHandle  *jobHandle)
{
    Aps_Result result;
    Queue     *queue;

    /* locate queue object */
    queue = QueueGetPtrFromHandle(queueHandle);
    if (! queue) return APS_INVALID_HANDLE;

    /* iterate through all and try filter */
    while (Aps_Succeeded(
        result = QueueIterateAllJobs(queueHandle, jobHandle))) {
        int match;
        result = QueueCheckFilter(queue, *jobHandle, & match);
        if (! Aps_Succeeded(result)) return result;

        /* break out only if it matched */
        if (match) break;
    }
    return result;
}

/* --------------------------------------------------------------------
 * QueueIterateAllJobs()
 *
 * Iterates through all jobs in a given queue. IGNORES FILTERS.
 *
 * Global queue viewing depends on new printers being added to the
 * tail of list. ie. ALL PRINTERS MUST REMAIN IN THE SAME ORDER AT
 * ALL TIMES UNTIL A SMARTER MECHANISM IS AVAILABLE.
 *
 * FUTURE: Printers should form a LINKED LIST.
 *
 * Parameters : queueHandle - handle of queue to query
 *              jobHandle   - handle of last job examined
 * Result     : (inparam) jobHandle - handle of next job in queue
 */
Aps_Result QueueIterateAllJobs(Aps_QueueHandle queueHandle,
    Aps_JobHandle *jobHandle)
{
    Aps_Result result;
    Queue *queue;
    char  *lastName = NULL;

    /* Check params */
    queue = QueueGetPtrFromHandle(queueHandle);
    if (! queue) return APS_INVALID_HANDLE;
    if (! jobHandle) return APS_INVALID_PARAM;

    /* If this queue is not global, then we can pass this message on
     * down to the transport directly.
     */
    if (queue->printerHandle) {
        Transport *trans;
        Printer   *printer;
        Aps_Result result;
        Job       *job;

        /* locate printer and transport */
        printer = PrinterGetPtrFromHandle(queue->printerHandle);
        ASSERT(printer);
        trans = printer->transport;
        ASSERT(trans);

        /* get next job */
        if (*jobHandle) {
            job = JobGetPtrFromHandle(*jobHandle);
            if (! job) return APS_INVALID_HANDLE;
        } else job = NULL;
        result = trans->vtbl->JobIterate(trans, printer, & job);
        *jobHandle = (Aps_Succeeded(result)) ? JobGetHandleFromPtr(job) : NULL;
        return result;
    }

    /* THIS IS THE GLOBAL QUEUE */
    /* If we are in the middle, then we can continue onwards... */
    if (*jobHandle) {
        Job *job = JobGetPtrFromHandle(*jobHandle);
        ASSERT(job->transport && job->printer);
        ASSERT(job->printer->name);

        /* Keep name in case this fails */
        lastName = alloca(strlen(job->printer->name) + 1); /* on stack */
        strcpy(lastName, job->printer->name);

        result = job->transport->vtbl->JobIterate(job->transport,
            job->printer, & job);
        *jobHandle = (Aps_Succeeded(result)) ? JobGetHandleFromPtr(job) : NULL;
        if (*jobHandle) return result;
    }

    /* Else, we will have to do some fancy footwork...
     * Find next printer then locate first job */
    {
        char **printerNames;
        int    count, i;
        Job   *job;

        /* Need to find next printer in chain... Let's hope that
         * we can find the last one we used, otherwise we'll have to
         * give up here -- or be forced to repeat ourselves...
         */
        result = Aps_GetPrinters(& printerNames, & count);
        if (result != APS_SUCCESS) return result; /* fail! */

        /* locate printer by name */
        if (lastName) {
            for (i = 0; i < count; i++) {
                ASSERT(printerNames[i]);
                /* stop if found */
                if (strcmp(printerNames[i], lastName) == 0) break;
            }
        } else i = -1; /* start from beginning */

        /* not found if i == count... */
        /* try incrementally... opening each printer if possible */
        while (++i < count) {
            Aps_PrinterHandle ph;

            /* open printer */
            result = Aps_OpenPrinter(printerNames[i], & ph);
            if (result == APS_SUCCESS) {
                Printer *printer;
                printer = PrinterGetPtrFromHandle(ph);
                ASSERT(printer && printer->transport);

                /* get first job */
                job = NULL;
                result = printer->transport->vtbl->JobIterate(
                    printer->transport, printer, & job);
                /* close printer */
                Aps_ReleaseHandle(ph);
                /* check job */
                *jobHandle = (Aps_Succeeded(result)) ?
                     JobGetHandleFromPtr(job) : NULL;
                if (*jobHandle) return result;
            }
        }
    }
    return APS_NO_MORE_DATA;
}

/* --------------------------------------------------------------------
 * QueueCheckFilter()
 *
 * Run a job through a filter. Handles FALLBACK.
 *
 * Parameters : queue  - pointer to queue structure
 *              job    - job handle
 * Result     : (inparam) match - TRUE if match, else FALSE.
 */
Aps_Result QueueCheckFilter(Queue *queue, Aps_JobHandle job, int *match)
{
    Aps_Result result = APS_SUCCESS;
    if (queue->baseClass.filterHandle) {
        /* check filter */
        result = Aps_FilterCheck(queue->baseClass.filterHandle,
            queue->baseClass.filterOpts, job, match);
        /* handle FALLBACK */
        if ((result == APS_FILTER_NOT_SUPPORTED) &&
            (queue->baseClass.filterOpts & APS_FILTER_FALLBACK)) {
            *match = TRUE;
            result = APS_PARTIAL_SUCCESS;
        }
    } else *match = TRUE;
    return result;
}

