/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: parseprintcapfile.c
 *
 * Description: Implementation of printcap file parser. The /etc/printcap
 *              file contains the printer configuration information used
 *              by lpr, LPRng and work-alikes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include "aps.h"
#include "apsinternal.h"
#include "readwriteprintcap.h"
#include "utils.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(printcap)

/* Text header that is placed at the beginning of newly-created printcap
 * files. */
static const char *PrintcapHeaderComment =
 "# This 'printcap' file contains the system's current configured printers.\n"
 "# The file was created using APS library functions developed by Corel\n"
 "# Corporation and distributed freely under GNU LGPL.\n"
 "#\n"
 "# The library presently recognizes the following tokens:\n"
 "#   af, br, cf, df, fc, ff, fo, fs, gf, hl, ic, if, lf, lo, lp, mx, nd, nf\n"
 "#   of, pc, pl, pw, px, py, rf, rg, rm, rp, rs, rw, sb, sc, sd, sf, sh, st\n"
 "#   tf, tr, vf\n"
 "#\n"
 "# Warning: When modifying this file, the library discards any comments or\n"
 "#          unrecognized mnemonic tokens.  It may also alter the order,\n"
 "#          position, or formatting of the entries.\n"
 "#\n"
 "# For more information, run 'man printcap' from your favorite shell\n\n";

/* Prototypes for helper functions private to this module. */
static PRINTCAP_OPTION_TYPE DetermineOptionType(char *optionType);
static int MoveFilePtrByX(FILE * fp, int moveBy, int from);
static int SmartStrcmp(char *string1, char *string2);
static int IsEqual(Aps_Printcap_Info * first, Aps_Printcap_Info * second);
static int ConsolidateApsPrintcapInfoArray(Aps_Printcap_Info ** printcapInfo,
                                           int size);
static int LoadPrintcapIntoBuffer(FILE * fp, TrackArray_PtrChar *buffer);
static void WriteIntegerValueInPrintcap(FILE * fp, int value, char *mnemonic);
static void WriteStringValueInPrintcap(FILE * fp, char *value,
                                       char *mnemonic);
static BOOL WritePrinterRecInPrintcapFile(FILE * fp,
                                          Aps_Printcap_Info * printerRec);
static int ParseLprPrintcapBuffers(char **buffer, int index,
                                   Aps_Printcap_Info ** printcapInfo);
static int ReadFileToEndOfLineAndFillBuffer(FILE * fp,
    TrackArrayElemRef_PtrChar buffer, int *spaceAvailable,
    int allocateBufferBy);

/*---------------------------------------------------------------------------
 * DetermineOptionType()
 *
 * Determines the type of printcap option depending on char string.
 *
 * Parameters: optionType - pointer to the option string.
 *
 *     Return: A PRINTCAP_OPTION_TYPE identifying the type of option, or
 *             OPTION_UNKNOWN if this was an unrecognized option type.
 * 
 */
static PRINTCAP_OPTION_TYPE DetermineOptionType(char *optionType)
{
    if (strcmp(optionType, "af") == 0)
        return OPTION_ACCOUNTING_FILE;
    if (strcmp(optionType, "br") == 0)
        return OPTION_BAUD_RATE;
    if (strcmp(optionType, "cf") == 0)
        return OPTION_CIFPLOT_FILTER;
    if (strcmp(optionType, "df") == 0)
        return OPTION_TEST_DATA_FILTER;
    if (strcmp(optionType, "fc") == 0)
        return OPTION_IS_TTY;
    if (strcmp(optionType, "ff") == 0)
        return OPTION_FORM_FEED;
    if (strcmp(optionType, "fo") == 0)
        return OPTION_FF_ON_OPEN;
    if (strcmp(optionType, "fs") == 0)
        return OPTION_IS_TTY_BITS;
    if (strcmp(optionType, "gf") == 0)
        return OPTION_IS_GRAPH_DATA_FILTER;
    if (strcmp(optionType, "hl") == 0)
        return OPTION_HEADER_PAGE_LAST;
    if (strcmp(optionType, "ic") == 0)
        return OPTION_IOCTL_SUPPORT;
    if (strcmp(optionType, "if") == 0)
        return OPTION_TEXT_FILTER;
    if (strcmp(optionType, "lf") == 0)
        return OPTION_LOG_FILE;
    if (strcmp(optionType, "lo") == 0)
        return OPTION_LOCK_FILE;
    if (strcmp(optionType, "lp") == 0)
        return OPTION_DEVICE;
    if (strcmp(optionType, "mx") == 0)
        return OPTION_MAX_FILE_SIZE;
    if (strcmp(optionType, "nd") == 0)
        return OPTION_NEXT_DIRECTORY;
    if (strcmp(optionType, "nf") == 0)
        return OPTION_DITROFF_FILTER;
    if (strcmp(optionType, "of") == 0)
        return OPTION_OUTPUT_FILTER;
    if (strcmp(optionType, "pc") == 0)
        return OPTION_PRICE_PER_FOOT;
    if (strcmp(optionType, "pl") == 0)
        return OPTION_LINES_PER_PAGE;
    if (strcmp(optionType, "pw") == 0)
        return OPTION_CHARS_ACROSS_PAGE;
    if (strcmp(optionType, "px") == 0)
        return OPTION_PAGE_WIDTH_PIXELS;
    if (strcmp(optionType, "py") == 0)
        return OPTION_PAGE_HEIGHT_PIXELS;
    if (strcmp(optionType, "rf") == 0)
        return OPTION_FORTRAN_STYLE_FILTER;
    if (strcmp(optionType, "rg") == 0)
        return OPTION_RESTRICTED_GROUP;
    if (strcmp(optionType, "rm") == 0)
        return OPTION_REMOTE_MACHINE;
    if (strcmp(optionType, "rp") == 0)
        return OPTION_REMOTE_PRINTER_ARGUMENT;
    if (strcmp(optionType, "rs") == 0)
        return OPTION_RESTRICT_TO_LOCAL_ACC;
    if (strcmp(optionType, "rw") == 0)
        return OPTION_READ_WRITE_DEVICE;
    if (strcmp(optionType, "sb") == 0)
        return OPTION_SHORT_BANNER;
    if (strcmp(optionType, "sc") == 0)
        return OPTION_SUPPRESS_COPIES;
    if (strcmp(optionType, "sd") == 0)
        return OPTION_SPOOL_DIR;
    if (strcmp(optionType, "sf") == 0)
        return OPTION_SUPPRESS_FORM_FEED;
    if (strcmp(optionType, "sh") == 0)
        return OPTION_SUPPRESS_BURST_PG_HDR;
    if (strcmp(optionType, "st") == 0)
        return OPTION_STATUS_FILE;
    if (strcmp(optionType, "tf") == 0)
        return OPTION_TROFF_FILTER;
    if (strcmp(optionType, "tr") == 0)
        return OPTION_TRAILING_STRING;
    if (strcmp(optionType, "vf") == 0)
        return OPTION_RASTER_FILTER;

    /* If we reach this point, then this printcap file has an option field 
     * not currently supported by the printcap parser.                     
     */
    WARN("Unrecognized printcap option type: %s", optionType);
    return OPTION_UNKNOWN;
}

/*---------------------------------------------------------------------------
 * ReadFileToEndOfLineAndFillBuffer()
 *
 * Read characters from a file util it reaches eof or a 
 * new line character.
 *
 * Parameters: fp               - file descriptor pointer
 *
 *             buffer           - pointer to the buffer pointer
 *                                TrackArray_PtrChar with component TrackMem's
 *
 *             spaceAvailable   - space available in buffer
 *
 *             allocateBufferBy - reallocate buffer by increments of this
 *                                value
 *
 *     Return: On error return -1
 * 
 */
static int ReadFileToEndOfLineAndFillBuffer(FILE * fp,
    TrackArrayElemRef_PtrChar buffer,
    int *spaceAvailable, int allocateBufferBy)
{
    char currentChar[2];

    if (*buffer == NULL) {
        ERROR("Buffer pointer argument was NULL\n");
        return -1;
    }
    while (fgets(currentChar, 2, fp) != NULL && (currentChar[0] != '\n')) {
        if (*spaceAvailable <= 0) {
            TRACE("Out of memory, allocating more...\n");
            *buffer = (char *)
                TrackMemRealloc(*buffer, strlen(*buffer) + 1 +
                    allocateBufferBy);
            if (! *buffer) {
                ERROR("Memory allocation failed\n");
                return -1;
            }
            *spaceAvailable = allocateBufferBy;
        }
        strncat(*buffer, (char *)&currentChar[0], 1);
        (*spaceAvailable)--;
    }
    return 0;
}

/*---------------------------------------------------------------------------
 * MoveFilePtrByX()
 *
 * Move file pointer back by specified counts.
 *
 * Parameters: fp     - file descriptor pointer
 *
 *             moveBy - move back by
 *
 *             from   - starting point for reference 
 *
 *     Return: -1 on error
 * 
 */
static int MoveFilePtrByX(FILE * fp, int moveBy, int from)
{
    return fseek(fp, ftell(fp) - moveBy, from);
}

/*---------------------------------------------------------------------------
 * LoadPrintcapIntoBuffer()
 *
 * Loads printcapfile into buffer.
 *
 * Parameters: fp     - file descriptor pointer 
 *
 *             buffer - pointer to a pointer to buffer
 *                      TrackArray_PtrChar with component TrackMem's
 *
 *     Return: Number of entries copied to the buffer
 * 
 */
static int LoadPrintcapIntoBuffer(FILE * fp, TrackArray_PtrChar *buffer)
{
    int index = 0;
    char currentChar[2];
    int spaceAvailable = 0;

    if (!fp) {
        ERROR("Invalid file pointer.\n");
        return -1;
    }
    TRACE("Starting with %d length of buffer\n", spaceAvailable);
    while (fgets(currentChar, 2, fp) != NULL) {
        switch (currentChar[0]) {
            case '\r':         /* carriage return */
                break;
            case '\n':         /* new line char */
                break;
            case '\t':         /* tab char */
                break;
            case ' ':          /* space char */
                break;
            case '\\':         /* line continuation char */
                break;
            case '#':          /* Comment marker */
                TRACE("Comment starts...\n");
                /*
                 * we go through the comment char by char till new line char
                 */
                while (fgets(currentChar, 2, fp) != NULL
                       && (currentChar[0] != '\n'));
                break;
            case ':':          /* colon */
                TRACE("Colon, adding whole line to the current entry...\n");
                MoveFilePtrByX(fp, 1, SEEK_SET);
                if (index == 0) {
                    /* Safeguard against a line continuation found if we */
                    /* haven't seen the start of an entry. In this case  */
                    /* parse up to the end of this line.                 */
                    while (fgets(currentChar, 2, fp) != NULL
                           && currentChar[0] != '\n');
                } else {
                    ReadFileToEndOfLineAndFillBuffer(fp,
                        & ((*buffer)[index - 1]), &spaceAvailable,
                        APSCFG_MAX_CONFIG_LINE_SIZE);
                }
                break;
            default:           /* All other chars */
                /* We come here when an entry starts with a char other than
                 * that has been dealt with in the above statements.
                 * This means this is the start of a new entry so we
                 * put a null character at the end of the last entry and then
                 * allocate space for the new one and copy the current line to 
                 * this new space. 
                 */
                {
                    /* Allocate space for a string to hold this line */
                    char *newString = (char *)TrackMemAlloc(*buffer,
                        APSCFG_MAX_CONFIG_LINE_SIZE, 0);

                    if (! newString) {
                        ERROR("Error allocating new line for array of"
                              "printcap lines\n");
                        return -1;
                    }
                    *newString = '\0'; /* make it empty */
                    /* Add to tail of array */
                    if (NULL == TrackArrayAddLast_PtrChar(buffer, newString)) {
                        ERROR("Error extending array of printcap lines\n");
                        return -1;
                    }
                    ASSERT( (*buffer)[index] == newString );
                }
                spaceAvailable = APSCFG_MAX_CONFIG_LINE_SIZE - 1;

                MoveFilePtrByX(fp, 1, SEEK_SET);
                ReadFileToEndOfLineAndFillBuffer(fp,
                    & ((*buffer)[index]), &spaceAvailable,
                    APSCFG_MAX_CONFIG_LINE_SIZE);
                ++index;
                break;
        }
    }
    return (index);
}

/*---------------------------------------------------------------------------
 * ParseLprPrintcapBuffers()
 *
 * Parse the buffers containing printcap enteries
 *
 * Parameters: buffer       - pointer to pointer to buffers containing 
 *                            printcap enteries
 *
 *             index        - number of enteries in provided buffer
 *
 *             printcapInfo - pointer to a pointer to buffers for 
 *                            translated ApsPrintcapInfo
 *
 *     Return: -1 on error
 */
int ParseLprPrintcapBuffers(char **buffer, int index,
                            Aps_Printcap_Info ** printcapInfo)
{

    int len = 0;
    BOOL option = FALSE;
    BOOL alreadyPassed = FALSE;
    BOOL nameCopied = FALSE;
    char optionTypeValue[4];
    PRINTCAP_OPTION_TYPE optionType = OPTION_UNKNOWN;
    int charsCopied = 0;
    char value[APSCFG_MAX_CONFIG_LINE_SIZE];

    int i = 0,
     j = 0;
    char *entry = NULL;

    memset(value, 0, sizeof(value));

    TRACE("Entered\n");
    FIXME("Code not fault-tolerant, eg. contains unchecked allocations");
    for (i = 0; i < index; i++) {
        /* for every entry in the buffer we have to 
           fill a printcap info structure. */
        len = strlen(buffer[i]);
        entry = buffer[i];
        TRACE("length %d", len);
        nameCopied = FALSE;
        for (j = 0; j < len; j++) {
            switch (entry[j]) {
                case ':':      /* options seperator */

                    value[charsCopied] = '\0';
                    if (!option) {
                        /* buffer is the name of the printer */
                        if (!nameCopied) {	/*!printcapInfo[i]->printerName) */
                            printcapInfo[i]->printerName =
                                (char *)calloc(1, charsCopied + 1);
                            strcpy(printcapInfo[i]->printerName, value);
                            nameCopied = TRUE;
                        }
                        if (value[0] == 'l' && value[1] == 'p') {
                            /* this is the default printer */
                            printcapInfo[i]->defaultPrinter = TRUE;
                            memset((void *)value, 0, charsCopied);
                            charsCopied = 0;
                        }
                    } else {
                        /* option value has ended
                         * if statement chain to copy value of 
                         *  option to the right place
                         */
                        optionType = DetermineOptionType(optionTypeValue);
                        switch (optionType) {
                            case OPTION_ACCOUNTING_FILE:
                                /* char* data */
                                printcapInfo[i]->accountingFile =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->accountingFile, value);
                                break;
                            case OPTION_BAUD_RATE:
                                /* int data */
                                printcapInfo[i]->baudRate = atoi(value);
                                break;
                            case OPTION_CIFPLOT_FILTER:
                                /* char* data */
                                printcapInfo[i]->cifplotFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->cifplotFilter, value);
                                break;
                            case OPTION_TEST_DATA_FILTER:
                                printcapInfo[i]->dftestDataFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->dftestDataFilter,
                                       value);
                                break;
                            case OPTION_IS_TTY:
                                printcapInfo[i]->isTTY = atoi(value);
                                break;
                            case OPTION_FORM_FEED:
                                printcapInfo[i]->formFeed =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->formFeed, value);
                                break;
                            case OPTION_FF_ON_OPEN:
                                printcapInfo[i]->formFeedOnOpen = atoi(value);
                                break;
                            case OPTION_IS_TTY_BITS:
                                printcapInfo[i]->isTTYBits = atoi(value);
                                break;
                            case OPTION_IS_GRAPH_DATA_FILTER:
                                printcapInfo[i]->graphDataFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->graphDataFilter,
                                       value);
                                break;
                            case OPTION_HEADER_PAGE_LAST:
                                printcapInfo[i]->headerPageLast = atoi(value);
                                break;
                            case OPTION_IOCTL_SUPPORT:
                                printcapInfo[i]->ioctlSupport = atoi(value);
                                break;
                            case OPTION_TEXT_FILTER:
                                printcapInfo[i]->textFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->textFilter, value);
                                break;
                            case OPTION_LOG_FILE:
                                printcapInfo[i]->logFile =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->logFile, value);
                                break;
                            case OPTION_LOCK_FILE:
                                printcapInfo[i]->lockFile =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->lockFile, value);
                                break;
                            case OPTION_DEVICE:
                                printcapInfo[i]->deviceName =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->deviceName, value);
                                break;
                            case OPTION_MAX_FILE_SIZE:
                                printcapInfo[i]->maxFileSize = atoi(value);
                                break;
                            case OPTION_NEXT_DIRECTORY:
                                printcapInfo[i]->nextDirectory =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->nextDirectory, value);
                                break;
                            case OPTION_DITROFF_FILTER:
                                printcapInfo[i]->ditroffFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->ditroffFilter, value);
                                break;
                            case OPTION_OUTPUT_FILTER:
                                printcapInfo[i]->outputFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->outputFilter, value);
                                break;
                            case OPTION_PRICE_PER_FOOT:
                                printcapInfo[i]->pricePerFoot = atoi(value);
                                break;
                            case OPTION_LINES_PER_PAGE:
                                printcapInfo[i]->pageLengthInLines =
                                    atoi(value);
                                break;
                            case OPTION_CHARS_ACROSS_PAGE:
                                printcapInfo[i]->pageWidthInChars =
                                    atoi(value);
                                break;
                            case OPTION_PAGE_WIDTH_PIXELS:
                                printcapInfo[i]->pageWidthInPixels =
                                    atoi(value);
                                break;
                            case OPTION_PAGE_HEIGHT_PIXELS:
                                printcapInfo[i]->pageLengthInPixels =
                                    atoi(value);
                                break;
                            case OPTION_FORTRAN_STYLE_FILTER:
                                printcapInfo[i]->fortranStyleFileFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->fortranStyleFileFilter,
                                       value);
                                break;
                            case OPTION_RESTRICTED_GROUP:
                                printcapInfo[i]->restrictedGroup =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->restrictedGroup,
                                       value);
                                break;
                            case OPTION_REMOTE_MACHINE:
                                printcapInfo[i]->remoteMachine =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->remoteMachine, value);
                                break;
                            case OPTION_REMOTE_PRINTER_ARGUMENT:
                                printcapInfo[i]->remotePrinterArgument =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->remotePrinterArgument,
                                       value);
                                break;
                            case OPTION_RESTRICT_TO_LOCAL_ACC:
                                printcapInfo[i]->
                                    restrictToRemoteUsersWithLocalAcc =
                                    atoi(value);
                                break;
                            case OPTION_READ_WRITE_DEVICE:
                                printcapInfo[i]->openPrinterDeviceForReadWrite
                                    = atoi(value);
                                break;
                            case OPTION_SHORT_BANNER:
                                printcapInfo[i]->shortBanner = atoi(value);
                                break;
                            case OPTION_SUPPRESS_COPIES:
                                printcapInfo[i]->suppressMultipleCopies =
                                    atoi(value);
                                break;
                            case OPTION_SPOOL_DIR:
                                printcapInfo[i]->spoolDirectory =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->spoolDirectory, value);
                                break;
                            case OPTION_SUPPRESS_FORM_FEED:
                                printcapInfo[i]->supressFormFeed = atoi(value);
                                break;
                            case OPTION_SUPPRESS_BURST_PG_HDR:
                                printcapInfo[i]->suppressBurstPageHeader =
                                    atoi(value);
                                break;
                            case OPTION_STATUS_FILE:
                                printcapInfo[i]->statusFileName =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->statusFileName, value);
                                break;
                            case OPTION_TROFF_FILTER:
                                printcapInfo[i]->troffDataFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->troffDataFilter,
                                       value);
                                break;
                            case OPTION_TRAILING_STRING:
                                printcapInfo[i]->trailingString =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->trailingString, value);
                                break;
                            case OPTION_RASTER_FILTER:
                                printcapInfo[i]->rasterImageFilter =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->rasterImageFilter,
                                       value);
                                break;
                            case OPTION_UNKNOWN:
                                /* An unrecognized option; that's okay. */
                                WARN("Ignoring unrecognized printcap option");
                                break;
                            default:
                                /* DetermineOptionType() returned a value */
                                /* that is not implemented here.          */
                                ASSERT(FALSE);
                        }
                        memset(optionTypeValue, 0, 4);
                        alreadyPassed = FALSE;
                        memset((void *)value, 0, charsCopied);
                        charsCopied = 0;

                    }
                    /* this is to avoid confusion because of overuse of colon
                     * in older printcaps. Once it passes end-of-line colon
                     * it will memset the value buffer and we will know that
                     * by checking this and should not try to memset again.
                     */
                    if (value[0] != '0' && value[1] != '0') {
                        alreadyPassed = FALSE;
                        option = FALSE;
                        memset((void *)value, 0, charsCopied);
                        charsCopied = 0;
                    }
                    break;
                case '\\':     /* ignore this character */

                    break;
                case '|':      /* alias seperator (used only in debian's 
                                   * magicfilterconfig script generated printcap 
                                   * entry for specifying default printer and 
                                   * printer nick name and Manufacturer and model
                                 */

                    if (!alreadyPassed) {
                        /* if buffer contains "lp" this is the default
                         * printer
                         */
                        if (value[0] == 'l' && value[1] == 'p') {
                            /*this is the default printer */
                            printcapInfo[i]->defaultPrinter = TRUE;
                            memset((void *)value, 0, charsCopied);
                            charsCopied = 0;
                        } else {
                            /* this is the printer name */
                            if (!nameCopied) {
                                printcapInfo[i]->printerName =
                                    (char *)calloc(1, charsCopied + 1);
                                strcpy(printcapInfo[i]->printerName, value);
                                nameCopied = TRUE;
                            }
                        }

                        /* else this is the short name and */
                        alreadyPassed = TRUE;
                    } else {
                        /* this is the printer name 
                         * alreadyPassed = FALSE; 
                         * not setting the already passed flag after
                         * passing it again to ignore the third entry(for
                         * default) and second for rest in the name       
                         */
                        if (!nameCopied) {
                            printcapInfo[i]->printerName =
                                (char *)calloc(1, charsCopied + 1);
                            strcpy(printcapInfo[i]->printerName, value);
                            nameCopied = TRUE;
                        }
                    }
                    break;
                case '#':
                    /* option=value seperator(legacy way of doing it) */

                case '=':
                    /* option=value seperator */

                    option = TRUE;
                    /* get the option type */
                    ASSERT((charsCopied < 4));
                    strncpy(optionTypeValue, value, charsCopied);
                    optionTypeValue[charsCopied] = '\0';
                    memset((void *)value, 0, charsCopied);
                    charsCopied = 0;
                    break;
                default:
                    /* copy the char */
                    value[charsCopied] = entry[j];
                    /* increment the copied char counter */
                    charsCopied++;
                    /* catch buffer overflow! -- NASTY! */
                    if (charsCopied >= sizeof(value)) return -1;
                    break;
            }                   /* switch statement */

        }                       /* for loop for characters in the entry */

    }                           /* for loop for entry */

    TRACE("Done\n");
    return 0;
}

/* ---------------------------------------------------------------------------
 * SmartStrcmp()
 *
 * Compares two strings. Can handle null pointers. 
 *
 * Parameters: string1 - pointer to first string
 *
 *             string2 - pointer to second string
 *
 *     Return: 1 if equal, 0 if not equal...
 * 
 */
static int SmartStrcmp(char *string1, char *string2)
{
    if (string1 == NULL) {
        if (string2 == NULL)    /* both are NULL hence equal */
            return 1;
    } else {
        if (string2 != NULL) {
            if (strcmp(string1, string2) == 0)
                return 1;
        }
    }
    return 0;
}

/* ---------------------------------------------------------------------------
 * IsEqual()
 *
 * Compares to Aps_Printcap_Info structures.
 *
 * Parameters: first  - pointer to first Aps_Printcap_Info structure
 *
 *             second - pointer to second Aps_Printcap_Info structure
 *
 *     Return: 1 if equal else 0
 * 
 */
static int IsEqual(Aps_Printcap_Info * first, Aps_Printcap_Info * second)
{
    /*
     * Other then Name of the printer if every thing is same 
     * it is the same printer.
     */
    if (SmartStrcmp(first->accountingFile, second->accountingFile) &&
        first->baudRate == second->baudRate &&
        SmartStrcmp(first->cifplotFilter, second->cifplotFilter) &&
        SmartStrcmp(first->dftestDataFilter, second->dftestDataFilter) &&
        first->isTTY == second->isTTY &&
        SmartStrcmp(first->formFeed, second->formFeed) &&
        first->formFeedOnOpen == second->formFeedOnOpen &&
        first->isTTYBits == second->isTTYBits &&
        SmartStrcmp(first->graphDataFilter, second->graphDataFilter) &&
        first->headerPageLast == second->headerPageLast &&
        first->ioctlSupport == second->headerPageLast &&
        SmartStrcmp(first->textFilter, second->textFilter) &&
        SmartStrcmp(first->logFile, second->logFile) &&
        SmartStrcmp(first->lockFile, second->lockFile) &&
        SmartStrcmp(first->deviceName, second->deviceName) &&
        first->maxFileSize == first->maxFileSize &&
        SmartStrcmp(first->nextDirectory, second->nextDirectory) &&
        SmartStrcmp(first->ditroffFilter, second->ditroffFilter) &&
        SmartStrcmp(first->outputFilter, second->outputFilter) &&
        first->pricePerFoot == second->pricePerFoot &&
        first->pageLengthInLines == second->pageLengthInLines &&
        first->pageWidthInChars == second->pageWidthInChars &&
        first->pageWidthInPixels == second->pageWidthInPixels &&
        first->pageLengthInPixels == second->pageLengthInPixels &&
        SmartStrcmp(first->fortranStyleFileFilter,
                    second->fortranStyleFileFilter) &&
        SmartStrcmp(first->restrictedGroup, second->restrictedGroup) &&
        SmartStrcmp(first->remoteMachine, second->remoteMachine) &&
        SmartStrcmp(first->remotePrinterArgument,
                    second->remotePrinterArgument) &&
        first->restrictToRemoteUsersWithLocalAcc ==
        second->restrictToRemoteUsersWithLocalAcc &&
        first->openPrinterDeviceForReadWrite ==
        second->openPrinterDeviceForReadWrite &&
        first->shortBanner == second->shortBanner &&
        first->suppressMultipleCopies == second->suppressMultipleCopies &&
        SmartStrcmp(first->spoolDirectory, second->spoolDirectory) &&
        first->supressFormFeed == second->supressFormFeed &&
        first->suppressBurstPageHeader == second->suppressBurstPageHeader &&
        SmartStrcmp(first->statusFileName, second->statusFileName) &&
        SmartStrcmp(first->troffDataFilter, second->troffDataFilter) &&
        SmartStrcmp(first->trailingString, second->trailingString) &&
        SmartStrcmp(first->rasterImageFilter, second->rasterImageFilter)) {
        return 1;
    }
    return 0;
}

/* ---------------------------------------------------------------------------
 * ConsolidateApsPrintcapInfoArray()
 *
 * Consolidates duplicate printcap enteries for default printer for enteries
 * made by printtool.
 *
 * Only considers a SINGLE duplicate of the default printer "lp" though
 * its duplicate may be called by another name.
 *
 * Parameters: printcapInfo - pointer to a pointer to Aps_Printcap_Info
 *                            buffer
 *
 *             size         - number of enteries in the buffer
 *
 *     Return: number of printcap enteries remaining in the buffer
 * 
 */
static int ConsolidateApsPrintcapInfoArray(Aps_Printcap_Info ** printcapInfo,
                                           int size)
{
    /* Some printcap file formats duplicate entries 
     * for default printer. We look for such duplicate 
     * entries and remove them.
     */
    int i,
     j,
     k,
     retVal = size;

    for (i = 0; i < size; i++) {
        if (printcapInfo[i]->defaultPrinter) {
            /* Find 'lp', this is the entry we wish to keep in case we find
             * another identical default printer.
             */
            if (SmartStrcmp(printcapInfo[i]->printerName, "lp")) {
                /* This is the printer. Compare it with the rest.
                 */
                for (j = 0; j < size; j++) {
                    if (i != j) {
                        /* IsEqual ignores name and defaultPrinter fields
                         * since the dupe may not be called 'lp' and
                         * only those called 'lp' are flagged with
                         * defaultPrinter, thus, only compare the settings.
                         */
                        if (IsEqual(printcapInfo[i], printcapInfo[j])) {
                            /* Printers are equal...
                             * Need to remove extra entry
                             * printcapInfo[i] is the one with 'lp'
                             * and default tag so change the name for that...
                             */
                            /* Copying the name pointer */
                            free(printcapInfo[i]->printerName);
                            printcapInfo[i]->printerName =
                                printcapInfo[j]->printerName;
                            printcapInfo[j]->printerName = NULL;

                            /* Free redundant printcapInfo */
                            FreePrintcapInfoStructure(printcapInfo[j]);

                            /* Now move up all the enteries
                             * from j+1 to size by one position...
                             */
                            for (k = j + 1; k < size; k++) {
                                /* j+1 is the next valid entry and 
                                 * j is the empty spot.
                                 */
                                printcapInfo[k - 1] = printcapInfo[k];
                            }

                            /* Last entry is now last-1 so getting rid of
                             * last entry
                             */
                            printcapInfo[size - 1] = NULL;
                            j = size;	/* to bail out */
                            i = size;
                            retVal--;	/* it 'll always be -1 */

                        }
                    }
                }
            }
        }
    }
    return retVal;
}

/* ---------------------------------------------------------------------------
 * LoadLprPrintcapFile()
 *
 * Loads printcap file into Aps_Printcap_Info buffers.
 *
 * Parameters: file path    - pointer to string containing path for 
 *                            printcap file.
 *
 *             printcapInfo - pointer to a pointer to a pointer for 
 *                            Aps_Printcap_Info buffers.
 *
 *     Return: number of enteries in the  printcap file.
 * 
 */
int LoadLprPrintcapFile(FILE * fp, Aps_Printcap_Info *** printcapInfo)
{
    TrackArray_PtrChar buffer;
    int numberOfEntries = 0;
    int rtn;
    int i = 0;

    /* Check parameters */
    if (! fp) {
        ERROR("Received invalid file pointer");
        return -1;
    }
    ASSERT(printcapInfo);

    /* Pre-initialize return value */
    *printcapInfo = NULL;

    /* read printcap into the buffer */
    buffer = TrackArrayNew_PtrChar(NULL, 0);
    if (! buffer) {
        ERROR("Unable to allocate memory for printcap line array");
        return -1;
    }
    numberOfEntries = LoadPrintcapIntoBuffer(fp, & buffer);

    /* Check for failure. */
    rtn = (numberOfEntries != -1);
    if (rtn) {
        TRACE("Number of Entries in Printcap file = %d\n", numberOfEntries);
        if (numberOfEntries > 0) {
            /* Create array */
            *printcapInfo = (Aps_Printcap_Info **)
                calloc(1, sizeof(Aps_Printcap_Info **) * numberOfEntries);
            if (*printcapInfo) {
                /* Alloc space for all entries */
                for (i = 0; i < numberOfEntries; i++) {
                    (*printcapInfo)[i] = (Aps_Printcap_Info *)
                    calloc(1, sizeof(Aps_Printcap_Info));
                    if (!(*printcapInfo)[i]) { rtn = FALSE; break; }
                }
                if (rtn) {
                    /* Parse buffer */
                    if (-1 != ParseLprPrintcapBuffers(buffer, numberOfEntries,
                        *printcapInfo)) {
                        /* Consolidating printcap entries maintly for
                         * Printtool printcaps */
                        numberOfEntries = ConsolidateApsPrintcapInfoArray(
                            *printcapInfo, numberOfEntries);
                        /* success!! */
                    } else ERROR("ParseLprPrintcapBuffers() failed\n");
                } else ERROR("Error allocating Aps_Printcap_Info elements\n");
                /* Back out of allocations */
                if (! rtn) {
                    for (i = 0; i < numberOfEntries; ++i) {
                        if ((*printcapInfo)[i]) free((*printcapInfo)[i]);
                    }
                    free(*printcapInfo);
                    *printcapInfo = NULL;
                }
            } else ERROR("Error allocating Aps_Printcap_Info container\n");
        } else WARN("Empty printcap, returning NULL array");
    }
    TrackArrayDelete_PtrChar(buffer);
    return (rtn) ? numberOfEntries : -1;
}

/* ---------------------------------------------------------------------------
 * WritePrinterRecInPrintcapFile()
 *
 * Writes the printer record in printcap file.
 *
 * Parameters:fp     -  pointer to the temporary printcap file which will 
 *                       be overwritten to printcap file. 
 *
 *            printerRec - pointer to the structure storing the printcap rec 
 *                          for a printer to be written in printcap.
 *
 *     Return: TRUE   -  if the operation is successful, FALSE otherwise.
 * 
 */

static BOOL WritePrinterRecInPrintcapFile(FILE * fp, Aps_Printcap_Info * printerRec)
{
    TRACE("fp = %p, printerRec = %p, printerRec->printerName = %s",
        fp, printerRec, (printerRec && printerRec->printerName) ?
        printerRec->printerName : "???");
    if ((printerRec == NULL) || (fp == NULL) ||
        (printerRec->printerName == NULL)) {
        ERROR("Got invalid arguments");
        return FALSE;
    }
    /* Write the printer name entry */
    if (printerRec->defaultPrinter == TRUE) {
        TRACE("Writing printer name as \"lp|%s:\\\"", printerRec->printerName);
        fprintf(fp, "lp|%s:\\\n", printerRec->printerName);
    } else {
        TRACE("Writing printer name as \"%s:\\\"", printerRec->printerName);
        fprintf(fp, "%s:\\\n", printerRec->printerName);
    }
    /* Write the value of different fields in the file */
    WriteStringValueInPrintcap(fp, printerRec->accountingFile, "af");
    WriteIntegerValueInPrintcap(fp, printerRec->baudRate, "br");
    WriteStringValueInPrintcap(fp, printerRec->cifplotFilter, "cf");
    WriteStringValueInPrintcap(fp, printerRec->dftestDataFilter, "df");
    WriteIntegerValueInPrintcap(fp, printerRec->isTTY, "fc");
    WriteStringValueInPrintcap(fp, printerRec->formFeed, "ff");
    WriteIntegerValueInPrintcap(fp, printerRec->formFeedOnOpen, "fo");
    WriteIntegerValueInPrintcap(fp, printerRec->isTTYBits, "fs");
    WriteStringValueInPrintcap(fp, printerRec->graphDataFilter, "gf");
    WriteIntegerValueInPrintcap(fp, printerRec->headerPageLast, "hl");
    WriteIntegerValueInPrintcap(fp, printerRec->ioctlSupport, "ic");
    WriteStringValueInPrintcap(fp, printerRec->textFilter, "if");
    WriteStringValueInPrintcap(fp, printerRec->logFile, "lf");
    WriteStringValueInPrintcap(fp, printerRec->lockFile, "lo");
    WriteStringValueInPrintcap(fp, printerRec->deviceName, "lp");
    WriteStringValueInPrintcap(fp, printerRec->nextDirectory, "nd");
    WriteStringValueInPrintcap(fp, printerRec->ditroffFilter, "nf");
    WriteStringValueInPrintcap(fp, printerRec->outputFilter, "of");
    WriteIntegerValueInPrintcap(fp, printerRec->pricePerFoot, "pc");
    WriteIntegerValueInPrintcap(fp, printerRec->pageLengthInLines, "pl");
    WriteIntegerValueInPrintcap(fp, printerRec->pageWidthInChars, "pw");
    WriteIntegerValueInPrintcap(fp, printerRec->pageWidthInPixels, "px");
    WriteIntegerValueInPrintcap(fp, printerRec->pageLengthInPixels, "py");
    WriteStringValueInPrintcap(fp, printerRec->fortranStyleFileFilter, "rf");
    WriteStringValueInPrintcap(fp, printerRec->restrictedGroup, "rg");
    WriteStringValueInPrintcap(fp, printerRec->remoteMachine, "rm");
    WriteStringValueInPrintcap(fp, printerRec->remotePrinterArgument, "rp");
    WriteIntegerValueInPrintcap(fp,
                       printerRec->restrictToRemoteUsersWithLocalAcc, "rs");
    WriteIntegerValueInPrintcap(fp,
                           printerRec->openPrinterDeviceForReadWrite, "rw");
    WriteIntegerValueInPrintcap(fp, printerRec->shortBanner, "sb");
    WriteIntegerValueInPrintcap(fp,
                                printerRec->suppressMultipleCopies, "sc");
    WriteStringValueInPrintcap(fp, printerRec->spoolDirectory, "sd");
    WriteIntegerValueInPrintcap(fp, printerRec->supressFormFeed, "sf");
    WriteStringValueInPrintcap(fp, printerRec->statusFileName, "st");
    WriteStringValueInPrintcap(fp, printerRec->troffDataFilter, "tf");
    WriteStringValueInPrintcap(fp, printerRec->trailingString, "tr");
    WriteStringValueInPrintcap(fp, printerRec->rasterImageFilter, "vf");
    if (!printerRec->suppressBurstPageHeader)
       fputs("\t:sh0:\\\n", fp);
    else
       fputs("\t:sh1\\\n", fp);
    fprintf(fp, "\t:%s%d:\n", "mx#", printerRec->maxFileSize);

    return TRUE;
}
/* ---------------------------------------------------------------------------
 * WriteStringValueInPrintcap()
 *
 * Writes the string mnemonic and its value in the temporary printcap file.
 *
 * Parameters: fp     -  pointer to the temporary printcap file which will 
 *                       be overwritten to printcap file. 
 *
 *             value  -  value of the mnemonic.
 *                     
 *             mnemonic - name of the mnemonic.
 *
 *     Return: void.
 * 
 */
static void WriteStringValueInPrintcap(FILE * fp, char *value, char *mnemonic)
{
    if (value != NULL)
        fprintf(fp, "\t:%s=%s:\\\n", mnemonic, value);
}
/*---------------------------------------------------------------------------
 * WriteIntegerValueInPrintcap()
 *
 * Writes the integer mnemonic and its value in the temporary printcap file.
 *
 * Parameters: fp     -  pointer to the temporary printcap file which will 
 *                       be overwritten to printcap file. 
 *
 *             value  -  value of the mnemonic.
 *                     
 *             mnemonic - name of the mnemonic.
 *
 *     Return: void.
 * 
 */
static void WriteIntegerValueInPrintcap(FILE * fp, int value, char *mnemonic)
{
    if (value != 0)
        fprintf(fp, "\t:%s=%d:\\\n", mnemonic, value);
}
/*---------------------------------------------------------------------------
 * WritePrinterEntriesInPrintcap()
 *
 * This function performs the add/modify/delete for the printers in the
 * printcap file.
 *
 * Parameters: thisBase - the pointer to the transport layer.
 *
 *     Return: APS_GENERIC_FAILURE, if there is a problem in creating the
 *             directory or allocation of memory and APS_SUCCESS otherwise.
 */

Aps_Result WritePrinterEntriesInPrintcap(char *printcapFileName, 
                                         Aps_Printcap_Info ** printcap,
                                         int numberOfPrintcapEntries)
{
    int fd;
    char *tmpFileName;
    Aps_Result result = APS_SUCCESS;

    /* Get temp name */
    tmpFileName = tempnam(NULL, "APSpc"); /* uses malloc */
    if (! tmpFileName) {
        ERROR("Could not get filename for temporary file");
        return APS_OUT_OF_MEMORY;
    }

    /* Open and set permissions */
    TRACE("Creating temporary file: %s", tmpFileName);
    errno = 0;
    fd = open(tmpFileName, O_CREAT | O_TRUNC | O_WRONLY,
        S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
    if (fd != -1) {
        FILE *fp;
        fchown(fd, 0, 0); /* owner: root, group: root */
        errno = 0;
        if ( (fp = fdopen(fd, "w")) ) { /* open for c-style access */
            int i;
            /* Write the Comments in the file */
            fputs(PrintcapHeaderComment, fp);
            /* Generate the printcap entries */
            for (i = 0; i < numberOfPrintcapEntries; i++) {
                int ret = WritePrinterRecInPrintcapFile(fp, printcap[i]);
                if (! ret) {
                    ERROR("WritePrinterRecInPrintcapFile() failed on entry"
                          " # %d", i);
                    result = APS_GENERIC_FAILURE;
                    break;
                }
            }
            /* Close (also closes fd) */
            fclose(fp);
        } else {
            /* Was unable to create file pointer */
            close(fd);
            result = (errno) ? GetResultFromErrno() : APS_GENERIC_FAILURE;
            ERROR("Could not get FILE * from FD");
        }
        /* Overwrite the current printcap file with backup printcap file */
        if (result == APS_SUCCESS) {
            FIXME("Backed-up writing algoritm is broken!");
            remove(printcapFileName);
            if (rename(tmpFileName, printcapFileName) == -1) {
                ERROR("Unable to rename new file over backup."
                      "Old file destroyed!!!");
                result = APS_GENERIC_FAILURE;
            }
        }
        /* ... or delete temp file if failed */
        if (result != APS_SUCCESS) {
            remove(tmpFileName);
        }
    } else result = (errno) ? GetResultFromErrno() : APS_GENERIC_FAILURE;

    /* Free tmp file name and quit! */
    free(tmpFileName);
    if (result != APS_SUCCESS) {
        ERROR("Failed with result: %s", RESULTSTR(result));
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * FreePrintcapInfoStructure()
 *
 * Frees the memory occupied by each Aps_Printcap_Info structure
 * member and the structure itself.
 *
 *
 * Parameters: rec - the pointer to the Aps_Printcap_Info structure to be
 *                   freed.
 *
 *     Return: void.
 */
void FreePrintcapInfoStructure(Aps_Printcap_Info * rec)
{
    TRACE("Freeing Aps_Printcap_Info structure: %p", rec);
    ASSERT(rec != NULL);
    if (rec->printerName != NULL)
        free(rec->printerName);
    if (rec->accountingFile != NULL)
        free(rec->accountingFile);
    if (rec->cifplotFilter != NULL)
        free(rec->cifplotFilter);
    if (rec->dftestDataFilter != NULL)
        free(rec->dftestDataFilter);
    if (rec->formFeed != NULL)
        free(rec->formFeed);
    if (rec->graphDataFilter != NULL)
        free(rec->graphDataFilter);
    if (rec->textFilter != NULL)
        free(rec->textFilter);
    if (rec->logFile != NULL)
        free(rec->logFile);
    if (rec->lockFile != NULL)
        free(rec->lockFile);
    if (rec->deviceName != NULL)
        free(rec->deviceName);
    if (rec->nextDirectory != NULL)
        free(rec->nextDirectory);
    if (rec->ditroffFilter != NULL)
        free(rec->ditroffFilter);
    if (rec->outputFilter != NULL)
        free(rec->outputFilter);
    if (rec->fortranStyleFileFilter != NULL)
        free(rec->fortranStyleFileFilter);
    if (rec->restrictedGroup != NULL)
        free(rec->restrictedGroup);
    if (rec->remoteMachine != NULL)
        free(rec->remoteMachine);
    if (rec->remotePrinterArgument != NULL)
        free(rec->remotePrinterArgument);
    if (rec->spoolDirectory != NULL)
        free(rec->spoolDirectory);
    if (rec->statusFileName != NULL)
        free(rec->statusFileName);
    if (rec->troffDataFilter != NULL)
        free(rec->troffDataFilter);
    if (rec->trailingString != NULL)
        free(rec->trailingString);
    if (rec->rasterImageFilter != NULL)
        free(rec->rasterImageFilter);

    free (rec);
}
