/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: trackarray.h
 *
 * Description: Builds on memory handling schemes implemented by
 *              TrackMem to construct compact dynamically-
 *              resized arrays of elements of an arbitrary size.
 *              Of course the contained elements must not lose
 *              coherence on shallow copy operations.
 */

#ifndef TRACK_ARRAY_H
#define TRACK_ARRAY_H

#include <stdlib.h>
#include "trackmem.h"

#ifdef TRACKARRAY_STORAGE
#define STORAGEQUALIFIER TRACKARRAY_STORAGE
#else
#define STORAGEQUALIFIER
#endif

/*** Types ***/
typedef TMEM void TrackArray;

/*** Protos ***/
/* managing arbitrarily-sized array buffers */
TrackArray* TrackArrayNew(TMEM void* memparent, size_t size, int initial);
TrackArray* TrackArrayClone(TMEM void* memparent, const TrackArray* src);
void TrackArrayDelete(TrackArray* array);
void *TrackArrayUnlink(TrackArray* src);
/* resizing */
int TrackArrayResize(TrackArray** array, int numElems);
int TrackArrayEnsureStorageForElem(TrackArray** array, int elem);
/* basic information */
int TrackArrayGetSize(const TrackArray* array);
size_t TrackArrayGetElemSize(const TrackArray* array);
int TrackArrayGetIndexByRef(const TrackArray* array, const void* elem);
/* set/get individual elements w/o worrying about memcpy()/memmove() */
void TrackArrayGetElemByRef(const TrackArray* array, const void* elem,
    void *data);
void TrackArraySetElemByRef(TrackArray* array, void* elem, const void *data);
/* add/remove single elements */
void* TrackArrayAddLast(TrackArray** array, const void* data);
void TrackArrayRemoveLast(TrackArray** array);
void* TrackArrayInsertAt(TrackArray** array, int elem, const void* data);
void TrackArrayRemoveAt(TrackArray** array, int elem);
/* add/remove/extract multiple elements */
void* TrackArrayInsertManyAt(TrackArray** targetArray, int targetElem,
    int count, const void *srcData);
void* TrackArrayAppendMany(TrackArray** targetArray, int count,
    const void *srcData);
void TrackArrayRemoveManyAt(TrackArray** srcArray, int srcElem,
    int count);
void TrackArrayGetManyAt(const TrackArray* srcArray, int srcElem,
    int count, void *targetData);
void TrackArraySetManyAt(TrackArray* targetArray, int targetElem,
    int count, const void *srcCount);
/* get element references */
void* TrackArrayElemAt(const TrackArray* array, int elem);
void* TrackArrayElemFirst(const TrackArray* array);
void* TrackArrayElemLast(const TrackArray* array);
void* TrackArrayElemEndMark(const TrackArray* array);
void* TrackArrayElemNext(const TrackArray* array, const void* ptrelem);
void* TrackArrayElemSkip(const TrackArray* array, const void* ptrelem,
    int num);

/*** TrackArray "Template" ***/
/*
 * The templates defined in these steps produce functions of
 * the form TrackArray[Operation]_[Type].  With the exception of
 * New, InsertAt and AddLast, their functionality mimics that
 * of the generic routines.
 *
 * The templates also produce type TrackArray_[Type] which is equivalent
 * to [Type]*.  Please use this whenever possible (except in places where
 * the public interface is not to disclose an underlying type) as it
 * improves overall readability of constructs using TrackArray.
 *
 * New types:
 *   TrackArray_[Type]        - base pointer to array
 *   TrackArrayElem_[Type]    - type of an element of that array
 *   TrackArrayElemRef_[Type] - type of a reference to an array element
 *                              stored inside an array (don't use for
 *                              arbitrary element references)
 *
 * Member function differences:
 *   TrackArrayNew_[Type]            - does not require element size
 *   TrackArrayInsertAt_[Type]       - accepts plain types (not buffers)
 *   TrackArrayAddLast_[Type]        - accepts plain types (not buffers)
 *   TrackArrayGetElemByRef_[Type]   - does not exist, use indirection
 *                                     and assignment operation on ref ptr
 *   TrackArraySetElemByRef_[Type]   - does not exist, use indirection
 *                                     and assignment operation on ref ptr
 *
 * Template instantiation must occur in two phases.
 * First declare the template members (prototypes) using
 *   the TrackArray_Declare() operation anywhere these operations are
 *   needed.
 * Then instantiate the template in one module using the
 *   TrackArray_Implement() operation.
 *
 * TrackArray_Declare() requires the specification of a name (1st arg) and
 * a base type (2nd arg).  The 1st arg is used in the function declarations
 * to distinguish between differently-types function (eg. Int), the 2nd
 * specifies the actual storage type of an array element and may be either
 * a primitive type or a compound type.  The 2nd arg should not contain
 * cv-modifiers (const, volatile) as that may result in compile-time
 * errors and is not really desirable anyhow.
 *
 * TrackArray_Implement() requires the name provided in a previous Declare()
 * operation (ie. its first arg).
 *
 * If TRACKARRAY_STORAGE is declared, it may be used to specify that
 * an implementation should be made "inline", "extern", or "static".
 * eg.  #define TRACKARRAY_STORAGE static inline
 *      TrackArray_Declare(Int, int)
 *      TrackArray_Implement(Int)
 *      #undef TRACKARRAY_STORAGE
 *
 * Declaring the operations as inline is safe and requires little
 * additional overhead (in many cases, it may even reduce the effective
 * code footprint) as the member functions are fairly trivial.  There
 * may be substantial gains from doing this, if at all possible.
 */

/*** define a TrackArray "template" declaration ***/
#define TrackArray_Declare( tName, tType ) \
    typedef TMEM tType*             TrackArray_##tName; \
    typedef tType                   TrackArrayElem_##tName; \
    typedef TrackArrayElem_##tName* TrackArrayElemRef_##tName; \
    \
    STORAGEQUALIFIER TrackArray_##tName \
      TrackArrayNew_##tName \
        ( TMEM void*, int ); \
    STORAGEQUALIFIER void \
      TrackArrayDelete_##tName \
        ( TrackArray_##tName ); \
    STORAGEQUALIFIER TrackArray_##tName \
      TrackArrayClone_##tName \
        ( TMEM void*, const TrackArray_##tName ); \
    STORAGEQUALIFIER TrackArrayElem_##tName* \
      TrackArrayUnlink_##tName \
        ( TrackArray_##tName ); \
    STORAGEQUALIFIER int \
      TrackArrayResize_##tName \
        ( TrackArray_##tName*, int ); \
    STORAGEQUALIFIER int \
      TrackArrayEnsureStorageForElem_##tName \
        ( TrackArray_##tName*, int ); \
    STORAGEQUALIFIER int \
      TrackArrayGetSize_##tName \
        ( const TrackArray_##tName ); \
    STORAGEQUALIFIER size_t \
      TrackArrayGetElemSize_##tName \
        ( const TrackArray_##tName ); \
    STORAGEQUALIFIER int \
      TrackArrayGetIndexByRef_##tName \
        ( const TrackArray_##tName, const TrackArrayElemRef_##tName ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayAddLast_##tName \
        ( TrackArray_##tName*, const TrackArrayElem_##tName ); \
    STORAGEQUALIFIER void \
      TrackArrayRemoveLast_##tName \
        ( TrackArray_##tName* ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayInsertAt_##tName \
        ( TrackArray_##tName*, int, const TrackArrayElem_##tName ); \
    STORAGEQUALIFIER void \
      TrackArrayRemoveAt_##tName \
        ( TrackArray_##tName*, int ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayInsertManyAt_##tName \
        ( TrackArray_##tName*, int, int, const TrackArrayElem_##tName* ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayAppendMany_##tName \
        ( TrackArray_##tName*, int, const TrackArrayElem_##tName* ); \
    STORAGEQUALIFIER void \
      TrackArrayRemoveManyAt_##tName \
        ( TrackArray_##tName*, int, int ); \
    STORAGEQUALIFIER void \
      TrackArrayGetManyAt_##tName \
        ( const TrackArray_##tName, int, int, TrackArrayElem_##tName* ); \
    STORAGEQUALIFIER void \
      TrackArraySetManyAt_##tName \
        ( TrackArray_##tName, int, int, const TrackArrayElem_##tName* ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemAt_##tName \
        ( const TrackArray_##tName, int ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemFirst_##tName \
        ( const TrackArray_##tName ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemLast_##tName \
        ( const TrackArray_##tName ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemEndMark_##tName \
        ( const TrackArray_##tName ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemNext_##tName \
        ( const TrackArray_##tName, const TrackArrayElemRef_##tName ); \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemSkip_##tName \
        ( const TrackArray_##tName, const TrackArrayElemRef_##tName, int ); \

/*** define a TrackArray "template" implementation ***/
#define TrackArray_Implement( tName ) \
    STORAGEQUALIFIER TrackArray_##tName \
      TrackArrayNew_##tName \
        ( TMEM void* memparent, int initialAlloc ) { \
        return (TrackArray_##tName)TrackArrayNew(memparent, \
            sizeof(TrackArrayElem_##tName), initialAlloc); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayDelete_##tName \
        ( TrackArray_##tName array ) { \
        TrackArrayDelete((TrackArray*)array); \
    } \
    STORAGEQUALIFIER TrackArray_##tName \
      TrackArrayClone_##tName \
        ( TMEM void* memparent, const TrackArray_##tName oldArray ) { \
        return (TrackArray_##tName)TrackArrayClone(memparent, oldArray); \
    } \
    STORAGEQUALIFIER TrackArrayElem_##tName* \
      TrackArrayUnlink_##tName \
        ( TrackArray_##tName array ) { \
        return (TrackArrayElem_##tName*)TrackArrayUnlink((TrackArray*)array); \
    } \
    STORAGEQUALIFIER int \
      TrackArrayResize_##tName \
        ( TrackArray_##tName* ptrarray, int size ) { \
        return TrackArrayResize((TrackArray**)ptrarray, size); \
    } \
    STORAGEQUALIFIER int \
      TrackArrayEnsureStorageForElem_##tName \
        ( TrackArray_##tName* ptrarray, int index ) { \
        return TrackArrayEnsureStorageForElem((TrackArray**)ptrarray, index); \
    } \
    STORAGEQUALIFIER int \
      TrackArrayGetSize_##tName \
        ( const TrackArray_##tName array ) { \
        return TrackArrayGetSize((const TrackArray*)array); \
    } \
    STORAGEQUALIFIER size_t \
      TrackArrayGetElemSize_##tName \
        ( const TrackArray_##tName array ) { \
         return TrackArrayGetElemSize((const TrackArray*)array); \
    } \
    STORAGEQUALIFIER int \
      TrackArrayGetIndexByRef_##tName \
        ( const TrackArray_##tName array, \
          const TrackArrayElemRef_##tName elem ) { \
         return TrackArrayGetIndexByRef((const TrackArray*)array, \
             (const void*)elem); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayAddLast_##tName \
        ( TrackArray_##tName* ptrarray, \
          const TrackArrayElem_##tName value ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayAddLast((TrackArray**)ptrarray, \
            (const void*)& value); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayRemoveLast_##tName \
        ( TrackArray_##tName* ptrarray ) { \
        return TrackArrayRemoveLast((TrackArray**)ptrarray); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayInsertAt_##tName \
        ( TrackArray_##tName* ptrarray, int index, \
          const TrackArrayElem_##tName value ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayInsertAt((TrackArray**)ptrarray, index, \
            (const void*)& value); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayRemoveAt_##tName \
        ( TrackArray_##tName* ptrarray, int index ) { \
        TrackArrayRemoveAt((TrackArray**)ptrarray, index); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayInsertManyAt_##tName \
        ( TrackArray_##tName* ptrarray, int index, int count, \
          const TrackArrayElem_##tName* source ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayInsertManyAt((TrackArray**)ptrarray, index, count, \
            (const void*)source); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayAppendMany_##tName \
        ( TrackArray_##tName* ptrarray, int count, \
          const TrackArrayElem_##tName* source ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayAppendMany((TrackArray**)ptrarray, count, \
            (const void*)source); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayRemoveManyAt_##tName \
        ( TrackArray_##tName* ptrarray, int index, int count ) { \
        TrackArrayRemoveManyAt((TrackArray**)ptrarray, index, count ); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayGetManyAt_##tName \
        ( const TrackArray_##tName array, int index, int count, \
          TrackArrayElem_##tName* target ) { \
        TrackArrayGetManyAt((const TrackArray*)array, index, count, \
            (void*)target); \
    } \
    STORAGEQUALIFIER void \
      TrackArraySetManyAt_##tName \
        ( TrackArray_##tName array, int index, int count, \
          const TrackArrayElem_##tName* source ) { \
        TrackArraySetManyAt((TrackArray*)array, index, count, \
            (const void*)source); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemAt_##tName \
        ( const TrackArray_##tName array, int index ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemAt((const TrackArray*)array, index); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemFirst_##tName \
        ( const TrackArray_##tName array ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemFirst((const TrackArray*)array); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemLast_##tName \
        ( const TrackArray_##tName array ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemLast((const TrackArray*)array); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemEndMark_##tName \
        ( const TrackArray_##tName array ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemEndMark((const TrackArray*)array); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemNext_##tName \
        ( const TrackArray_##tName array, \
          const TrackArrayElemRef_##tName elem ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemNext((const TrackArray*)array, (const void*)elem); \
    } \
    STORAGEQUALIFIER TrackArrayElemRef_##tName \
      TrackArrayElemSkip_##tName \
        ( const TrackArray_##tName array, \
          const TrackArrayElemRef_##tName elem, \
          int count ) { \
        return (TrackArrayElemRef_##tName) \
            TrackArrayElemSkip((const TrackArray*)array,\
            (const void*)elem, count); \
    }

/*** TrackArrayIndirect "Template" ***/
/*
 * The templates defined in these steps produce functions of
 * the form TrackArrayIndirect[Operation]_[Type].  These template
 * functions build upon the TrackArray templates to provide transparent
 * access to members of an array of pointers.  Specifically, this allows
 * the user to construct an array of pointers to objects while adding
 * and removing individual elements of the array as complete objects.
 *
 * Fundamentally, each operation prefixed with Indirect implies that the
 * target of the operation is not the pointer type of the array, but
 * the object type of the contents.
 *
 * The array is comprised of a set of pointers and a set of objects.
 * By default, these objects are allocated as children of the array's
 * base pointer, but they may be reassigned by the user to displace
 * individual entitites.  Similarly, it is legal for the user to directly
 * modify the contents of the array of pointers so long as he ensures that
 * ALL pointers are valid members or NULL.
 *
 *
 *
 * Summary of new operations:
 *
 * newArray = TrackArrayIndirectNew_[Type](parent, num)
 *   - creates an array linked to parent with initially num objects
 *     (array pointers reference new uninitialized objects)
 *
 * *elem = TrackArrayIndirectNewElem_[Type](parent)
 *   - creates storage for a new element of this array
 *
 * TrackArrayIndirectDelete_[Type](array)
 *   - deletes all individually-references objects of the array, then
 *     deletes array of pointers
 *
 * TrackArrayIndirectDeleteElem_[Type](elem)
 *   - frees the storage for an element of this array
 *
 * newArray = TrackArrayIndirectClone_[Type](array)
 *   - clones all individually-referenced objects of the array and
 *     attaches them to a new array with a newly built list of pointers
 *
 * success = TrackArrayIndirectResize_[Type](ptrarray, num)
 *   - resizes the array, creating/deleting objects at the tail as necessary
 *
 * success = TrackArrayEnsureStorageForElem_[Type](ptrarray, index)
 *   - ensures that element 'index' will fit, resizing if necessary
 *
 * size = TrackArrayIndirectGetSize_[Type](array)
 *   - returns number of elements in the array
 *
 * size = TrackArrayIndirectGetElemSize_[Type](array)
 *   - returns default size of the objects of the array in multiples of
 *     sizeof(char *)
 *
 * index = TrackArrayIndirectGetIndexByRef_[Type](array, *elem)
 *   - returns the index number of a given element by references as found
 *     by scanning the list of pointers for a matching element (ASSERT's on
 *     failure)
 *
 * *elem = TrackArrayIndirectAddLast_[Type](ptrarray, elemsource)
 * *elem = TrackArrayIndirectAddLastByRef_[Type](ptrarray, *elemsource)
 *   - returns a reference to a new object added at the tail of the array
 *     (initializes with shallow-copy of elemsource if not NULL)
 *
 * *elem = TrackArrayIndirectInsertAt_[Type](ptrarray, index, elemsource)
 * *elem = TrackArrayIndirectInsertAtByRef_[Type](ptrarray, index, *elemsource)
 *   - returns a reference to a new object inserted at position 'index'
 *     (initializes with shallow-copy of elemsource if not NULL)
 *
 * TrackArrayIndirectRemoveLast_[Type](ptrarray)
 *   - removes the last element and frees using TrackMemFree()
 *
 * TrackArrayIndirectRemoveAt_[Type](ptrarray)
 *   - removes the element at position 'index' and frees using TrackMemFree()
 *
 * *elem = TrackArrayIndirectInsertManyAt_[Type](ptrarray, index, count,
 *                                               *elemsource)
 *   - inserts a series of elements at position 'index' and initializes
 *     via shallow-copy from 'elemsource' which is a contiguous array of
 *     elements of same size, unless NULL
 *     <initialization unimplemented -- procedure may change>
 *
 * *elem = TrackArrayIndirectAppendMany_[Type](ptrarray, count, *elemsource)
 *   - add a series of elements to the tail of the array as if
 *     TrackArrayIndirectInsertManyAt_[Type]() had been called on that
 *     position.
 *     <initialization unimplemented -- procedure may change>
 *
 * TrackArrayIndirectRemoveManyAt_[Type](ptrarray, index, count)
 *   - removes a series of elements at position 'index' as if they had been
 *     iteratively removed by TrackArrayIndirectRemoveAt_[Type]().
 *
 * *elem = TrackArrayIndirectElemAt_[Type](array, index)
 *   - returns a pointer to the object of the element at a particular index
 *
 * *elem = TrackArrayIndirectElemFirst_[Type](array)
 *   - returns a pointer to the first object
 *
 * *elem = TrackArrayIndirectElemLast_[Type](array)
 *   - returns a pointer to the last object
 *
 *** Maybe someday someone will get around to completing this interface... ***
 */

/*** define a TrackArrayIndirect "template" declaration ***/
#define TrackArrayIndirect_Declare( tName, tPtrName, tType ) \
    typedef TMEM TrackArray_##tPtrName TrackArrayIndirect_##tName; \
    typedef tType                      TrackArrayIndirectElem_##tName; \
    typedef TrackArrayIndirectElem_##tName*  \
        TrackArrayIndirectElemRef_##tName; \
    \
    STORAGEQUALIFIER TrackArrayIndirect_##tName \
      TrackArrayIndirectNew_##tName \
        ( TMEM void*, int ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectNewElem_##tName \
        ( TMEM void* memparent ); \
    STORAGEQUALIFIER void \
      TrackArrayIndirectDelete_##tName \
        ( TrackArrayIndirect_##tName ); \
    STORAGEQUALIFIER void \
      TrackArrayIndirectDeleteElem_##tName \
        ( TrackArrayIndirectElemRef_##tName elem ); \
    STORAGEQUALIFIER TrackArrayIndirect_##tName \
      TrackArrayIndirectClone_##tName \
        ( TMEM void*, const TrackArrayIndirect_##tName ); \
    STORAGEQUALIFIER int \
      TrackArrayIndirectResize_##tName \
        ( TrackArrayIndirect_##tName*, int ); \
    STORAGEQUALIFIER int \
      TrackArrayIndirectEnsureStorageForElem_##tName \
        ( TrackArrayIndirect_##tName*, int ); \
    STORAGEQUALIFIER int \
      TrackArrayIndirectGetSize_##tName \
        ( const TrackArrayIndirect_##tName ); \
    STORAGEQUALIFIER size_t \
      TrackArrayIndirectGetElemSize_##tName \
        ( const TrackArrayIndirect_##tName ); \
    STORAGEQUALIFIER int \
      TrackArrayIndirectGetIndexByRef_##tName \
        ( const TrackArrayIndirect_##tName, \
          const TrackArrayIndirectElemRef_##tName ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAddLast_##tName \
        ( TrackArrayIndirect_##tName*, \
          const TrackArrayIndirectElem_##tName ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAddLastByRef_##tName \
        ( TrackArrayIndirect_##tName*, \
          const TrackArrayIndirectElemRef_##tName ); \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveLast_##tName \
        ( TrackArrayIndirect_##tName* ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertAt_##tName \
        ( TrackArrayIndirect_##tName*, int, \
          const TrackArrayIndirectElem_##tName ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertAtByRef_##tName \
        ( TrackArrayIndirect_##tName*, int, \
          const TrackArrayIndirectElemRef_##tName ); \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveAt_##tName \
        ( TrackArrayIndirect_##tName*, int ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertManyAt_##tName \
        ( TrackArrayIndirect_##tName*, int, int, \
          const TrackArrayIndirectElem_##tName* ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAppendMany_##tName \
        ( TrackArrayIndirect_##tName*, int, \
          const TrackArrayIndirectElem_##tName* ); \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveManyAt_##tName \
        ( TrackArrayIndirect_##tName*, int, int ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemAt_##tName \
        ( const TrackArrayIndirect_##tName, int ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemFirst_##tName \
        ( const TrackArrayIndirect_##tName ); \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemLast_##tName \
        ( const TrackArrayIndirect_##tName );

/*** define a TrackArrayIndirect "template" implementation ***/
#define TrackArrayIndirect_Implement( tName, tPtrName ) \
    STORAGEQUALIFIER TrackArrayIndirect_##tName \
      TrackArrayIndirectNew_##tName \
        ( TMEM void* memparent, int initialAlloc ) { \
        TrackArray_##tPtrName array; \
        array = TrackArrayNew_##tPtrName(memparent, initialAlloc); \
        if (array) { \
            int i; \
            for (i = 0; i < initialAlloc; ++i) { \
                array[i] = TrackArrayIndirectNewElem_##tName(array); \
                if (! array[i]) { \
                    TrackArrayDelete_##tPtrName(array); \
                    array = NULL; \
                    break; \
                } \
            } \
        } \
        return array; \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectNewElem_##tName \
        ( TMEM void* memparent ) { \
        return (TrackArrayIndirectElemRef_##tName)TrackMemAlloc(memparent, \
            sizeof(TrackArrayIndirectElem_##tName), 0); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayIndirectDelete_##tName \
        ( TrackArrayIndirect_##tName array ) { \
        int size = TrackArrayGetSize_##tPtrName(array); \
        int i; \
        for (i = 0; i < size; ++i) { \
            if (array[i]) TrackArrayIndirectDeleteElem_##tName(array[i]); \
            array[i] = NULL; \
        } \
        TrackArrayDelete_##tPtrName(array); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayIndirectDeleteElem_##tName \
        ( TrackArrayIndirectElemRef_##tName elem ) { \
        TrackMemFree((void*)elem); \
    } \
    STORAGEQUALIFIER TrackArrayIndirect_##tName \
      TrackArrayIndirectClone_##tName \
        ( TMEM void* memparent, const TrackArrayIndirect_##tName oldArray ) { \
        TrackArray_##tPtrName array; \
        int size = TrackArrayGetSize_##tPtrName(oldArray); \
        array = TrackArrayNew_##tPtrName(memparent, size); \
        if (array) { \
            int i; \
            for (i = 0; i < size; ++i) { \
                if (oldArray[i]) { \
                    array[i] = (TrackArrayIndirectElemRef_##tName) \
                        TrackMemClone(array, (void*)oldArray[i]); \
                    if (! array[i]) { \
                        TrackArrayDelete_##tPtrName(array); \
                        array = NULL; \
                        break; \
                    } \
                } else array[i] = NULL; \
            } \
        } \
        return array; \
    } \
    STORAGEQUALIFIER int \
      TrackArrayIndirectResize_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int size ) { \
        int oldSize; \
        ASSERT(ptrarray); \
        oldSize = TrackArrayGetSize_##tPtrName(*ptrarray); \
        if (oldSize > size) { \
            TrackArrayIndirectRemoveManyAt_##tName(ptrarray, size, \
                oldSize - size); \
        } else if (oldSize < size) { \
            return (NULL != TrackArrayIndirectAppendMany_##tName(ptrarray, \
                oldSize - size, NULL)); \
        } \
        return TRUE; \
    } \
    STORAGEQUALIFIER int \
      TrackArrayIndirectEnsureStorageForElem_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index ) { \
        int oldSize; \
        ASSERT(ptrarray); \
        oldSize = TrackArrayGetSize_##tPtrName(*ptrarray); \
        if (index >= oldSize) { \
            return (NULL != TrackArrayIndirectAppendMany_##tName(ptrarray, \
                oldSize - index + 1, NULL)); \
        } \
        return TRUE; \
    } \
    STORAGEQUALIFIER int \
      TrackArrayIndirectGetSize_##tName \
        ( const TrackArrayIndirect_##tName array ) { \
        return TrackArrayGetSize_##tPtrName(array); \
    } \
    STORAGEQUALIFIER size_t \
      TrackArrayIndirectGetElemSize_##tName \
        ( const TrackArrayIndirect_##tName array ) { \
        return TrackArrayGetElemSize_##tPtrName(array); \
    } \
    STORAGEQUALIFIER int \
      TrackArrayIndirectGetIndexByRef_##tName \
        ( const TrackArrayIndirect_##tName array, \
          const TrackArrayIndirectElemRef_##tName elem ) { \
        int size, index; \
        size = TrackArrayGetSize_##tPtrName(array); \
        for (index = 0; index < size; ++index) { \
            if (array[index] == elem) break; \
        } \
        ASSERT(index != size); \
        return index; \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAddLast_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, \
          const TrackArrayIndirectElem_##tName value ) { \
        return TrackArrayIndirectAddLastByRef_##tName(ptrarray, \
            (const TrackArrayIndirectElemRef_##tName)&value); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAddLastByRef_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, \
          const TrackArrayIndirectElemRef_##tName value ) { \
        TrackArrayIndirectElemRef_##tName elem = \
            TrackArrayIndirectNewElem_##tName(*ptrarray); \
        if (elem) { \
            if (value) \
                memcpy(elem, value, sizeof(TrackArrayIndirectElem_##tName)); \
            if (! TrackArrayAddLast_##tPtrName(ptrarray, elem)) { \
                TrackArrayIndirectDeleteElem_##tName(elem); \
                elem = NULL; \
            } \
        } \
        return elem; \
    } \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveLast_##tName \
        ( TrackArrayIndirect_##tName* ptrarray ) { \
        int index = TrackArrayGetSize_##tPtrName(*ptrarray) - 1; \
        TrackArrayIndirectElemRef_##tName elem; \
        ASSERT(index >= 0); \
        elem = (*ptrarray)[index]; \
        TrackArrayRemoveLast_##tPtrName(ptrarray); \
        TrackArrayIndirectDeleteElem_##tName(elem); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertAt_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index, \
          const TrackArrayIndirectElem_##tName value ) { \
        return TrackArrayIndirectInsertAtByRef_##tName(ptrarray, \
            index, (const TrackArrayIndirectElemRef_##tName)&value); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertAtByRef_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index, \
          const TrackArrayIndirectElemRef_##tName value ) { \
        TrackArrayIndirectElemRef_##tName elem = \
            TrackArrayIndirectNewElem_##tName(*ptrarray); \
        if (elem) { \
            if (value) \
                memcpy(elem, value, sizeof(TrackArrayIndirectElem_##tName)); \
            if (! TrackArrayInsertAt_##tPtrName(ptrarray, index, elem)) { \
                TrackArrayIndirectDeleteElem_##tName(elem); \
                elem = NULL; \
            } \
        } \
        return elem; \
    } \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveAt_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index ) { \
        TrackArrayIndirectElemRef_##tName elem; \
        elem = (*ptrarray)[index]; \
        TrackArrayRemoveAt_##tPtrName(ptrarray, index); \
        TrackArrayIndirectDeleteElem_##tName(elem); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectInsertManyAt_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index, int count, \
          const TrackArrayIndirectElem_##tName* elemsource ) { \
        ASSERT(ptrarray); \
        if (! TrackArrayInsertManyAt_##tPtrName(ptrarray, index, count, \
            NULL)) { \
            int i; \
            for (i = index; i < index + count; ++i) { \
                (*ptrarray)[i] = TrackArrayIndirectNewElem_##tName(ptrarray); \
                if (! (*ptrarray)[i]) break; \
            } \
            if (i == (index + count)) return (*ptrarray)[index]; \
            while (i < index + count) (*ptrarray)[i++] = NULL; \
            TrackArrayRemoveManyAt_##tPtrName(ptrarray, index, count); \
        } \
        return NULL; \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectAppendMany_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int count, \
          const TrackArrayIndirectElem_##tName* elemsource ) { \
        int oldSize; \
        ASSERT(ptrarray); \
        oldSize = TrackArrayGetSize_##tPtrName(*ptrarray); \
        return TrackArrayIndirectInsertManyAt_##tName(ptrarray, oldSize -1, \
            count, elemsource); \
    } \
    STORAGEQUALIFIER void \
      TrackArrayIndirectRemoveManyAt_##tName \
        ( TrackArrayIndirect_##tName* ptrarray, int index, int count) { \
        int i; \
        ASSERT(ptrarray); \
        for (i = index; i < index + count; ++i) { \
            if ((*ptrarray)[i]) { \
                TrackArrayIndirectDeleteElem_##tName( (*ptrarray)[i] ); \
                (*ptrarray)[i] = NULL; \
            } \
        } \
        TrackArrayRemoveManyAt_##tPtrName(ptrarray, index, count); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemAt_##tName \
        ( const TrackArrayIndirect_##tName array, int index ) { \
        return *(TrackArrayElemAt_##tPtrName(array, index)); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemFirst_##tName \
        ( const TrackArrayIndirect_##tName array ) { \
        return *(TrackArrayElemFirst_##tPtrName(array)); \
    } \
    STORAGEQUALIFIER TrackArrayIndirectElemRef_##tName \
      TrackArrayIndirectElemLast_##tName \
        ( const TrackArrayIndirect_##tName array ) { \
        return *(TrackArrayElemLast_##tPtrName(array)); \
    }


/* FUTURE: (details not yet decided)
 *
 * newArray = TrackArrayIndirectUnlink_[Type](array)
 *   - creates a new C-style array consisting of a contiguous concatenation
 *     of the objects of the array with the assumption that they are of
 *     equal length, then builds a list of pointers to these objects
 *     <unimplemented>
 *
 * TrackArrayIndirectGetManyAt_[Type](array, index, count, *elemtarget)
 *   - fills an array with the contiguous concatenation of a series of
 *     elements
 *     <unimplemented>
 *
 * TrackArrayIndirectSetManyAt_[Type](array, index, count, *elemsource)
 *   - initializes a series of elements using the data in elemsource
 *     <unimplemented>
 *
 */

#endif /* (! MEM_TRACKARRAY_H) */
