/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: transport.c
 *
 * Description: Implementation of the generic transport abstraction layer
 *              interface.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>

#include "aps.h"
#include "apsinternal.h"
#include "transport.h"
#include "printer.h"
#include "lprtransport.h"
#include "job.h"
#include "utils.h"
#include "resultcode.h"

DEBUG_CHANNEL_DEFAULT(transprt)

#if (APSCFG_CUPS_BUILD)
#include "cupstransport.h"
#endif /* APSCFG_CUPS_BUILD */

/* Static variables that maintain global transports. */
static int numTransports = -1;
static Transport *transportArray[APSCFG_TRANSPORT_MAX_TRANSPORTS];

/* ---------------------------------------------------------------------------
 * TransGetAllTransports()
 *
 * Obtains an array of all available printer transports.
 *
 * Parameters: count - A pointer to an int to receive the number of
 *                     transports in the array returned by this function.
 *
 *     Return: A pointer to an array of Transport pointers.
 */
Transport **TransGetAllTransports(int *count)
{
    /* If the transports have not been created yet, then do so now. */
    /* Only try once though so we don't bog down the library if nothing
     * was found... */
    if (numTransports == -1) {
        numTransports = 0;
#if (APSCFG_CUPS_BUILD)
        /* Create a CUPS transport. */
        ASSERT(numTransports < APSCFG_TRANSPORT_MAX_TRANSPORTS);
        transportArray[numTransports] = CupsCreateTransport();
        if (transportArray[numTransports] != NULL)
            ++numTransports;
#endif
#if (APSCFG_LPR_BUILD)
        /* Create an lpr transport. */
        ASSERT(numTransports < APSCFG_TRANSPORT_MAX_TRANSPORTS);
        transportArray[numTransports] = LprCreateTransport();
        if (transportArray[numTransports] != NULL)
            ++numTransports;
#endif
    }

    /* Pass the tranport array and count of transports back to the
     * caller.
     */
    if (numTransports == 0) {
        WARN("No transports have been registered, APS will not be able to\n"
             "manipulate printers or print jobs.\n");
    }
    *count = numTransports;
    return (transportArray);
}

/* ---------------------------------------------------------------------------
 * TransGetFromPrinterName()
 *
 * Obtains a pointer to the transport used to access a particular printer,
 * given the name of that printer.
 *
 * Parameters: name - A string containing the name of the printer to
 *                    obtain the transport for.
 *
 *     Return: A pointer to a Transport, or NULL on failure.
 */
Transport *TransGetFromPrinterName(const char *name)
{
    int numTransports;
    Transport **transports;
    int i;

    transports = TransGetAllTransports(&numTransports);
    for (i = 0; i < numTransports; ++i) {
        if (transports[i]->vtbl->IsPrinterKnown(transports[i], name)) {
            return (transports[i]);
        }
    }
    return (NULL);
}

/* ---------------------------------------------------------------------------
 * TransGetPrinter()
 *
 * Finds the transport associated with the named printer, and obtains a
 * pointer to a Printer instance for this printer from the transport. The
 * PrinterRelease() function should be called on this printer when it is no
 * longer needed.
 *
 * Parameters: name    - A string containing the name of the printer to
 *                       obtain the transport for.
 *
 *             printer - A pointer to a Printer * to receive a pointer to the
 *                       newly created printer, if successful.
 *
 *     Return: APS_SUCCESS if a new printer instance has been obtained
 */
Aps_Result TransGetPrinter(const char *name, Printer ** printer)
{
    Transport *transport = TransGetFromPrinterName(name);

    if (transport == NULL)
        return (APS_NOT_FOUND);

    *printer = transport->vtbl->CreatePrinterInstance(transport, name);
    if (*printer == NULL)
        return (APS_GENERIC_FAILURE);

    return (APS_SUCCESS);
}

/* ---------------------------------------------------------------------------
 * TransJobCreate()
 *
 * Creates a Job based on the provided printer and QuickInfo structure.
 *
 * Parameters: printer - Pointer to a Printer for new job.
 *             info    - Pointer to an Aps_QuickJobInfo for new job.
 *             job     - Pointer to Job* to receive job.
 *
 *     Return: APS_SUCCESS if a new job has been created.
 */
Aps_Result TransJobCreate(Printer *printer, Aps_QuickJobInfo *info,
    Job **xjob) {
    Aps_Result result;
    Aps_JobHandle jobHandle;
    Job  *job;

    ASSERT(printer && info && xjob);

    /* Create new job structure */
    *xjob = NULL;
    result = JobCreate(PrinterGetHandleFromPtr(printer), & jobHandle);
    if (result != APS_SUCCESS) {
        ERROR("Unable to create job structure : %s", RESULTSTR(result));
        return result;
    }

    *xjob = job = JobGetPtrFromHandle(jobHandle);

    /* Copy settings from info structure */
    /** Version 0 **/
    /* Set by JobCreate
     * job->info->jobHandle = info->jobHandle;
     */
    FIXME("Does not check for failure of memory allocations in strupdate()");
    job->info->jobStatus = info->jobStatus;
    strupdate(& job->info->jobHost, info->jobHost);
    strupdate(& job->info->jobName, info->jobName);
    strupdate(& job->info->jobFilename, info->jobFilename);
    job->info->jobID = info->jobID;
    job->info->jobSize = info->jobSize;
    job->info->jobCreationTime = info->jobCreationTime;
    strupdate(& job->info->jobFormat, info->jobFormat);
    job->info->jobOrder = info->jobOrder;
    job->info->jobPriority = info->jobPriority;
    /* Set by JobCreate
     *  job->info->printerHandle = info->printerHandle;
     *  strupdate(& info->job->printerName, info->printerName);
     */
    job->info->printerStatus = info->printerStatus;
    strupdate(& job->info->docTitle, info->docTitle);
    strupdate(& job->info->docRevision, info->docRevision);
    strupdate(& job->info->docComments, info->docComments);
    strupdate(& job->info->docAuthor, info->docAuthor);
    strupdate(& job->info->docType, info->docType);
    strupdate(& job->info->docCreator, info->docCreator);
    strupdate(& job->info->ownerName, info->ownerName);
    job->info->ownerID = info->ownerID;
    strupdate(& job->info->localHost, info->localHost);
    strupdate(& job->info->localFile, info->localFile);
    strupdate(& job->info->spoolHost, info->spoolHost);
    strupdate(& job->info->spoolFile, info->spoolFile);
    /** Version 1, here **/
    if ((job->info->version >= 1) && (info->version >= 1)) { }

    return APS_SUCCESS;
}

/* This module contains two isolated streams of code designed to
 * simplify the creation of transports which only support a limited
 * set of spooling operations. At this time, jobs may either be
 * sent to the system's print queue as complete files, or may
 * be written progressively.
 *
 * To support this scheme, this layer provides default implementations
 * of the job dispatch mechanism with the assumption that at least
 * one of these procedures will be supported directly.
 *
 * PROG: JobStart()     : begin spooling operation
 * PROG: JobWrite()     : write data to open spool
 * PROG: JobEnd()       : end spooling operation
 * PROG: JobGetFileDescriptor() : get file descriptor for
 *                                writing to spool (real or
 *                                emulated)
 *
 * BULK: JobDispatch()  : send file to spool, immediately
 *
 * For transports supporting progressive data transfer:
 *   MUST define JobStart(), JobWrite(), JobEnd() [AS A SET].
 *
 *   MAY  define JobGetFileDescriptor().
       else JobGetFileDescriptor() emulated via a pipe.
 *   MAY  define JobDispatch().
 *     else JobDispatch() sent as [multiple] JobWrite()'s
 *        inside a JobStart/End() bracket.
 *
 * For transports supporting bulk file data transfer:
 *   MUST define JobDispatch().
 *
 *   MAY  define JobStart(), JobWrite(), JobEnd() [AS A SET].
 *     else JobStart/Write/End() spooled internally and
 *       dispatched in a single operation.
 *
 *   IF [JobStart/Write/End] implemented internally,
 *     MAY  define JobGetFileDescriptor().
 *       else JobGetFileDescriptor() emulated via a pipe.
 *
 * If the default entry points are to be used, the DefaultXXX()
 * functions must explicitly be specified in the vector table
 * provided upon transport initialization. This behaviour may
 * change in the future to provide better isolation between
 * layers.  For now, transports which support BULK transfers
 * should specify all DefaultBULK_xxx() functions and those
 * which support PROGRESSIVE transports should specify all
 * DefaultPROG_xxx() functions.
 *
 * It is illegal for both sets of default functions to be
 * specified simultaneously as this will result in never-
 * ending loops.
 */

/*************************************************
 *** DEFAULT IMPLEMENTATION FOR ANY TRANSPORT ***
 *************************************************/

/***********************************************************
 *** DEFAULT IMPLEMENTATION OF PROGRESSIVE DATA TRANSFER ***
 ***   FOR TRANSPORTS WHICH SUPPORT BULK DATA TRANSFER   ***
 ***********************************************************/

typedef struct {
    int      spoolDescriptor;
    char    *spoolFilename;
} DefaultJobContext_Bulk;

/* ---------------------------------------------------------------------------
 * DefaultBULK_JobStart()
 *
 * Create a spooling context for emulating progressive data transfer
 * for transports which only support bulk data transfer.  Opens
 * spool file.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             printer   - pointer to the Printer the job will be sent to
 *             job       - pointer to the Job to use
 *
 * Result    : A standard APS result code
 */
Aps_Result DefaultBULK_JobStart(Transport *thisBase, Job *job)
{
    DefaultJobContext_Bulk *context;
    Aps_Result result;

    /* Sanity check -- should already be verified by high-level API */
    ASSERT(thisBase && job);
    ASSERT(! job->transportDefaultData); /* check for sequence */

    /* Create control structure */
    job->transportDefaultData =
        context = malloc(sizeof(DefaultJobContext_Bulk));
    if (context) {
        context->spoolDescriptor = -1;
        context->spoolFilename = tempnam(NULL, "APS__"); /* uses malloc */
        if (context->spoolFilename) {
            /* Set some information */
            job->info->jobSize = 0;
            strupdate(& job->info->localFile, context->spoolFilename);
            strupdatehostname(& job->info->localHost);

            /* Open file */
            errno = 0;
            context->spoolDescriptor =
                creat(context->spoolFilename, O_WRONLY);
            if (context->spoolDescriptor != -1) {
                fchmod(context->spoolDescriptor,
                       S_IRUSR | S_IWUSR);  /* fix problems with attribs */
                fchown(context->spoolDescriptor,
                       getuid(), getgid()); /* send job from "real" user */
                return APS_SUCCESS;
            }
            result = GetResultFromErrno();
            free(context->spoolFilename);
        } else {
            ERROR("Failed to generate temporary spool filename");
            result = APS_OUT_OF_MEMORY;
        }
        free(context);
        job->transportDefaultData = NULL;
    } else {
        ERROR("Failed to allocate space for job control structure");
        result = APS_OUT_OF_MEMORY;
    }
    job->info->jobStatus = APS_JOB_FAILED;
    ERROR("Failed with result code : %s", RESULTSTR(result));
    return result;
}

/* ---------------------------------------------------------------------------
 * DefaultBULK_JobWrite()
 *
 * Appends a block of data to the spool.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *             data      - pointer to the block of data to write
 *             size      - size of data ( >= 0)
 *
 * Result    : A standard APS result code
 */
Aps_Result DefaultBULK_JobWrite(Transport *thisBase, Job *job, const void *data,
    size_t size)
{
    DefaultJobContext_Bulk *context;
    Aps_Result result = APS_SUCCESS;

    /* Locate control structure */
    ASSERT(thisBase && job && (size >= 0));
    context = (DefaultJobContext_Bulk *)job->transportDefaultData;
    if (! context) {
        ERROR("Tried to write to job without control structure");
        return APS_GENERIC_FAILURE;
    }

    /* Append data to file */
    if (size > 0) {
        int numWritten;

        /* attempt to write data */
        errno = 0;
        numWritten = write(context->spoolDescriptor, data, size);
        if ((numWritten != size) || (errno)) {
            result = GetResultFromErrno();
            ERROR("write() failed with result code: %s", RESULTSTR(result));
        } else job->info->jobSize += size; /* update info */
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * DefaultBULK_JobGetFileDescriptor()
 *
 * Get a file descriptor for outputting file data.  This implementation
 * returns the real file descriptor that we are using.  It is assumed
 * that the descriptor will not be close()'d by the application.
 *
 * The application should ensure that all buffers are flushed when
 * switching between use of this file descriptor and the JobWrite()
 * method.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *
 * Result    : file descriptor or -1 if not available.
 *             A standard APS result code
 */
Aps_Result DefaultBULK_JobGetFileDescriptor(Transport *thisBase, Job *job,
    int *fd)
{
    DefaultJobContext_Bulk *context;
    Aps_Result result = APS_SUCCESS;

    ASSERT(thisBase && job && fd);
    *fd = -1;

    /* Locate control structure */
    context = (DefaultJobContext_Bulk *)job->transportDefaultData;
    if (! context) {
        ERROR("Tried to get FD for job without control structure");
        return APS_GENERIC_FAILURE;
    }

    /* Get descriptor */
    if (context->spoolDescriptor != -1) {
        *fd = context->spoolDescriptor;
    } else {
        ERROR("Stored file descriptor was -1 !?!");
    }
    return result;
}

/* ---------------------------------------------------------------------------
 * DefaultBULK_JobEnd()
 *
 * Terminates the creation phase.  If data was actually created,
 * publish it out to the system using JobDispatch().  If no data
 * was written, the job will be discarded.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *
 * Result    : A standard APS result code
 */
Aps_Result DefaultBULK_JobEnd(Transport *thisBase, Job *job)
{
    DefaultJobContext_Bulk *context;
    Aps_Result result = APS_SUCCESS;

    ASSERT(thisBase && job);

    /* Locate control structure */
    context = (DefaultJobContext_Bulk *)job->transportDefaultData;
    if (! context) {
        ERROR("Tried to end job without control structure");
        job->info->jobStatus = APS_JOB_FAILED;
        return APS_GENERIC_FAILURE;
    }

    /* Dispatch job */
    if (context->spoolDescriptor) {
        close(context->spoolDescriptor);
        context->spoolDescriptor = -1;
        result = thisBase->vtbl->JobDispatch(thisBase, job,
            context->spoolFilename);
        unlink(context->spoolFilename);
        free(context->spoolFilename);
        context->spoolFilename = NULL;
    } else {
    /* No data was written, discard and set status to completed. */
        job->info->jobStatus = APS_JOB_COMPLETED;
    }

    /* Free control structure */
    free(context);
    job->transportDefaultData = NULL;
    return result;
}

/* ---------------------------------------------------------------------------
 * DefaultBULK_JobAbort()
 *
 * Terminates the creation phase.  Discards any data that was written
 * to the print job.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *
 * Result    : A standard APS result code
 */
Aps_Result DefaultBULK_JobAbort(Transport *thisBase, Job *job)
{
    DefaultJobContext_Bulk *context;
    Aps_Result result = APS_SUCCESS;

    /* Locate control structure */
    ASSERT(thisBase && job);
    job->info->jobStatus = APS_JOB_ABORTED;
    context = (DefaultJobContext_Bulk *)job->transportDefaultData;
    if (! context) {
        ERROR("Tried to abort job without control structure");
        return APS_GENERIC_FAILURE;
    }

    /* Discard job */
    if (context->spoolDescriptor) {
        close(context->spoolDescriptor);
        context->spoolDescriptor = -1;
        unlink(context->spoolFilename);
        free(context->spoolFilename);
        context->spoolFilename = NULL;
    }

    /* Free control structure */
    free(context);
    job->transportDefaultData = NULL;
    return result;
}

/****************************************************************
 *** DEFAULT IMPLEMENTATION OF BULK DATA TRANSFER             ***
 ***   FOR TRANSPORTS WHICH SUPPORT PROGRESSIVE DATA TRANSFER ***
 ****************************************************************/

/* ---------------------------------------------------------------------------
 * DefaultPROG_JobDispatch()
 *
 * Emulate a bulk data transfer by sending progressively through JobStart/Write/
 * End.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             printer   - pointer to the Printer the job will be sent to
 *             job       - pointer to the Job to use
 *             filename  - name of file to print
 *
 * Result    : A standard APS result code
 */
Aps_Result DefaultPROG_JobDispatch(Transport *thisBase,
    Job *job, const char *filename)
{
    int  fd;
    Aps_Result result = APS_SUCCESS;

    /* Sanity check -- should already be verified by high-level API */
    ASSERT(thisBase && job && filename);
    ASSERT(! job->transportDefaultData); /* check for sequence */

    /* cause assert to fail on circular program flow */
    job->transportDefaultData = (void *)(1);

    /* setup some information */
    strupdate(& job->info->localFile, filename);
    strupdatehostname(& job->info->localHost);
    job->info->jobSize = 0;

    /* setup job with JobStart() */
    result = thisBase->vtbl->JobStart(thisBase, job);
    if (result == APS_SUCCESS) {
        /* Open file and create buffer */
        errno = 0;
        fd = open(filename, O_RDONLY);
        if (fd) {
            char *buffer = malloc(APSCFG_COPY_BUFFER_SIZE);
            if (buffer) {
                /* Transfer data */
                int numRead;

                while ((numRead = read(fd, buffer,
                    APSCFG_COPY_BUFFER_SIZE)) > 0) {
                    result = thisBase->vtbl->JobWrite(thisBase, job,
                        buffer, numRead);
                    if (! Aps_Succeeded(result)) {
                        ERROR("JobWrite() failed with result code: %s",
                            RESULTSTR(result));
                        break;
                    }
                }
                free(buffer); /* Free buffer */
            } else {
                ERROR("Unable to allocate copy buffer");
                result = APS_OUT_OF_MEMORY;
            }
            close(fd); /* Close file */
        } else {
            result = GetResultFromErrno(); /* file not found */
            ERROR("Unable to open job source file %s, result code: %s",
                RESULTSTR(result));
        }

        /* if we ran into an error, abort the job, else end it */
        if (result == APS_SUCCESS) {
            result = thisBase->vtbl->JobEnd(thisBase, job);
        } else {
            ERROR("Failed after job started, aborting, result code: %s",
                RESULTSTR(result));
            thisBase->vtbl->JobAbort(thisBase, job);
        }
    } else {
        ERROR("JobStart() failed with result code: %s", RESULTSTR(result));
    }
    job->transportDefaultData = NULL;
    return result;
}

/* ---------------------------------------------------------------------------
 * DefaultPROG_JobGetFileDescriptor()
 *
 * Get a file descriptor for outputting file data.  It is assumed
 * that the descriptor will not be close()'d by the application.
 *
 * The application should ensure that all buffers are flushed when
 * switching between use of this file descriptor and the JobWrite()
 * method.
 *
 * Parameters: thisBase  - pointer to the Transport
 *             job       - pointer to the Job to use
 *
 * Result    : file descriptor or -1 if not available.
 *             A standard APS result code
 */
Aps_Result DefaultPROG_JobGetFileDescriptor(Transport *thisBase, Job *job,
    int *fd)
{
    *fd = -1;
    FIXME("Not implemented");
    return APS_NOT_IMPLEMENTED;
}

