/*
 *   kmoon - a moon phase indicator
 *   $Id: kmoon.cpp,v 1.25 2000/09/12 11:01:41 gebauer Exp $
 *   Copyright (C) 1998,2000  Stephan Kulow
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <assert.h>
#include <unistd.h>

#include <qwidget.h>
#include <qwmatrix.h>
#include <qbitmap.h>
#include <qtooltip.h>
#include <qpainter.h>
#include <qimage.h>
#include <qpopupmenu.h>

#include <dcopclient.h>
#include <kdebug.h>
#include <kapp.h>
#include <kwin.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <klocale.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kcmdlineargs.h>
#include <kstdaction.h>
#include <kiconloader.h>
#include <kconfig.h>

#include "version.h"
#include "kmoondlg.h"
#include "kmoon.h"

extern double moonphasebylunation(int lun, int phi);
extern time_t JDtoDate(double jd, struct tm *event_date);

MoonWidget::MoonWidget(QWidget *parent, const char *name)
    : QWidget(parent, name)
{
    struct tm * t;
    time_t clock;
	
    counter = -1;
	KConfig *config = KGlobal::config();
	config->setGroup("General");
	angle = config->readNumEntry("Rotation", 0);
    old_w = old_h = old_counter = -1;
    startTimer(100000);

    time(&clock);
    t = localtime(&clock);
    calcStatus(mktime(t));
	if (!parent) {
		popup = new QPopupMenu();
		popup->insertItem(kapp->miniIcon(),
						  i18n("&About %1...").arg("kmoon"), this,
						  SLOT(showAbout()));
		popup->insertItem(i18n("&Settings..."), this,
						  SLOT(settings()));
		popup->insertSeparator();
		popup->insertItem(SmallIcon("exit"), i18n("&Quit"),
						  kapp, SLOT(quit()));
	} else
		popup = 0;

}

void MoonWidget::calcStatus( time_t time )
{
    double JDE;
    uint lun = 0;
    struct tm event_date;
    struct tm last_event;
    time_t last_new, next_new;

    do {
		last_event = event_date;
		JDE = moonphasebylunation(lun, 0);
		next_new = last_new;
		last_new = JDtoDate(JDE, &event_date);
		lun++;
    } while (last_new < time);
	
    counter = qRound(float(time - next_new) / ( 60 * 60 * 24));
    if (counter >= 29)
		counter -= 29;
}

QImage MoonWidget::loadMoon(int index)
{
	if (index == 0) // the new moon has the wrong filename
		index = 29;
	QString filename = QString("kmoon/pics/moon%1.png").arg(index);
	QString path = locate("data", filename);
	if (path.isNull()) 
		kdFatal() << "cound't find " << filename << ". Exiting.\n";
	return QImage(path);
}

const char *description = I18N_NOOP("Moon Phase Indicator for KDE");

void MoonWidget::settings()
{
	KMoonDlg dlg(angle, this, "moondlg");
	if (dlg.exec() == KMoonDlg::Accepted) {
		setAngle(dlg.getAngle());
		KConfig *config = KGlobal::config();
		config->setGroup("General");
		config->writeEntry("Rotation", angle);
	}
}

void MoonWidget::showAbout()
{
    KMessageBox::about(0,
					   i18n(description) + QString::fromLatin1("\n\n") +
					   i18n("Written by Stephan Kulow <coolo@kde.org>\n"
							"\n"
							"Lunar code by Chris Osburn "
							"<chris@speakeasy.org>\n"
							"\n"
							"Moon graphics by Tim Beauchamp "
							"<timb@googol.com>"),
					   i18n("About Moon Phase Indicator"));
}


void MoonWidget::timerEvent( QTimerEvent * )
{
    struct tm * t;
    time_t clock;
    time(&clock);
    t = localtime(&clock);
    calcStatus(mktime(t));
    renderGraphic();
    repaint( false );
}

void MoonWidget::setAngle(int value)
{
	angle = value;
	renderGraphic();
	repaint(false);
}

void MoonWidget::paintEvent( QPaintEvent * )
{
    renderGraphic();
    bitBlt(this, 0, 0, &pixmap, 0, 0);
}

void MoonWidget::resizeEvent( QResizeEvent *)
{
    renderGraphic();
    repaint();
}

void MoonWidget::renderGraphic()
{
    if (old_counter == counter && old_w == width() && old_h == height() && old_a == angle)
		return;
    old_counter = counter;
    old_w = width();
    old_h = height();
	QImage im = loadMoon(counter);
	im = im.convertDepth(32, 0);

	QWMatrix m;
	m.rotate(angle);
	int mw = QMIN(width(), height());
    if (!pixmap.convertFromImage(im.smoothScale(mw * 2, mw * 2), 0))
		return;
	QPixmap rotated = pixmap.xForm(m);

	QBitmap mask(pixmap.size());
	mask.fill(Qt::color0);
	QRegion r(QRect(0, 0, pixmap.width(), pixmap.height()), QRegion::Ellipse);
	QPainter p;
	p.begin(&mask);
	p.setClipRegion(r);
	p.fillRect(0, 0, pixmap.width(), pixmap.height(), Qt::color1);
	p.end();
	
	bitBlt(&pixmap, 0, 0, &rotated, (rotated.width() - pixmap.width()) / 2,
		   (rotated.height() - pixmap.height()) / 2, pixmap.width(),
		   pixmap.height(), CopyROP);
	rotated = pixmap;
	rotated.setMask( mask );
	pixmap.fill(QApplication::palette().normal().background());
	p.begin(&pixmap);
	p.drawPixmap(0, 0, rotated);
	p.end();

	im = pixmap.convertToImage();

	QImage dest = im.copy(0, 0, mw, mw);
	for (int y = 0; y < mw; y++) {
		QRgb *destline = (QRgb*)dest.scanLine(y);
		QRgb *sourceline1 = (QRgb*)im.scanLine(2*y);
		QRgb *sourceline2 = (QRgb*)im.scanLine(2*y + 1);
		for (int x = 0; x < mw; x++) {
			int r = qRed(sourceline1[2*x]) + qRed(sourceline1[2*x+1]);
			r = r + qRed(sourceline2[2*x]) + qRed(sourceline2[2*x+1]);
			int g = qGreen(sourceline1[2*x]) + qGreen(sourceline1[2*x+1]);
			g = g + qGreen(sourceline2[2*x]) + qGreen(sourceline2[2*x+1]);
			int b = qBlue(sourceline1[2*x]) + qBlue(sourceline1[2*x+1]);
			b = b + qBlue(sourceline2[2*x]) + qBlue(sourceline2[2*x+1]);
			destline[x] = qRgb(qRound(r / 4), qRound(g / 4),
							   qRound(b / 4));
		}
	}

    if (!pixmap.convertFromImage(dest, 0))
		return;

    QToolTip::remove(this);
	QString phase;
	switch (counter) {
	case 0:
		phase = i18n("New Moon");
		break;
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
		phase = i18n("Waxing Crescent");
		break;
	case 6:
	case 7:
	case 8:
		phase = i18n("First Quarter");
		break;
	case 9:
	case 10:
	case 11:
	case 12:
	case 13:
	case 14:
		phase = i18n("Waxing Gibbous");
		break;
    case 15:
        phase = i18n("Full Moon");
        break;
	case 16:
	case 17:
	case 18:
	case 19:
	case 20:
		phase = i18n("Waning Gibbous");
		break;
	case 21:
	case 22:
	case 23:
		phase = i18n("Last Quarter");
		break;
	case 24:
	case 25:
	case 26:
	case 27:
	case 28:
		phase = i18n("Waning Crescent");
		break;
	default:
		kdFatal() << "coolo can't count\n";
	}
	
    tooltip = i18n("Moon has passed %1 days since last new moon. Current phase: %2").arg(counter).arg(phase);
    QToolTip::add(this, tooltip);
}

void MoonWidget::mousePressEvent( QMouseEvent *e)
{
	if (!popup)
		return;

    if (e->button() == RightButton) {
		popup->popup(mapToGlobal(e->pos()));
		popup->exec();
    }
    if (e->button() == LeftButton) {
		showAbout();
    }
}

static KCmdLineOptions options[] =
{
    { "o", 0, 0 },
	{ "offset <days>", I18N_NOOP("Set the moon some days off"), "0" },
	{ 0, 0, 0}
};

int main( int argc, char **argv)
{
    KAboutData aboutData( "kmoon", I18N_NOOP("KMoon"),
						  version, description, KAboutData::License_GPL,
						  "(c) 1998, Stephan Kulow");
    aboutData.addAuthor("Stephan Kulow",0, "coolo@kde.org");
    KCmdLineArgs::init( argc, argv, &aboutData );
    KCmdLineArgs::addCmdLineOptions( options );
	
    KApplication a;
    MoonWidget *moon = new MoonWidget();
	KWin::setSystemTrayWindowFor(moon->winId(),0);
    kapp->setTopWidget(new QWidget());
    a.setMainWidget(moon);
    moon->show();
    KWin::appStarted();
    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
    int offset = args->getOption("offset").toInt();
    args->clear();
    if (offset)
		moon->calcStatus(time(0) + offset * 24 * 60 * 60);
    return a.exec();
}

#include "kmoon.moc"
