
package kdocParser;

use Carp;
use File::Basename;

use Ast;
use kdocAstUtil;
use kdocParseDoc;

use strict;

use vars qw/ %stats %rootNodes $cNode $currentfile $cSourceNode @classStack
	$inExtern $docNode $lastLine $declNodeType $cpp $cppcmd $docincluded
	$rootNode @includes $striphpath/;

sub BEGIN 
{
		# a lot of these need to be set externally.

		%stats = ();
		%rootNodes = ();
		$cNode = undef;
		$currentfile = undef;
		$cSourceNode = undef;
		@classStack = ();
		$inExtern = 0;
		$docNode = undef;
		$lastLine = undef;
		$declNodeType = undef;
		$cpp = 0;
		$cppcmd = "";
		$docincluded = 0;
		$rootNode = undef;
		@includes = ();
		$striphpath = 0;
}

sub parseFiles
{
	foreach $currentfile ( @ARGV ) {
		my $lang = "CXX";

		if ( $currentfile =~ /\.idl\s*$/ ) {
			# IDL file
			$lang = "IDL";
		}

		# assume cxx file
		if( $cpp ) {
			# pass through preprocessor
			my $cmd = $cppcmd;
			foreach my $dir ( @includes ) {
				$cmd .= " -I $dir ";
			}

			$cmd .= " -DQOBJECTDEFS_H $currentfile";

			open( INPUT, "$cmd |" )
				|| croak "Can't preprocess $currentfile";
		}
		else {
			open( INPUT, "$currentfile" ) 
				|| croak "Can't read from $currentfile";
		}

		print $main::exe.": processing $currentfile\n" unless $main::quiet;

		# reset vars
		$rootNode = getRoot( $lang );


		# add to file lookup table
		my $showname = $striphpath ? basename( $currentfile )
						: $currentfile;

		$cSourceNode = Ast::New( $showname );
		$cSourceNode->AddProp( "NodeType", "source" );
		$cSourceNode->AddProp( "Path", $currentfile );
		$rootNode->AddPropList( "Sources", $cSourceNode );


		# reset state
		@classStack = ();
		$cNode = $rootNode;
		$inExtern = 0;

		# parse
		my $k = undef;
		while ( defined ($k = readDecl()) ) {
			print "\nDecl: <$k>[$declNodeType]\n" if $main::debug;
			if( identifyDecl( $k ) && $k =~ /{/ ) {
				readCxxCodeBlock();
			} 
		}
		close INPUT;
	}
}

###### Parser routines

=head2 readSourceLine

	Returns a raw line read from the current input file.
	This is used by routines outside main, since I don't know
	how to share fds.

=cut

sub readSourceLine
{
	return <INPUT>;
}

=head2 readCxxLine

	Reads a C++ source line, skipping comments, blank lines,
	preprocessor tokens and the Q_OBJECT macro

=cut

sub readCxxLine
{
	my( $p );
	my( $l );
	
	while( 1 ) {
		return undef if !defined ($p = <INPUT>);

		$p =~ s#//.*$##g;			# C++ comment
		$p =~ s#/\*(?!\*).*?\*/##g;		# C comment

		# join all multiline comments
		if( $p =~ m#/\*(?!\*)#s ) {
			# unterminated comment
LOOP:
			while( defined ($l = <INPUT>) ) {
				$l =~ s#//.*$##g;		# C++ comment
				$p .= $l;
				$p =~ s#/\*(?!\*).*?\*/##sg;	# C comment
				last LOOP unless $p =~ m#(/\*(?!\*))|(\*/)#sg;
			}
		}

		next if ( $p =~ /^\s*$/s 		# blank lines
			|| $p =~ /^\s*Q_OBJECT/		# QObject macro
			);

		next if ( $p =~ /^\s*Q_ENUMS/           # ignore Q_ENUMS
                        || $p =~ /^\s*Q_PROPERTY/       # and Q_PROPERTY
                        || $p =~ /^\s*K_SYCOCATYPE/     # and K_SYCOCA stuff
                        || $p =~ /^\s*K_SYCOCAFACTORY/  # 
                        || $p =~ /^\s*K_DCOP/           # and DCOP stuff
			);

		# remove all preprocessor macros
		if( $p =~ /^\s*#\s*(\w+)/ ) {
			if (!$docincluded && $p =~ /^\s*#\s*[0-9]+\s*\".*$/ 
					&& not($p =~ /\"$currentfile\"/)) {
				# include file markers
				while( <INPUT> ) {
					last if(/\"$currentfile\"/);
						print "Overread $_" if $main::debug;
				};
				print "Cont: $_" if $main::debug;
			} 
			else {
				# multiline macros
				while ( defined $p && $p =~ m#\\\s*$# ) {
					$p = <INPUT>;
				}
			}
			next;
		}

		$lastLine = $p;
		return $p;
	}
}

=head2 readCxxCodeBlock

	Reads a C++ code block (recursive curlies), returning the last line
	or undef on error.

	Parameters: none

=cut

sub readCxxCodeBlock
{
# Code: begins in a {, ends in }\s*;?
# In between: cxx source, including {}
	my ( $count ) = 0;
	my $l = undef;
	
	if ( defined $lastLine ) {
		print "lastLine: '$lastLine'" if $main::debug;

		my $open = kdocUtil::countReg( $lastLine, "{" );
		my $close = kdocUtil::countReg( $lastLine, "}" );
		$count = $open - $close;

		return $lastLine if ( $open || $close) && $count == 0;
	}

	# find opening brace
	if ( $count == 0 ) {
		while( $count == 0 ) {
			$l = readCxxLine();
			return undef if !defined $l;
			$l =~ s/\\.//g;
			$l =~ s/'.?'//g;
			$l =~ s/".*?"//g;

			$count += kdocUtil::countReg( $l, "{" );
			print "c ", $count, " at '$l'" if $main::debug;
		}
		$count -= kdocUtil::countReg( $l, "}" );
	}

	# find associated closing brace
	while ( $count > 0 ) {
		$l = readCxxLine();
		croak "Confused by unmatched braces" if !defined $l;
		$l =~ s/\\.//g;
		$l =~ s/'.?'//g;
		$l =~ s/".*?"//g;

		my $add = kdocUtil::countReg( $l, "{" );
		my $sub = kdocUtil::countReg( $l, "}" );
		$count += $add - $sub;

		print "o ", $add, " c ", $sub, " at '$l'" if $main::debug;
	}

	undef $lastLine;
	return $l;
}

=head2 readDecl

	Returns a declaration and sets the $declNodeType variable.

	A decl starts with a type or keyword and ends with [{};]
	The entire decl is returned in a single line, sans newlines.

	declNodeType values: undef for error, "a" for access specifier,
	"c" for doc comment, "d" for other decls.

	readCxxLine is used to read the declaration.

=cut

sub readDecl
{
	undef $declNodeType;
	my $l = readCxxLine();
	my ( $decl ) = "";

	if( !defined $l ) {
		return undef;
	}
	elsif ( $l =~ /^\s*(private|public|protected|signals)
		(\s+\w+)?\s*:/x ) { # access specifier
		$declNodeType = "a";

		return $l;
	}
	elsif ( $l =~ m#^\s*/\*\*# ) {	# doc comment
		$declNodeType = "c";
		return $l;
	}

	do {
		$decl .= $l;

		if ( $l =~ /[{};]/ ) {
			$decl =~ s/\n/ /gs;
			$declNodeType = "d";
			return $decl;
		}
		return undef if !defined ($l = readCxxLine());

	} while ( 1 );
}

#### AST Generator Routines

=head2 getRoot

	Return a root node for the given type of input file.

=cut

sub getRoot
{
	my $type = shift;
	carp "getRoot called without type" unless defined $type;

	if ( !exists $rootNodes{ $type } ) {
		my $node = Ast::New( "Global" );	# parent of all nodes
		$node->AddProp( "NodeType", "root" );
		$node->AddProp( "RootType", $type );
		$node->AddProp( "Compound", 1 );
		$node->AddProp( "KidAccess", "public" );

		$rootNodes{ $type } = $node;
	}
	print "getRoot: call for $type\n" if $main::debug;

	return $rootNodes{ $type };
}

=head2 allRoots

	Return ref to a hash of all root nodes.

=cut

sub allRoots
{
	return \%rootNodes;
}

=head2 identifyDecl

	Parameters: decl

	Identifies a declaration returned by readDecl. If a code block
	needs to be skipped, this subroutine returns a 1, or 0 otherwise.

=cut

sub identifyDecl
{
	my( $decl ) = @_;

	my $newNode = undef;
	my $skipBlock = 0;

	# Doc comment
	if ( $declNodeType eq "c" ) {
		$docNode = kdocParseDoc::newDocComment( $decl, \&readSourceLine );

		# if it's the main doc, it is attached to the root node
		if ( defined $docNode->{LibDoc} ) {
			kdocParseDoc::attachDoc( $rootNode, $docNode,
				$rootNode );
			undef $docNode;
		}

	}
	elsif ( $declNodeType eq "a" ) {
		newAccess( $decl );
	}

	# Typedef struct/class
	elsif ( $decl =~ /^\s*typedef
			\s+(struct|union|class|enum)
			\s*([_\w\:]*)
			\s*([;{]) 
			/xs ) {
		my ($type, $name, $endtag, $rest ) = ($1, $2, $3, $' );
		$name = "--" if $name eq "";

		warn "typedef '$type' n:'$name'\n" if $main::debug;

		if ( $rest =~ /}\s*([\w_]+(?:::[\w_])*)\s*;/ ) {
			# TODO: Doesn't parse members yet!
			$endtag = ";";
			$name = $1;
		}

		$newNode = newTypedefComp( $type, $name, $endtag );
	}

	# Typedef
	elsif ( $decl =~ /^\s*typedef\s+
			(?:typename\s+)?        # `typename' keyword
			(.*?\s*[\*&]?)		# type
			\s+([-\w_\:]+)		# name
			\s*((?:\[[-\w_\:<>\s]*\])*)	# array
			\s*[{;]\s*$/xs  ) {

		print "Typedef: <$1 $3> <$2>\n" if $main::debug;
		$newNode = newTypedef( $1." ".$3, $2 );
	}

	# Enum
	elsif ( $decl =~ /^\s*enum(\s+[-\w_:]*)?\s*\{(.*)/s  ) {

		print "Enum: <$1>\n" if $main::debug;
		my $enumname = defined $2 ? $1 : "";

		$newNode = newEnum( $enumname );
	}

	# Class/Struct
	elsif ( $decl =~ /^\s*((?:template\s*<.*>)?)      # 1 template
					\s*(class|struct|union|namespace) # 2 struct type
					\s+([\w_]+                        # 3 name
							(?:<[\w_ :,]+?>)?		  # maybe explicit template
									#    (eat chars between <> non-hungry)
							(?:::[\w_]+)*                 #   maybe nested
					   )
					(.*?)                             # 4 inheritance
					([;{])/xs ) {                     # 5 rest

		print "Class: [$1]\n\t[$2]\n\t[$3]\n\t[$4]\n\t[$5]\n" if $main::debug;
		my ( $tmpl, $ntype, $name, $rest, $endtag ) =
			( $1, $2, $3, $4, $5 );

		my @inherits = ();
		my $isCompound = 1;

		$tmpl =~ s/<(.*)>/$1/ if $tmpl ne "";

		unless ( $rest =~ /^\s*$/ ) {
			if(  $rest =~ /^\s*:\s*/ ) { # inheritance 
				$rest = $';
				@inherits = parseInheritance( $rest );
			}
			else { # this is a var decl with "struct/class" etc prefixed.
				my $val = "";

				if ( $rest =~ /^((?:\s*[&*])+)/ ) {	# real name
					$name .= $1;
					$rest = $';
				}
				if ( $rest =~ /=\s*(.+)/ ) { # value
					$rest = $`;
					$val = $1;
				}

				$newNode = newVar( $ntype." ".$name, $rest, $val );
				$isCompound = 0;
			}
		}

		if ( $isCompound ) {
			$newNode = newClass( $tmpl, $ntype, $name, $endtag, @inherits );
		}
	}
	# IDL compound node
	elsif( $decl =~ /^\s*(module|interface|exception) # struct type
			\s+([-\w_]+)			# name
			(.*?)				# inheritance?
			([;{])/xs ) {
		
		my ( $type, $name, $rest, $fwd, $complete ) 
			= ( $1, $2, $3, $4 eq ";" ? 1 : 0,
				0 );
		my @in = ();
		print "IDL: [$type] [$name] [$rest] [$fwd]\n" if $main::debug;

		if( $rest =~ /^\s*:\s*/ ) {
			$rest = $';
			$rest =~ s/\s+//g;
			@in = split ",", $rest;
		}
		if( $decl =~ /}\s*;/ ) {
			$complete = 1;
		}

		$newNode = newIDLstruct( $type, $name, $fwd, $complete, @in );
	}
	# Method
	elsif ( $decl =~ /^\s*([^=]+?(?:operator\s*(?:\(\)|.?=)\s*)?) # ret+nm
		\( (.*?) \)		# parameters
		\s*((?:const)?)\s*
		\s*((?:=\s*0(?:L?))?)\s*	# Pureness. is "0L" allowed?
		\s*[;{]+/xs ) {	# rest

		print "Method: R+N:[$1]\n\tP:[$2]\n\t[$3]\n" if $main::debug;

		my $tpn = $1; # type + name
		my $params = $2;

		my $const = $3 eq "" ? 0 : 1;
		my $pure = $4 eq "" ? 0 : 1;

		if ( $tpn =~ /((?:\w+\s*::\s*)?operator.*?)\s*$/	# operator
					|| $tpn =~ /((?:\w*\s*::\s*~?)?[-\w:]+)\s*$/ ) { # normal
				my $name = $1;
				$tpn = $`;
				$newNode = newMethod( $tpn, $name, 
								$params, $const, $pure );
		}

		$skipBlock = 1; # FIXME check end token before doing this!
	}
	# Using: import namespace
	elsif ( $decl =~ /^\s*using\s+namespace\s+(\w+)/ ) {
		newNamespace( $1 );

	}

	# extern block
	elsif ( $decl =~ /^\s*extern\s*"(.*)"\s*{/ ) {
		$inExtern = 1 unless $decl =~ /}/;
	}

	# Single variable
	elsif ( $decl =~ /^
			\s*( (?:[\w_:]+(?:\s+[\w_:]+)*? )# type
				\s*(?:<.+>)?		# template
				\s*(?:[\&\*])?		# ptr or ref
				(?:\s*(?:const|volatile))* )
			\s*([\w_:]+)			# name
			\s*( (?:\[[^\[\]]*\] (?:\s*\[[^\[\]]*\])*)? ) # array
			\s*((?:=.*)?)			# value
			\s*([;{])\s*$/xs ) {
		my $type = $1;
		my $name = $2;
		my $arr  = $3;
		my $val	 = $4;
		my $end	 = $5;

		print "Var: [$name] type: [$type$arr] val: [$val]\n" 
				if $main::debug;

		$newNode = newVar( $type.$arr, $name, $val );

		$skipBlock = 1 if $end eq '{';

	}

	# Multi variables
	elsif ( $decl =~ m/^
		\s*( (?:[\w_:]+(?:\s+[\w_:]+)*? )	# type
		\s*(?:<.+>)?)						# template

		\s*( (?:\s*(?: [\&\*][\&\*\s]*)? 	# ptr or ref
			[\w_:]+)						# name
		\s*(?:\[[^\[\]]*\] (?:\s*\[[^\[\]]*\])*)? # array
		\s*(?:,								# extra vars
			\s*(?: [\&\*][\&\*\s]*)? 		# ptr or ref
			\s*(?:[\w_:]+)					# name
			\s*(?:\[[^\[\]]*\] (?:\s*\[[^\[\]]*\])*)? # array
			)* 
		\s*(?:=.*)?)						# value
		\s*[;]/xs ) {

		my $type = $1;
		my $names = $2;
		my $end = $3;
		my $doc = $docNode;

		print "Multivar: type: [$type] names: [$names] \n" if $main::debug;

		foreach my $vardecl ( split( /\s*,\s*/, $names ) ) {
			next unless $vardecl =~ m/
			\s*((?: [\&\*][\&\*\s]*)?) 	# ptr or ref
			\s*([\w_:]+)			# name
			\s*( (?:\[[^\[\]]*\] (?:\s*\[[^\[\]]*\])*)? ) # array
			\s*((?:=.*)?)			# value
				/xs;
			my ($ptr, $name, $arr, $val) = ($1, $2, $3, $4);

			print "Split: type: [$type$ptr$arr] ",
				" name: [$name] val: [$val] \n" if $main::debug;

			my $node = newVar( $type.$ptr.$arr, $name, $val );

			$docNode = $doc;	# reuse docNode for each
			postInitNode( $node ) unless !defined $node;
		}

		$skipBlock = 1 if $end eq '{';
	}
	# end of an "extern" block
	elsif ( $decl =~ /^\s*}\s*$/ ) {
		$inExtern = 0;
	}
	# end of an in-block declaration
	elsif ( $decl =~ /^\s*}\s*(.*?)\s*;\s*$/ ) {

		if ( $cNode->{astNodeName} eq "--" ) {
			# structure typedefs should have no name preassigned.
			# If they do, then the name in 
			# "typedef struct <name> { ..." is kept instead.
			# TODO: Buglet. You should fix YOUR code dammit. ;)


			$cNode->{astNodeName} = $1;
			my $siblings = $cNode->{Parent}->{KidHash};
			undef $siblings->{"--"};
			$siblings->{ $1 } = $cNode;
		}

		if ( $#classStack < 0 ) {
			confess "close decl found, but no class in stack!" ;
			$cNode = $rootNode;
		}
		else {
			$cNode = pop @classStack;
			print "end decl: popped $cNode->{astNodeName}\n" 
				if $main::debug;
		}
	}
	# unidentified block start
	elsif ( $decl =~ /{/ ) {
		print "Unidentified block start: $decl\n" if $main::debug;
		$skipBlock = 1;
	}
	else {

		## decl is unidentified.
		warn "Unidentified decl: '$decl'\n";
	}

	# once we get here, the last doc node is already used.
	# postInitNode should NOT be called for forward decls
	postInitNode( $newNode ) unless !defined $newNode;

	return $skipBlock;
}

sub postInitNode
{
	my $newNode = shift;

	carp "Cannot postinit undef node." if !defined $newNode;

	# The reasoning here:
	# Forward decls never get a source node.
	# Once a source node is defined, don't assign another one.

	if ( $newNode->{NodeType} ne "Forward" && !defined $newNode->{Source}) {
		$newNode->AddProp( "Source", $cSourceNode );
	}
	elsif ($main::debug) {
		print "postInit: skipping fwd: $newNode->{astNodeName}\n";
	}

	if( defined $docNode ) {
		kdocParseDoc::attachDoc( $newNode, $docNode, $rootNode );
		undef $docNode;
	}
}


##### Node generators

=head2 newEnum

	Reads the parameters of an enumeration.

	Returns the parameters, or undef on error.

=cut

sub newEnum
{
	my ( $enum ) = @_;
	my $k = undef;
	my $params = "";

	$k = $lastLine if defined $lastLine;

	if( defined $lastLine && $lastLine =~ /{/ ) {
		$params = $';
		if ( $lastLine =~ /}(.*?);/ ) {
			return initEnum( $enum, $1, $params );
		}
	}

	while ( defined ( $k = readCxxLine() ) ) {
		$params .= $k;

		if ( $k =~ /}(.*?);/ ) {
			return initEnum( $enum, $1, $params );
		}
	}

	return undef;
}

=head2 initEnum

	Parameters: name, (ref) params

	Returns an initialized enum node.

=cut

sub initEnum
{
	my( $name, $end, $params ) = @_;

	($name = $end) if $name eq "" && $end ne "";

	$params =~ s#\s+# #sg; # no newlines
	$params = $1 if $params =~ /^\s*{?(.*)}/;
	print "$name params: [$params]\n" if $main::debug;


	my ( $node ) = Ast::New( $name );
	$node->AddProp( "NodeType", "enum" );
	$node->AddProp( "Params", $params );
	kdocAstUtil::attachChild( $cNode, $node );

	return $node;
}

=head2 newIDLstruct

	Parameters: type, name, forward, complete, inherits...

	Handles an IDL structure definition (ie module, interface,
	exception).

=cut

sub newIDLstruct
{
	my ( $type, $name, $fwd, $complete ) = @_;

	my $node = exists $cNode->{KidHash} ? 
		$cNode->{KidHash}->{ $name } : undef;

	if( !defined $node ) {
		$node = Ast::New( $name );
		$node->AddProp( "NodeType", $fwd ? "Forward" : $type );
		$node->AddProp( "KidAccess", "public" );
		$node->AddProp( "Compound", 1 ) unless $fwd;
		kdocAstUtil::attachChild( $cNode, $node );
	}
	elsif ( $fwd ) {
		# If we have a node already, we ignore forwards.
		return undef;
	}
	elsif ( $node->{NodeType} eq "Forward" ) {
		# we are defining a previously forward node.
		$node->AddProp( "NodeType", $type );
		$node->AddProp( "Compound", 1 );
		$node->AddProp( "Source", $cSourceNode );
	}

	# register ancestors.
	foreach my $ances ( splice ( @_, 4 ) ) {
		my $n = kdocAstUtil::newInherit( $node, $ances );
	}

	if( !( $fwd || $complete) ) {
		print "newIDL: pushing $cNode->{astNodeName},",
			" new is $node->{astNodeName}\n"
				if $main::debug;
		push @classStack, $cNode;
		$cNode = $node;
	}

	return $node;
}

=head2 newClass

	Parameters: tmplArgs, cNodeType, name, endTag, @inheritlist

	Handles a class declaration (also fwd decls).

=cut

sub newClass
{
	my( $tmplArgs, $cNodeType, $name, $endTag ) = @_;

	my $access = "private";
	$access = "public" if $cNodeType ne "class";

	# try to find an exisiting node, or create a new one
	my $oldnode = kdocAstUtil::findRef( $cNode, $name );
	my $node = defined $oldnode ? $oldnode : Ast::New( $name );

	if ( $endTag ne "{" ) {
		# forward
		if ( !defined $oldnode ) {
			# new forward node
			$node->AddProp( "NodeType", "Forward" );
			$node->AddProp( "KidAccess", $access );
			kdocAstUtil::attachChild( $cNode, $node );
		}
		return $node;
	}

	# this is a class declaration

	print "ClassName: $name\n" if $main::debug;

	$node->AddProp( "NodeType", $cNodeType );
	$node->AddProp( "Compound", 1 );
	$node->AddProp( "Source", $cSourceNode );

	$node->AddProp( "KidAccess", $access );
	$node->AddProp( "Tmpl", $tmplArgs ) unless $tmplArgs eq "";

	if ( !defined $oldnode ) {
		kdocAstUtil::attachChild( $cNode, $node );
	}

	# inheritance

	foreach my $ances ( splice (@_, 4) ) {
		my $type = "";
		my $name = $ances;
		my $intmpl = undef;

WORD:
		foreach my $word ( split ( /([\w:]+(:?\s*<.*>)?)/, $ances ) ) {
			next WORD unless $word =~ /^[\w:]/;
			if ( $word =~ /(private|public|protected|virtual)/ ) {
				$type .= "$1 ";
			}
			else {
				
				if ( $word =~ /<(.*)>/ ) {
					# FIXME: Handle multiple tmpl args
					$name = $`;
					$intmpl = $1;
				}
				else {
					$name = $word;
				}

				last WORD;
			}
		}

		# set inheritance access specifier if none specified
		if ( $type eq "" ) {
			$type = $cNodeType eq "class" ? "private ":"public ";
		}
		chop $type;

		# attach inheritance information
		my $n = kdocAstUtil::newInherit( $node, $name );
		$n->AddProp( "Type", $type );

		$n->AddProp( "TmplType", $intmpl ) if defined $intmpl;

		print "In: $name type: $type, tmpl: $intmpl\n" if $main::debug;
	}

	# new current node
	print "newClass: Pushing $cNode->{astNodeName}\n" if $main::debug;
	push ( @classStack, $cNode );
	$cNode = $node;

	return $node;
}


=head3 parseInheritance

	Param: inheritance decl string
	Returns: list of superclasses (template decls included)

=cut

sub parseInheritance
{
	my $instring = shift;
	my @inherits = ();

	my $accum = "";
	foreach $instring ( split (/\s*,\s*/, $instring) ) {
		$accum .= $instring.", ";
		next unless  (kdocUtil::countReg( $accum, "<" )
			- kdocUtil::countReg( $accum, ">" ) ) == 0;

		# matching no. of < and >, so assume the parent is
		# complete
		$accum =~ s/,\s*$//;
		print "Inherits: '$accum'\n" if $main::debug;
		push @inherits, $accum;
		$accum = "";
	}

	return @inherits;
}


=head2 newNamespace

	Param: namespace name.
	Returns nothing.

	Imports a namespace into the current node, for ref searches etc.
	Triggered by "using namespace ..."

=cut

sub newNamespace
{
	$cNode->AddPropList( "ImpNames", shift );
}



=head2 newTypedef

	Parameters: realtype, name

	Handles a type definition.

=cut

sub newTypedef
{
	my ( $realtype, $name ) = @_;

	my ( $node ) = Ast::New( $name );

	$node->AddProp( "NodeType", "typedef" );
	$node->AddProp( "Type", $realtype );

	kdocAstUtil::attachChild( $cNode, $node );

	return $node;
}

=head2 newTypedefComp

	Params: realtype, name endtoken

	Creates a new compound type definition.

=cut

sub newTypedefComp
{
	my ( $realtype, $name, $endtag ) = @_;

	my ( $node ) = Ast::New( $name );

	$node->AddProp( "NodeType", "typedef" );
	$node->AddProp( "Type", $realtype );

	kdocAstUtil::attachChild( $cNode, $node );

	if ( $endtag eq '{' ) {
		print "newTypedefComp: Pushing $cNode->{astNodeName}\n" 
			if $main::debug;
		push ( @classStack, $cNode );
		$cNode = $node;
	}

	return $node;
}


=head2 newMethod

	Parameters: retType, name, params, const, pure?

	Handles a new method declaration or definition.

=cut

sub newMethod
{
	my ( $retType, $name, $params, $const, $pure ) = @_;
	my $parent = $cNode;
	my $class;

	print "Cracked: [$retType] [$name]\n\t[$params]\n\t[$const]\n" 
		if $main::debug;

    if ( $retType =~ /([\w\s_<>,]+)\s*::\s*$/ ) {
		# check if stuff before :: got into rettype by mistake.
		$retType = $`;
		($name = $1."::".$name);
		$name =~ s/\s+/ /g;
		print "New name = \"$name\" and type = '$retType'\n" if $main::debug;
	}

	if( $name =~ /^\s*(.*?)\s*::\s*(.*?)\s*$/ ) {
		# Fully qualified method name.
		$name = $2;
		$class = $1;

		if( $class =~ /^\s*$/ ) {
			$parent = $rootNode;
		}
		elsif ( $class eq $cNode->{astNodeName} ) {
			$parent = $cNode;
		}
		else {
			my $node = kdocAstUtil::findRef( $cNode, $class );

			if ( !defined $node ) {
				# if we couldn't find the name, try again with
				# all template parameters stripped off:
				my $strippedClass = $class;
				$strippedClass =~ s/<[^<>]*>//g;

				$node = kdocAstUtil::findRef( $cNode, $strippedClass );

				# if still not found: give up
				if ( !defined $node ) {
						warn $main::exe.": Unidentified class: $class ".
								"in $currentfile\:$.\n";
						return undef;
				}
			}

			$parent = $node;
		}
	}
	else {
		# Within current class/global
	}


	# flags

	my $flags = "";

	if( $retType =~ /static/ ) {
		$flags .= "s";
		$retType =~ s/static//g;
	}

	if( $const ) {
		$flags .= "c";
	}

	if( $pure ) {
		$flags .= "p";
	}

	if( $retType =~ /virtual/ ) {
		$flags .= "v";
		$retType =~ s/virtual//g;
	}

	print "\n" if $flags ne "" && $main::debug;

	if ( !defined $parent->{KidAccess} ) {
		warn "'", $parent->{astNodeName}, "' has no KidAccess ",
		exists $parent->{Forward} ? "(forward)\n" :"\n";
	}

	if ( $parent->{KidAccess} =~ /slot/ ) {
		$flags .= "l";
	}
	elsif ( $parent->{KidAccess} =~ /signal/ ) {
		$flags .= "n";
	}

	# node
	
	my $node = Ast::New( $name );
	$node->AddProp( "NodeType", "method" );
	$node->AddProp( "Flags", $flags );
	$node->AddProp( "ReturnType", $retType );
	$node->AddProp( "Params", $params );

	$parent->AddProp( "Pure", 1 ) if $pure;
	kdocAstUtil::attachChild( $parent, $node );

	return $node;
}


=head2 newAccess

	Parameters: access

	Sets the default "Access" specifier for the current class node. If
	the access is a "slot" type, "_slots" is appended to the access
	string.

=cut

sub newAccess
{
	my ( $access ) = @_;

	return undef unless ($access =~ /^\s*(\w+)\s*(slots)?/);

	print "Access: [$1] [$2]\n" if $main::debug;

	$access = $1;

	if ( defined $2 && $2 ne "" ) {
		$access .= "_" . $2;
	}

	$cNode->AddProp( "KidAccess", $access );

	return $cNode;
}


=head2 newVar

	Parameters: type, name, value

	New variable. Value is ignored if undef

=cut

sub newVar
{
	my ( $type, $name, $val ) = @_;

	my $node = Ast::New( $name );
	$node->AddProp( "NodeType", "var" );

	my $static = 0;
	if ( $type =~ /static/ ) {
	#	$type =~ s/static//;
		$static = 1;
	}

	$node->AddProp( "Type", $type );
	$node->AddProp( "Flags", 's' ) if $static;
	$node->AddProp( "Value", $val ) if defined $val;
	kdocAstUtil::attachChild( $cNode, $node );

	return $node;
}

1;
