/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2001 by Matthias Kiefer <matthias.kiefer@gmx.de>
                    
  based on code of Andrea Rizzi <rizzi@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef _TAG_EXTRACTOR_H_
#define _TAG_EXTRACTOR_H_

#include <qlist.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>


/**
 * holds a description of what a tag is 
 */
class TagListEntry
{
public:
  TagListEntry();
  TagListEntry(QString t,bool isReg);
  
  bool operator==(const TagListEntry& t2);
  QString asString() const { return tag; }
  QRegExp asRegExp() const { return reg; }
  bool isRegExp() const { return isReg; }

  /**
   * @return the first match of this tag in st,
   * QString::null, if it does not match
   */
  QString firstMatch(const QString st) const;

private:
  QString tag;
  QRegExp reg;
  bool isReg;
};
 

struct TagInfo{
    uint index;
    QString tag;
};


/**
 * class to extract tags from a string
 * @author Andrea Rizzi <rizzi@kde.org>
 */
class TagExtractor 
{

public:
  /**
   * Create a tag extractor for "string"
   */ 
  TagExtractor(QString string=QString::null);

  /**
   * Set the string of this tag extractor
   */
  void setString(QString string);

  /**
   * @return the number of tags found
   */
  uint countTags();

  /**
   * @return the first tag, and set the internal cursor to
   * the beginning
   */
  QString firstTag();

  /**
   * @return the next tag and move cursor forward
   */
  QString nextTag();

  /**
   * @return the n-th tag. It also moves the cursor.
   */
  QString tag(uint tagnumber);

  /**
   * @return the next tag and move cursor forward
   */
  QString prevTag();

  /**
   * @return the last tag and move the cursor to the end
   */
  QString lastTag();
  
  /**
   * @return a list of all tags
   */
  QStringList tags();

  /**
   * @return the string without tags
   *
   * @param keepPos if false, the tags are just removed and so the
   * position of the other words in the string will change. If true,
   * the tags are replaced with ' ' and therefore the position of the
   * words will not change
   */
  QString plainString(bool keepPos=false);

  /**
   * @return the string, where tags are replaced with the given string
   */
  QString tagsReplaced(const QString& replace);


  //Functions that allow user to define his own tags.
  
  /**
   * Add a tag to the list of tag identifier.
   * It may be a tag (e.g. <HTML> or <BODY>)
   * or a regexp (if the isRegExp flag is true)
   * for tags. 
   */
   void addTagIdentifier(QString tag,bool isRegExp);
   

  /**
   * Delete from the tag list the tag identified by
   * tag and isRegExp specified.
   */
  void deleteTagIdentifier(QString tag,bool isRegExp);

 
protected:
  /**
   * @return the static list of tags
   */
  static QValueList<TagListEntry> *tagList();
  void processString();
  QList<TagInfo> _tags;
  QString _string;
  
private: 
  static QValueList<TagListEntry> *_tagList; 
};




#endif
