/*
    Copyright (c) 2001 Dawit Alemayehu <adawit@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License (LGPL) as published by the Free Software Foundation;
    either version 2 of the License, or (at your option) any later
    version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <sys/utsname.h>

#include <kapp.h>
#include <kdebug.h>
#include <kaction.h>
#include <klocale.h>
#include <kglobal.h>
#include <ktrader.h>
#include <kio/job.h>
#include <kinstance.h>
#include <kpopupmenu.h>
#include <dcopclient.h>
#include <khtml_part.h>
#include <ksimpleconfig.h>
#include <kprotocolmanager.h>

#include "uachangerplugin.h"

#define UA_PTOS(x) (*it)->property(x).toString()
#define QFL(x) QString::fromLatin1(x)


UAChangerPlugin::UAChangerPlugin( QObject* parent, const char* name )
                :KParts::Plugin( parent, name )
{
  m_pUAMenu = new KActionMenu( i18n("Change Browser &Identification"), "agent",
                               actionCollection(), "changeuseragent" );
  m_pUAMenu->setDelayed( false );
  connect( m_pUAMenu->popupMenu(), SIGNAL( aboutToShow() ),
           this, SLOT( slotAboutToShow() ) );

  if ( parent && parent->inherits( "KHTMLPart" ) )
  {
    KHTMLPart* part = static_cast<KHTMLPart*>(parent);
    connect( part, SIGNAL(started(KIO::Job*)), this,
             SLOT(slotStarted(KIO::Job*)) );
  }
  else
    m_pUAMenu->setEnabled( false );

  m_iSelectedItem = -1;
  m_bReloadConfig = false;
  m_config = new KSimpleConfig( "kio_httprc" );
  parseDescFiles();
}

UAChangerPlugin::~UAChangerPlugin()
{
    delete m_config;
    m_config = 0L;
}

void UAChangerPlugin::parseDescFiles()
{
  KTrader::OfferList list = KTrader::self()->query("UserAgentStrings");
  if ( list.count() == 0 )
    return;

  m_mapAlias.clear();
  m_lstAlias.clear();
  m_lstIdentity.clear();
  KTrader::OfferList::ConstIterator it = list.begin();
  for ( ; it != list.end(); ++it )
  {
    QString tmp = UA_PTOS("X-KDE-UA-FULL");
    if ( (*it)->property("X-KDE-UA-DYNAMIC-ENTRY").toBool() )
    {
      int pos;
      struct utsname utsn;
      uname( &utsn );

      if ( (pos=tmp.find("appSysName")) != -1 )
        tmp.replace( pos, 10, QString(utsn.sysname) );
      if ( (pos=tmp.find("appSysRelease")) != -1 )
        tmp.replace( pos, 13, QString(utsn.release) );
      if ( (pos=tmp.find("appMachineType")) != -1 )
        tmp.replace( pos, 14, QString(utsn.machine) );

      QStringList languageList = KGlobal::locale()->languageList();
      if ( languageList.count() )
      {
        QStringList::Iterator it = languageList.find(QString::fromLatin1("C"));
        if( it != languageList.end() )
        {
          if( languageList.contains( QString::fromLatin1("en") ) > 0 )
            languageList.remove( it );
          else
            (*it) = QString::fromLatin1("en");
        }
      }
      if ( (pos=tmp.find("appLanguage")) != -1 )
        tmp.replace( pos, 11, QString("%1").arg(languageList.join(", ")) );
      if ( (pos=tmp.find("appPlatform")) != -1 )
        tmp.replace( pos, 11, QString::fromLatin1("X11") );
    }

    if ( !m_lstIdentity.contains(tmp) )
      m_lstIdentity << tmp;
    else
      continue; // Ignore dups!

    tmp = QString("%1 %2").arg(UA_PTOS("X-KDE-UA-SYSNAME")).arg(UA_PTOS("X-KDE"
                  "-UA-SYSRELEASE"));
    if ( tmp.stripWhiteSpace().isEmpty() )
      tmp = QString("%1 %2").arg(UA_PTOS("X-KDE-UA-"
                    "NAME")).arg(UA_PTOS("X-KDE-UA-VERSION"));
    else
      tmp = QString("%1 %2 on %3").arg(UA_PTOS("X-KDE-UA-"
                    "NAME")).arg(UA_PTOS("X-KDE-UA-VERSION")).arg(tmp);
    m_lstAlias << tmp;
    m_mapAlias[UA_PTOS("X-KDE-UA-TAG")].append( m_lstAlias.count()-1 );
  }
}

void UAChangerPlugin::slotStarted( KIO::Job* )
{
  KHTMLPart* part = dynamic_cast<KHTMLPart*>( parent() );
  if ( !part )
  {
    m_pUAMenu->setEnabled( false );
    return;
  }

  KURL u = part->url();
  bool isEnabled = (!u.isMalformed() && !u.isLocalFile());
  if ( isEnabled )
    m_prevHost = u.host();

  m_pUAMenu->setEnabled( true );
}

void UAChangerPlugin::slotAboutToShow()
{
  m_pUAMenu->popupMenu()->clear();

  AliasConstIterator map = m_mapAlias.begin();
  for( ; map != m_mapAlias.end(); ++map )
  {
    BrowserGroup::ConstIterator e = map.data().begin();
    for( ; e != map.data().end(); ++e )
      m_pUAMenu->popupMenu()->insertItem( m_lstAlias[(*e)], this,
                                          SLOT(slotItemSelected(int)),
                                          0, (*e) );
    m_pUAMenu->popupMenu()->insertSeparator();
  }

  m_pUAMenu->popupMenu()->setItemChecked( m_iSelectedItem, true );

  uint count = m_pUAMenu->popupMenu()->count();
  QString activeUA;
  if ( m_iSelectedItem > -1 && m_iSelectedItem < (int) count )
    activeUA = m_pUAMenu->popupMenu()->text( m_iSelectedItem );

  m_pUAMenu->popupMenu()->insertItem( i18n("Save Identification"), this,
                                      SLOT(slotSave()), 0, count );
  m_pUAMenu->popupMenu()->setItemEnabled( count, m_iSelectedItem != -1 );
  m_pUAMenu->popupMenu()->insertItem( i18n("Reload Identifications"), this,
                                      SLOT(parseDescFiles()), 0, ++count );
  m_pUAMenu->popupMenu()->insertSeparator();
  m_pUAMenu->popupMenu()->insertItem( i18n("Default Identification"), this,
                                      SLOT(slotReset()), 0, ++count );
  // we might show the default UA (uaForHost(m_prevHost)) together with "Default Identification"
  bool isDefaultUA = (KProtocolManager::defaultUserAgent() == activeUA) || (m_iSelectedItem == -1);
  m_pUAMenu->popupMenu()->setItemEnabled( count, !isDefaultUA );
}

void UAChangerPlugin::slotItemSelected( int id )
{
  kdDebug() << "Selected item ID: " << id << endl;
  kdDebug() << "Total number of items: " << m_lstIdentity.count() << endl;

  // A new UA string was selected...
  KHTMLPart* part = dynamic_cast<KHTMLPart*>( parent() );
  if ( !part ) return;

  if ( m_iSelectedItem != id )
  {
    m_iSelectedItem = id;
    KParts::URLArgs args;
    kdDebug() << "Selected Item: " << m_lstIdentity[m_iSelectedItem] << endl;
    args.metaData()["UserAgent"] = m_lstIdentity[m_iSelectedItem];
    args.reload = true;
    part->browserExtension()->setURLArgs( args );
    part->openURL( part->url() );
  }
}

void UAChangerPlugin::updateIOSlaves()
{
  // Inform running http(s) io-slaves about the change...
  if ( !kapp->dcopClient()->isAttached() )
    kapp->dcopClient()->attach();

  {
      QByteArray data;
      QDataStream stream( data, IO_WriteOnly );
      stream << QString("http");
      kapp->dcopClient()->send( "*", "KIO::Scheduler",
                                "reparseSlaveConfiguration(QString)", data );
  }
  {
      QByteArray data;
      QDataStream stream( data, IO_WriteOnly );
      stream << QString("https");
      kapp->dcopClient()->send( "*", "KIO::Scheduler",
                                "reparseSlaveConfiguration(QString)", data );
  }
}

void UAChangerPlugin::slotReset()
{
  KHTMLPart* part = dynamic_cast<KHTMLPart*>( parent() );
  if ( !part ) return;

  QString host = part->url().host();
  m_config->setGroup( host );
  m_config->deleteEntry("UserAgent", false);
  m_config->deleteGroup( host, false );
  m_config->sync();
  updateIOSlaves();

  // Reset some internal variables...
  m_iSelectedItem = -1;
  KParts::URLArgs args;
  args.reload = true;
  part->browserExtension()->setURLArgs( args );
  args.metaData()["UserAgent"] = KProtocolManager::defaultUserAgent();
  part->openURL( part->url() );
}

void UAChangerPlugin::slotSave()
{
  KHTMLPart* part = dynamic_cast<KHTMLPart*>( parent() );
  if ( !part ) return;

  m_config->setGroup( part->url().host() );
  m_config->writeEntry( "UserAgent", m_lstIdentity[m_iSelectedItem] );
  m_config->sync();
  updateIOSlaves();

  // Reset some internal variables...
  m_iSelectedItem = -1;
}

// not used at the moment
QString UAChangerPlugin::uaForHost( const QString& hostname )
{
  if( m_config->hasGroup( hostname ) )
  {
    m_config->setGroup( hostname );
    QString ua = m_config->readEntry( "UserAgent" );
    if ( !ua.isEmpty() )
      return ua;
  }
  return KProtocolManager::defaultUserAgent();
}

UAChangerPluginFactory::UAChangerPluginFactory( QObject* parent,
                                                const char* name )
                       :KLibFactory( parent, name )
{
  s_instance = new KInstance("uachangerplugin");
}

UAChangerPluginFactory::~UAChangerPluginFactory()
{
  delete s_instance;
  s_instance = 0L;
}

QObject* UAChangerPluginFactory::createObject( QObject* parent,
                                               const char* name,
                                               const char*,
                                               const QStringList & )
{
  return new UAChangerPlugin( parent, name );
}

extern "C"
{
  void* init_libuachangerplugin()
  {
    KGlobal::locale()->insertCatalogue("uachangerplugin"); 
    return new UAChangerPluginFactory;
  }
}
KInstance* UAChangerPluginFactory::s_instance = 0L;

#include "uachangerplugin.moc"
