/**
 ** On-Screen-Display for Noatun
 ** Copyright (c) 2001, Flaming Sword Productions
 ** All Rights Reserved under the BSD License
 **/
#include <kdebug.h>
#include <kwin.h>
#include <klocale.h>
#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qfont.h>
#include <qobject.h>
#include <qpainter.h>
#include <qwidget.h>
#include <qtimer.h>
#include <qlabel.h>
#include "kosd.h"

/**
 ** KOSD::KOSD(): CONSTRUCTOR. 
 **/
KOSD::KOSD()
	: QLabel(0L, "KOSD", WStyle_Customize | WStyle_StaysOnTop | WStyle_NoBorder ),
	_position (Bottom),
	_status (Stopped),
	_font (QFont("helvetica", 18, QFont::Bold)),
	_color (Qt::green),
	_displayTime (1000),
	_volume( -1),
	_displayTimer(new QTimer(this))
{

	KWin::setState( winId(), NET::Modal | NET::StaysOnTop | NET::SkipTaskbar | NET::SkipPager );

	emit setTextFont( _font );
	emit setTextColor( _color );

	setText( QString::null );
	setAlignment( AlignBottom );

	connect( _displayTimer, SIGNAL(timeout()), this, SLOT(hide()));
	connect( _displayTimer, SIGNAL(timeout()), this, SLOT(clear()));
	hide();
}

/**
 ** KOSD::~KOSD(): DESTRUCTOR. 
 **/
KOSD::~KOSD()
{
}

/**
 ** KOSD::setPosition(): Update our position.
 **/
void KOSD::setPosition( Position p )
{
	_position = p;
	_setPosition();
}

/**
 ** KOSD::setStatus(): Set our current operational status.
 **/
void KOSD::setStatus( Status s )
{
	_status = s;
	showText(_lastMessage);
}

/**
 ** KOSD::setDisplayTime(): Adjust the delay between text updates.
 **/
void KOSD::setDisplayTime( u_int time )
{
	_displayTime = time;
	_displayTimer->changeInterval( time );
}

/**
 ** KOSD::setTextColor(): Set the text color.
 **/
void KOSD::setTextColor( const QColor& c )
{
	_color = c;

	QPalette p(Qt::black);
	p.setColor( QColorGroup::Foreground, _color );
	setPalette( p );
}

/**
 ** KOSD::setFont(): Set the text color.
 **/
void KOSD::setTextFont( const QFont& c )
{
	_font = c;
	setFont( _font );
}

/**
 ** KOSD::setVolume(): Someone adjusted our volume.
 **/
void KOSD::setVolume( int volume )
{
	_volume = volume;
	showText(_lastMessage);
}

/**
 ** KOSD::showText(): Emit this signal when you want KOSD to show
 ** something.
 **/
void KOSD::showText( const QString& s )
{
	QString show = _statusString();
	switch(_status)
	{
	case Stopped:
		// do nothing
		break;
	case Playing:
	case Paused:
		show += "\n" + _volumeString();
		if(s.length()) show += "\n" + s;
		break;
	};

	_lastMessage = s;

	setText( show );
	_setPosition();
	emit _showLabel();
}

/**
 ** KOSD::_setPosition(): Based on the screen geometry and configured parameters,
 ** move the label to a point on the screen.
 **/
void KOSD::_setPosition()
{
	QFontMetrics metrics(_font);
	int scnum = QApplication::desktop()->screenNumber(QCursor::pos());
	QRect screenGeometry = QApplication::desktop()->screenGeometry(scnum);

	int height = metrics.lineSpacing() * (text().contains('\n') + 1);
	int width = screenGeometry.width();
	int x = 5 + screenGeometry.x();
	int y = (_position == Bottom ? screenGeometry.height() - height : 0);
        y += screenGeometry.y();

	setGeometry(x, y, width, height);
}

/**
 ** KOSD::_showLabel(): Emit this to show the label.  It will set
 ** a hide() timer automagically.
 **/
void KOSD::_showLabel()
{
	if(_displayTimer->isActive())
		_displayTimer->stop();
	_setPosition();
	updateMask();
	show();
	repaint(true);
	_displayTimer->start( _displayTime, false );
}

/**
 ** KOSD::_statusString(): Return a QString representing our current
 ** status.
 **/
QString KOSD::_statusString()
{
	QString status;
	switch(_status)
	{
	case(Playing):
		status = i18n("Play");
		break;
	case(Stopped):
		status = i18n("Stop");
		break;
	case(Paused):
		status = i18n("Pause");
		break;
	}
	return status;
}

/**
 ** KOSD::_volumeString(): Return a QString representing our current
 ** volume level.  The volume should be set by a signal connected to
 ** KOSD::setVolume().
 **/
QString KOSD::_volumeString()
{
	if(_volume < 0) return QString::null;

	QString volume = "";
	for ( int i = 0; i < (_volume / 2); i++ ) volume.append("|");
	for ( int i = (_volume / 2) + 1; i <= 50; i++ ) volume.append("-");

	return i18n("Volume: ") + volume;
}

void KOSD::updateMask(void)
{
	QPixmap mask(size());
	mask.fill();
	QPainter painter(&mask);
	drawContents(&painter);
	painter.end();
	QBitmap bm = mask.createHeuristicMask(false);
	setMask(bm);
}

#include "kosd.moc"
