/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdio.h>

#include <qregexp.h>

#include "dmaport.h"
#include "interrupt.h"
#include "ioaddress.h"
#include "isapnpscanner.h"
#include "memaddress.h"
#include "resbaseconf.h"


ISAPnPScanner::ISAPnPScanner()
{
  _parser = new FileParser("/proc/isapnp");

  if (_parser->eof()) fprintf(stderr, "ALSA isapnp module not loaded\n");
}


ISAPnPScanner::~ISAPnPScanner()
{
  delete _parser;
}


/*
 * Public methods
 *****************/

Device  *ISAPnPScanner::firstDevice()
{
  Device  *dev;
  
  if (_parser->reset()) 
  {
    fixupRange(dev = device());
    return dev;
  }
  
  return 0L;
}


Device  *ISAPnPScanner::nextDevice()
{
  Device *dev;
  
  if (!_parser->eof()) 
  {
    fixupRange(dev = device());
    return dev;
  }
  
  return 0L;
}


/*
 * Private methods
 ******************/
 
void  ISAPnPScanner::addActiveResource(const QString& line, ISAPnPDevice *isapnpdev)
{
  QString  pnpline = line.right(line.length()-6).simplifyWhiteSpace();
  QString  portstr, dmastr;
  int      pos;
  uint     iorange, port, irq, dma;
  ushort   bits;
    
  if (pnpline.left(4) == "port")
  {
    //debug("    ADD ACTIVE RESOURCE: PORT");
    
    pnpline = pnpline.right(pnpline.length()-4).simplifyWhiteSpace();
    
    do
    {
      pos     = pnpline.find(',');	    
      portstr = (pos >= 0) ? pnpline.left(pos) : pnpline;
      
      sscanf(pnpline.ascii(), "%x", &port);
      
      iorange = ioRange(port);
      
      isapnpdev->addResource(new IOAddress(port, iorange, _ioindex++));
      
      pnpline = (pos >= 0) ? pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace() : QString("");
    } while (pos >= 0);
  }
  else if (pnpline.left(3) == "IRQ")
  {
    //debug("    ADD ACTIVE RESOURCE: IRQ");

    pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
    
    sscanf(pnpline.ascii(), "%u", &irq);
    
    isapnpdev->addResource(new Interrupt(irq, _irqindex++));     
  }
  else if (pnpline.left(3) == "DMA")
  {
    //debug("    ADD ACTIVE RESOURCE: DMA");

    pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
    bits    = 8;
    
    do
    {
      if (!(pos = pnpline.find(','))) break;
      	   
      dmastr = (pos >= 0) ? pnpline.left(pos) : pnpline;
      
      sscanf(pnpline.ascii(), "%u", &dma);
      
      isapnpdev->addResource(new DMAPort(dma, bits, _dmaindex++));
      
      bits   += 8;
      pnpline = (pos >= 0) ? pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace() : QString("");
    } while (pos >= 0);   
  } //else debug("    ADD ACTIVE RESOURCE: UNKNOWN");
}


void  ISAPnPScanner::addBaseConfiguration(ISAPnPDevice *isapnpdev)
{
  QString             pnpline, irqstr, dmastr, numstr;
  QRegExp             commaslash("[,/]");
  ResourceBaseConfig  *baseconf = new ResourceBaseConfig();
  ResourceList	      *reslist  = 0L;
  uint                fromport, toport, align, size, irq, dma, bits;
  int		      pos;
  QList<uint>         dmalist;
  
  //debug("    ADD BASE CONFIGURATION"); 
  _irqindex = _dmaindex = _ioindex = _memindex = 0;
  
  while ((pnpline = _parser->nextLine()).left(8) == "Priority");
  
  _parser->traceBack();

  for (pnpline = _parser->nextLine();
       (pnpline.left(4) == "Port") || (pnpline.left(3) == "IRQ") || (pnpline.left(3) == "DMA");
       pnpline = _parser->nextLine())
  {
    if (pnpline.left(4) == "Port")
    {
      baseconf->append(reslist = new ResourceList());
      
      pnpline = pnpline.right(pnpline.length()-4).simplifyWhiteSpace();
      
      sscanf(pnpline.ascii(), "%x-%x, align %x, size %x", &fromport, &toport, &align, &size);
      //debug("      PORT 0x%04x-0x%04x ALIGN 0x%02x SIZE 0x%02x", fromport, toport, align, size);
      
      reslist->addIOBaseConfig(new IOBaseConfig(fromport, toport, align, size, _ioindex++));
    }
    else if (pnpline.left(3) == "IRQ")
    {
      baseconf->append(reslist = new ResourceList());
    
      pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
      irqstr  = "";				// only needed for debugging
      
      while ((pos = pnpline.find(commaslash)) >= 0)
      {      
        sscanf(pnpline.left(pos).ascii(), "%u", &irq);
	
        if (!irqstr.isEmpty()) irqstr += ", ";	// only needed for debugging
	irqstr += numstr.setNum(irq);	
	
	pnpline = pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace();
      
        reslist->append(new Interrupt(irq, _irqindex));
      }
      
      sscanf(pnpline.ascii(), "%u", &irq);
      reslist->append(new Interrupt(irq, _irqindex++));      
      
      if (!irqstr.isEmpty()) irqstr += ", ";	// only needed for debugging
      irqstr += numstr.setNum(irq);	
      //debug("      IRQ %s", irqstr.data());
    }
    else if (pnpline.left(3) == "DMA")
    {
      baseconf->append(reslist = new ResourceList());
      
      pnpline = pnpline.right(pnpline.length()-3).simplifyWhiteSpace();
      dmastr  = "";				// only needed for debugging
      
      dmalist.setAutoDelete(true);
      dmalist.clear();
      
      while ((pos = pnpline.find(',')) >= 0)
      {      
        sscanf(pnpline.left(pos).ascii(), "%u", &dma);
	dmalist.append(new uint(dma));
	
        if (!dmastr.isEmpty()) dmastr += ", ";	// only needed for debugging
	dmastr += numstr.setNum(dma);	
	
	pnpline = pnpline.right(pnpline.length()-pos-1).simplifyWhiteSpace();
      }
            
      sscanf(pnpline.ascii(), "%u %u-bit", &dma, &bits);
      dmalist.append(new uint(dma));
      
      for (uint *dmaptr = dmalist.first(); dmaptr; dmaptr = dmalist.next())      
        reslist->append(new DMAPort(*dmaptr, bits, _dmaindex));
        
      _dmaindex++;  
        
      if (!dmastr.isEmpty()) dmastr += ", ";	// only needed for debugging
      dmastr += numstr.setNum(dma);	
      //debug("      DMA %s\t%2u BITS", dmastr.data(), bits);
    }
  }
    
  isapnpdev->addBaseConfig(baseconf);  
  _parser->traceBack();
}


ISAPnPDevice  *ISAPnPScanner::device()
{
  static QString      cardidstr;		// Seems we have to clean up sooner or later...
  QString             logidstr;
  static int	      cardno;
  int                 logno;
  
  QString             pnpline, logname;
  QString             devidstr;
  static QString      vendorid, devname;
  static uint         deviceid;
  char                mark = 39;
  uint                logid;
  int		      pos;

  _irqindex = _dmaindex = _ioindex = _memindex = 0;
  
  while (!_parser->eof())
  {
    pnpline = _parser->nextLine();
    
    if ((pnpline.left(6) == "Device") || (pnpline.left(4) == "Card"))	// We have a physical device
    {
      sscanf(pnpline.ascii(), "%*s %i", &cardno);
    
      if ((pos = pnpline.find(mark)) >= 0)
      {
        pnpline   = pnpline.right(pnpline.length()-pos-1); 	
	cardidstr = pnpline.left(7);
        vendorid  = pnpline.left(3); 
        devname   = pnpline.right(pnpline.length()-8);
	
	sscanf(pnpline.ascii()+3, "%x", &deviceid);
	
	devname.truncate(devname.find(mark)); 	
      }
      
      //debug("PHYSICAL DEVICE <%s>", devname.data());      
    }
    else if (pnpline.left(14) == "Logical device")			// We have a logical device
    {
      sscanf(pnpline.ascii(), "%*s %*s %i", &logno);
      
      if ((pos = pnpline.find(mark)) >= 0)
      {
        pnpline  = pnpline.right(pnpline.length()-pos-1);
        logidstr = pnpline.left(7);
	logname  = pnpline.right(pnpline.length()-8);	
	devidstr = pnpline.mid(3, 4);	
	
	sscanf(devidstr.ascii(), "%x", &logid);
	
	logname.truncate(logname.find(mark)); 	
      }
    
      return logicalDevice(new ISAPnPDevice(cardno-1, cardidstr, logno, logidstr,
      					    vendorid, deviceid, devname, logid, logname));
    }
  }

  return 0L;
}


/*
 * If there is no driver loaded for this device, we don't get
 * address range information from /proc/ioports, so in this
 * case we have to look at the device base configurations (if any)
 ******************************************************************/
void  ISAPnPScanner::fixupRange(Device *device)
{
  ResourceBaseConfig  *baseconf;
  Resource            *res;
  ulong               range;
  Resource::restype   type; 
  
  if (!device) return;
  
  for (res = device->firstResource(); res; res = device->nextResource())
  {
    type  = res->resourceType();
    range = res->resourceRange();
  
    if (((type == Resource::ResIOAddress) || (type == Resource::ResMemAddress)) && (range == 1))
    {
      // We don't know the address range
    
      for (baseconf = device->firstConfig(); baseconf; baseconf = device->nextConfig())
      {
        if ((range = baseconf->resourceRange(res)) > 0)
        {
          res->setResourceRange(range);
          break;
        }
      }
    }
  }
  
}


ISAPnPDevice  *ISAPnPScanner::logicalDevice(ISAPnPDevice *isapnpdev)
{
  QString  pnpline;

  //debug("  LOGICAL DEVICE <%s>", isapnpdev->deviceName().data());

  for (pnpline = _parser->nextLine(); 
       (pnpline.left(10) == "Compatible") || (pnpline.left(9) == "Device is");
       pnpline = _parser->nextLine())
  {
    if (pnpline.left(16) == "Device is active") 
    {
      //debug("  DEVICE IS ACTIVE");
      isapnpdev->setActive(true);
    }
  }

  _parser->traceBack();
  
  while ((pnpline = _parser->nextLine()).left(6) == "Active") addActiveResource(pnpline, isapnpdev);

  _parser->traceBack();
  
  while (_parser->nextLine().left(9) == "Resources")	// What does this tag mean?
  {    
    for (pnpline = _parser->nextLine();
         (pnpline.left(8) == "Priority") || (pnpline.left(19) == "Alternate resources");
         pnpline = _parser->nextLine())
      addBaseConfiguration(isapnpdev);
    
    _parser->traceBack();
  }

  _parser->traceBack();
  
  return isapnpdev;
}


