/*
  KPF - Public fileserver for KDE

  Copyright 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <cmath>

#include <qdir.h>
#include <qstring.h>
#include <qcache.h>

#include "Defines.h"
#include "DirectoryLister.h"

namespace KPF
{
  class DirectoryLister::Private
  {
    public:

      Private()
      {
      }

      static QCache<QString> * cache;
  };

  QCache<QString> * DirectoryLister::Private::cache = 0L;

  QString buildHTML(const QString & title, const QString & body)
  {
    QString s;

    s +=
      "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\"\n"
      "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"
      "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
      "<head>\n"
      "<title>";

    s += title;

    s +=
      "</title>\n"
      "</head>\n"
      "<body>\n";

    s += body;

    s +=
      "</body>\n"
      "</html>\n";

    return s;
  }

  QString prettySize(uint size)
  {
    QString suffix(" bytes");

    if (size > 1023)
    {
      if (size > 1048575)
      {
        size = static_cast<uint>(ceil(size / 1048576));
        suffix = " MB";
      }
      else
      {
        size = static_cast<uint>(ceil(size / 1024));
        suffix = " kB";
      }
    }

    return QString("%1%2").arg(size).arg(suffix);
  }

  DirectoryLister * DirectoryLister::instance_ = 0L;

    DirectoryLister *
  DirectoryLister::instance()
  {
    if (0 == instance_)
      instance_ = new DirectoryLister;

    return instance_;
  }

  DirectoryLister::DirectoryLister()
  {
    Private::cache = new QCache<QString>(32 * 1024 /* 32 kb */);

    d = new Private;
  }

  DirectoryLister::~DirectoryLister()
  {
    delete d;

    delete Private::cache;
    Private::cache = 0;
  }

    QString
  DirectoryLister::html(const QString & path)
  {
    QDir d(path);

    if (!d.exists())
    {
      return buildHTML
        ("Error", "Directory does not exist: " + path);
    }

    const QFileInfoList * infoList = d.entryInfoList();

    if (0 == infoList)
    {
      return buildHTML
        ("Error", "Directory unreadable: " + path);
    }

    QString s;

    s += "<table";
    s += " class=\"kpf_directory_listing\"";
    s += " summary=\"kpf directory listing\">\n";

    s += "<tr>\n";
    s += "<th>Type</th>\n";
    s += "<th>Name</th>\n";
    s += "<th>Size</th>\n";
    s += "</tr>\n";

    for (QPtrListIterator<QFileInfo> it(*infoList); it.current(); ++it)
    {
      QFileInfo * fi(it.current());

      s += "<tr>\n";
      s += "<td>" + QString(fi->isDir() ? "Dir" : "File") + "</td>\n";
      s += "<td>" + fi->fileName() + "</td>\n";
      s += "<td>" + prettySize(fi->size()) + "</td>\n";
      s += "</tr>\n";
    }

    s += "</table>\n";

    return buildHTML("Directory listing for " + path, s);
  }

} // End namespace KPF

// vim:ts=2:sw=2:tw=78:et
