/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: attrprovider.h
 *
 * Description: Base for all attribute providers.
 *
 */

#ifndef ATTRPROVIDER_H
#define ATTRPROVIDER_H

typedef struct AttrProvider_ AttrProvider;

#include "jobattributes.h"
#include "printer.h"

typedef struct {
    void (*Destructor) (AttrProvider * this);
    Aps_Result(*AttrCreateCopy) (AttrProvider * this,
                                 AttrProvider ** newProvider,
                                 Aps_Handle associatedObject);
    Aps_Result(*AttrGetList) (AttrProvider * this,
                              const char *group,
                              TrackArray_PtrChar *attributeIDs);
    Aps_Result(*AttrGetSubGroups) (AttrProvider * this,
                                   const char *group,
                                   TrackArray_PtrChar *subGroupNames);
    Aps_Result(*AttrGetTranslatedName) (AttrProvider * this,
                                        const char *attributeID,
                                        char **translatedName);
    Aps_Result(*AttrGetMainData) (AttrProvider * this,
                                  const char *attributeID,
                                  char **mainData);
    Aps_Result(*AttrGetTranslatedData) (AttrProvider * this,
                                        const char *attributeID,
                                        char **translatedData);
    Aps_Result(*AttrGetType) (AttrProvider * this,
                              const char *attributeID,
                              Aps_AttrType * attributeType);
    Aps_Result(*AttrGetRange) (AttrProvider * this,
                               const char *attributeID,
                               double *minSetting,
                               double *maxSetting);
    Aps_Result(*AttrGetOptions) (AttrProvider * this,
                                 const char *attributeID,
                                 TrackArrayIndirect_AttrOption *options);
    Aps_Result(*AttrGetSetting) (AttrProvider * this,
                                 const char *attributeID,
                                 char **setting);
    Aps_Result(*AttrSetSetting) (AttrProvider * this,
                                 const char *attributeID,
                                 const char *setting);
    Aps_Result(*AttrCheckConstraints) (AttrProvider * this,
                                       const char *attributeID,
                                       const char *setting,
                                       char **conflictingAttribute,
                                       char **conflictingSetting);
    Aps_Result(*AttrWriteBlock) (AttrProvider * this,
                                 Aps_JobHandle job,
                                 Aps_BlockType blockType);
    Aps_Result(*AttrResetToModelDefaults)(AttrProvider * this,
                                          Printer * printer);
    Aps_Result(*AttrSaveAsPrinterDefaults)(AttrProvider * this,
                                           Printer * printer);
    Aps_Result(*AttrSetToPrinterDefaults)(AttrProvider * this,
                                          Printer * printer);
    Aps_Result(*AttrRemovePrinterDefaults)(AttrProvider * this,
                                           Printer * printer);
} AttrProviderVtbl;

struct AttrProvider_ {
    AttrProviderVtbl *vtbl;
    Aps_Handle associatedObject;
};

/* Initialization of a new attribute provider instance. */
Aps_Result AttrProvInitialize(AttrProvider * this,
                              Aps_Handle associatedObject);

/* Attribute provider destruction. */
void AttrProvDelete(AttrProvider * this);
                              
/* Base-class implementations for attribute provider virtual functions. */
void AttrProvDestructor(AttrProvider * this);
Aps_Result AttrProvCreateCopy(AttrProvider * this,
                              AttrProvider ** newProvider,
                              Aps_Handle associatedObject);
Aps_Result AttrProvGetList(AttrProvider * this,
                           const char *group,
                           TrackArray_PtrChar *attributeIDs);
Aps_Result AttrProvGetSubGroups(AttrProvider * this,
                                const char *group,
                                TrackArray_PtrChar *subGroupNames);
Aps_Result AttrProvGetTranslatedName(AttrProvider * this,
                                     const char *attributeID,
                                     char **translatedName);
Aps_Result AttrProvGetMainData(AttrProvider * this,
                               const char *attributeID,
                               char **mainData);
Aps_Result AttrProvGetTranslatedData(AttrProvider * this,
                                     const char *attributeID,
                                     char **translatedData);
Aps_Result AttrProvGetType(AttrProvider * this,
                           const char *attributeID,
                           Aps_AttrType * attributeType);
Aps_Result AttrProvGetRange(AttrProvider * this,
                            const char *attributeID,
                            double *minSetting,
                            double *maxSetting);
Aps_Result AttrProvGetOptions(AttrProvider * this,
                              const char *attributeID,
                              TrackArrayIndirect_AttrOption *options);
Aps_Result AttrProvGetSetting(AttrProvider * this,
                              const char *attributeID,
                              char **setting);
Aps_Result AttrProvSetSetting(AttrProvider * this,
                              const char *attributeID,
                              const char *setting);
Aps_Result AttrProvCheckConstraints(AttrProvider * this,
                                    const char *attributeID,
                                    const char *setting,
                                    char **conflictingAttribute,
                                    char **conflictingSetting);
Aps_Result AttrProvWriteBlock(AttrProvider * this,
                              Aps_JobHandle job,
                              Aps_BlockType blockType);
Aps_Result AttrProvResetToModelDefaults(AttrProvider * this,
                                        Printer * printer);
Aps_Result AttrProvSaveAsPrinterDefaults(AttrProvider * this,
                                         Printer * printer);
Aps_Result AttrProvSetToPrinterDefaults(AttrProvider * this,
                                        Printer * printer);
Aps_Result AttrProvRemovePrinterDefaults(AttrProvider * this,
                                         Printer * printer);
#endif /* ATTRPROVIDER_H */
