/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: filterspecial.c
 *
 * Description: APIs for composing and applying filters to special objects.
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "filter.h"
#include "job.h"
#include "utils.h"

DEBUG_CHANNEL_DEFAULT(filter)

/* This collection of functions constructs filters of specific
 * types.  This helps to eliminate some of the redundancy of this
 * process.  (tough luck)
 */
static Aps_Result FilterBuildXString(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XString filterData, FilterSpecial filterSpecial);
static Aps_Result FilterBuildXInt(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XInt filterData, FilterSpecial filterSpecial);
static Aps_Result FilterBuildXTime(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XTime filterData, FilterSpecial filterSpecial);
static Aps_Result FilterBuildXBitfield(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XBitfield filterData,
    FilterSpecial filterSpecial);

/* This collection of functions performs comparisons of fields of
 * specific types.  These take into account the mode flags.
 */
static Aps_Result FilterCompareXString(FilterQual *filterQual, int *match,
    XString filterData);
static Aps_Result FilterCompareXInt(FilterQual *filterQual, int *match,
    XInt filterData);
static Aps_Result FilterCompareXTime(FilterQual *filterQual, int *match,
    XTime filterData);
static Aps_Result FilterCompareXBitfield(FilterQual *filterQual, int *match,
    XBitfield filterData);

/* ---------------------------------------------------------------------------
 * FilterBuildX[datatype]()
 *
 * Builds and links a filter object of the specified type.
 *
 * Parameters: filterHandle - pointer to handle storage location, if not
 *                            NULL, filter is appended to old one
 *             filterMode   - one of Aps_FilterMode modes
 *             filterData   - data to match (stored)
 *
 *     Result: A standard APS result code indicating the cause of failure.
 */
/*** STRINGS ***/
static Aps_Result FilterBuildXString(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XString filterData, FilterSpecial filterSpecial)
{
    Aps_Result  result;
    FilterQual *qual;
    /* check params */
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;
    if (! filterData) filterData = "";
    /* create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_SPECIAL, filterSpecial,
        filterMode, strtotallen(filterData));
    if (! qual) return APS_OUT_OF_MEMORY;
    strcpy((char *)qual->data, filterData);
    /* associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}
/*** INTEGERS ***/
static Aps_Result FilterBuildXInt(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XInt filterData, FilterSpecial filterSpecial)
{
    Aps_Result  result;
    FilterQual *qual;

    /* Check params */
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_SPECIAL, filterSpecial,
        filterMode, sizeof(XInt));
    if (! qual) return APS_OUT_OF_MEMORY;
    *((XInt *)qual->data) = filterData;

    /* Associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}
/*** TIME ***/
static Aps_Result FilterBuildXTime(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XTime filterData, FilterSpecial filterSpecial)
{
    Aps_Result  result;
    FilterQual *qual;

    /* Check params */
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_SPECIAL, filterSpecial,
        filterMode, sizeof(XTime));
    if (! qual) return APS_OUT_OF_MEMORY;
    *((XTime *)qual->data) = filterData;

    /* Associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}
/*** BITFIELDS ***/
static Aps_Result FilterBuildXBitfield(Aps_FilterHandle *filterHandle,
    Aps_FilterMode filterMode, XBitfield filterData, FilterSpecial filterSpecial)
{
    Aps_Result  result;
    FilterQual *qual;

    /* Check params */
    if (! filterHandle) return APS_INVALID_PARAM;
    if (! (filterMode & APS_FILTER_MODE_MASK)) return APS_INVALID_PARAM;

    /* Create qualifier */
    qual = FilterAllocQual(FILTER_TYPE_SPECIAL, filterSpecial,
        filterMode, sizeof(XBitfield));
    if (! qual) return APS_OUT_OF_MEMORY;
    *((XBitfield *)qual->data) = filterData;

    /* Associate with filter */
    result = FilterLink(filterHandle, qual);
    if (result != APS_SUCCESS) free(qual);
    return result;
}

/* ---------------------------------------------------------------------------
 * FilterCompareX[datatype]()
 *
 * Compares fields of a specific type.
 *
 * Parameters: filterQual - filter qualifier data
 *             filterData - source data to compare to (from object
 *                          being tested)
 *
 *     Result: (inparam) match - TRUE if match, else FALSE.
 *             A standard APS result code indicating the cause of failure.
 */
/*** STRINGS ***/
static Aps_Result FilterCompareXString(FilterQual *filterQual, int *match,
    XString filterData)
{
    int temp;
    /* match wildcards -- case sensitive or not */
    if (filterQual->filterMode & APS_FILTER_WILDCARDS) {
        temp = strcmpwild((char *)filterQual->data, filterData,
            filterQual->filterMode & APS_FILTER_NOCASE);
    } else {
        /* no wildcards, but check case sensitivity */
        if (filterQual->filterMode & APS_FILTER_NOCASE) {
            temp = stricmp((char *)filterQual->data, filterData);
        } else {
            temp = strcmp((char *)filterQual->data, filterData);
        }
    }
    /* handle greater than / less than / equal */
    if (filterQual->filterMode & APS_FILTER_GREATERTHAN) {
        if (temp > 0) *match = TRUE;
    } else if (filterQual->filterMode & APS_FILTER_LESSTHAN) {
        if (temp < 0) *match = TRUE;
    } else *match = (temp == 0);
    return APS_SUCCESS;
}
/*** INTEGERS ***/
static Aps_Result FilterCompareXInt(FilterQual *filterQual, int *match,
    XInt filterData)
{
    XInt source = *((XInt *)filterQual->data);
    /* handle greater than / less than / equal */
    if (filterQual->filterMode & APS_FILTER_GREATERTHAN) {
        if (filterData > source) *match = TRUE;
    } else if (filterQual->filterMode & APS_FILTER_LESSTHAN) {
        if (filterData < source) *match = TRUE;
    } else *match = (source == filterData);

    return APS_SUCCESS;
}
/*** TIME ***/
static Aps_Result FilterCompareXTime(FilterQual *filterQual, int *match,
    XTime filterData)
{
    XTime source = *((XInt *)filterQual->data);
    /* handle greater than / less than / equal */
    if (filterQual->filterMode & APS_FILTER_GREATERTHAN) {
        if (filterData > source) *match = TRUE;
    } else if (filterQual->filterMode & APS_FILTER_LESSTHAN) {
        if (filterData < source) *match = TRUE;
    } else *match = (source == filterData);
    return APS_SUCCESS;
}
/*** BITFIELDS ***/
static Aps_Result FilterCompareXBitfield(FilterQual *filterQual, int *match,
    XBitfield filterData)
{
    XBitfield source = *((XBitfield *)filterQual->data);
    /* handle all bits / any bit */
    if (filterQual->filterMode & APS_FILTER_ALLBITS) {
        *match = (source == filterData);
    } else {
        *match = ((source & filterData) != 0);
    }
    return APS_SUCCESS;
}

/*----------------------------------------------------------------------------
 * Aps_Result Aps_Filter[ObjectClass]By[FieldName]()
 *
 * Compose filters of the form:
 *   Aps_Result Aps_Filter[ObjectClass]By[FieldName](
 *     Aps_FilterHandle *filterHandle,
 *     Aps_FilterMode    filterMode,
 *     [FieldType]       matchField);
 *
 * Parameters: filterHandle - pointer to handle storage location, if not
 *                            NULL, filter is appended to old one
 *             filterMode   - one of Aps_FilterMode modes
 *             matchField   - data with which to match field
 *
 *     Result: A standard APS result code indicating the cause of failure.
 */

/* To ease the gradual addition of new filters, we will rely on a
 * simple macro to create filters of a particular type.  In
 * the long run, it may be better to export this functionality
 * directly to the application or define it through a set of
 * inline operations rather than have the library define all these
 * entry points.
 */
#define MAKE_FILTERFUNC(_xname, _xtype, _xastype, _xspecial) \
    Aps_Result Aps_Filter ## _xname ( \
        Aps_FilterHandle filterHandle, Aps_FilterMode filterMode, \
        _xtype filterData) { \
        return FilterBuild ## _xastype (filterHandle, filterMode, \
            (_xastype)filterData, _xspecial); \
    }

MAKE_FILTERFUNC( JobByStatus, Aps_JobStatus,
    XBitfield, FILTER_SPECIAL_JOB_jobStatus)
MAKE_FILTERFUNC( JobByHost, const char *,
    XString, FILTER_SPECIAL_JOB_jobHost)
MAKE_FILTERFUNC( JobByName, const char *,
    XString, FILTER_SPECIAL_JOB_jobName)
MAKE_FILTERFUNC( JobByFilename, const char *,
    XString, FILTER_SPECIAL_JOB_jobFilename)
MAKE_FILTERFUNC( JobByID, int,
    XInt, FILTER_SPECIAL_JOB_jobID)
MAKE_FILTERFUNC( JobBySize, int,
    XInt, FILTER_SPECIAL_JOB_jobSize)
MAKE_FILTERFUNC( JobByCreationTime, time_t,
    XTime, FILTER_SPECIAL_JOB_jobCreationTime)
MAKE_FILTERFUNC( JobByFormat, const char *,
    XString, FILTER_SPECIAL_JOB_jobFormat)
MAKE_FILTERFUNC( JobByOrder, int,
    XInt, FILTER_SPECIAL_JOB_jobOrder)
MAKE_FILTERFUNC( JobByPriority, int,
    XInt, FILTER_SPECIAL_JOB_jobPriority)
MAKE_FILTERFUNC( JobByPrinterStatus, Aps_PrinterStatus,
    XBitfield, FILTER_SPECIAL_JOB_printerStatus)
MAKE_FILTERFUNC( JobByPrinterName, const char *,
    XString, FILTER_SPECIAL_JOB_printerName)
MAKE_FILTERFUNC( JobByOwnerName, const char *,
    XString, FILTER_SPECIAL_JOB_ownerName)
MAKE_FILTERFUNC( JobByOwnerID, int,
    XInt, FILTER_SPECIAL_JOB_ownerID)
MAKE_FILTERFUNC( JobByLocalHost, const char *,
    XString, FILTER_SPECIAL_JOB_localHost)
MAKE_FILTERFUNC( JobByLocalFile, const char *,
    XString, FILTER_SPECIAL_JOB_localFile)
MAKE_FILTERFUNC( JobBySpoolHost, const char *,
    XString, FILTER_SPECIAL_JOB_spoolHost)
MAKE_FILTERFUNC( JobBySpoolFile, const char *,
    XString, FILTER_SPECIAL_JOB_spoolFile)

/* --------------------------------------------------------------------
 * FilterCheckSpecial()
 *
 * Run an object through a filter to determine if it is a match or not.
 *
 * Parameters : filterQual - filter qualifications
 *              objHandle  - handle of object to check
 *
 * Result     : matches    - set to TRUE if match, else FALSE.
 */
Aps_Result FilterCheckSpecial(FilterQual *qual, Aps_Handle objHandle,
    int *match)
{
    *match = FALSE;

    /* first separate by object type */
    switch ((int)qual->filterSpecial & (~ 0x0fff)) {
        /* JOB object */
        case FILTER_SPECIAL_JOB_BASE: {
            /* locate job */
            Aps_QuickJobInfo *info;
            Job *job = JobGetPtrFromHandle((Aps_JobHandle)objHandle);
            if (! job) return APS_INVALID_HANDLE;
            /* get info */
            info = job->info;
            if (! info) return APS_GENERIC_FAILURE;

            /* next separate by field */
            switch (qual->filterSpecial) {
                case FILTER_SPECIAL_JOB_jobStatus:
                    return FilterCompareXBitfield(qual, match,
                        (XBitfield)info->jobStatus);
                case FILTER_SPECIAL_JOB_jobHost:
                    return FilterCompareXString(qual, match,
                        (XString) info->jobHost);
                case FILTER_SPECIAL_JOB_jobName:
                    return FilterCompareXString(qual, match,
                        (XString) info->jobName);
                case FILTER_SPECIAL_JOB_jobFilename:
                    return FilterCompareXString(qual, match,
                        (XString) info->jobFilename);
                case FILTER_SPECIAL_JOB_jobID:
                    return FilterCompareXInt(qual, match,
                        (XInt) info->jobID);
                case FILTER_SPECIAL_JOB_jobSize:
                    return FilterCompareXInt(qual, match,
                        (XInt) info->jobSize);
                case FILTER_SPECIAL_JOB_jobCreationTime:
                    return FilterCompareXTime(qual, match,
                        (XTime) info->jobCreationTime);
                case FILTER_SPECIAL_JOB_jobFormat:
                    return FilterCompareXString(qual, match,
                        (XString) info->jobFormat);
                case FILTER_SPECIAL_JOB_jobOrder:
                    return FilterCompareXInt(qual, match,
                        (XInt) info->jobOrder);
                case FILTER_SPECIAL_JOB_jobPriority:
                    return FilterCompareXInt(qual, match,
                        (XInt) info->jobPriority);
                case FILTER_SPECIAL_JOB_printerStatus:
                    return FilterCompareXBitfield(qual, match,
                        (XBitfield)info->printerStatus);
                case FILTER_SPECIAL_JOB_printerName:
                    return FilterCompareXString(qual, match,
                        (XString) info->printerName);
                case FILTER_SPECIAL_JOB_docTitle:
                    return FilterCompareXString(qual, match,
                        (XString) info->docTitle);
                case FILTER_SPECIAL_JOB_docRevision:
                    return FilterCompareXString(qual, match,
                        (XString) info->docRevision);
                case FILTER_SPECIAL_JOB_docComments:
                    return FilterCompareXString(qual, match,
                        (XString) info->docComments);
                case FILTER_SPECIAL_JOB_docAuthor:
                    return FilterCompareXString(qual, match,
                        (XString) info->docAuthor);
                case FILTER_SPECIAL_JOB_docType:
                    return FilterCompareXString(qual, match,
                        (XString) info->docType);
                case FILTER_SPECIAL_JOB_docCreator:
                    return FilterCompareXString(qual, match,
                        (XString) info->docCreator);
                case FILTER_SPECIAL_JOB_ownerName:
                    return FilterCompareXString(qual, match,
                        (XString) info->ownerName);
                case FILTER_SPECIAL_JOB_ownerID:
                    return FilterCompareXInt(qual, match,
                        (XInt) info->ownerID);
                case FILTER_SPECIAL_JOB_localHost:
                    return FilterCompareXString(qual, match,
                        (XString) info->localHost);
                case FILTER_SPECIAL_JOB_localFile:
                    return FilterCompareXString(qual, match,
                        (XString) info->localFile);
                case FILTER_SPECIAL_JOB_spoolHost:
                    return FilterCompareXString(qual, match,
                        (XString) info->spoolHost);
                case FILTER_SPECIAL_JOB_spoolFile:
                    return FilterCompareXString(qual, match,
                        (XString) info->spoolFile);
                /* hunh? */
                default:
            }
        } break;
        /* ... some other object
         * case FILTER_SPECIAL_XXX_BASE:
         */
        /* hunh? */
        default:
    }
    return APS_GENERIC_FAILURE;
}