/*
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: ppdmanager.c
 *
 * Description: PPD Manager is a library with an interface allowing the user 
 *              to query PPD (PostScript Printer Descriction) files, and to 
 *              create custom PPD files.  The library also contains methods 
 *              specific to building user interfaces.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "ppdmanager.h"


/* The following defines are possible values of the member variable 
 * mainKeyType of the structure PPDKey.  They can be combined when looking 
 * for more then one type of main key.
 */
#define PPD_MAIN_KEY_TYPE_STANDARD          0x001
#define PPD_MAIN_KEY_TYPE_OPEN_UI           0x002
#define PPD_MAIN_KEY_TYPE_OPEN_GROUP        0x004
#define PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP     0x008
#define PPD_MAIN_KEY_TYPE_JCL_OPEN_UI	    0x010
#define PPD_MAIN_KEY_TYPE_CLOSE_UI          0x020
#define PPD_MAIN_KEY_TYPE_CLOSE_GROUP       0x040
#define PPD_MAIN_KEY_TYPE_CLOSE_SUBGROUP    0x080
#define PPD_MAIN_KEY_TYPE_JCL_CLOSE_UI      0x100
#define PPD_MAIN_KEY_TYPE_ALL               0x200

/* The following defines are possible values of the member variable
 * valueType of the structure PPDKey.
 */
#define PPD_VALUE_TYPE_NONE                0
#define PPD_VALUE_TYPE_INVOCATION          1
#define PPD_VALUE_TYPE_QUOTED              2
#define PPD_VALUE_TYPE_SYMBOL              3
#define PPD_VALUE_TYPE_STRING              4
#define PPD_VALUE_TYPE_TRANSLATION_STRING  5

/* The following defines a default order dependency when one is not found. */
#define PPD_ORDER_DEPENDENCY_NOT_DEFINED   9999999

/* The following defines the longest valid line size in a PPD file. */
#define PPD_MAX_LINE_SIZE 255

/* The following defines strings found in a PPD file. */
#define PPD_NTS_INCLUDE                 "*Include"
#define PPD_NTS_ORDER_DEPENDENCY        "*OrderDependency"
#define PPD_NTS_NON_UI_ORDER_DEPENDENCY "*NonUIOrderDependency"
#define PPD_NTS_PAGE_SIZE               "*PageSize"
#define PPD_NTS_PAGE_REGION             "*PageRegion"
#define PPD_NTS_INPUT_SLOT              "*InputSlot"
#define PPD_NTS_UI_CONSTRAINTS          "*UIConstraints"
#define PPD_NTS_NON_UI_CONSTRAINTS      "*NonUIConstraints"
#define PPD_NTS_DEFAULT                 "*Default"
#define PPD_NTS_JOB_PATCH_FILE          "*JobPatchFile"

#define PPD_NTS_TRUE                      "True"
#define PPD_NTS_FALSE                     "False"
#define PPD_NTS_UNKNOWN                   "Unknown"

#define PPD_NTS_CUSTOMIZED_PPD          "*% Locally Customized PPD"
#define PPD_NTS_EMBEDDED_CUSTOMIZED_PPD "*% Embedded Locally Customized PPD"
#define PPD_NTS_COREL_CUSTOM_PPD        "*% Corel Corporation Custom PPD"

#define PPD_NTS_CUSTOM_PAGE_SIZE                     "*CustomPageSize"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE               "*ParamCustomPageSize"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_WIDTH         "Width"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_HEIGHT        "Height"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_WIDTH_OFFSET  "WidthOffset"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_HEIGHT_OFFSET "HeightOffset"
#define PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_ORIENTATION   "Orientation"

#define PPD_NTS_OPEN_UI                 "*OpenUI"
#define PPD_NTS_CLOSE_UI                "*CloseUI"
#define PPD_NTS_OPEN_GROUP              "*OpenGroup"
#define PPD_NTS_CLOSE_GROUP             "*CloseGroup"
#define PPD_NTS_OPEN_SUBGROUP           "*OpenSubGroup"
#define PPD_NTS_CLOSE_SUBGROUP          "*CloseSubGroup"

#define PPD_NTS_JCL_BEGIN               "*JCLBegin"
#define PPD_NTS_JCL_TO_PS               "*JCLToPSInterpreter"
#define PPD_NTS_JCL_END                 "*JCLEnd"
#define PPD_NTS_JCL_OPEN_UI             "*JCLOpenUI"
#define PPD_NTS_JCL_CLOSE_UI            "*JCLCloseUI"

#define PPD_NTS_UI_TYPE_BOOLEAN         "Boolean"
#define PPD_NTS_UI_TYPE_PICK_ONE        "PickOne"
#define PPD_NTS_UI_TYPE_PICK_MANY       "PickMany"

/* The following defines strings found in a PS document. */
#define PPD_NTS_PS_PROCEDURE_BEGIN      "[{"
#define NTS_NTS_PS_PROCEDURE_END        "} stopped cleartomark"
#define PPD_NTS_PS_BEGIN_FEATURE        "%%BeginFeature: "
#define NTS_NTS_PS_END_FEATURE          "%%EndFeature"


/* The following structure describes and store one key (statement) from the 
 * PPD file.
 */
typedef struct PPDKey_ {

    int   mainKeyType;
    char *mainKey;

    char *optionKey;
    char *optionKeyTrns;

    int   valueType;
    char *value;
    char *valueTrns;

    struct PPDKey_ *next;
    struct PPDKey_ *parent;
    struct PPDKey_ *child;

} PPDKey;


/* The following structure describes and store one UI Constraint key 
 * (statement) from the PPD file.
 */
typedef struct PPDUIConstraints_ {

    char *firstMainKey;
    char *firstOptionKey;

    char *secondMainKey;
    char *secondOptionKey;

    struct PPDUIConstraints_ *next;

} PPDUIConstraints;


/* The following structure describes an iterate handle.  The structure 
 * contains all the information necessary to iterate. 
 */
typedef struct PPDIterate_ {

    char   *searchValue;
    PPDKey *currentKey;

} PPDIterate;


/* The following structure describes and stores a string, and a value used 
 * for the sorting  With a pointer to the next entry, this forms a link 
 * list of strings.
 */
typedef struct PPDSort_ {

    char            *string;
    double           order;
    struct PPDSort_ *next;

} PPDSort;


/* The following structure describes an instance of the PPD Manager. */
typedef struct PPDManager_ {

    char *ppdFilename;
    char *ppdIncludeFilename;
    int   isOk;

    char **stringMemoryArray;
    int    stringMemoryArraySize;

    PPDIterate **iterateMemoryArray;
    int          iterateMemoryArraySize;

    char **includeMemoryArray;
    int    includeMemoryArraySize;

    PPDUIConstraints *headUIConstraints;
    PPDUIConstraints *lastUIConstraints;

    PPDKey *headAllKeys;
    PPDKey *lastAllKeys;

    double useCustomPageSize;
    double customPageSizeWidth;
    double customPageSizeHeight;
    double customPageSizeWidthOffset;
    double customPageSizeHeightOffset;
    int    customPageSizeOrientation;

} PPDManager;


/* String helper methods */
int PPDStringAddChar(char **string, char character);
int PPDStringAddString(char **string, const char *characters);
int PPDStringAddStringOfLength(char **string, const char *characters, 
                               int length);
int PPDStringDelete(char **string);
int PPDStringSkipWhiteSpaces(char **string);
int PPDStringSkipLine(char **string);
int PPDStringSkipToNextKey(char **string);

/* String Memory Array helper methods */
int PPDAddToStringMemoryArray(PPDManager * ppdManager, char **characters);
int PPDRemoveFromStringMemoryArray(PPDManager * ppdManager, 
                                   char **characters);
int PPDClearStringMemoryArray(PPDManager * ppdManager);

/* Iterate Memory Array helper methods */
int PPDAddToIterateMemoryArray(PPDManager * ppdManager, 
                               PPDIterate **iterate);
int PPDRemoveFromIterateMemoryArray(PPDManager * ppdManager, 
                                    PPDIterate **iterate);
int PPDClearIterateMemoryArray(PPDManager * ppdManager);

/* Include Memory Array helper methods */
int PPDAddToIncludeMemoryArray(PPDManager * ppdManager, const char *string);
int PPDClearIncludeMemoryArray(PPDManager * ppdManager);

/* PPD Manager helper methods */
int PPDClearUIConstraints(PPDUIConstraints ** constraints);
int PPDCreateCopyOfUIConstraints(PPDUIConstraints ** constraints, 
                                 PPDUIConstraints ** newConstraints);
int PPDClearAllKeys(PPDKey ** key);
int PPDCreateCopyOfAllKeys(PPDKey ** key, PPDKey ** newKey);

int PPDCleanUp(PPDManager * ppdManager);

int PPDParseFilename(PPDManager * ppdManager, const char *ppdFilename);
int PPDParseString(PPDManager * ppdManager, const char *string);

int PPDAddKey(PPDManager * ppdManager,
              char *currentMainKey, int currentMainKeyLength,
              char *currentOptionKey, int currentOptionKeyLength,
              char *currentOptionKeyTrns, int currentOptionKeyTrnsLength,
              char *currentValue, int currentValueLength,
              char *currentValueTrns, int currentValueTrnsLength);

PPDKey *PPDGetKeyRecursive(PPDKey * current, PPDKey ** searchKey,
                           const char *mainKey, int mainKeyType,
                           const char *optionKey, const char *value,
                           int checkAllLevels);
int PPDCleanString(char **dest, const char *source, int type);
int PPDConvertHexValue(char character);
int PPDGetOrderDependency(PPDManager * ppdManager, const char *string,
                          double *order, int isUI);
int PPDGetSorted(PPDSort * headSort, char **string);
int PPDSetSorted(PPDSort ** headSort, const char *string, double order);
int PPDClearSort(PPDSort ** key);
int PPDGetCustomPagePSSetup(PPDManager *ppdManager, char **psString);
int PPDCreateCustomPPD(PPDManager *ppdManager, const char *ppdFilename, 
			           int embedded);


/* ---------------------------------------------------------------------------
 * Public interface methods.
 * 
 * All methods of the public interface returns 0 if an error occured, and 1 if
 * everything was fine.  Not finding a specific key is not concidered an 
 * error.  For that, check the returning strings if they are NULL.
 */

/* ---------------------------------------------------------------------------
 * PPDCreate()
 *
 * Public interface method to create an instance of the PPD Manager. One PPD
 * file at a time can be parsed with one instance of the PPD Manager.  One
 * instance can be used for multiple parsing though.
 *
 * Parameters: none
 *
 *     Return: Handle to newly created instance of the PPD Manager, NULL if
 *             failed.
 */
PPDHandle PPDCreate()
{
    PPDHandle handle = NULL;


    /* Allocate memory for one instance of the PPD Manager */
    handle = malloc(sizeof(PPDManager));

    /* Initialize all members of PPD Manager */
    if (handle != NULL) {
        memset(handle, '\0', sizeof(PPDManager));
    }

    return (handle);
}

/* ---------------------------------------------------------------------------
 * PPDDestroy()
 *
 * Public interface method to delete an instance of the PPD Manager created
 * by 'PPDCreate'.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager to be deleted
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDDestroy(PPDHandle ppdHandle)
{
    int ok = 1;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }

    PPDCleanUp((PPDManager *) ppdHandle);
    free(ppdHandle);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCreateCopy()
 *
 * Public interface method to create a copy of an instance of the PPD Manager.
 *
 * Parameters: none
 *
 *     Return: Handle to newly created instance of the PPD Manager, NULL if
 *             failed.
 */
PPDHandle PPDCreateCopy(PPDHandle ppdHandle)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDManager *newPPDManager = NULL;


    /* Make sure we have valid pointer. */
    if (ppdHandle == NULL) {
        return (NULL);
    }
    ppdManager = (PPDManager *) ppdHandle;


    /* Allocate memory for one instance of the PPD Manager. */
    newPPDManager = (PPDManager *) malloc(sizeof(PPDManager));
    if (newPPDManager == NULL) {
        return (NULL);
    }

    /* Initialize all members of new PPD Manager. */
    memset(newPPDManager, '\0', sizeof(PPDManager));

    /* Set the member variables. */
    ok &= PPDStringAddString(&(newPPDManager->ppdFilename), 
                             ppdManager->ppdFilename);
    ok &= PPDStringAddString(&(newPPDManager->ppdIncludeFilename), 
                             ppdManager->ppdIncludeFilename);
    newPPDManager->isOk = ppdManager->isOk;
    newPPDManager->useCustomPageSize = ppdManager->useCustomPageSize;
    newPPDManager->customPageSizeWidth = ppdManager->customPageSizeWidth;
    newPPDManager->customPageSizeHeight = ppdManager->customPageSizeHeight;
    newPPDManager->customPageSizeWidthOffset = 
                             ppdManager->customPageSizeWidthOffset;
    newPPDManager->customPageSizeHeightOffset = 
                             ppdManager->customPageSizeHeightOffset;
    newPPDManager->customPageSizeOrientation = 
                             ppdManager->customPageSizeOrientation;

    /* Copy the link list of UI contraints. */
    ok &= PPDCreateCopyOfUIConstraints(&ppdManager->headUIConstraints, 
                                       &newPPDManager->headUIConstraints);

    /* Copy the link list of all keys. */
    ok &= PPDCreateCopyOfAllKeys(&ppdManager->headAllKeys, 
                                 &newPPDManager->headAllKeys);

    /* If there was any error, don't return the copy. */
    if (ok == 0) {
        PPDCleanUp(newPPDManager);
        free(newPPDManager);
	    newPPDManager = NULL;
    }

    return (newPPDManager);
}

/* ---------------------------------------------------------------------------
 * PPDSetFilename()
 *
 * Public interface method to set the current PPD file in the PPD Manager and
 * parse it.  If successfull, you will be able to query the PPD Manager.
 *
 * Parameters: ppdHandle   - Handle to instance of the PPD Manager
 *             ppdFilename - PPD file to parse
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDSetFilename(PPDHandle ppdHandle, const char *ppdFilename)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    char *value = NULL;


    /* Make sure we have valid pointers. */
    if ((ppdHandle == NULL) || (ppdFilename == NULL)) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* We have a file to parse and must first clean-up since this might not
     * be the first time a file was parsed with the specified PPD Manager.
     */
    PPDCleanUp(ppdManager);

    /* Set member variables of PPD Manager */
    ok = PPDStringAddString(&(ppdManager->ppdFilename), ppdFilename);
    if (ok == 1) {
        /* Parse the file */
        ok = PPDParseFilename(ppdManager, ppdManager->ppdFilename);

        /* Check if we successfully parsed the file */
        if (ok == 0) {
            PPDCleanUp(ppdManager);
        } else {
            ppdManager->isOk = 1;
	        ppdManager->lastUIConstraints = NULL;
	        ppdManager->lastAllKeys = NULL;
        }
    }

    /* Confirm that this is a valid PPD file */
    if (ok == 1) {
        ok = PPDGetKey(ppdManager, "*PPD-Adobe", &value, NULL);
	    
        if ((ok == 0) || (value == NULL)) {
            PPDCleanUp(ppdManager);
	        ok = 0;
	    }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetFilename()
 *
 * Public interface method to get the current PPD filename in the PPD Manager.
 *
 * Parameters: ppdHandle   - Handle to instance of the PPD Manager
 *             ppdFilename - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetFilename(PPDHandle ppdHandle, char **ppdFilename)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;


    /* Make sure we have a valid pointer. */
    if (ppdFilename == NULL) {
        return (0);
    }
    *ppdFilename = NULL;

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Set member variables of PPD Manager */
    ok = PPDStringAddString(ppdFilename, ppdManager->ppdFilename);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDIsCustomPPD()
 *
 * Public interface method to tell if the current PPD file in the PPD Manager
 * is a Custom PPD file.
 *
 * Parameters: ppdHandle   - Handle to instance of the PPD Manager
 *
 *     Return: 1 if a custom PPD file, otherwise 0
 */
int PPDIsCustomPPD(PPDHandle ppdHandle)
{
    int         ok         = 0;
    PPDManager *ppdManager = NULL;
    FILE       *pPPDFile   = NULL;
    char        szLine[PPD_MAX_LINE_SIZE+1];


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a file to read. */
    if ((ppdManager->ppdFilename == NULL) || 
        (strlen(ppdManager->ppdFilename) == 0)) {
        return (0);
    }


    /* Lets open the file for reading. */
    pPPDFile = fopen(ppdManager->ppdFilename, "rb");
    if(NULL == pPPDFile) {
        return(0);
    }

    /* Get the first line to look if it is one of our Custom PPD. */
    if(NULL != fgets(szLine, PPD_MAX_LINE_SIZE, pPPDFile))
    {
        if(0 == strncmp(szLine, PPD_NTS_CUSTOMIZED_PPD, 
                        strlen(PPD_NTS_CUSTOMIZED_PPD)) ||
           0 == strncmp(szLine, PPD_NTS_EMBEDDED_CUSTOMIZED_PPD, 
                        strlen(PPD_NTS_EMBEDDED_CUSTOMIZED_PPD)) ||
           0 == strncmp(szLine, PPD_NTS_COREL_CUSTOM_PPD, 
                        strlen(PPD_NTS_COREL_CUSTOM_PPD))) {
	        ok = 1;
	    }
    }

    /* Close the file */
    fclose(pPPDFile);

    return(ok);
}

/* ---------------------------------------------------------------------------
 * PPDSaveToCustomPPD()
 *
 * Public interface method to creates a Corel Custom PPD file where the values 
 * of all the default keys of selectable features will be saved. A link to the
 * original PPD file will then be appended, using the '*Include' statement. 
 * The file specified will be overwritten.
 *
 * Parameters: ppdHandle   - Handle to instance of the PPD Manager
 *             ppdFilename - Custom PPD file to write to
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDSaveToCustomPPD(PPDHandle ppdHandle, const char *ppdFilename)
{
    int ok = 0;

    ok = PPDCreateCustomPPD((PPDManager *)ppdHandle, ppdFilename, 0);

    return(ok);
}

/* ---------------------------------------------------------------------------
 * PPDSaveToEmbeddedCustomPPD()
 *
 * Public interface method to creates a Corel Embedded Custom PPD file where 
 * the values of all the default keys of selectable features will be saved. A
 * copy of the original PPD file will then be appended, instead of using the 
 * '*Include' statement.  The file specified will be overwritten.
 *
 * Parameters: ppdHandle   - Handle to instance of the PPD Manager
 *             ppdFilename - Custom PPD file to write to
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDSaveToEmbeddedCustomPPD(PPDHandle ppdHandle, const char *ppdFilename)
{
    int ok = 0;

    ok = PPDCreateCustomPPD((PPDManager *)ppdHandle, ppdFilename, 1);

    return(ok);
}

/* ---------------------------------------------------------------------------
 * PPDDeleteString()
 *
 * Public interface method to delete a string returned by the other public 
 * interface methods.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             ppdString - Pointer to string to delete
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDDeleteString(PPDHandle ppdHandle, char **ppdString)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;


    /* Make sure we have valid pointers. */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Call helper method to remove from the memory array and delete it. */
    ok = PPDRemoveFromStringMemoryArray(ppdManager, ppdString);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetJCLHeader()
 *
 * Public interface method that will return a string containing the JCL code 
 * that must be inserted at the very beginning of the PS document.  All the 
 * selectable JCL features for the device described by the PPD will be 
 * included.  The returning string will be NULL if JCL is not supported by the 
 * device.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             jclString - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetJCLHeader(PPDHandle ppdHandle, char **jclString)
{
    int ok = 1;
    double order = 0;
    char *optionKey = NULL;
    char *value = NULL;
    char *jcl = NULL;
    PPDPosition pos = NULL;
    PPDManager *ppdManager = NULL;
    PPDSort *headSort = NULL;


    /* Make sure we have a valid pointer. */
    if (jclString == NULL) {
        return (0);
    }
    *jclString = NULL;

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Call public interface method to get the JCL Begin statement.  If 
     * there is an error or it cannot be found, then there is no need to go 
     * further.
     */
    ok &= PPDGetKey(ppdHandle, PPD_NTS_JCL_BEGIN, &jcl, NULL);
    ok &= PPDStringAddString(jclString, jcl);
    PPDDeleteString(ppdHandle, &jcl);
    if ((ok == 0) || (*jclString == NULL)) {
        PPDStringDelete(jclString);
        return (ok);
    }


    /* Now that we have the Begin statement, get all selectable JCL 
     * features.  We will store them in a list of PPDKeys.  As we get one
     * we will sort them with it's order dependency.
     */
    ok = PPDGetKeysIterateStart(ppdHandle, &pos, PPD_NTS_JCL_OPEN_UI);
    if (ok == 0) {
        PPDStringDelete(jclString);
        return (0);
    }

    while (pos != NULL) {
        ok &= PPDGetKeysIterateNext(ppdHandle, &pos, NULL, &optionKey, NULL,
                                    NULL, NULL);
        ok &= PPDGetDefaultKey(ppdHandle, optionKey, &value);
        ok &= PPDGetKeyWithOption(ppdHandle, optionKey, value, NULL, &jcl, NULL);

        /* Save the JCL feature string and order dependency value. */
        if (jcl != NULL) {
            ok &= PPDGetOrderDependency(ppdManager, optionKey, &order, 1);
            ok &= PPDSetSorted(&headSort, jcl, order);
        }

        /* Clean up. */
        PPDDeleteString(ppdHandle, &optionKey);
        PPDDeleteString(ppdHandle, &value);
        PPDDeleteString(ppdHandle, &jcl);

        /* Check for errors/ */
        if (ok == 0) {
            PPDIterateEnd(ppdHandle, &pos);
            PPDClearSort(&headSort);
            return (0);
        }
    }

    PPDIterateEnd(ppdHandle, &pos);


    /* Now that the selectable JCL features are sorted we can output them. */
    ok &= PPDGetSorted(headSort, jclString);
    ok &= PPDClearSort(&headSort);


    /* Call public interface method to get the JCL To PS Interpreter 
     * statement.
     */
    ok &= PPDGetKey(ppdHandle, PPD_NTS_JCL_TO_PS, &jcl, NULL);
    ok &= PPDStringAddString(jclString, jcl);
    PPDDeleteString(ppdHandle, &jcl);


    /* Add the string for the user to the memory array. */
    ok &= PPDAddToStringMemoryArray(ppdManager, jclString);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetJCLFooter()
 *
 * Public interface method that will return a string containing the JCL code 
 * that must be inserted at the very end of the PS document.  The returning 
 * string will be NULL if JCL is not supported by the device.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             jclString - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetJCLFooter(PPDHandle ppdHandle, char **jclString)
{
    int ok = 1;


    /* Make sure we have a valid pointer. */
    if (jclString == NULL) {
        return (0);
    }
    *jclString = NULL;


    /* Call public interface method to get the JCL end statement. */
    ok = PPDGetKey(ppdHandle, PPD_NTS_JCL_END, jclString, NULL);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetJobPatch()
 *
 * Public interface method that will return a string containing the PS code 
 * that must be inserted before the prolog or any other PS code.  It fixes 
 * a bug or sets up an initial state for a job. The returning string will 
 * be NULL if no features needs to be set.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             psString  - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetJobPatch(PPDHandle ppdHandle, char **psString)
{
    int ok = 1;
    char *optionKey = NULL;
    char *ps = NULL;
    PPDPosition pos = NULL;
    PPDManager *ppdManager = NULL;


    /* Make sure we have a valid pointer. */
    if (psString == NULL) {
        return (0);
    }
    *psString = NULL;

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Get all *JobPatchFile keys. */
    ok = PPDGetKeysIterateStart(ppdHandle, &pos, PPD_NTS_JOB_PATCH_FILE);
    if (ok == 0) {
        return (0);
    }

    while (pos != NULL) {
        ok &= PPDGetKeysIterateNext(ppdHandle, &pos, NULL, &optionKey, NULL, 
	                            &ps, NULL);
        /* Save the Job Patch string and order dependency value. */
        if (ps != NULL) {
				    ok &= PPDStringAddString(psString, "\r\n");
            ok &= PPDStringAddString(psString, PPD_NTS_PS_PROCEDURE_BEGIN);
            ok &= PPDStringAddString(psString, "\r\n");
            ok &= PPDStringAddString(psString, PPD_NTS_PS_BEGIN_FEATURE);
            ok &= PPDStringAddString(psString, PPD_NTS_JOB_PATCH_FILE);
            ok &= PPDStringAddString(psString, " ");
            ok &= PPDStringAddString(psString, optionKey);
            ok &= PPDStringAddString(psString, "\r\n");
            ok &= PPDStringAddString(psString, ps);
            ok &= PPDStringAddString(psString, "\r\n");
            ok &= PPDStringAddString(psString, NTS_NTS_PS_END_FEATURE);
            ok &= PPDStringAddString(psString, "\r\n");
            ok &= PPDStringAddString(psString, NTS_NTS_PS_PROCEDURE_END);
            ok &= PPDStringAddString(psString, "\r\n");
        }

        PPDDeleteString(ppdHandle, &ps);
	    PPDDeleteString(ppdHandle, &optionKey);

        if (ok == 0) {
            PPDIterateEnd(ppdHandle, &pos);
            return (0);
        }
    }

    PPDIterateEnd(ppdHandle, &pos);

    /* Add this string for the user to the memory array. */
    ok &= PPDAddToStringMemoryArray(ppdManager, psString);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetPSSetup()
 *
 * Public interface method that will return a string containing the PS code 
 * that must be inserted in the setup section of the PS document.  All the 
 * selectable PS features for the device described by the PPD will be 
 * included, and order dependencies will be taken care of.  The returning 
 * string will be NULL if no features needs to be set.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             psString  - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetPSSetup(PPDHandle ppdHandle, char **psString)
{
    int ok = 1;
    double order = 0;
    char *optionKey = NULL;
    char *value = NULL;
    char *ps = NULL;
    char *buffer = NULL;
    PPDPosition pos = NULL;
    PPDManager *ppdManager = NULL;
    PPDSort *headSort = NULL;


    /* Make sure we have a valid pointer. */
    if (psString == NULL) {
        return (0);
    }
    *psString = NULL;

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Get all selectable PS features. */
    ok = PPDGetKeysIterateStart(ppdHandle, &pos, PPD_NTS_OPEN_UI);
    if (ok == 0) {
        return (0);
    }

    while (pos != NULL) {
        ok &= PPDGetKeysIterateNext(ppdHandle, &pos, NULL, &optionKey, NULL,
                                    NULL, NULL);

        /* We will take care of '*PageSize', '*PageRegion' and '*InputSlot 
	     * later.
         */
        if ((strcmp(optionKey, PPD_NTS_PAGE_REGION) != 0) &&
            (strcmp(optionKey, PPD_NTS_INPUT_SLOT) != 0) &&
            (strcmp(optionKey, PPD_NTS_PAGE_SIZE) != 0)) {

            ok &= PPDGetDefaultKey(ppdHandle, optionKey, &value);
            ok &= PPDGetKeyWithOption(ppdHandle, optionKey, value, NULL, &ps,
                                      NULL);
            /* Save the PS feature string and order dependency value. */
            if ((optionKey != NULL) && (value != NULL)) {
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_PROCEDURE_BEGIN);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_BEGIN_FEATURE);
                ok &= PPDStringAddString(&buffer, optionKey);
                ok &= PPDStringAddString(&buffer, " ");
                ok &= PPDStringAddString(&buffer, value);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, ps);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_END_FEATURE);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_PROCEDURE_END);
                ok &= PPDStringAddString(&buffer, "\r\n");

                ok &= PPDGetOrderDependency(ppdManager, optionKey, &order, 1);
                ok &= PPDSetSorted(&headSort, buffer, order);

                PPDStringDelete(&buffer);
            }

            PPDDeleteString(ppdHandle, &value);
            PPDDeleteString(ppdHandle, &ps);
        }

        PPDDeleteString(ppdHandle, &optionKey);

        if (ok == 0) {
            PPDIterateEnd(ppdHandle, &pos);
            PPDClearSort(&headSort);
            return (0);
        }
    }

    PPDIterateEnd(ppdHandle, &pos);


    /* If we want to use a custom page size, add it to the list of 
     * features. 
     */
    if (ppdManager->useCustomPageSize == 1) {

        ok &= PPDGetCustomPagePSSetup(ppdManager, &buffer);
        ok &= PPDGetOrderDependency(ppdManager, PPD_NTS_CUSTOM_PAGE_SIZE, 
	                            &order, 0);
        ok &= PPDSetSorted(&headSort, buffer, order);
        PPDStringDelete(&buffer);

    } else {
        /* We do not want to use both '*PageSize' and '*PageRegion'.  If no input
         * slot was selected, then we will use '*PageSize'.
         */
        ok &= PPDGetDefaultKey(ppdHandle, PPD_NTS_INPUT_SLOT, &value);

	    if ((value == NULL) || (strcmp(value, PPD_NTS_UNKNOWN) == 0)) {

            PPDDeleteString(ppdHandle, &value);

            /* Get '*PageSize' default state. */
            ok &= PPDGetDefaultKey(ppdHandle, PPD_NTS_PAGE_SIZE, &value);
            ok &= PPDGetKeyWithOption(ppdHandle, PPD_NTS_PAGE_SIZE, value, 
	                              NULL, &ps, NULL);
            /* Save the PS feature string and order dependency value. */
            if (value != NULL) {
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_PROCEDURE_BEGIN);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_BEGIN_FEATURE);
                ok &= PPDStringAddString(&buffer, PPD_NTS_PAGE_SIZE);
                ok &= PPDStringAddString(&buffer, " ");
                ok &= PPDStringAddString(&buffer, value);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, ps);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_END_FEATURE);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_PROCEDURE_END);
                ok &= PPDStringAddString(&buffer, "\r\n");

                ok &= PPDGetOrderDependency(ppdManager, PPD_NTS_PAGE_SIZE, &order, 1);
                ok &= PPDSetSorted(&headSort, buffer, order);

                PPDStringDelete(&buffer);
            }

            PPDDeleteString(ppdHandle, &value);
            PPDDeleteString(ppdHandle, &ps);

	    } else {
            /* Get '*InputSlot' default state. */
            ok &= PPDGetKeyWithOption(ppdHandle, PPD_NTS_INPUT_SLOT, value, 
	                              NULL, &ps, NULL);
            /* Save the PS feature string and order dependency value. */
            if (value != NULL) {
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_PROCEDURE_BEGIN);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_BEGIN_FEATURE);
                ok &= PPDStringAddString(&buffer, PPD_NTS_INPUT_SLOT);
                ok &= PPDStringAddString(&buffer, " ");
                ok &= PPDStringAddString(&buffer, value);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, ps);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_END_FEATURE);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_PROCEDURE_END);
                ok &= PPDStringAddString(&buffer, "\r\n");

                ok &= PPDGetOrderDependency(ppdManager, PPD_NTS_INPUT_SLOT, &order, 1);
                ok &= PPDSetSorted(&headSort, buffer, order);

                PPDStringDelete(&buffer);
            }

            PPDDeleteString(ppdHandle, &value);
            PPDDeleteString(ppdHandle, &ps);


            /* Get '*PageRegion' default state. */
            ok &= PPDGetDefaultKey(ppdHandle, PPD_NTS_PAGE_REGION, &value);
            ok &= PPDGetKeyWithOption(ppdHandle, PPD_NTS_PAGE_REGION, value, 
	                              NULL, &ps, NULL);
            /* Save the PS feature string and order dependency value. */
            if (value != NULL) {
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_PROCEDURE_BEGIN);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, PPD_NTS_PS_BEGIN_FEATURE);
                ok &= PPDStringAddString(&buffer, PPD_NTS_PAGE_REGION);
                ok &= PPDStringAddString(&buffer, " ");
                ok &= PPDStringAddString(&buffer, value);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, ps);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_END_FEATURE);
                ok &= PPDStringAddString(&buffer, "\r\n");
                ok &= PPDStringAddString(&buffer, NTS_NTS_PS_PROCEDURE_END);
                ok &= PPDStringAddString(&buffer, "\r\n");

                ok &= PPDGetOrderDependency(ppdManager, PPD_NTS_PAGE_REGION, &order, 1);
                ok &= PPDSetSorted(&headSort, buffer, order);

                PPDStringDelete(&buffer);
            }

            PPDDeleteString(ppdHandle, &value);
            PPDDeleteString(ppdHandle, &ps);
	    }
    }


    /* Now that the selectable PS features are sorted we can output them. */
    ok &= PPDGetSorted(headSort, psString);
    ok &= PPDClearSort(&headSort);


    /* Add this string for the user to the memory array. */
    ok &= PPDAddToStringMemoryArray(ppdManager, psString);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDUseCustomPageSize()
 *
 * Public interface method to tell the PPD Manager to use a custom page size
 * based on the information provided.  This will only effect the returning 
 * string in PPDGetPSSetup().  The method will fail if the current PPD file
 * does not support custom page size.
 *
 * Parameters: ppdHandle    - Handle to instance of the PPD Manager
 *             width        - Width of custom page size
 *             height       - Height of custom page size
 *             widthOffset  - Width offset of custom page size
 *             heightOffset - Height offset of custom page size
 *             orientation  - Orientation of custom page size
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDUseCustomPageSize(PPDHandle ppdHandle, double width, double height,
                         double widthOffset, double heightOffset, 
			             int orientation)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    char *value = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Check for support for '*CustomPageSize' statement. */
    ok = PPDGetKeyWithOption(ppdHandle, PPD_NTS_CUSTOM_PAGE_SIZE,
                             PPD_NTS_TRUE, NULL, &value, NULL);
    if ((ok == 1) && (value != NULL)) {
        ppdManager->useCustomPageSize = 1;
        ppdManager->customPageSizeWidth = width;
        ppdManager->customPageSizeHeight = height;
        ppdManager->customPageSizeWidthOffset = widthOffset;
        ppdManager->customPageSizeHeightOffset = heightOffset;
        ppdManager->customPageSizeOrientation = orientation;
    } else {
        ok = 0;
    }

    PPDDeleteString(ppdHandle, &value);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDUseStandardPageSize()
 *
 * Public interface method to tell the PPD Manager to use a standard page 
 * size, not the custom page size.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDUseStandardPageSize(PPDHandle ppdHandle)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }
    
    ppdManager->useCustomPageSize = 0;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDDoesKeyExist()
 *
 * Public interface method that will determine whether or not the specified
 * main key exists in the PPD file.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             mainKey         - Main key to look for
 *
 *     Return: 0 on failure, 1 if the key was found
 */
int PPDDoesKeyExist(PPDHandle ppdHandle, const char *mainKey)
{
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;

    /* We need at least 2 characters to have a valid main key */
    if ((mainKey != NULL) && (strlen(mainKey) < 2)) {
        return (0);
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Go through all keys and find the matching entry. */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, mainKey,
                               PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);

    return (found != NULL);
}

/* ---------------------------------------------------------------------------
 * PPDGetKey()
 *
 * Public interface method that will return strings containing the value and
 * translated value of the specified main key.  If the main key can have 
 * multiple values because it is defined with a option key, then the strings 
 * will be NULL.  A returning string will be NULL if no match was found, or no 
 * content is available.  If a pointer to a returning string is NULL, this 
 * means the user does want the content for that string.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             mainKey         - Main key to look for
 *             value           - Pointer to returning string
 *             valueTranslated - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetKey(PPDHandle ppdHandle, const char *mainKey, char **value,
              char **valueTranslated)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;


    /* We will return NULL pointer if no string is found. */
    if (value != NULL) {
        *value = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (valueTranslated != NULL) {
        *valueTranslated = NULL;
    }

    /* We need at least 2 characters to have a valid main key */
    if ((mainKey != NULL) && (strlen(mainKey) < 2)) {
        return (0);
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Go through all keys and find the matching entry. */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, mainKey,
                               PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);
    if ((found != NULL) && (found->optionKey == NULL)) {
        /* Check for a value and if the user wants it */
        if ((value != NULL) && (found->value != NULL) &&
            (strlen(found->value) > 0)) {

            ok &= PPDCleanString(value, found->value, found->valueType);
            ok &= PPDAddToStringMemoryArray(ppdManager, value);
        }
        /* Check for a translated value and if the user wants it */
        if ((valueTranslated != NULL) && (found->valueTrns != NULL) &&
            (strlen(found->valueTrns) > 0)) {

            ok &= PPDCleanString(valueTranslated, found->valueTrns,
                                 PPD_VALUE_TYPE_TRANSLATION_STRING);
            ok &= PPDAddToStringMemoryArray(ppdManager, valueTranslated);
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetKeyWithOption()
 *
 * Public interface method that will return strings containing the translated 
 * option key, value and translated value of the specified main key and option 
 * key.  A returning string will be NULL if no match was found, or no content
 * is available.  If a pointer to a returning string is NULL, this means the 
 * user does want the content for that string.
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             mainKey             - Main key to look for
 *             optionKey           - Option key to look for
 *             optionKeyTranslated - Pointer to returning string
 *             value               - Pointer to returning string
 *             valueTranslated     - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetKeyWithOption(PPDHandle ppdHandle, const char *mainKey,
                        const char *optionKey, char **optionKeyTranslated,
                        char **value, char **valueTranslated)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;


    /* We will return NULL pointer if no string is found. */
    if (optionKeyTranslated != NULL) {
        *optionKeyTranslated = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (value != NULL) {
        *value = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (valueTranslated != NULL) {
        *valueTranslated = NULL;
    }

    /* We need at least 2 characters to have a valid main key */
    if ((mainKey != NULL) && (strlen(mainKey) < 2)) {
        return (0);
    }

    /* We need at least 1 character to have a valid option key */
    if ((optionKey != NULL) && (strlen(optionKey) < 1)) {
        return (0);
    }

    /* Make sure we have a PPD Manager handle. */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully. */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Go through all keys and find the matching entry. */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, mainKey,
                               PPD_MAIN_KEY_TYPE_ALL, optionKey, NULL, 1);
    if (found != NULL) {
        /* Check for an option key translated and if the user wants it */
        if ((optionKeyTranslated != NULL) && (found->optionKeyTrns != NULL) &&
            (strlen(found->optionKeyTrns) > 0)) {

            ok &= PPDCleanString(optionKeyTranslated, found->optionKeyTrns,
                                 PPD_VALUE_TYPE_TRANSLATION_STRING);
            ok &= PPDAddToStringMemoryArray(ppdManager, optionKeyTranslated);
        }

        /* Check for a value and if the user wants it */
        if ((value != NULL) && (found->value != NULL) &&
            (strlen(found->value) > 0)) {

            ok &= PPDCleanString(value, found->value, found->valueType);
            ok &= PPDAddToStringMemoryArray(ppdManager, value);
        }

        /* Check for a value translated and if the user wants it */
        if ((valueTranslated != NULL) && (found->valueTrns != NULL) &&
            (strlen(found->valueTrns) > 0)) {

            ok &= PPDCleanString(valueTranslated, found->valueTrns,
                                 PPD_VALUE_TYPE_TRANSLATION_STRING);
            ok &= PPDAddToStringMemoryArray(ppdManager, valueTranslated);
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetDefaultKey()
 *
 * Public interface method to get the default value of the specified main key.
 * If the main key could not be found, then defaultValue will be NULL.  A 
 * returning string will be NULL if no match was found, or no  content is 
 * available.  If a pointer to a returning string is NULL, this means the user 
 * does want the content for that string.
 *
 * Parameters: ppdHandle    - Handle to instance of the PPD Manager
 *             mainKey      - Main key to look for
 *             defaultValue - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetDefaultKey(PPDHandle ppdHandle, const char *mainKey,
                     char **defaultValue)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    char *defaultKey = NULL;


    /* We will return NULL pointer if no string is found. */
    if (defaultValue != NULL) {
        *defaultValue = NULL;
    }

    /* We need at least 2 characters to have a valid main key */
    if ((mainKey == NULL) || ((mainKey != NULL) && (strlen(mainKey) < 2))) {
        return (0);
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Build the default key name based on the specified main key */
    ok &= PPDStringAddString(&defaultKey, "*Default");
    ok &= PPDStringAddString(&defaultKey, &mainKey[1]);

    if (ok == 1) {
        /* Go through all keys and find the matching entry. */
        found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, defaultKey,
                                   PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);
        if (found != NULL) {
            /* Check for a default value and if the user wants it */
            if ((defaultValue != NULL) && (found->value != NULL) &&
                (strlen(found->value) > 0)) {

                ok &= PPDStringAddString(defaultValue, found->value);
                ok &= PPDAddToStringMemoryArray(ppdManager, defaultValue);
            }
        }

        PPDStringDelete(&defaultKey);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDSetDefaultKey()
 *
 * Public interface method to set the default value of the specified main key.
 *
 * Parameters: ppdHandle    - Handle to instance of the PPD Manager
 *             mainKey      - Main key to look for
 *             defaultValue - Value to be set
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDSetDefaultKey(PPDHandle ppdHandle, const char *mainKey,
                     const char *defaultValue)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    char *defaultKey = NULL;


    /* We need at least 2 characters to have a valid main key */
    if ((mainKey != NULL) && (strlen(mainKey) < 2)) {
        return (0);
    }

    /* We need at least 1 character to have a valid default value */
    if ((defaultValue != NULL) && (strlen(defaultValue) < 1)) {
        return (0);
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Build the default key name based on the specified main key */
    ok &= PPDStringAddString(&defaultKey, "*Default");
    ok &= PPDStringAddString(&defaultKey, &mainKey[1]);

    if (ok == 1) {
        /* Go through all keys and find the matching entry. */
        found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, defaultKey,
                                   PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);
        if (found != NULL) {
            /* Check for a default value and remove it */
            if (found->value != NULL) {
                PPDStringDelete(&(found->value));
            }

            ok &= PPDStringAddString(&(found->value), defaultValue);
        }

        PPDStringDelete(&defaultKey);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCheckForUIConstraints()
 *
 * Public interface method to be called before changing the default setting of 
 * a UI, to check for possible conflicts. The returned string pointers contain 
 * the conflicting main key and value.  If there is no conflicts, the 
 * returning strings will be NULL.  You can still change the default setting 
 * of a UI even if there is a conflict.  This method is only for information 
 * purposes.
 *
 * Parameters: ppdHandle          - Handle to instance of the PPD Manager
 *             mainKey            - Main key to test
 *             value              - Value to test
 *             conflictingMainKey - Pointer to returning string
 *             conflictingValue   - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDCheckForUIConstraints(PPDHandle ppdHandle, const char *mainKey,
                             const char *value, char **conflictingMainKey,
                             char **conflictingValue)
{
    int ok = 1;
    int found = 0;
    PPDManager *ppdManager = NULL;
    PPDUIConstraints *constraints = NULL;
    char *defaultValue = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have valid pointers */
    if ((conflictingMainKey == NULL) || (conflictingValue == NULL) ||
        (mainKey == NULL) || (value == NULL)) {
        return (0);
    }

    *conflictingMainKey = NULL;
    *conflictingValue = NULL;


    /* Look at each entry in the list */
    constraints = ppdManager->headUIConstraints;
    while (constraints != NULL) {

        /* Does the main key match the first main key. */
        if (strcmp(constraints->firstMainKey, mainKey) == 0) {
            found = 0;

            /* A first option key might not be specified. */
            if (constraints->firstOptionKey != NULL) {
                if (strcmp(value, constraints->firstOptionKey) == 0) {
                    found = 1;
                }
            } else {
                if ((strcmp(value, "None") != 0) &&
                    (strcmp(value, "False") != 0)) {
                    found = 1;
                }
            }

            if (found == 1) {

				defaultValue = NULL;

				/* The '*CustomPageSize' key doesn't have an associated
				 * default key.
  			     */
			    if(strcmp(constraints->secondMainKey, 
				          PPD_NTS_CUSTOM_PAGE_SIZE) == 0) {

				    if(ppdManager->useCustomPageSize == 1) {
                        PPDStringAddString(&defaultValue, PPD_NTS_TRUE);
					} else {
					    PPDStringAddString(&defaultValue, PPD_NTS_FALSE);
					}

					PPDAddToStringMemoryArray(ppdManager, &defaultValue);

				} else {
                    /* Get the current default value of the second main key */
                    if (PPDGetDefaultKey(ppdHandle, constraints->secondMainKey,
                                     &defaultValue) == 0) {
                        return (0);
				    }
                }

                /* Make sure we have a value. */
                if (defaultValue != NULL) {
					/* If a second option key is specified, then that value must be 
					 * equal to the current default value to have a conflict.
					 */
					if (constraints->secondOptionKey != NULL) {
						if (strcmp(defaultValue, constraints->secondOptionKey) == 0) {
							ok &= PPDStringAddString(conflictingMainKey,
													 constraints->secondMainKey);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingMainKey);

							ok &= PPDStringAddString(conflictingValue,
													 defaultValue);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingValue);

							PPDDeleteString(ppdHandle, &defaultValue);

							break;
						}
					} else {
						/* If no second option key is specified, then the current 
						 * default value must not be equal to 'None' and 'False' to 
						 * have a conflict.
						 */
						if ((strcmp(defaultValue, "None") != 0) &&
							(strcmp(defaultValue, "False") != 0)) {
							ok &= PPDStringAddString(conflictingMainKey,
													 constraints->secondMainKey);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingMainKey);

							ok &= PPDStringAddString(conflictingValue,
													 defaultValue);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingValue);

							PPDDeleteString(ppdHandle, &defaultValue);

							break;
						}
					}

					PPDDeleteString(ppdHandle, &defaultValue);
				}
            }
        }


        /* Does the main key match the second main key. */
        if (strcmp(constraints->secondMainKey, mainKey) == 0) {
            found = 0;

            /* A second option key might not be specified. */
            if (constraints->secondOptionKey != NULL) {
                if (strcmp(value, constraints->secondOptionKey) == 0) {
                    found = 1;
                }
            } else {
                if ((strcmp(value, "None") != 0) &&
                    (strcmp(value, "False") != 0)) {
                    found = 1;
                }
            }

            if (found == 1) {

				defaultValue = NULL;

				/* The '*CustomPageSize' key doesn't have an associated
				 * default key.
  			     */
			    if(strcmp(constraints->firstMainKey, 
				          PPD_NTS_CUSTOM_PAGE_SIZE) == 0) {

				    if(ppdManager->useCustomPageSize == 1) {
                        PPDStringAddString(&defaultValue, PPD_NTS_TRUE);
					} else {
					    PPDStringAddString(&defaultValue, PPD_NTS_FALSE);
					}

					PPDAddToStringMemoryArray(ppdManager, &defaultValue);

				} else {
                    /* Get the current default value of the first main key */
                    if (PPDGetDefaultKey(ppdHandle, constraints->firstMainKey,
                                     &defaultValue) == 0) {
                        return (0);
				    }
                }

                /* Make sure we have a value. */
                if (defaultValue != NULL) {
					/* If a first option key is specified, then that value must be 
					 * equal to the current default value to have a conflict.
					 */
					if (constraints->firstOptionKey != NULL) {
						if (strcmp(defaultValue, constraints->firstOptionKey) == 0) {
							ok &= PPDStringAddString(conflictingMainKey,
													 constraints->firstMainKey);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingMainKey);

							ok &= PPDStringAddString(conflictingValue,
													 defaultValue);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingValue);

							PPDDeleteString(ppdHandle, &defaultValue);

							break;
						}
					} else {
						/* If no first option key is specified, then the current 
						 * default value must not be equal to 'None' and 'False' to 
						 * have a conflict.
						 */
						if ((strcmp(defaultValue, "None") != 0) &&
							(strcmp(defaultValue, "False") != 0)) {
							ok &= PPDStringAddString(conflictingMainKey,
													 constraints->firstMainKey);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingMainKey);

							ok &= PPDStringAddString(conflictingValue,
													 defaultValue);
							ok &= PPDAddToStringMemoryArray(ppdManager,
													  conflictingValue);

							PPDDeleteString(ppdHandle, &defaultValue);

							break;
						}
					}

					PPDDeleteString(ppdHandle, &defaultValue);
				}
			}
        }

        constraints = constraints->next;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDIterateEnd()
 *
 * Public interface method to end iteration process pointed by 'pos'.
 *
 * Parameters: ppdHandle - Handle to instance of the PPD Manager
 *             pos       - Iteration handle to end
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDIterateEnd(PPDHandle ppdHandle, PPDPosition * pos)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;


    /* Make sure we have valid pointers. */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Call helper method to remove from the memory array and delete it. */
    ok = PPDRemoveFromIterateMemoryArray(ppdManager, (PPDIterate **)pos);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetKeysIterateStart()
 *
 * Public interface method to start the iteration process for a specified main 
 * key.  This method, allong with 'PPDGetKeysIterateNext' allows you to go 
 * through all available keys one at a time.  If no main key is specified, 
 * then the iteration process will go through all keys.  The parameter pos is 
 * an iteration handle and is set to NULL if there is no entries available.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             pos             - Iteration handle
 *             mainKey         - Main key to look for
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetKeysIterateStart(PPDHandle ppdHandle, PPDPosition * pos,
                           const char *mainKey)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* We will return NULL pointer if no entry is found. */
    *pos = NULL;


    /* Find the first matching entry, starting at the root entry and looking 
     * at any sub-levels. 
     */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, mainKey,
                               PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);
    if (found != NULL) {

        /* Allocate memory for one instance of the Iterate Position */
        iterate = (PPDIterate *) malloc(sizeof(PPDIterate));
        if (iterate == NULL) {
            return (0);
        }
    
        /* Set the search value */
        iterate->searchValue = NULL;
        if (mainKey != NULL) {
            ok = PPDStringAddString(&(iterate->searchValue), mainKey);
            if (ok == 0) {
                free(iterate);
                return (0);
            }
        }

        /* Set the position of the first position */
        iterate->currentKey = found;

	    ok = PPDAddToIterateMemoryArray(ppdManager, &iterate);
        *pos = (void *)iterate;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetKeysIterateNext()
 *
 * Public interface method to get the next available entry.  This method, 
 * allong with 'PPDGetKeysIterateStart' allows you to go through all available 
 * keys one at a time.  The parameter pos is an iteration handle and is set to 
 * NULL if there is no more entries available.  A returning string will be 
 * NULL if no content is available.  If a pointer to a returning string is 
 * NULL, this means the user does want the content for that string.
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             pos                 - Iteration handle
 *             mainKey             - Pointer to returning string
 *             optionKey           - Pointer to returning string
 *             optionKeyTranslated - Pointer to returning string
 *             value               - Pointer to returning string
 *             valueTranslated     - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetKeysIterateNext(PPDHandle ppdHandle, PPDPosition * pos, char **mainKey,
                          char **optionKey, char **optionKeyTranslated,
                          char **value, char **valueTranslated)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* We will return NULL pointer if no string is found. */
    if (mainKey != NULL) {
        *mainKey = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (optionKey != NULL) {
        *optionKey = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (optionKeyTranslated != NULL) {
        *optionKeyTranslated = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (value != NULL) {
        *value = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (valueTranslated != NULL) {
        *valueTranslated = NULL;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* There is no more values available */
    if (*pos == NULL) {
        return (ok);
    }
    iterate = (PPDIterate *) (*pos);


    /* Check for an option key translated and if the user wants it */
    if ((mainKey != NULL) && (iterate->currentKey->mainKey != NULL) &&
        (strlen(iterate->currentKey->mainKey) > 0)) {

        ok &= PPDStringAddString(mainKey, iterate->currentKey->mainKey);
        ok &= PPDAddToStringMemoryArray(ppdManager, mainKey);
    }

    /* Check for an option key translated and if the user wants it */
    if ((optionKey != NULL) && (iterate->currentKey->optionKey != NULL) &&
        (strlen(iterate->currentKey->optionKey) > 0)) {

        ok &= PPDStringAddString(optionKey, iterate->currentKey->optionKey);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKey);
    }

    /* Check for an option key translated and if the user wants it */
    if ((optionKeyTranslated != NULL) &&
        (iterate->currentKey->optionKeyTrns != NULL) &&
        (strlen(iterate->currentKey->optionKeyTrns) > 0)) {

        ok &= PPDCleanString(optionKeyTranslated,
                             iterate->currentKey->optionKeyTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKeyTranslated);
    }

    /* Check for a value and if the user wants it */
    if ((value != NULL) && (iterate->currentKey->value != NULL) &&
        (strlen(iterate->currentKey->value) > 0)) {

        ok &= PPDCleanString(value, iterate->currentKey->value,
                             iterate->currentKey->valueType);
        ok &= PPDAddToStringMemoryArray(ppdManager, value);
    }

    /* Check for a value translated and if the user wants it */
    if ((valueTranslated != NULL) &&
        (iterate->currentKey->valueTrns != NULL) &&
        (strlen(iterate->currentKey->valueTrns) > 0)) {

        ok &= PPDCleanString(valueTranslated, iterate->currentKey->valueTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, valueTranslated);
    }

    if (ok == 1) {
        /* Find the next matching entry, starting at the root entry and 
         * looking at any sub-levels. 
         */
        found = PPDGetKeyRecursive(ppdManager->headAllKeys,
                                   &(iterate->currentKey),
                                   iterate->searchValue,
                                   PPD_MAIN_KEY_TYPE_ALL, NULL, NULL, 1);
        if (found != NULL) {
            iterate->currentKey = found;
        } else {
	        ok = PPDRemoveFromIterateMemoryArray(ppdManager, &iterate);
            *pos = NULL;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetGroupsIterateStart()
 *
 * Public interface method to start the iteration process for all groups. This 
 * method, allong with 'PPDGetGroupsIterateStart' allows you to go through all 
 * available groups one at a time.  The parameter pos is an iteration handle 
 * and is set to NULL if there is no entries available.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             pos             - Iteration handle
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetGroupsIterateStart(PPDHandle ppdHandle, PPDPosition * pos)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* We will return NULL pointer if no entry is found. */
    *pos = NULL;


    /* Find the first group entry, starting at the root entry and not looking 
     * at any sub-levels. The key '*OpenGroup' only exists at the root level. 
     */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, NULL,
                               PPD_MAIN_KEY_TYPE_OPEN_GROUP, NULL, NULL, 0);
    if (found != NULL) {

        /* Allocate memory for one instance of the Iterate Position */
        iterate = (PPDIterate *) malloc(sizeof(PPDIterate));
        if (iterate == NULL) {
            return (0);
        }

        /* Set the search value */
        iterate->searchValue = NULL;

        /* Set the position of the first position */
        iterate->currentKey = found;

        ok = PPDAddToIterateMemoryArray(ppdManager, &iterate);
        *pos = (void *)iterate;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetGroupsIterateNext()
 *
 * Public interface method to get the next available entry.  This method, 
 * allong with 'PPDGetGroupsIterateStart' allows you to go through all 
 * available groups one at a time.  The parameter pos is an iteration handle 
 * and is set to NULL if there is no more entries available.  A returning 
 * string will be NULL if no content is available.  If a pointer to a 
 * returning string is NULL, this means the user does want the content for 
 * that string.
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             pos                 - Iteration handle
 *             value               - Pointer to returning string
 *             valueTranslated     - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetGroupsIterateNext(PPDHandle ppdHandle, PPDPosition * pos,
                            char **value, char **valueTranslated)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* We will return NULL pointer if no string is found. */
    if (value != NULL) {
        *value = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (valueTranslated != NULL) {
        *valueTranslated = NULL;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* There is no more values available */
    if (*pos == NULL) {
        return (ok);
    }
    iterate = (PPDIterate *) (*pos);


    /* Check for a value and if the user wants it */
    if ((value != NULL) && (iterate->currentKey->value != NULL) &&
        (strlen(iterate->currentKey->value) > 0)) {

        ok &= PPDCleanString(value, iterate->currentKey->value,
                             iterate->currentKey->valueType);
        ok &= PPDAddToStringMemoryArray(ppdManager, value);
    }

    /* Check for a value translated and if the user wants it */
    if ((valueTranslated != NULL) &&
        (iterate->currentKey->valueTrns != NULL) &&
        (strlen(iterate->currentKey->valueTrns) > 0)) {

        ok &= PPDCleanString(valueTranslated, iterate->currentKey->valueTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, valueTranslated);
    }


    if (ok == 1) {
        /* Find the next group entry, starting at the current entry and not 
         * looking at any sub-levels. The key '*OpenGroup' only exists at 
         * the root level. 
         */
        found = PPDGetKeyRecursive(iterate->currentKey->next, NULL, NULL,
                               PPD_MAIN_KEY_TYPE_OPEN_GROUP, NULL, NULL, 0);
        if (found != NULL) {
            iterate->currentKey = found;
        } else {
            ok = PPDRemoveFromIterateMemoryArray(ppdManager, &iterate);
            *pos = NULL;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetSubGroupsIterateStart()
 *
 * Public interface method to start the iteration process for all the 
 * sub-groups of the specified group. This method, allong with 
 * 'PPDGetSubGroupsIterateNext' allows you to go through all available 
 * sub-groups one at a time.  The parameter pos is an iteration handle 
 * and is set to NULL if there is no entries available.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             pos             - Iteration handle
 *             group           - Group to find
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetSubGroupsIterateStart(PPDHandle ppdHandle, PPDPosition * pos,
                                const char *group)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* We will return NULL pointer if no entry is found. */
    *pos = NULL;

    /* We need a group to be specified.  Subgroups cannot exist alone. */
    if ((group == NULL) || (strlen(group) < 1)) {
        return (0);
    }


    /* Find the specified group entry, starting at the root entry and 
     * looking at any sub-levels.  We must look at Groups and Subgroups.
     */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, NULL,
                               PPD_MAIN_KEY_TYPE_OPEN_GROUP |
                               PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP,
                               NULL, group, 1);
    if (found != NULL) {
        /* Find the first subgroup entry, of the specified entry and not 
         * looking at any sub-levels. 
         */
        found = PPDGetKeyRecursive(found->child, NULL, NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP, NULL,
                                   NULL, 0);
        if (found != NULL) {

            /* Allocate memory for one instance of the Iterate Position */
            iterate = (PPDIterate *) malloc(sizeof(PPDIterate));
            if (iterate == NULL) {
                return (0);
            }

            /* Set the search value */
            iterate->searchValue = NULL;

            /* Set the position of the first position */
            iterate->currentKey = found;

            ok = PPDAddToIterateMemoryArray(ppdManager, &iterate);
            *pos = (void *)iterate;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetSubGroupsIterateNext()
 *
 * Public interface method to get the next available entry.  This method, 
 * allong with 'PPDGetSubGroupsIterateStart' allows you to go through all 
 * available sub-groups one at a time.  The parameter pos is an iteration 
 * handle and is set to NULL if there is no more entries available.  A 
 * returning string will be NULL if no content is available.  If a pointer to 
 * a returning string is NULL, this means the user does want the content for 
 * that string.
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             pos                 - Iteration handle
 *             value               - Pointer to returning string
 *             valueTranslated     - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetSubGroupsIterateNext(PPDHandle ppdHandle, PPDPosition * pos,
                               char **value, char **valueTranslated)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* We will return NULL pointer if no string is found. */
    if (value != NULL) {
        *value = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (valueTranslated != NULL) {
        *valueTranslated = NULL;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* There is no more values available */
    if (*pos == NULL) {
        return (ok);
    }
    iterate = (PPDIterate *) (*pos);


    /* Check for a value and if the user wants it */
    if ((value != NULL) && (iterate->currentKey->value != NULL) &&
        (strlen(iterate->currentKey->value) > 0)) {

        ok &= PPDCleanString(value, iterate->currentKey->value,
                             iterate->currentKey->valueType);
        ok &= PPDAddToStringMemoryArray(ppdManager, value);
    }

    /* Check for a value translated and if the user wants it */
    if ((valueTranslated != NULL) &&
        (iterate->currentKey->valueTrns != NULL) &&
        (strlen(iterate->currentKey->valueTrns) > 0)) {

        ok &= PPDCleanString(valueTranslated, iterate->currentKey->valueTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, valueTranslated);
    }


    if (ok == 1) {
        /* Find the next subgroup entry, starting at the current entry and 
         * not looking at any sub-levels.
         */
        found = PPDGetKeyRecursive(iterate->currentKey->next, NULL, NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP, NULL,
                                   NULL, 0);
        if (found != NULL) {
            iterate->currentKey = found;
        } else {
            ok = PPDRemoveFromIterateMemoryArray(ppdManager, &iterate);
            *pos = NULL;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetUIsIterateStart()
 *
 * Public interface method to start the iteration process for all the UIs of
 * the specified group. This method, allong with 'PPDGetUIsIterateNext' allows 
 * you to go through all available UIs one at a time.  If no group is 
 * specified, then the iteration process will go through all UIs not enclosed
 * in a group. The parameter pos is an iteration handle and is set to NULL if 
 * there is no entries available.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             pos             - Iteration handle
 *             group           - Group to find
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetUIsIterateStart(PPDHandle ppdHandle, PPDPosition * pos,
                          const char *group)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* We will return NULL pointer if no entry is found. */
    *pos = NULL;


    /* If no group is specified, then we start at the root entry */
    if (group == NULL) {
        found = ppdManager->headAllKeys;
    } else {
        /* Find the specified group entry, starting at the root entry and 
         * looking at any sub-levels.  We must look at Groups and Subgroups.
         */
        found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_GROUP |
                                   PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP,
                                   NULL, group, 1);
        if (found != NULL) {
            found = found->child;
        }
    }

    if (found != NULL) {
        /* Find the first UI entry, starting at the current entry and not 
         * looking at any sub-levels. 
         */
        found = PPDGetKeyRecursive(found, NULL, NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_UI |
                                   PPD_MAIN_KEY_TYPE_JCL_OPEN_UI,
                                   NULL, NULL, 0);
        if (found != NULL) {

            /* Allocate memory for one instance of the Iterate Position */
            iterate = (PPDIterate *) malloc(sizeof(PPDIterate));
            if (iterate == NULL) {
                return (0);
            }
            /* Set the search value */
            iterate->searchValue = NULL;

            /* Set the position of the first position */
            iterate->currentKey = found;

            ok = PPDAddToIterateMemoryArray(ppdManager, &iterate);
            *pos = (void *)iterate;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetUIsIterateNext()
 *
 * Public interface method to get the next available entry.  This method, 
 * allong with 'PPDGetUIsIterateStart' allows you to go through all available 
 * UIs one at a time.  The parameter pos is an iteration handle and is set to 
 * NULL if there is no more entries available.  A returning string will be 
 * NULL if no content is available.  If a pointer to a returning string is 
 * NULL, this means the user does want the content for that string.  The 
 * parameter 'type' will describe the style of the UI interface:
 *                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              
 *   PPD_UI_TYPE_UNDEFINED - not defined
 *   PPD_UI_TYPE_BOOLEAN   - only one selection from two choices available
 *   PPD_UI_TYPE_PICK_ONE  - only one selection from a list
 *   PPD_UI_TYPE_PICK_MANY - more than one selection from a list
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             pos                 - Iteration handle
 *             mainKey             - Pointer to returning string
 *             optionKey           - Pointer to returning string
 *             optionKeyTranslated - Pointer to returning string
 *             type                - Type of UI
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetUIsIterateNext(PPDHandle ppdHandle, PPDPosition * pos,
                         char **optionKey, char **optionKeyTranslated,
                         int *type)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* We will return NULL pointer if no string is found. */
    if (optionKey != NULL) {
        *optionKey = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (optionKeyTranslated != NULL) {
        *optionKeyTranslated = NULL;
    }

    /* We will return default if no type is found. */
    if (type != NULL) {
        *type = PPD_UI_TYPE_UNDEFINED;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* There is no more values available */
    if (*pos == NULL) {
        return (ok);
    }
    iterate = (PPDIterate *) (*pos);


    /* Check for an option key and if the user wants it */
    if ((optionKey != NULL) && (iterate->currentKey->optionKey != NULL) &&
        (strlen(iterate->currentKey->optionKey) > 0)) {

        ok &= PPDStringAddString(optionKey, iterate->currentKey->optionKey);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKey);
    }

    /* Check for an option key translated and if the user wants it */
    if ((optionKeyTranslated != NULL) &&
        (iterate->currentKey->optionKeyTrns != NULL) &&
        (strlen(iterate->currentKey->optionKeyTrns) > 0)) {

        ok &= PPDCleanString(optionKeyTranslated,
                             iterate->currentKey->optionKeyTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKeyTranslated);
    }

    /* Check for a type and if the user wants it */
    if (type != NULL) {
        if (iterate->currentKey->value == NULL) {
            *type = PPD_UI_TYPE_UNDEFINED;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_BOOLEAN)
                   == 0) {
            *type = PPD_UI_TYPE_BOOLEAN;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_PICK_ONE)
                   == 0) {
            *type = PPD_UI_TYPE_PICK_ONE;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_PICK_MANY)
                   == 0) {
            *type = PPD_UI_TYPE_PICK_MANY;
        } else {
            *type = PPD_UI_TYPE_UNDEFINED;
        }
    }

    if (ok == 1) {
        /* Find the next UI entry, starting at the current entry and 
         * not looking at any sub-levels.
         */
        found = PPDGetKeyRecursive(iterate->currentKey->next, NULL, NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_UI |
                                   PPD_MAIN_KEY_TYPE_JCL_OPEN_UI,
                                   NULL, NULL, 0);
        if (found != NULL) {
            iterate->currentKey = found;
        } else {
            ok = PPDRemoveFromIterateMemoryArray(ppdManager, &iterate);
            *pos = NULL;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetAllUIsIterateStart()
 *
 * Public interface method to start the iteration process for all the UIs, 
 * regardless of groupings.  This method, allong with 
 * 'PPDGetAllUIsIterateNext' allows you to go through all available UIs one at 
 * a time.  The parameter pos is an iteration handle and is set to NULL if 
 * there is no entries available.
 *
 * Parameters: ppdHandle       - Handle to instance of the PPD Manager
 *             pos             - Iteration handle
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetAllUIsIterateStart(PPDHandle ppdHandle, PPDPosition * pos)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* We will return NULL pointer if no entry is found. */
    *pos = NULL;


    /* Find the first UI entry, starting at the root entry and looking at any 
     * sub-levels. 
     */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, NULL,
                               PPD_MAIN_KEY_TYPE_OPEN_UI |
                               PPD_MAIN_KEY_TYPE_JCL_OPEN_UI,
                               NULL, NULL, 1);
    if (found != NULL) {
        /* Allocate memory for one instance of the Iterate Position */
        iterate = (PPDIterate *) malloc(sizeof(PPDIterate));
        if (iterate == NULL) {
            return (0);
        }
        /* Set the search value */
        iterate->searchValue = NULL;

        /* Set the position of the first position */
        iterate->currentKey = found;

        ok = PPDAddToIterateMemoryArray(ppdManager, &iterate);
        *pos = (void *)iterate;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetAllUIsIterateNext()
 *
 * Public interface method to get the next available entry.  This method, 
 * along with 'PPDGetAllUIsIterateStart' allows you to go through all 
 * available UIs one at a time.  The parameter pos is an iteration handle and 
 * is set to NULL if there is no more entries available.  A returning string 
 * will be NULL if no content is available.  If a pointer to a returning 
 * string is NULL, this means the user does want the content for that string.
 * The parameter 'type' will describe the style of the UI interface:
 *                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              
 *   PPD_UI_TYPE_UNDEFINED - not defined
 *   PPD_UI_TYPE_BOOLEAN   - only one selection from two choices available
 *   PPD_UI_TYPE_PICK_ONE  - only one selection from a list
 *   PPD_UI_TYPE PICK_MANY - more than one selection from a list
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             pos                 - Iteration handle
 *             mainKey             - Pointer to returning string
 *             optionKey           - Pointer to returning string
 *             optionKeyTranslated - Pointer to returning string
 *             type                - Type of UI
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetAllUIsIterateNext(PPDHandle ppdHandle, PPDPosition * pos,
                            char **optionKey, char **optionKeyTranslated,
                            int *type)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;
    PPDIterate *iterate = NULL;


    /* We will return NULL pointer if no string is found. */
    if (optionKey != NULL) {
        *optionKey = NULL;
    }

    /* We will return NULL pointer if no string is found. */
    if (optionKeyTranslated != NULL) {
        *optionKeyTranslated = NULL;
    }

    /* We will return default if no type is found. */
    if (type != NULL) {
        *type = PPD_UI_TYPE_UNDEFINED;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure we have a pointer to a position */
    if (pos == NULL) {
        return (0);
    }

    /* There is no more values available */
    if (*pos == NULL) {
        return (ok);
    }
    iterate = (PPDIterate *) (*pos);


    /* Check for an option key and if the user wants it */
    if ((optionKey != NULL) && (iterate->currentKey->optionKey != NULL) &&
        (strlen(iterate->currentKey->optionKey) > 0)) {

        ok &= PPDStringAddString(optionKey, iterate->currentKey->optionKey);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKey);
    }

    /* Check for an option key translated and if the user wants it */
    if ((optionKeyTranslated != NULL) &&
        (iterate->currentKey->optionKeyTrns != NULL) &&
        (strlen(iterate->currentKey->optionKeyTrns) > 0)) {

        ok &= PPDCleanString(optionKeyTranslated,
                             iterate->currentKey->optionKeyTrns,
                             PPD_VALUE_TYPE_TRANSLATION_STRING);
        ok &= PPDAddToStringMemoryArray(ppdManager, optionKeyTranslated);
    }

    /* Check for a type and if the user wants it */
    if (type != NULL) {
        if (iterate->currentKey->value == NULL) {
            *type = PPD_UI_TYPE_UNDEFINED;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_BOOLEAN)
                   == 0) {
            *type = PPD_UI_TYPE_BOOLEAN;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_PICK_ONE)
                   == 0) {
            *type = PPD_UI_TYPE_PICK_ONE;
        } else if (strcmp(iterate->currentKey->value, PPD_NTS_UI_TYPE_PICK_MANY)
                   == 0) {
            *type = PPD_UI_TYPE_PICK_MANY;
        } else {
            *type = PPD_UI_TYPE_UNDEFINED;
        }
    }

    if (ok == 1) {
        /* Find the next UI entry, starting at the root entry and 
         * looking at any sub-levels.
         */
        found = PPDGetKeyRecursive(ppdManager->headAllKeys,
                                   &(iterate->currentKey), NULL,
                                   PPD_MAIN_KEY_TYPE_OPEN_UI |
                                   PPD_MAIN_KEY_TYPE_JCL_OPEN_UI,
                                   NULL, NULL, 1);
        if (found != NULL) {
            iterate->currentKey = found;
        } else {
            ok = PPDRemoveFromIterateMemoryArray(ppdManager, &iterate);
            *pos = NULL;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetUI()
 *
 * Public interface method to get translated name and type of a specified UI 
 * entry.  A returning string will be NULL if no content is available.  If a 
 * pointer to a returning string is NULL, this means the user does want the 
 * content for that string.  The parameter 'type' will describe the style of 
 * the UI interface:
 *                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              
 *   PPD_UI_TYPE_UNDEFINED - not defined
 *   PPD_UI_TYPE_BOOLEAN   - only one selection from two choices available
 *   PPD_UI_TYPE_PICK_ONE  - only one selection from a list
 *   PPD_UI_TYPE PICK_MANY - more than one selection from a list
 *
 * Parameters: ppdHandle           - Handle to instance of the PPD Manager
 *             optionKey           - UI to look for
 *             optionKeyTranslated - Pointer to returning string
 *             type                - Type of UI
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetUI(PPDHandle ppdHandle, const char *optionKey, 
             char **optionKeyTranslated, int *type)
{
    int ok = 1;
    PPDManager *ppdManager = NULL;
    PPDKey *found = NULL;


    /* We need at least 1 character to have a valid option key */
    if ((optionKey != NULL) && (strlen(optionKey) < 1)) {
        return (0);
    }

    /* We will return NULL pointer if no string is found. */
    if (optionKeyTranslated != NULL) {
        *optionKeyTranslated = NULL;
    }

    /* We will return default if no type is found. */
    if (type != NULL) {
        *type = PPD_UI_TYPE_UNDEFINED;
    }

    /* Make sure we have a PPD Manager handle */
    if (ppdHandle == NULL) {
        return (0);
    }
    ppdManager = (PPDManager *) ppdHandle;

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }


    /* Go through all keys and find the matching entry. */
    found = PPDGetKeyRecursive(ppdManager->headAllKeys, NULL, NULL,
                               PPD_MAIN_KEY_TYPE_OPEN_UI |
                               PPD_MAIN_KEY_TYPE_JCL_OPEN_UI, 
			                   optionKey, NULL, 1);
    if ((found != NULL) && (found->optionKey != NULL)) {
        /* Check for an option key translated and if the user wants it */
        if ((optionKeyTranslated != NULL) && (found->optionKeyTrns != NULL) &&
            (strlen(found->optionKeyTrns) > 0)) {

            ok &= PPDCleanString(optionKeyTranslated, found->optionKeyTrns,
                                 PPD_VALUE_TYPE_TRANSLATION_STRING);
            ok &= PPDAddToStringMemoryArray(ppdManager, optionKeyTranslated);
        }

	    /* Check for a type and if the user wants it */
        if (type != NULL) {
            if (found->value == NULL) {
                *type = PPD_UI_TYPE_UNDEFINED;
            } else if (strcmp(found->value, PPD_NTS_UI_TYPE_BOOLEAN) == 0) {
                *type = PPD_UI_TYPE_BOOLEAN;
            } else if (strcmp(found->value, PPD_NTS_UI_TYPE_PICK_ONE) == 0) {
                *type = PPD_UI_TYPE_PICK_ONE;
            } else if (strcmp(found->value, PPD_NTS_UI_TYPE_PICK_MANY) == 0) {
                *type = PPD_UI_TYPE_PICK_MANY;
            } else {
                *type = PPD_UI_TYPE_UNDEFINED;
            }
        }
    }

    return (ok);
}


/* ---------------------------------------------------------------------------
 * Helper methods
 */

/* ---------------------------------------------------------------------------
 * PPDStringAddChar()
 *
 * String helper method to append a character from 'character' into 
 * 'string'. This method takes care of allocating and re-allocating enough 
 * memory to store all characters.
 *
 * Parameters: string    - Pointer to string to append to
 *             character - Character to append
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDStringAddChar(char **string, char character)
{
    int ok = 1;
    char characters[2];


    /* Call main string helper method to do the work */
    characters[0] = character;
    characters[1] = '\0';

    ok = PPDStringAddStringOfLength(string, characters, 1);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringAddString()
 *
 * String helper method to append all characters from 'characters' into 
 * 'string'. This method takes care of allocating and re-allocating enough 
 * memory to store all characters.
 *
 * Parameters: string     - Pointer to string to append to
 *             characters - Pointer to source string
 *             length     - number of characters to copy
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDStringAddString(char **string, const char *characters)
{
    int ok = 1;


    /* No need to go further if we have no pointer */
    if (characters == NULL) {
        return (ok);
    }

    /* Call main string helper method to do the work */
    ok = PPDStringAddStringOfLength(string, characters, strlen(characters));

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringAddStringOfLength()
 *
 * String helper method to append a 'length' number of characters from 
 * 'characters' into 'string'. This method takes care of allocating and 
 * re-allocating enough memory to store all characters.
 *
 * Parameters: string     - Pointer to string to append to
 *             characters - Pointer to source string
 *             length     - number of characters to copy
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDStringAddStringOfLength(char **string, const char *characters,
                               int length)
{
    int ok = 1;


    /* No need to go further if we have no pointer */
    if ((string == NULL) || (characters == NULL)) {
        return (ok);
    }

    /* If the pointer is NULL, then we need to allocate some memory for a 
     * new string.  Else, we need to allocate more memory.
     */
    if (*string == NULL) {
        *string = (char *)malloc(length + 1);
        if (*string == NULL) {
            /* Failed to allocate memory */
            return (0);
        }
        memset(*string, '\0', length + 1);
    } else {
        *string = (char *)realloc(*string, strlen(*string) + length + 1);
        if (*string == NULL) {
            /* Failed to re-allocate memory */
            return (0);
        }
        memset(&(*string)[strlen(*string) + 1], '\0', length);
    }

    /* Store the characters in the string */
    if (length != 0) {
        strncpy(&(*string)[strlen(*string)], characters, length);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringDelete()
 *
 * String helper method to delete a character string, pointed by 'string'.  
 * This method takes care of checking for NULL pointers.
 *
 * Parameters: string - Pointer to string to be deleted.
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDStringDelete(char **string)
{
    int ok = 1;

    /* Make sure we have a pointer */
    if (*string == NULL) {
        return (0);
    }

    free(*string);
    *string = NULL;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringSkipWhiteSpaces()
 *
 * Helper method to skip all white spaces and new lines until a character is
 * found.
 *
 * Parameters: string  - Pointer into the string to parse
 *
 *     Return: 0 on end of string, 1 on success
 */
int PPDStringSkipWhiteSpaces(char **string)
{
    int ok = 1;
    char *currentPosition = *string;


    /* Look at all characters until we are done */
    while (1) {
        /* Check for end of string */
        if (*currentPosition == '\0') {
            ok = 0;
            break;
        }

        /* If it's any other characters then a white space or tab then we 
         * are done here.
         */
        if ((*currentPosition != ' ') && (*currentPosition != '\t')) {
            break;
        }

        currentPosition++;
    }

    /* Send back our current position */
    *string = currentPosition;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringSkipLine()
 *
 * Helper method to skip all characters of the current line.
 *
 * Parameters: string  - Pointer into the string to parse
 *
 *     Return: 0 on end of string, 1 on success
 */
int PPDStringSkipLine(char **string)
{
    int ok = 1;
    char *currentPosition = *string;


    /* Look at all characters until we are done */
    while (1) {
        /* Check for end of string */
        if (*currentPosition == '\0') {
            ok = 0;
            break;
        }

        /* If it's a carriage return or line feed then we are done here */
        if ((*currentPosition == '\r') || (*currentPosition == '\n')) {
            break;
        }

        currentPosition++;
    }

    /* Look at all characters until we are done */
    while (1) {
        /* Check for end of string */
        if (*currentPosition == '\0') {
            ok = 0;
            break;
        }

        /* If it's any other characters then a carriage return or line feed 
         * then we are done here.
         */
 
        if ((*currentPosition != '\r') && (*currentPosition != '\n')) {
            break;
        }
 
        currentPosition++;
    }

    /* Send back our current position */
    *string = currentPosition;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDStringSkipToNextKey()
 *
 * Helper method to skip all comments, and possible white spaces and new lines
 * until a key is found.
 *
 * Parameters: string  - Pointer into the string to parse
 *
 *     Return: 0 on end of string, 1 on success
 */
int PPDStringSkipToNextKey(char **string)
{
    int ok = 1;
    char *currentPosition = *string;


    /* There could be more then one comment, so we must keep looping */
    while (1) {
        /* Check for the start of a new Main Key */
        if (*currentPosition == '*') {
            /* We must look ahead and check if it's a comment */
            currentPosition++;

            /* Check for end of string */
            if (*currentPosition == '\0') {
                ok = 0;
                break;
            }
            /* If we found a comment, skip that line, and loop again */
            if (*currentPosition == '%') {
                ok = PPDStringSkipLine(&currentPosition);
                if (ok == 0) {
                    /* We reached end of string */
                    break;
                }
            } else if ((*currentPosition == '\r') ||
                       (*currentPosition == '\n') ||
                       (*currentPosition == ' ') ||
                       (*currentPosition == '\t')) {
                /* We should have a valid character for a key, which means
                 * that there is a problem in the PPD file. TBD: This could 
                 * be an error.
                 */
                ok = PPDStringSkipLine(&currentPosition);
                if (ok == 0) {
                    /* We reached end of string */
                    break;
                }
            } else {
                /* We found a valid charcater for a key stop here. */
                currentPosition--;
                break;
            }
        } else {
            /* The first character of this line is not a '*', which means
             * that there is a problem in the PPD file. TBD: This could be 
             * an error.
             */
            ok = PPDStringSkipLine(&currentPosition);
            if (ok == 0) {
                /* We reached end of string */
                break;
            }
        }
    }

    /* Send back our current position */
    *string = currentPosition;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDAddToStringMemoryArray()
 *
 * Helper method to add a string pointer to the Memory Array.  
 *
 * The Memory Array is an array of string pointers, so strings allocated and
 * returned to the caller can be accounted for and prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             string      - String pointer to be added
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDAddToStringMemoryArray(PPDManager * ppdManager, char **string)
{
    int ok = 1;
    int i = 0;


    /* If there is nothing to delete, then we are done. */
    if ((string == NULL) || (*string == NULL)) {
        return (ok);
    }

    /* Make sure we have valid pointer */
    if (ppdManager == NULL) {
        PPDStringDelete(string);
        return (0);
    }

    /* If the pointer is NULL, then we need to allocate some memory for a 
     * new string pointer.  Else, we need to allocate more memory.
     */
    if (ppdManager->stringMemoryArray == NULL) {
        ppdManager->stringMemoryArray = (char **)malloc(sizeof(char *));
        /* If failed to re-allocate memory */
        if (ppdManager->stringMemoryArray == NULL) {
            ppdManager->stringMemoryArraySize = 0;
            PPDStringDelete(string);
            return (0);
        }

        i = 0;
        ppdManager->stringMemoryArraySize = 1;
    } else {
        /* Look in the list for a free spot from a previous delete. */
        for (i = 0; i < ppdManager->stringMemoryArraySize; i++) {
            if (ppdManager->stringMemoryArray[i] == NULL) {
                break;
            }
        }

        /* There was no free entry, so let's increase the array. */
        if (i == ppdManager->stringMemoryArraySize) {

            ppdManager->stringMemoryArray = (char **)realloc(
                ppdManager->stringMemoryArray, 
                sizeof(char *) * (ppdManager->stringMemoryArraySize + 1));

            /* If failed to re-allocate memory */
            if (ppdManager->stringMemoryArray == NULL) {
                ppdManager->stringMemoryArraySize = 0;
                PPDStringDelete(string);
                return (0);
            }

            ppdManager->stringMemoryArraySize += 1;
        }
    }

    /* Store the string pointer */
    ppdManager->stringMemoryArray[i] = *string;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDRemoveFromStringMemoryArray()
 *
 * Helper method to remove and delete a string pointer to the Memory Array.  
 *
 * The Memory Array is an array of string pointers, so strings allocated and
 * returned to the caller can be accounted for and prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             string      - String pointer to be added
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDRemoveFromStringMemoryArray(PPDManager * ppdManager, char **string)
{
    int ok = 1;
    int i = 0;


    /* If there is nothing to delete, then we are done. */
    if ((string == NULL) || (*string == NULL)) {
        return (ok);
    }

    /* Make sure we have valid pointers */
    if ((ppdManager == NULL) || (ppdManager->stringMemoryArray == NULL)) {
        return (0);
    }

    /* We need to go through the list of string for a matching pointer */
    for (i = 0; i < ppdManager->stringMemoryArraySize; i++) {
        if (ppdManager->stringMemoryArray[i] == *string) {
            PPDStringDelete(&(ppdManager->stringMemoryArray[i]));
            *string = NULL;
            ok = 1;
            break;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearStringMemoryArray()
 *
 * Helper method to delete all entries in the list, and the list itself.
 *
 * The Memory Array is an array of string pointers, so strings allocated and
 * returned to the caller can be accounted for and prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearStringMemoryArray(PPDManager * ppdManager)
{
    int ok = 1;
    int i = 0;

    /* Make sure we have valid pointer */
    if (ppdManager->stringMemoryArray == NULL) {
        return (ok);
    }

    /* Go through array of strings for entries that has not been freed. */
    for (i = 0; i < ppdManager->stringMemoryArraySize; i++) {
        if (ppdManager->stringMemoryArray[i] != NULL) {
            PPDStringDelete(&(ppdManager->stringMemoryArray[i]));
        }
    }

    /* Delete the array of strings. */
    free(ppdManager->stringMemoryArray);
    ppdManager->stringMemoryArray = NULL;
    ppdManager->stringMemoryArraySize = 0;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDAddToIterateMemoryArray()
 *
 * Helper method to add a string pointer to the Memory Array.  
 *
 * The Memory Array is an array of Iterate structure pointers, so Iterate 
 * structures allocated and returned to the caller can be accounted for and
 * prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             iterate     - Iterate structure to add
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDAddToIterateMemoryArray(PPDManager * ppdManager, PPDIterate **iterate)
{
    int ok = 1;
    int i = 0;


    /* If there is nothing to add, then we are done. */
    if ((iterate == NULL) || (*iterate == NULL)) {
        return (ok);
    }
    
    /* Make sure we have valid pointer */
    if (ppdManager == NULL) {
        PPDStringDelete(&((*iterate)->searchValue));
        free(*iterate);
        *iterate = NULL;
        return (0);
    }

    /* If the pointer is NULL, then we need to allocate some memory for a 
     * new string pointer.  Else, we need to allocate more memory.
     */
    if (ppdManager->iterateMemoryArray == NULL) {
        ppdManager->iterateMemoryArray = 
           (PPDIterate **)malloc(sizeof(PPDIterate *));

        /* If failed to re-allocate memory */
        if (ppdManager->iterateMemoryArray == NULL) {
            ppdManager->iterateMemoryArraySize = 0;
            PPDStringDelete(&((*iterate)->searchValue));
            free(*iterate);
            *iterate = NULL;
            return (0);
        }

        i = 0;
        ppdManager->iterateMemoryArraySize = 1;
    } else {
        /* Look in the list for a free spot from a previous delete. */
        for (i = 0; i < ppdManager->iterateMemoryArraySize; i++) {
            if (ppdManager->iterateMemoryArray[i] == NULL) {
                break;
            }
        }

        /* There was no free entry, so let's increase the array. */
        if (i == ppdManager->iterateMemoryArraySize) {
            ppdManager->iterateMemoryArray = 
	       (PPDIterate **)realloc(ppdManager->iterateMemoryArray,
               sizeof(PPDIterate *) * (ppdManager->iterateMemoryArraySize + 1));

            /* If failed to re-allocate memory */
            if (ppdManager->iterateMemoryArray == NULL) {
                ppdManager->iterateMemoryArraySize = 0;
                PPDStringDelete(&((*iterate)->searchValue));
                free(*iterate);
                *iterate = NULL;
                return (0);
            }

            ppdManager->iterateMemoryArraySize += 1;
        }
    }

    /* Store the string pointer */
    ppdManager->iterateMemoryArray[i] = *iterate;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDRemoveFromIterateMemoryArray()
 *
 * Helper method to remove and delete a string pointer to the Memory Array.  
 *
 * The Memory Array is an array of Iterate structure pointers, so Iterate 
 * structures allocated and returned to the caller can be accounted for and
 * prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             iterate     - Iterate structure to delete
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDRemoveFromIterateMemoryArray(PPDManager * ppdManager, PPDIterate **iterate)
{
    int ok = 1;
    int i = 0;


    /* If there is nothing to delete, then we are done. */
    if ((iterate == NULL) || (*iterate == NULL)) {
        return (ok);
    }

    /* Make sure we have valid pointers */
    if ((ppdManager == NULL) || (ppdManager->iterateMemoryArray == NULL)) {
        return (0);
    }

    /* We need to go through the list of Iterate structures for a matching 
     * pointer.
     */
    for (i = 0; i < ppdManager->iterateMemoryArraySize; i++) {
        if (ppdManager->iterateMemoryArray[i] == *iterate) {
	    PPDStringDelete(&(ppdManager->iterateMemoryArray[i]->searchValue));
	    free(ppdManager->iterateMemoryArray[i]);
	    ppdManager->iterateMemoryArray[i] = NULL;
            ok = 1;
            break;
        }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearIterateMemoryArray()
 *
 * Helper method to delete all entries in the list, and the list itself.
 *
 * The Memory Array is an array of Iterate structure pointers, so Iterate 
 * structures allocated and returned to the caller can be accounted for and
 * prevent memory leaks.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearIterateMemoryArray(PPDManager * ppdManager)
{
    int ok = 1;
    int i = 0;

    /* Make sure we have valid pointer */
    if (ppdManager->iterateMemoryArray == NULL) {
        return (ok);
    }

    /* Go through array of Iterate structures for entries that has not been 
     * freed. 
     */
    for (i = 0; i < ppdManager->iterateMemoryArraySize; i++) {
        if (ppdManager->iterateMemoryArray[i] != NULL) {
	    PPDStringDelete(&(ppdManager->iterateMemoryArray[i]->searchValue));
	    free(ppdManager->iterateMemoryArray[i]);
	    ppdManager->iterateMemoryArray[i] = NULL;
        }
    }

    /* Delete the array of Iterate structures. */
    free(ppdManager->iterateMemoryArray);
    ppdManager->iterateMemoryArray = NULL;
    ppdManager->iterateMemoryArraySize = 0;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDAddToIncludeMemoryArray()
 *
 * Helper method to add an Include string pointer to the Memory Array.  The
 * method will fail if the string already exist.
 *
 * The Include Array is an array of filenames that has already been included 
 * by PPD files.  This is to prevent a circular inclusion.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             string      - String to be added
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDAddToIncludeMemoryArray(PPDManager * ppdManager, const char *string)
{
    int ok = 1;
    int i = 0;
    char *includeFilename = NULL;


    /* If there is nothing to add, then we are done. */
    if (string == NULL) {
        return (ok);
    }

    /* Make sure we have valid pointer */
    if (ppdManager == NULL) {
        return (0);
    }


    /* Look if the include filename matches the current PPD filename. */
    if (strcmp(string, ppdManager->ppdFilename) == 0) {
        return (0);
    }

    /* Look if the include filename matches any of the other include 
     * filenames.
     */
    if (ppdManager->includeMemoryArray != NULL) {

        /* Go through array of strings. */
        for (i = 0; i < ppdManager->includeMemoryArraySize; i++) {
            if (ppdManager->includeMemoryArray[i] != NULL) {

                /* Look if the include filename matches the current include 
                 * filename. 
                 */
                if (strcmp(string, ppdManager->includeMemoryArray[i]) == 0) {
                    return (0);
                }
            }
        }
    }


    /* We have not found a matching one, so we will save it. Make a copy of 
     * the string we want to add. 
     */
    if (PPDStringAddString(&includeFilename, string) == 0) {
        return (0);
    }

    /* If the pointer is NULL, then we need to allocate some memory for a 
     * new string pointer.  Else, we need to allocate more memory.
     */
    if (ppdManager->includeMemoryArray == NULL) {
        ppdManager->includeMemoryArray = (char **)malloc(sizeof(char *));
        /* If failed to re-allocate memory */
        if (ppdManager->includeMemoryArray == NULL) {
            ppdManager->includeMemoryArraySize = 0;
            PPDStringDelete(&includeFilename);
            return (0);
        }

        i = 0;
        ppdManager->includeMemoryArraySize = 1;
    } else {
        /* Look in the list for a free spot from a previous delete. */
        for (i = 0; i < ppdManager->includeMemoryArraySize; i++) {
            if (ppdManager->includeMemoryArray[i] == NULL) {
                break;
            }
        }

        /* There was no free entry, so let's increase the array. */
        if (i == ppdManager->includeMemoryArraySize) {

            ppdManager->includeMemoryArray = (char **)realloc(
                ppdManager->includeMemoryArray, 
                sizeof(char *) * (ppdManager->includeMemoryArraySize + 1));

            /* If failed to re-allocate memory */
            if (ppdManager->includeMemoryArray == NULL) {
                ppdManager->includeMemoryArraySize = 0;
                PPDStringDelete(&includeFilename);
                return (0);
            }

            ppdManager->includeMemoryArraySize += 1;
        }
    }

    /* Store the string pointer. */
    ppdManager->includeMemoryArray[i] = includeFilename;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearIncludeMemoryArray()
 *
 * Helper method to delete all entries in the list, and the list itself.
 *
 * The Include Array is an array of filenames that has already been included 
 * by PPD files.  This is to prevent a circular inclusion.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearIncludeMemoryArray(PPDManager * ppdManager)
{
    int ok = 1;
    int i = 0;

    /* Make sure we have valid pointer */
    if (ppdManager->includeMemoryArray == NULL) {
        return (ok);
    }

    /* Go through array of strings for entries that has not been freed. */
    for (i = 0; i < ppdManager->includeMemoryArraySize; i++) {
        if (ppdManager->includeMemoryArray[i] != NULL) {
            PPDStringDelete(&(ppdManager->includeMemoryArray[i]));
        }
    }

    /* Delete the array of strings. */
    free(ppdManager->includeMemoryArray);
    ppdManager->includeMemoryArray = NULL;
    ppdManager->includeMemoryArraySize = 0;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearUIConstraints()
 *
 * Helper method to clear a link list of UI Constraints.  The call is 
 * recursive.
 *
 * Parameters: constraints - Pointer to a UI Contraints structure
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearUIConstraints(PPDUIConstraints ** constraints)
{
    int ok = 1;

    if (*constraints != NULL) {

        /* Delete the next entry */
        ok = PPDClearUIConstraints(&(*constraints)->next);

        /* Delete strings owned by this structure. */
        PPDStringDelete(&(*constraints)->firstMainKey);
        PPDStringDelete(&(*constraints)->firstOptionKey);
        PPDStringDelete(&(*constraints)->secondMainKey);
        PPDStringDelete(&(*constraints)->secondOptionKey);

        free(*constraints);
        *constraints = NULL;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCreateCopyOfUIConstraints()
 *
 * Helper method to create a copy of a link list of UI Constraints.  The call 
 * is recursive.
 *
 * Parameters: constraints - Pointer to current UI Contraints structure
 *             constraints - Pointer to new UI Contraints structure
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDCreateCopyOfUIConstraints(PPDUIConstraints ** constraints, 
                             PPDUIConstraints ** newConstraints)
{
    int ok = 1;

    if (*constraints != NULL) {
        /* Create an instance of UI Constraints structure. */
        *newConstraints = (PPDUIConstraints *) malloc(sizeof(PPDUIConstraints));
        if (*newConstraints == NULL) {
            /* Could not allocate memory for our structure, not good. */
            return (0);
        }

        memset(*newConstraints, '\0', sizeof(PPDUIConstraints));

        /* Make a copy of member variables. */
        ok &= PPDStringAddString(&(*newConstraints)->firstMainKey, 
	                             (*constraints)->firstMainKey);
        ok &= PPDStringAddString(&(*newConstraints)->firstOptionKey, 
	                             (*constraints)->firstOptionKey);
        ok &= PPDStringAddString(&(*newConstraints)->secondMainKey, 
	                             (*constraints)->secondMainKey);
        ok &= PPDStringAddString(&(*newConstraints)->secondOptionKey, 
	                             (*constraints)->secondOptionKey);

        /* If there was a problem creating strings, return right away. */
    	if (ok == 0) {
	        return (0);
	    }

        /* Copy the next entry. */
        ok = PPDCreateCopyOfUIConstraints(&(*constraints)->next, 
	                                      &(*newConstraints)->next);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearAllKeys()
 *
 * Helper method to clear a link list of keys.  The call is recursive.
 *
 * Parameters: key - Pointer to a key
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearAllKeys(PPDKey ** key)
{
    int ok = 1;

    if (*key != NULL) {

        /* Delete the next entry */
        ok = PPDClearAllKeys(&(*key)->next);

        /* Delete the child entry */
        ok = PPDClearAllKeys(&(*key)->child);

        /* Delete strings owned by this structure. */
        PPDStringDelete(&(*key)->mainKey);
        PPDStringDelete(&(*key)->optionKey);
        PPDStringDelete(&(*key)->optionKeyTrns);
        PPDStringDelete(&(*key)->value);
        PPDStringDelete(&(*key)->valueTrns);

        free(*key);
        *key = NULL;
    }
    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCreateCopyOfAllKeys()
 *
 * Helper method to create a copy of a link list of keys.  The call is 
 * recursive.
 *
 * Parameters: key    - Pointer to current key
 *             newKey - Pointer to new key
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDCreateCopyOfAllKeys(PPDKey ** key, PPDKey ** newKey)
{
    int ok = 1;

    if (*key != NULL) {
        /* Create an instance of UI Constraints structure. */
        *newKey = (PPDKey *) malloc(sizeof(PPDKey));
        if (*newKey == NULL) {
            /* Could not allocate memory for our structure, not good. */
            return (0);
        }
        memset(*newKey, '\0', sizeof(PPDKey));

        /* Make a copy of member variables. */
	    (*newKey)->mainKeyType = (*key)->mainKeyType;
        ok &= PPDStringAddString(&(*newKey)->mainKey, 
	                         (*key)->mainKey);

        ok &= PPDStringAddString(&(*newKey)->optionKey, 
	                         (*key)->optionKey);
        ok &= PPDStringAddString(&(*newKey)->optionKeyTrns, 
	                         (*key)->optionKeyTrns);

	    (*newKey)->valueType = (*key)->valueType;
        ok &= PPDStringAddString(&(*newKey)->value, 
	                         (*key)->value);
        ok &= PPDStringAddString(&(*newKey)->valueTrns, 
	                         (*key)->valueTrns);

        /* If there was a problem creating strings, return right away. */
	    if (ok == 0) {
	        return (0);
	    }

        /* Copy the next entry. */
	    ok = PPDCreateCopyOfAllKeys(&(*key)->next, &(*newKey)->next);
	    if (ok == 1) {
	        /* Copy the child entry. */
	        ok = PPDCreateCopyOfAllKeys(&(*key)->child, &(*newKey)->child);
	        if ((*newKey)->child != NULL) {
                (*newKey)->child->parent = *newKey;
	        }
	    }
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCleanUp()
 *
 * Helper method to clear a PPD Manager of all current settings.
 *
 * Parameters: ppdManager - Pointer to PPD Manager instance
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDCleanUp(PPDManager * ppdManager)
{
    int ok = 1;

    
    PPDStringDelete(&(ppdManager->ppdFilename));
    PPDStringDelete(&(ppdManager->ppdIncludeFilename));
    ppdManager->isOk = 0;

    PPDClearStringMemoryArray(ppdManager);
    PPDClearIterateMemoryArray(ppdManager);
    PPDClearIncludeMemoryArray(ppdManager);

    PPDClearUIConstraints(&ppdManager->headUIConstraints);
    PPDClearAllKeys(&ppdManager->headAllKeys);

    ppdManager->useCustomPageSize = 0.0;
    ppdManager->customPageSizeWidth = 0.0;
    ppdManager->customPageSizeHeight = 0.0;
    ppdManager->customPageSizeWidthOffset = 0.0;
    ppdManager->customPageSizeHeightOffset = 0.0;
    ppdManager->customPageSizeOrientation = 0;

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDParseFilename()
 *
 * Helper method to parse a PPD file.  You specify the filename because this
 * method will be called again if we find an '*include' key.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             ppdFilename - PPD file to read in and parse.
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDParseFilename(PPDManager * ppdManager, const char *ppdFilename)
{
    int ok = 0;
    FILE *filePointer = NULL;
    long size = 0;
    char *buffer = NULL;


    /* Make sure we have a valid pointers */
    if ((ppdManager == NULL) || (ppdFilename == NULL)) {
        return (0);
    }

    /* Make sure there is a file to open. */
    if (strlen(ppdFilename) == 0) {
	      return (0);
    }


    /* Open the file for read-only */
    filePointer = fopen(ppdFilename, "rb");
    if (filePointer == NULL) {
        return (0);
    }

    /* Get the size of the file */
    fseek(filePointer, 0, SEEK_END);
    size = ftell(filePointer);
    fseek(filePointer, 0, SEEK_SET);
    if (size == 0) {
        fclose(filePointer);
        return (0);
    }

    /* Create a string big enough to hold the whole file */
    if (PPDStringAddStringOfLength(&buffer, "", size) == 0) {
        fclose(filePointer);
        return (0);
    }

    /* Read the whole file */
    if (fread(buffer, sizeof(char), size, filePointer) == 0) {
        PPDStringDelete(&buffer);
        fclose(filePointer);
        return (0);
    }

    /* Close the file and check if an error occured. */
    if (fclose(filePointer)) {
        PPDStringDelete(&buffer);
        return (0);
    }

    /* Get the string parsed */
    ok = PPDParseString(ppdManager, buffer);

    PPDStringDelete(&buffer);

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDParseString()
 *
 * Helper method to parse a string.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             string      - String to be parsed
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDParseString(PPDManager * ppdManager, const char *string)
{
    int ok = 1;
    int quotedString = 0;
    char *currentPosition = (char *)string;
    char *currentMainKey = NULL;
    int currentMainKeyLength = 0;
    char *currentOptionKey = NULL;
    int currentOptionKeyLength = 0;
    char *currentOptionKeyTrns = NULL;
    int currentOptionKeyTrnsLength = 0;
    char *currentValue = NULL;
    int currentValueLength = 0;
    char *currentValueTrns = NULL;
    int currentValueTrnsLength = 0;


    /* There could be more then one key, so we must keep looping. */
    while (ok == 1) {

        /* Initalize all pointers so we do not confuse with a previous key. */
        currentMainKey = NULL;
        currentMainKeyLength = 0;

        currentOptionKey = NULL;
        currentOptionKeyLength = 0;

        currentOptionKeyTrns = NULL;
        currentOptionKeyTrnsLength = 0;

        currentValue = NULL;
        currentValueLength = 0;

        currentValueTrns = NULL;
        currentValueTrnsLength = 0;


        /*
         * Parsing the Main Key
         */

        /* Remove any posible comments and white spaces, and go to the start 
         * of a next available key. 
    	 */
        if (PPDStringSkipToNextKey(&currentPosition) == 0) {
            break;
        }

        /* Found the start of the Main Key, and we know we have at least one 
         * valid character.
         */
        currentMainKey = currentPosition;
        currentMainKeyLength = 1;

        /* Find all other characters of the Main Key, if any. */
        for (;;) {
            currentPosition++;

            /* Check for possible problems, or an end of Main Key 
             * character.
             */
            if ((*currentPosition == '\0') || (*currentPosition == '\r') ||
                (*currentPosition == '\n') || (*currentPosition == ':') ||
                (*currentPosition == ' ') || (*currentPosition == '\t')) {
                break;
            }
        
            /* This current character is part of the name. */
            currentMainKeyLength++;
        }

        /* Check for end of string because we might have left the while loop 
         * because of it.
         */
        if (*currentPosition == '\0') {
            break;
        }

        /* Check for new line characters because we might have left the 
         * while loop because of it.
         */
        if ((*currentPosition == '\r') || (*currentPosition == '\n')) {
            continue;
        }


        /*
         * Parsing the Option Key
         */

        /* If there is an Option Key, go get it.  Like the name says, this 
         * key is optional and may not be present. 
         */
        if ((*currentPosition == ' ') || (*currentPosition == '\t')) {

            /* Skip any extra white spaces that could be present */
            if (PPDStringSkipWhiteSpaces(&currentPosition) == 0) {
                break;
            }
   
            /* Any of these characters means that there is a problem in the 
             * PPD file. TBD: This could be an error.
             */
            if ((*currentPosition == '\r') || (*currentPosition == '\n') ||
                (*currentPosition == '/') || (*currentPosition == ':')) {
                continue;
            }
   
            /* Found the start of the Option Key. */
            currentOptionKey = currentPosition;
            currentOptionKeyLength = 1;

            /* Find all other characters of the Option Key, if any. */
            for (;;) {
                currentPosition++;

                /* Check for possible problems, or an end of Option Key 
                 * character.
                 */
                if ((*currentPosition == '\0') || (*currentPosition == '\r') ||
                    (*currentPosition == '\n') || (*currentPosition == ':') ||
                    (*currentPosition == '/')) {
                    break;
                }
      
                /* This current character is part of the name. */
                currentOptionKeyLength++;
            }

            /* Check for end of string because we might have left the while 
             * loop because of it.
             */
            if (*currentPosition == '\0') {
                break;
            }
      
            /* Check for new line characters because we might have left the 
             * while loop because of it.
             */
            if ((*currentPosition == '\r') || (*currentPosition == '\n')) {
                continue;
            }
      

            /*
             * Parsing the Translated Option Key
             */

            /* If there is an Translated Option Key, go get it.  Again this
             * key is optional and may not be present. */
            if (*currentPosition == '/') {
                currentPosition++;

                /* Check for end of string. */
                if (*currentPosition == '\0') {
                    break;
                }
                /* Check for new line characters. */
                if ((*currentPosition == '\r') || (*currentPosition == '\n')) {
                    continue;
                }
                /* If we found and end of translation string marker, then no 
                 * need to keep going futher.
                 */
                if (*currentPosition != ':') {
                    /* Found the start of the Translated Option Key. */
                    currentOptionKeyTrns = currentPosition;
                    currentOptionKeyTrnsLength = 1;

                    /* Find all other characters of the Translated Option 
                     * Key, if any. 
                     */
                    for (;;) {
                        currentPosition++;

                        /* Check for possible problems, or an end of Option Key 
                         * character.
                         */
                        if ((*currentPosition == '\0') ||
                            (*currentPosition == '\r') ||
                            (*currentPosition == '\n') ||
                            (*currentPosition == ':')) {
                            break;
                        }

                        /* This current character is part of the name. */
                        currentOptionKeyTrnsLength++;
                    }

                    /* Check for end of string because we might have left 
                     * the while loop because of it.
                     */
                    if (*currentPosition == '\0') {
                        break;
                    }

                    /* Check for new line characters because we might have 
                     * left the while loop because of it.
                     */
                    if ((*currentPosition == '\r') ||
                        (*currentPosition == '\n')) {
                        continue;
                    }
                }
            }
        }


        /*
         * Parsing the Value
         */

        /* We must have the marker to the start of the Value. Any other 
         * characters means that there is a problem in the PPD file. 
         * TBD: This could be an error.
         */
        if (*currentPosition != ':') {
            continue;
        }
        currentPosition++;

        /* Skip any extra white spaces that could be present */
        if (PPDStringSkipWhiteSpaces(&currentPosition) == 0) {
            break;
        }

        /* Any of these characters means that there is a no value to this 
         * key 
         */
        if ((*currentPosition != '\r') && (*currentPosition != '\n')) {
            /* Found the start of the Value. */
            currentValue = currentPosition;
            currentValueLength = 1;

            /* If the Value starts with a double quote, then it is possible 
             * to have a multi-line value.
             */
            if (*currentPosition == '"') {
                quotedString = 1;
            } else {
                quotedString = 0;
            }

            /* Find all other characters of the Value, if any. */
            for (;;) {
                currentPosition++;

                /* Check for end of string. */
                if (*currentPosition == '\0') {
                    break;
                }
                if (quotedString == 1) {
                    /* Check for end of the Value. */
                    if (*currentPosition == '"') {
                        currentValueLength++;
                        break;
                    }
                } else {
                    /* Check for end of the Value. */
                    if ((*currentPosition == '\r') ||
                        (*currentPosition == '\n') ||
                        (*currentPosition == '/')) {
                        break;
                    }
                }

                /* This current character is part of the name. */
                currentValueLength++;
            }

            /* Check for end of string because we might have left the while 
             * loop because of it.
             */
            if (*currentPosition == '\0') {
                break;
            }

            /* If we had a quoted value, then let's look ahead for a possible 
             * Translated Value marker.
             */
            if (quotedString == 1) {
                /* Skip any extra white spaces that could be present */
                if (PPDStringSkipWhiteSpaces(&currentPosition) == 0) {
                    break;
                }
            }
        }


        /*
         * Parsing the Translated Value
         */

        /* Check if we have the marker to the start of the Translated Value. */
        if (*currentPosition == '/') {
            currentPosition++;

            /* Check for end of string. */
            if (*currentPosition == '\0') {
                break;
            }

            /* Any of these characters means that there is a no Translated 
             * Value to this key 
             */
            if ((*currentPosition != '\r') && (*currentPosition != '\n')) {

                /* Found the start of the Translated Value. */
                currentValueTrns = currentPosition;
                currentValueTrnsLength = 1;

                /* If the Translated Value starts with a double quote, then 
                 * it is possible to have a multi-line value.
                 */
                if (*currentPosition == '"') {
                    quotedString = 1;
                } else {
                    quotedString = 0;
                }

                /* Find all other characters of the TranslatedValue, if any. */
                for (;;) {
                    currentPosition++;

                    /* Check for end of string. */
                    if (*currentPosition == '\0') {
                        break;
                    }
                    if (quotedString == 1) {
                        /* Check for end of the Value. */
                        if (*currentPosition == '"') {
                            break;
                        }
                    } else {
                        /* Check for end of the Value. */
                        if ((*currentPosition == '\r') ||
                            (*currentPosition == '\n') ||
                            (*currentPosition == '/')) {
                            break;
                        }
                    }

                    /* This current character is part of the name. */
                    currentValueTrnsLength++;
                }

                /* Check for end of string because we might have left the 
                 * while loop because of it.
                 */
                if (*currentPosition == '\0') {
                    break;
                }
            }
        }

        /* We gathered all the information on the current key, now let's add 
         * it to the PPD Manager's list. 
         */
        ok = PPDAddKey(ppdManager, currentMainKey, currentMainKeyLength,
                       currentOptionKey, currentOptionKeyLength,
                       currentOptionKeyTrns, currentOptionKeyTrnsLength,
                       currentValue, currentValueLength,
                       currentValueTrns, currentValueTrnsLength);
    }


    /* We gathered all the information on the current key, now let's add 
     * it to the PPD Manager's list. We need to try to add it here outside 
     * the while loop because we might have exited at various point because
     * we reached the end of the string.  But this doesn't mean that we might
     * not have a valid key to add.
     */
    if (ok == 1) {
        ok = PPDAddKey(ppdManager, currentMainKey, currentMainKeyLength,
                       currentOptionKey, currentOptionKeyLength,
                       currentOptionKeyTrns, currentOptionKeyTrnsLength,
                       currentValue, currentValueLength,
                       currentValueTrns, currentValueTrnsLength);
    }

    /* This return value is not an error.  This is to stop parsing after
     * an *include statement.
     */
    if (ok == 2) {
        ok = 1;
    }

    return (ok);
}


/* ---------------------------------------------------------------------------
 * PPDAddKey()
 *
 * Helper method add the specified key information.
 *
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             string      - String to be parsed
 *
 *     Return: 0 on failure and stop parsing
 *             1 on success and keep parsing
 *             2 on success and stop parsing
 */
int PPDAddKey(PPDManager * ppdManager,
              char *currentMainKey, int currentMainKeyLength,
              char *currentOptionKey, int currentOptionKeyLength,
              char *currentOptionKeyTrns, int currentOptionKeyTrnsLength,
              char *currentValue, int currentValueLength,
              char *currentValueTrns, int currentValueTrnsLength)
{
    int ok = 1;
    PPDKey *key = NULL;
    int i = 0;


    /* No need to go further if we don't have a Main Key. */
    if ((currentMainKey == NULL) || (currentMainKeyLength == 0)) {
        return (ok);
    }


    /* 
     * Check if this is an '*Include" statement.  If it is, we must open the
     * specified filename and parse it.
     */
    if (strncmp(currentMainKey, PPD_NTS_INCLUDE,
                strlen(PPD_NTS_INCLUDE)) == 0) {

        if ((currentValue != NULL) && (currentValueLength > 2)) {

            char *filename = NULL;
            
	        ok = PPDStringAddStringOfLength(&filename, &currentValue[1],
                                            currentValueLength-2);
            if (ok == 1) {

                /* Check if the include filename has been included before, if 
                 * so then we have a circular inclusion.
                 */
                if (PPDAddToIncludeMemoryArray(ppdManager, filename) == 0) {
                    ok = 0;
                }

                /* Save the include filename, but only the first occurence
                 * of it. To be used when creating a Custom PPD file.
                 */
                if ((ok == 1) && (ppdManager->ppdIncludeFilename == NULL)) {
    	            ok = PPDStringAddString(&(ppdManager->ppdIncludeFilename), 
                                            filename);
                }

                /* Call helper method to parse the include filename. */
                if ((ok == 1) && (PPDParseFilename(ppdManager, filename) == 1)) {
                    /* Success, but we do not want to continue 
                     * parsing because all statements after an 
                     * '*include" should be ignored.
                     */
                    ok = 2;
                } else {
                    /* Failed the parsing, not good. */
                    ok = 0;
                }

                PPDStringDelete(&filename);
            }
        }

        /* Let's stop here and not add this key to the standard list. */
        return (ok);
    }


    /* 
     * Check if this is an '*UIContraints' statement.  If it is, we will add 
     * it to the UI Constraints list.
     */
    if ((strncmp(currentMainKey, PPD_NTS_UI_CONSTRAINTS, 
				 strlen(PPD_NTS_UI_CONSTRAINTS)) == 0) ||
		(strncmp(currentMainKey, PPD_NTS_NON_UI_CONSTRAINTS, 
				 strlen(PPD_NTS_NON_UI_CONSTRAINTS)) == 0)) {

        PPDUIConstraints *constraints = NULL;
        char *value = NULL;
        int found = 0;
        char *token = NULL;
        char *first = NULL;
        char *second = NULL;
        char *third = NULL;
        char *fourth = NULL;


        /* There is no reason to go further is we have nothing. */
        if ((currentValue != NULL) && (currentValueLength != 0)) {

            /* Make a copy so we can make changes to it */
            ok = PPDStringAddStringOfLength(&value, currentValue,
                                            currentValueLength);
            if (ok == 0) {
                /* Could not allocate memory for our string, not good. */
                return (ok);
            }

            /* Create an instance of our UI Constraints structure. */
            constraints = (PPDUIConstraints *) malloc(sizeof(PPDUIConstraints));
            if (constraints == NULL) {
                /* Could not allocate memory for our structure, not good. */
                PPDStringDelete(&value);
                return (0);
            }

            memset(constraints, '\0', sizeof(PPDUIConstraints));

            /* Get all four possible keywords the string could 
             * have.
             */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                ok &= PPDStringAddString(&first, token);
            }
            token = strtok(NULL, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                ok &= PPDStringAddString(&second, token);
            }
            token = strtok(NULL, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                ok &= PPDStringAddString(&third, token);
            }
            token = strtok(NULL, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                ok &= PPDStringAddString(&fourth, token);
            }

            /* We must now try to figure out what keywords 
             * combination we have. 
             */
            if ((first != NULL) && (first[0] == '*')) {
                if ((second != NULL) && (second[0] == '*') &&
                    (third != NULL)) {

                    ok &= PPDStringAddString(&constraints->firstMainKey,
                                             first);
                    ok &= PPDStringAddString(&constraints->secondMainKey,
                                             second);
                    ok &= PPDStringAddString(&constraints->secondOptionKey,
                                             third);
                    found = 1;
                } else if ((second != NULL) && (third != NULL) &&
                           (third[0] == '*') && (fourth != NULL)) {
                    ok &= PPDStringAddString(&constraints->firstMainKey,
                                             first);
                    ok &= PPDStringAddString(&constraints->firstOptionKey,
                                             second);
                    ok &= PPDStringAddString(&constraints->secondMainKey,
                                             third);
                    ok &= PPDStringAddString(&constraints->secondOptionKey,
                                             fourth);
                    found = 1;
                } else if ((second != NULL) && (third != NULL) &&
                           (third[0] == '*')) {
                    ok &= PPDStringAddString(&constraints->firstMainKey,
                                             first);
                    ok &= PPDStringAddString(&constraints->firstOptionKey,
                                             second);
                    ok &= PPDStringAddString(&constraints->secondMainKey,
                                             third);
                    found = 1;
                }
            }

            /* If everything went well, we can then add the entry to 
             * the list.
             */
            if ((ok == 1) && (found == 1)) {
                /* If our list is empty, put the new entry at the 
                 * begining.  Else, add it the the end.
                 */
                if (ppdManager->headUIConstraints == NULL) {
                    ppdManager->headUIConstraints = constraints;
                } else {
                    ppdManager->lastUIConstraints->next = constraints;
                }

                ppdManager->lastUIConstraints = constraints;

            } else {
                /* We failed, clean up. */
                PPDStringDelete(&constraints->firstMainKey);
                PPDStringDelete(&constraints->firstOptionKey);
                PPDStringDelete(&constraints->secondMainKey);
                PPDStringDelete(&constraints->secondOptionKey);
                free(constraints);
            }

            /* Clean up. */
            PPDStringDelete(&value);
            PPDStringDelete(&first);
            PPDStringDelete(&second);
            PPDStringDelete(&third);
            PPDStringDelete(&fourth);

            /* We had a problem, so we can't go futher and add this UI 
             * Contraints key to the standard list.
             */
            if (ok == 0) {
                return (0);
            }
        }
    }


    /* 
     * All keys will be put in the standard list.
     */
    key = (PPDKey *) malloc(sizeof(PPDKey));
    if (key == NULL) {
        /* Could not allocate memory for our structure, not good. */
        return (0);
    }

    memset(key, '\0', sizeof(PPDKey));

    /* Fill up with all the information we have. */
    ok &= PPDStringAddStringOfLength(&key->mainKey, currentMainKey,
                                     currentMainKeyLength);
    ok &= PPDStringAddStringOfLength(&key->optionKey, currentOptionKey,
                                     currentOptionKeyLength);
    ok &= PPDStringAddStringOfLength(&key->optionKeyTrns, currentOptionKeyTrns,
                                     currentOptionKeyTrnsLength);
    ok &= PPDStringAddStringOfLength(&key->value, currentValue,
                                     currentValueLength);
    ok &= PPDStringAddStringOfLength(&key->valueTrns, currentValueTrns,
                                     currentValueTrnsLength);

    /* If there was any problems creating the strings, we cannot go any 
     * further.
     */
    if (ok == 0) {
        PPDStringDelete(&key->mainKey);
        PPDStringDelete(&key->optionKey);
        PPDStringDelete(&key->optionKeyTrns);
        PPDStringDelete(&key->value);
        PPDStringDelete(&key->valueTrns);
        free(key);

        return (0);
    }

    /* Find the type of Main key that we have found. */
    if (strcmp(key->mainKey, PPD_NTS_OPEN_UI) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_OPEN_UI;

    } else if (strcmp(key->mainKey, PPD_NTS_OPEN_GROUP) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_OPEN_GROUP;

    } else if (strcmp(key->mainKey, PPD_NTS_OPEN_SUBGROUP) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP;

    } else if (strcmp(key->mainKey, PPD_NTS_JCL_OPEN_UI) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_JCL_OPEN_UI;

    } else if (strcmp(key->mainKey, PPD_NTS_CLOSE_UI) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_CLOSE_UI;

    } else if (strcmp(key->mainKey, PPD_NTS_CLOSE_GROUP) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_CLOSE_GROUP;

    } else if (strcmp(key->mainKey, PPD_NTS_CLOSE_SUBGROUP) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_CLOSE_SUBGROUP;

    } else if (strcmp(key->mainKey, PPD_NTS_JCL_CLOSE_UI) == 0) {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_JCL_CLOSE_UI;

    } else {
        key->mainKeyType = PPD_MAIN_KEY_TYPE_STANDARD;
    }

    /* Find the type of Value that we have found. */
    if (key->value == NULL) {
        key->valueType = PPD_VALUE_TYPE_NONE;

    } else if (key->value[0] == '^') {
        key->valueType = PPD_VALUE_TYPE_SYMBOL;

    } else if (key->value[0] == '"') {
        if (strncmp(key->mainKey, "*JCL", 4) == 0) {
            key->valueType = PPD_VALUE_TYPE_QUOTED;

        } else {
            if (key->optionKey == NULL) {
                key->valueType = PPD_VALUE_TYPE_QUOTED;

            } else {
                key->valueType = PPD_VALUE_TYPE_INVOCATION;
            }
        }
    } else {
        key->valueType = PPD_VALUE_TYPE_STRING;

	    /* A String value could have white spaces at the end, remove them 
	    * until we reach any other character.
	    */
        for (i = strlen(key->value) - 1; i >= 0; i--) {
            if ((key->value[i] != '\t') && (key->value[i] != ' ')) {
                break;
	        }

            key->value[i] = '\0';
        }

	    if(strlen(key->value) == 0) {
            PPDStringDelete(&key->value);
	    }
    }

    /* If our list is empty, put the new entry at the begining.  Else, we 
     * must be carefull of it's insertion.
     */
    if (ppdManager->headAllKeys == NULL) {
        ppdManager->headAllKeys = key;
        ppdManager->lastAllKeys = key;
    } else {
        if ((ppdManager->lastAllKeys->mainKeyType ==
             PPD_MAIN_KEY_TYPE_OPEN_UI) ||
            (ppdManager->lastAllKeys->mainKeyType ==
             PPD_MAIN_KEY_TYPE_OPEN_GROUP) ||
            (ppdManager->lastAllKeys->mainKeyType ==
             PPD_MAIN_KEY_TYPE_OPEN_SUBGROUP) ||
            (ppdManager->lastAllKeys->mainKeyType ==
             PPD_MAIN_KEY_TYPE_JCL_OPEN_UI)) {

            /* Add it as a child entry to the last entry. */
            key->parent = ppdManager->lastAllKeys;
            ppdManager->lastAllKeys->child = key;

        } else if ((ppdManager->lastAllKeys->mainKeyType ==
                    PPD_MAIN_KEY_TYPE_CLOSE_UI) ||
                   (ppdManager->lastAllKeys->mainKeyType ==
                    PPD_MAIN_KEY_TYPE_CLOSE_GROUP) ||
                   (ppdManager->lastAllKeys->mainKeyType ==
                    PPD_MAIN_KEY_TYPE_CLOSE_SUBGROUP) ||
                   (ppdManager->lastAllKeys->mainKeyType ==
                    PPD_MAIN_KEY_TYPE_JCL_CLOSE_UI)) {

            /* Make sure we have a pointer. */
            if (ppdManager->lastAllKeys->parent != NULL) {
                /* Add it as a next entry to the parent of the last entry. */
                key->parent = ppdManager->lastAllKeys->parent->parent;
                ppdManager->lastAllKeys->parent->next = key;
            } else {
                /* Error in the structure of the PPD file.  Add it as a 
                 * next entry to the last entry. 
                 */
                key->parent = ppdManager->lastAllKeys->parent;
                ppdManager->lastAllKeys->next = key;
            }
        } else {
            /* Add it as a next entry to the last entry. */
            key->parent = ppdManager->lastAllKeys->parent;
            ppdManager->lastAllKeys->next = key;
        }

        ppdManager->lastAllKeys = key;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetKeyRecursive()
 *
 * Helper method to recursively find a key.  The parameter 'current' represent
 * the current location in the tree.  The 'searchKey' is and optional position 
 * that we which to go to, before starting to compare values in keys.  This 
 * allows recusive calls to be stacked and usefull when wanting to look at 
 * keys stored lower in the tree (closer to the root).  If a main key is not
 * specified, then the main key type will be used.  The type allows searching
 * for more then one type of key at the same time.  Is an option key and value
 * are specified, then it will be used for the find.  The parameter 
 * 'checkAllLevels' indicates if the search should only take places at the 
 * current level in the tree.  That is if the child and parent of the keys 
 * will be looked at.
 *
 * Parameters: ppdManager     - Pointer to PPD Manager instance
 *             mainKey        - Main key to look for
 *             mainKeyType    - Type of main key to look for
 *             optionKey      - Option key to look for
 *             value          - Value to look for
 *             checkAllLevels - Should child and parent keys be looked at
 *
 *     Return: 0 on failure and stop parsing
 *             1 on success and keep parsing
 */
PPDKey *PPDGetKeyRecursive(PPDKey * current, PPDKey ** searchKey,
                           const char *mainKey, int mainKeyType,
                           const char *optionKey, const char *value,
                           int checkAllLevels)
{
    PPDKey *found = NULL;


    /* To confirm we have something to look at. */
    if (current != NULL) {
        if ((searchKey != NULL) && (*searchKey != NULL)) {
            /* Look at the key entry if it matches. */
            if (current == *searchKey) {
                *searchKey = NULL;
            }
        } else {
            /* If we have no main key or type to match, then we are done. */
            if ((mainKey == NULL) && (mainKeyType & PPD_MAIN_KEY_TYPE_ALL)) {
                return (current);
            }
            /* Check for a match, using the type if no string is 
             * specified, else using the string. */
            if (((mainKey == NULL) && (current->mainKeyType & mainKeyType)) ||
                ((mainKey != NULL) && (current->mainKey != NULL) &&
                 (strcmp(current->mainKey, mainKey) == 0))) {

                /* If we have no option key to match, then check the value. */
                if (optionKey == NULL) {
                    /* If we have no value to match, then we are done. */
                    if (value == NULL) {
                        return (current);
                    }

                    /* Look at the value if it matches. */
                    if ((current->value != NULL) &&
                        (strcmp(current->value, value)) == 0) {
                        return (current);
                    }
                } else {
                    /* Look at the option key if it matches. */
                    if ((current->optionKey != NULL) &&
                        (strcmp(current->optionKey, optionKey)) == 0) {

                        /* If we have no value to match, then we are done. */
                        if (value == NULL) {
                            return (current);
                        }

                        /* Look at the value if it matches. */
                        if ((current->value != NULL) &&
                            (strcmp(current->value, value)) == 0) {
                            return (current);
                        }
                    }
                }
            }
        }

        if (((searchKey != NULL) && (*searchKey != NULL)) ||
            (checkAllLevels == 1)) {

            /* Look at the child entry and if we found a match. */
            found = PPDGetKeyRecursive(current->child, searchKey, mainKey,
                                       mainKeyType, optionKey, value,
                                       checkAllLevels);
            if ((found != NULL) || (((searchKey == NULL) ||
                (*searchKey == NULL)) && (checkAllLevels == 0))) {

                return (found);
            }
        }

        /* Look at the next entry for a match. */
        found = PPDGetKeyRecursive(current->next, searchKey, mainKey,
                                   mainKeyType, optionKey, value,
                                   checkAllLevels);
    }
    
    return (found);
}

/* ---------------------------------------------------------------------------
 * PPDCleanString()
 *
 * Helper method to remove double quotes and hex substring.
 *
 * Parameters: dest   - Pointer to returning string
 *             source - String to be converted
 *             type   - Type of string
 *
 *     Return: Integer of converted string
 */
int PPDCleanString(char **dest, const char *source, int type)
{
    int ok = 1;
    int length = 0;
    int i = 0;
    int openBracket = 0;
    char firstHexChar = '\0';
    char character = '\0';


    /* Make sure we have valid destination pointer. */
    if (dest == NULL) {
        return (0);
    }
    *dest = NULL;

    /* Check if there is a value to translate. */
    if ((source == NULL) || (type == PPD_VALUE_TYPE_NONE)) {
        return (ok);
    }

    /* Check if there is a value to translate. */
    if (strlen(source) == 0) {
	return (ok);
    }


    /* If it's a Symbol or String value, there is nothing to do. */
    if ((type == PPD_VALUE_TYPE_SYMBOL) || (type == PPD_VALUE_TYPE_STRING)) {
	ok = PPDStringAddString(dest, source);
        return (ok);
    }


    /* If it's an Invocation value, remove the double quotes. */
    if (type == PPD_VALUE_TYPE_INVOCATION) {

        /* We don't want to send back an empty string, so check the length 
	     * after having removed the double quotes. 
	     */
	    if (strlen(source) <= 2) {
	        return (ok);
	    }

        ok = PPDStringAddStringOfLength(dest, &(source[1]), strlen(source)-2);
        return (ok);
    }

    
    /* We have a Quoted Value or a Translation String.  We must remove the 
     * double quotes and the hex substrings. 
     */
    openBracket = 0;
    firstHexChar = '\0';
    character = '\0';
    length = strlen(source);

    for (i = 0; i < length; i++) {
        /* Skip any quotes if a Quoted Value. */
        if ((source[i] == '"') && (type == PPD_VALUE_TYPE_QUOTED)) {
            continue;
        }
     
	    /* Characters after the open bracket must be converted. */
        if (source[i] == '<') {
            openBracket = 1;
            firstHexChar = '\0';
            continue;
        }
        
	    /* Characters after the close bracket need not be converted. */
        if (source[i] == '>') {
            openBracket = 0;
            firstHexChar = '\0';
            continue;
        }
        
	    /* We have encounted an open bracket, which means the next characters
         * are hex values.
         */
        if (openBracket == 1) {
            /* Ignore any spaces and tabs between brackets. */
            if ((source[i] == '\t') || (source[i] == ' ')) {
                continue;
            }
        
    	    /* We need 2 characters to be able to convert. Save this one and
             * go get another. 
             */
            if (firstHexChar == '\0') {
                firstHexChar = source[i];
                continue;
            }
            
	        /* These characters must be converted from their hex format. */
            character = (PPDConvertHexValue(firstHexChar) * 16) +
                PPDConvertHexValue(source[i]);

            if (PPDStringAddChar(dest, character) == 0) {
                return (0);
            }
    
            firstHexChar = '\0';
            continue;
        }

        if (PPDStringAddChar(dest, source[i]) == 0) {
            return (0);
        }
    }

    /* We don't want to send back an empty string. */
    if((*dest != NULL) && (strlen(*dest) == 0)) {
        PPDStringDelete(dest);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDConvertHexValue()
 *
 * Helper method to return the interger value of a character representing a 
 * hex value.
 *
 * Parameters: character - character to convert
 *
 *     Return: Integer of converted string
 */
int PPDConvertHexValue(char character)
{
    switch (character) {
        case '1':
            return (1);
        case '2':
            return (2);
        case '3':
            return (3);
        case '4':
            return (4);
        case '5':
            return (5);
        case '6':
            return (6);
        case '7':
            return (7);
        case '8':
            return (8);
        case '9':
            return (9);
        case 'A':
        case 'a':
            return (10);
        case 'B':
        case 'b':
            return (11);
        case 'C':
        case 'c':
            return (12);
        case 'D':
        case 'd':
            return (13);
        case 'E':
        case 'e':
            return (14);
        case 'F':
        case 'f':
            return (15);
        default:
            return (0);
    }

    return (0);
}

/* ---------------------------------------------------------------------------
 * PPDGetOrderDependency()
 *
 * Helper method to get the order dependency value matching 'mainKey', for
 * features with UIs or Non UIs, depending on the parameter 'isUI'.
 *
 * Parameters: ppdManager - Pointer to PPD Manager instance
 *             mainKey    - Main key to look for
 *             order      - Returning integer with order dependency value
 *             isUIs      - Indicate if feature is a UI or Non UI.
 *
 *     Return: 0 on failure and stop parsing
 *             1 on success and keep parsing
 */
int PPDGetOrderDependency(PPDManager * ppdManager, const char *mainKey,
                          double *order, int isUI)
{
    int ok = 1;
    PPDPosition pos = NULL;
    char *value = NULL;
    char *token = NULL;
    int found = 0;


    /* Make sure we have valid pointers */
    if ((order == NULL) && (ppdManager == NULL)) {
        return (0);
    }
    *order = PPD_ORDER_DEPENDENCY_NOT_DEFINED;

    /* Check if there is a key to look for. */
    if (mainKey == NULL) {
        return (1);
    }

    /* Get all Order Dependency or Non UI Order Dependency keys and look for 
     * a match. 
     */
    if (isUI == 1) {
        ok = PPDGetKeysIterateStart(ppdManager, &pos,
                                    PPD_NTS_ORDER_DEPENDENCY);
    } else {
        ok = PPDGetKeysIterateStart(ppdManager, &pos,
                                    PPD_NTS_NON_UI_ORDER_DEPENDENCY);
    }

    if (ok == 0) {
        return (0);
    }

    while (pos != NULL) {
        ok &= PPDGetKeysIterateNext(ppdManager, &pos, NULL, NULL, NULL,
                                    &value, NULL);
        if (ok == 0) {
            PPDDeleteString(ppdManager, &value);
            break;
        }
        if (value != NULL) {
            /* Get the order dependency value. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                *order = atof(token);
            }
    
            /* Skip sencond token. */
            token = strtok(NULL, "\t ");

            /* Get the token and compare for match. */
            token = strtok(NULL, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                if (strcmp(mainKey, token) == 0) {
                    found = 1;
                    PPDDeleteString(ppdManager, &value);
                    break;
                }
            }

            PPDDeleteString(ppdManager, &value);
        }
    }

    PPDIterateEnd(ppdManager, &pos);

    /* If we did not find a matching order dependency, then set the order
     * to not defined.
     */
    if (found == 0) {
        *order = PPD_ORDER_DEPENDENCY_NOT_DEFINED;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetSorted()
 *
 * Helper method returning a string containing all the stringsselectable features 
 * sorted by order dependency, from the specified list of keys.
 *
 * Parameters: headKey - Head of list of keys
 *             string  - Pointer to returning string
 *
 *     Return: 0 on failure and stop parsing
 *             1 on success and keep parsing
 */
int PPDGetSorted(PPDSort * headSort, char **string)
{
    int ok = 1;
    PPDSort *currentSort = NULL;


    /* Append each value. */
    currentSort = headSort;
    while (currentSort != NULL) {
        ok &= PPDStringAddString(string, currentSort->string);
        currentSort = currentSort->next;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDSetSorted()
 *
 * Helper method to save 'string' in a list of sort entries, sorted by 
 * 'order'.
 *
 * Parameters: headSort - Head of list of sort entries
 *             string   - String to save
 *             order    - Value to be used for sorting
 *
 *     Return: 0 on failure and stop parsing
 *             1 on success and keep parsing
 */
int PPDSetSorted(PPDSort ** headSort, const char *string, double order)
{
    int ok = 1;
    PPDSort *sort = NULL;
    PPDSort *currentSort = NULL;
    PPDSort *previousSort = NULL;


    /* Make sure we have valid pointer. */
    if (headSort == NULL) {
        return (0);
    }

    /* Check if there is a string to add. */
    if (string == NULL) {
        return (ok);
    }

    /* Create a sort to store the value and order dependency. */
    sort = (PPDSort *) malloc(sizeof(PPDSort));
    if (sort == NULL) {
        return (0);
    }

    memset(sort, '\0', sizeof(PPDSort));

    ok &= PPDStringAddString(&sort->string, string);
    sort->order = order;
    if (ok == 0) {
        PPDStringDelete(&sort->string);
        free(sort);
        return (0);
    }

    /* Insert by order dependency. */
    currentSort = *headSort;
    previousSort = NULL;
    while (1) {
        if (currentSort == NULL) {
            if (previousSort == NULL) {
                *headSort = sort;
                break;
            } else {
                previousSort->next = sort;
                break;
            }
        } else {
            if (sort->order < currentSort->order) {
                if (previousSort == NULL) {
                    sort->next = currentSort;
                    *headSort = sort;
                    break;
                } else {
                    sort->next = currentSort;
                    previousSort->next = sort;
                    break;
                }
            }
        }

        previousSort = currentSort;
        currentSort = currentSort->next;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDClearSort()
 *
 * Helper method to clear a link list of sort entries.  The call is recursive.
 *
 * Parameters: sort - Pointer to a sort entry.
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDClearSort(PPDSort ** sort)
{
    int ok = 1;

    if (*sort != NULL) {

        /* Delete the next entry */
        ok = PPDClearSort(&(*sort)->next);

        /* Delete strings owned by this structure. */
        PPDStringDelete(&(*sort)->string);

        free(*sort);
        *sort = NULL;
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDGetCustomPagePSSetup()
 *
 * Helper method build the PS code to set a custom page. 
 * 
 * Parameters: ppdManager - Pointer to PPD Manager instance
 *             string     - Pointer to returning string
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDGetCustomPagePSSetup(PPDManager *ppdManager, char **psString)
{
    int      ok          = 1;
    double   order       = 1;
    char    *ps          = NULL;
    char    *buffer      = NULL;
    char    *value       = NULL;
    char    *token       = NULL;
    char     format[256] = "";
    PPDSort *headSort    = NULL;


    /* Make sure we have a valid pointer. */
    if (psString == NULL) {
        return (0);
    }
    *psString = NULL;

    /* Make sure we have a PPD Manager handle */
    if (ppdManager == NULL) {
        return (0);
    }

 
    /* Get the Custom Page key and build the command, if available. */
    ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_CUSTOM_PAGE_SIZE,
                              PPD_NTS_TRUE, NULL, &ps, NULL);
    if (ps != NULL) {
        /* Start feature. */
        ok &= PPDStringAddString(&buffer, PPD_NTS_PS_PROCEDURE_BEGIN);
        ok &= PPDStringAddString(&buffer, "\r\n");
        ok &= PPDStringAddString(&buffer, PPD_NTS_PS_BEGIN_FEATURE);
        ok &= PPDStringAddString(&buffer, PPD_NTS_CUSTOM_PAGE_SIZE);
        ok &= PPDStringAddString(&buffer, "\r\n");

        /* Get the Width key and save it in order. */
        ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_PARAM_CUSTOM_PAGE_SIZE,
                                  PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_WIDTH, 
				                  NULL, &value, NULL);
        if (value != NULL) {
            /* Get the order. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                order = atof(token);
                sprintf(format, "%f ", ppdManager->customPageSizeWidth);
                ok &= PPDSetSorted(&headSort, format, order);
            }

            PPDDeleteString(ppdManager, &value);
        }

        /* Get the Height key and save it in order. */
        ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_PARAM_CUSTOM_PAGE_SIZE,
                                  PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_HEIGHT, 
				                  NULL, &value, NULL);
        if (value != NULL) {
            /* Get the order. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                order = atof(token);
                sprintf(format, "%f ", ppdManager->customPageSizeHeight);
                ok &= PPDSetSorted(&headSort, format, order);
            }

            PPDDeleteString(ppdManager, &value);
        }

        /* Get the Width Offset key and save it in order. */
        ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_PARAM_CUSTOM_PAGE_SIZE,
                                  PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_WIDTH_OFFSET, 
				                  NULL, &value, NULL);
        if (value != NULL) {
            /* Get the order. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                order = atof(token);
                sprintf(format, "%f ", ppdManager->customPageSizeWidthOffset);
                ok &= PPDSetSorted(&headSort, format, order);
            }

            PPDDeleteString(ppdManager, &value);
        }

        /* Get the Height Offset key and save it in order. */
        ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_PARAM_CUSTOM_PAGE_SIZE,
                                  PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_HEIGHT_OFFSET, 
				                  NULL, &value, NULL);
        if (value != NULL) {
            /* Get the order. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                order = atof(token);
                sprintf(format, "%f ", ppdManager->customPageSizeHeightOffset);
                ok &= PPDSetSorted(&headSort, format, order);
            }

            PPDDeleteString(ppdManager, &value);
        }

        /* Get the Orientation key and save it in order. */
        ok &= PPDGetKeyWithOption(ppdManager, PPD_NTS_PARAM_CUSTOM_PAGE_SIZE,
                                  PPD_NTS_PARAM_CUSTOM_PAGE_SIZE_ORIENTATION, 
				                  NULL, &value, NULL);
        if (value != NULL) {
            /* Get the order. */
            token = strtok(value, "\t ");
            if ((token != NULL) && (strlen(token) > 0)) {
                order = atof(token);
                sprintf(format, "%d ", ppdManager->customPageSizeOrientation);
                ok &= PPDSetSorted(&headSort, format, order);
            }

            PPDDeleteString(ppdManager, &value);
        }

        /* Output the sorted custom page parameters. */
        ok &= PPDGetSorted(headSort, &buffer);

        /* Output the custom page PS code. */
	    ok &= PPDStringAddString(&buffer, ps);
        ok &= PPDStringAddString(&buffer, "\r\n");

        /* End feature. */
        ok &= PPDStringAddString(&buffer, NTS_NTS_PS_END_FEATURE);
        ok &= PPDStringAddString(&buffer, "\r\n");
        ok &= PPDStringAddString(&buffer, NTS_NTS_PS_PROCEDURE_END);
        ok &= PPDStringAddString(&buffer, "\r\n");

        /* If there was no errors, save into the return string. */
    	if (ok == 1 ) {
	        ok &= PPDStringAddString(psString, buffer);
	    }

        PPDStringDelete(&buffer);
        PPDDeleteString(ppdManager, &ps);
    }

    return (ok);
}

/* ---------------------------------------------------------------------------
 * PPDCreateCustomPPD()
 *
 * Helper method to create a Corel Custom PPD file, using the '*include' 
 * statement or embedding the original PPD file.
 * 
 * Parameters: ppdManager  - Pointer to PPD Manager instance
 *             ppdFilename - Custom PPD file to write to
 *             embedded    - Should we embbed
 *
 *     Return: 0 on failure, 1 on success
 */
int PPDCreateCustomPPD(PPDManager *ppdManager, const char *ppdFilename, 
                       int embedded)
{
    int ok = 1;
    FILE *filePointer = NULL;
    FILE *originalFilePointer = NULL;
    char *optionKey = NULL;
    char *value = NULL;
    PPDPosition pos = NULL;
    long size = 0;
    char *buffer = NULL;
    char *ppdIncludeFilename = NULL;


    /* Make sure we have valid pointers. */
    if ((ppdManager == NULL) || (ppdFilename == NULL)) {
        return (0);
    }

    /* Make sure we have parsed a PPD file successfully */
    if (ppdManager->isOk == 0) {
        return (0);
    }

    /* Make sure there is a file to open. */
    if (strlen(ppdFilename) == 0) {
        return (0);
    }


    /* Pick the include filename, depending if the current PPD file is  
     * a Custom PPD file or not.
     */
    if(PPDIsCustomPPD(ppdManager) == 1) {
        ppdIncludeFilename = ppdManager->ppdIncludeFilename;
    } 
    
    /* If no include is selected at this point, just the the current 
     * PPD file.
     */
    if (ppdIncludeFilename == NULL){
        ppdIncludeFilename = ppdManager->ppdFilename;
    }

    /* If there is no file to include, we have a problem. */
    if(ppdIncludeFilename == NULL) {
        return (0);
    }

    /* If the file to include is the same as the Custom PPD file to create, we
     * have a problem.
     */
    if(strcmp(ppdIncludeFilename, ppdFilename) == 0) {
        return (0);
    }


    /* Open the Custom PPD file for writting and discarding it's content. */
    filePointer = fopen(ppdFilename, "wb");
    if (filePointer == NULL) {
        return (0);
    }

    /* Check what type of Custom PPD we want, and output proper header. */
    if (embedded == 0) {
      fputs(PPD_NTS_CUSTOMIZED_PPD, filePointer);
      fputs("\r\n\r\n", filePointer);
    } else {
      fputs(PPD_NTS_EMBEDDED_CUSTOMIZED_PPD, filePointer);
      fputs("\r\n\r\n", filePointer);
    }


    /* Get all selectable PS and JCL features. */
    ok = PPDGetAllUIsIterateStart(ppdManager, &pos);
    if (ok == 0) {
        fclose(filePointer);
        return (0);
    }
    
    while (pos != NULL) {
        ok &= PPDGetAllUIsIterateNext(ppdManager, &pos, &optionKey, NULL, NULL);
        ok &= PPDGetDefaultKey(ppdManager, optionKey, &value);

        /* If we have strings, then output a '*Default' statement. */
        if ((ok == 1) && (optionKey != NULL) && (strlen(optionKey) > 1) && 
	    (value != NULL)) {

            fputs(PPD_NTS_DEFAULT, filePointer);
            fputs(&optionKey[1], filePointer);
            fputs(": ", filePointer);
            fputs(value, filePointer);
            fputs("\r\n", filePointer);
        }

        PPDDeleteString(ppdManager, &optionKey);
        PPDDeleteString(ppdManager, &value);

        if (ok == 0) {
            break;
        }
    }

    PPDIterateEnd(ppdManager, &pos);
	fputs("\r\n", filePointer);


    /* Check for possible errors. */
    if (ok == 0) {
        fclose(filePointer);
        return (0);
    }


    /* Check what type of Custom PPD we want. */
    if (embedded == 0) {
        /* Include statement to original file. */
        fputs(PPD_NTS_INCLUDE, filePointer);
        fputs(": \"", filePointer);
        fputs(ppdIncludeFilename, filePointer);
        fputs("\"", filePointer);
        fputs("\r\n", filePointer);
    } else {
        /* Append the original file. */

        /* Open the file for read-only */
        originalFilePointer = fopen(ppdIncludeFilename, "rb");
        if (originalFilePointer == NULL) {
            PPDStringDelete(&ppdIncludeFilename);
            fclose(filePointer);
            return (0);
        }
        
	    /* Get the size of the file */
        fseek(originalFilePointer, 0, SEEK_END);
        size = ftell(originalFilePointer);
        fseek(originalFilePointer, 0, SEEK_SET);
        if (size == 0) {
            fclose(originalFilePointer);
	        fclose(filePointer);
            return (0);
        }

        /* Create a string big enough to hold the whole file */
        if (PPDStringAddStringOfLength(&buffer, "", size) == 0) {
            fclose(originalFilePointer);
	        fclose(filePointer);
            return (0);
        }

        /* Read the whole file */
        if (fread(buffer, sizeof(char), size, originalFilePointer) == 0) {
            PPDStringDelete(&buffer);
            fclose(originalFilePointer);
	        fclose(filePointer);
            return (0);
        }

        /* Write the whole file */
        if (fwrite(buffer, sizeof(char), strlen(buffer), 
	        filePointer) == 0) {
            PPDStringDelete(&buffer);
            fclose(originalFilePointer);
	        fclose(filePointer);
            return (0);
        }

        PPDStringDelete(&buffer);

        /* Close the file and check if an error occured. */
        if (fclose(originalFilePointer)) {
	        fclose(filePointer);
            return (0);
        }
    }


    /* Close the file and check if an error occured. */
    if (fclose(filePointer)) {
        return (0);
    }

    return (ok);
}
