/* 
 * APPLICATION PRINT SERVICES LIBRARY
 * (C) Copyright 2000 Corel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *
 *
 *        File: printer.c
 *
 * Description: Implementation of basic functions that manipulate a
 *              generic Printer.
 *
 */

#include <stdlib.h>
#include <string.h>

#include "aps.h"
#include "apsinternal.h"
#include "printer.h"
#include "jobattributes.h"
#include "metaconfig.h"

DEBUG_CHANNEL_DEFAULT(printer)

/* ---------------------------------------------------------------------------
 * PrinterGetPtrFromHandle()
 *
 * Obtains a pointer to a generic Printer instance given an Aps_PrinterHandle.
 *
 * Parameters: handle - The handle received from the application.
 *
 *     Return: A Printer pointer; NULL on failure.
 */
Printer *PrinterGetPtrFromHandle(Aps_PrinterHandle handle)
{
    Printer *printer;

    if (handle == NULL)
        return (NULL);

    printer = (Printer *) handle;

    /* Check to see that this looks like a valid Printer object. */
    if (printer->baseClass.identifier != PRINTER_HANDLE)
        return (NULL);

    return (printer);
}

/* ---------------------------------------------------------------------------
 * PrinterGetHandleFromPtr()
 *
 * Obtains a handle that the client application can use to refer to a Printer
 * instance.
 *
 * Parameters: printer - A pointer to a Printer instance
 *
 *     Return: The resulting Aps_PrinterHandle.
 */
Aps_PrinterHandle PrinterGetHandleFromPtr(Printer * printer)
{
    ASSERT(printer != NULL);

    return ((Aps_PrinterHandle) printer);
}

/* ---------------------------------------------------------------------------
 * PrinterInitialize()
 *
 * Called to initialize a newly allocated Printer instance.
 *
 * Parameters: printer   - A pointer to the newly allocated Printer instance.
 *
 *             name      - The name to be assigned to this printer.
 *
 *             transport - A pointer to the transport to be associated with
 *                         this printer.
 *
 *     Return: APS_SUCCESS if initialization was completed, or a standard
 *             Aps_Result error code on failure.
 */
Aps_Result PrinterInitialize(Printer * printer, const char *name,
                             void *transport)
{
    Aps_Result result;
    char *manufacturerName = NULL;
    char *modelName = NULL;
    BOOL partialSuccess = FALSE;

    ASSERT(printer != NULL);
    ASSERT(name != NULL);

    /* Start with all member variables uninitialized. */
    printer->name = NULL;
    printer->defaultJobAttributes = NULL;
    printer->model = NULL;
                                         
    /* Store a pointer back to the associated transport. */
    printer->transport = transport;
    
    /* Initialize the base class information for this APS object, marking */
    /* it as a printer object.                                            */
    result = ObjectInitialize(&printer->baseClass, PRINTER_HANDLE);
    if (result != APS_SUCCESS)
        goto cleanup;

    /* Store the name of this printer in the Printer object. */
    printer->name = malloc(strlen(name) + 1);
    if (printer->name == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }
    strcpy(printer->name, name);

    /* Attempt to obtain the model object for this printer. */
    result = MetaRead(&manufacturerName, name, "manufacturer");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        goto cleanup;

    result = MetaRead(&modelName, name, "model");
    if (result != APS_SUCCESS && result != APS_NOT_FOUND)
        goto cleanup;

    if (manufacturerName != NULL && modelName != NULL) {
        /* The manufacturer and model names have been configured for this
         * printer, so attempt to obtain that model object from the model
         * database.
         */
        result = Aps_GetModel(manufacturerName, modelName, &printer->model);
        if (result != APS_SUCCESS && result != APS_NOT_FOUND)
            goto cleanup;
    } 
    
    if (printer->model == NULL) {
        /* The manufacturer and model names have not been configured for this
         * printer, so ask the transport to attempt to autodetect what type
         * of printer it is.
         */
        result = printer->transport->vtbl->ProbeModelForPrinter(
            printer->transport, printer, &printer->model);
        if (result != APS_SUCCESS && result != APS_MODEL_UNKNOWN)
            goto cleanup;
    }

    /* Create a default JobAttributes instance for this printer. */
    printer->defaultJobAttributes = JobAttrCreate();
    if (printer->defaultJobAttributes == NULL) {
        result = APS_OUT_OF_MEMORY;
        goto cleanup;
    }

    /* Initialize the job attributes with the defaults for this model. */
    result = JobAttrSetToModelDefaults(printer->defaultJobAttributes,
                                       printer);
    if (result != APS_SUCCESS)
        partialSuccess = TRUE;
    
    /* Override the job attributes with the defaults for this printer. */
    result = JobAttrSetToPrinterDefaults(printer->defaultJobAttributes,
                                         printer);
    if (result != APS_SUCCESS)
        partialSuccess = TRUE;

    /* We've now initialized this printer, so return with either full or
     * partial success.
     */
    result = partialSuccess ?  APS_PARTIAL_SUCCESS : APS_SUCCESS;
        
cleanup:
    /* On failure, cleanup any members of this printer that were created. */
    if (result != APS_SUCCESS && result != APS_PARTIAL_SUCCESS) {
        if (printer->name != NULL)
            free(printer->name);
            
        if (printer->defaultJobAttributes != NULL)
            Aps_ReleaseHandle(JobAttrGetHandleFromPtr(
                printer->defaultJobAttributes));
                
        if (printer->model != NULL)
            Aps_ReleaseHandle(printer->model);
    }

    /* Release any temporary resources allocated by this function. */
    if (manufacturerName != NULL)
        free(manufacturerName);
    if (modelName != NULL)
        free(modelName);

    return result;
}

/* ---------------------------------------------------------------------------
 * PrinterDelete()
 *
 * Destroys a Printer instance.
 *
 * Parameters: printer - A pointer to a Printer instance
 *
 *     Return: None.
 */
void PrinterDelete(Printer * printer)
{
    ASSERT(printer != NULL);

    /* Call subclass destructor first. */
    printer->vtbl->Destructor(printer);

    /* Deallocate printer name buffer. */
    free(printer->name);

    /* Release the associated default job settings instance. */
    Aps_ReleaseHandle(JobAttrGetHandleFromPtr(printer->defaultJobAttributes));

    /* Release the associated model handle, if there is one. */
    if (printer->model != NULL) {
        Aps_ReleaseHandle(printer->model);
    }

    /* Mark this object as invalid, to give ourselves a fighting chance of
     * catching subsequent attempts to access this handle.
     */
    printer->baseClass.identifier = INVALID_HANDLE;

    /* Deallocate memory used for the Printer structure. */
    free(printer);
}
